/*
 * pc2400m_spi.c
 *
 * Copyright (C) 2007 Nokia Corporation
 * Author: Juuso Oikarinen <juuso.oikarinen@nokia.com>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */



#define __KERNEL_SYSCALLS__

#include <linux/version.h>
#ifdef MODULE
#ifdef MODVERSIONS
#include <linux/modversions.h>
#endif
#include <linux/module.h>
#else
#define MOD_INC_USE_COUNT
#define MOD_DEC_USE_COUNT
#endif

#include <linux/init.h>
#include <linux/netdevice.h>
#include <linux/clk.h>
#include <linux/spi/spi.h>
#include <linux/platform_device.h>
#include <linux/string.h>
#include <linux/firmware.h>
#include <asm/bitops.h>

#ifndef CONFIG_NET_PC2400M_SIM

#include <asm/arch/gpio.h>

#if !defined(CONFIG_FW_LOADER) && !defined(CONFIG_FW_LOADER_MODULE)
#error No Firmware Loading configured in the kernel !
#endif

#include <asm/irq.h>
#include <asm/hardware.h>
#include <asm/arch/board.h>

#include "pc2400m_if.h"
#include "pc2400m_spi.h"
#include "pc2400m_com.h"
#include "pc2400m_al.h"

/* device name for identification */
#define DEVICE_NAME "pc2400m"

/* treshold for using PIO instead of DMA */
#define PC2400M_DMA_THRESHOLD  12

/* irq control flags */
#define PC2400M_IRQ_PENDING   0x00000001
#define PC2400M_IRQ_EXECUTING 0x00000002

/* list to maintain all pc2400m devices managed by this driver */
LIST_HEAD(pc2400m_spi_wimax_device);

/* global variable holding the currently executing context -
   this is valid only within the common driver. */
struct net_device *pc2400m_com_running_context;

/*
 * Global lock for the adaptation library. This is global, because the driver
 * lib needs to be concurrency protected from multiple accesses regardless
 * of the number of instances with one single lock. */
struct semaphore pc2400m_com_mutex;

/* function structure for adaptation layers */
struct adaptation_if pc2400m_drv_adapt;

static irqreturn_t pc2400m_spi_irq(int irq, void *config);
static int pc2400m_spi_probe(struct spi_device *spi);
static int pc2400m_spi_remove(struct spi_device *spi);
static int pc2400m_spi_suspend(struct spi_device *spi, pm_message_t mesg);
static int pc2400m_spi_resume(struct spi_device *spi);

static struct spi_driver pc2400m_driver = {
        .driver = {
                .name           = DEVICE_NAME,
                .bus            = &spi_bus_type,
                .owner          = THIS_MODULE,
        },

        .probe          = pc2400m_spi_probe,
        .remove         = __devexit_p(pc2400m_spi_remove),
	.suspend        = pc2400m_spi_suspend,
	.resume         = pc2400m_spi_resume,
};


/**
 * pc2400m_spi_h2d - queue an SPI write
 * @ndev: pointer to network device 
 * @data: a buffer containing the data to write
 * @length: length of the buffer to write (in bytes)
 *
 * Queue an SPI write to be executed upon the next commit
 */
static s32 pc2400m_spi_h2d(struct net_device *ndev, u8* data, s32 length) 
{

	struct net_local *nl = netdev_priv(ndev);
	struct spi_transfer *t;

	if (!nl->spi_msg) {
		nl->spi_msg = kmalloc(sizeof(struct spi_message), GFP_KERNEL);
		if (!nl->spi_msg)
			return -ENOMEM;
		spi_message_init((struct spi_message*)nl->spi_msg);
		((struct spi_message*)nl->spi_msg)->is_dma_mapped = 1;
	}

	t = (struct spi_transfer*)
		kmalloc(sizeof(struct spi_transfer), GFP_KERNEL);
	if (!t) return -ENOMEM;
	t->cs_change = 0;
	t->bits_per_word = 0;
	t->speed_hz = 0;
	t->tx_buf = data;
	t->rx_buf = NULL;
	t->tx_dma = length<=PC2400M_DMA_THRESHOLD?0:virt_to_phys(data);
	t->rx_dma = 0;
	t->len = length;
	spi_message_add_tail(t, (struct spi_message*)nl->spi_msg);
	
	return length;
}

/**
 * pc2400m_spi_h2d - queue an SPI read
 * @ndev: pointer to network device 
 * @data: a buffer containing space for the read data
 * @length: length of the data to read (in bytes)
 *
 * Queue an SPI write to be executed upon the next commit
 */
static s32 pc2400m_spi_d2h(struct net_device *ndev, u8* buf, s32 length) 
{

	struct net_local *nl = netdev_priv(ndev);
	struct spi_transfer *t;

	if (!nl->spi_msg) {
		nl->spi_msg = kmalloc(sizeof(struct spi_message), GFP_KERNEL);
		if (!nl->spi_msg)
			return -ENOMEM;
		spi_message_init((struct spi_message*)nl->spi_msg);
		((struct spi_message*)nl->spi_msg)->is_dma_mapped = 1;
	}

	t = (struct spi_transfer*)
		kmalloc(sizeof(struct spi_transfer), GFP_KERNEL);
	if (!t) return -ENOMEM;
	t->cs_change = 0;
	t->bits_per_word = 0;
	t->speed_hz = 0;
	t->tx_buf = NULL;
	t->rx_buf = buf;
	t->tx_dma = 0;
	t->rx_dma = length<=PC2400M_DMA_THRESHOLD?0:virt_to_phys(buf);
	t->len = length;
	spi_message_add_tail(t, (struct spi_message*)nl->spi_msg);
	
	return length;

}

/**
 * pc2400m_spi_commit - commit queued SPI transactions
 * @ndev: pointer to network device 
 * @cb: call back function to be called when transactions complete
 * @data: a data pointer passed as is to the completion callback
 *
 * Perform all queued transactions on the SPI bus
 */
static int pc2400m_spi_commit(struct net_device *ndev, spi_commit_cb cb, 
			      void* data)
{

	struct net_local *nl = netdev_priv(ndev);
	struct spi_transfer *pos;
	struct spi_transfer *n;
	int ret = 0;

	/* asynchronous operations are not implemented by this interface */
	BUG_ON((int)cb);

	spi_sync(nl->spi_dev, (struct spi_message*)nl->spi_msg);

	/* release memory taken by the message and count bytes */
	nl->spi_total = 0;

	list_for_each_entry_safe(
		pos, n, &(((struct spi_message*)(nl->spi_msg))->transfers), 
		transfer_list) {
	        nl->spi_total += pos->len;
		kfree(pos);
	}
	ret = nl->spi_total;
	kfree(nl->spi_msg);
	nl->spi_msg = NULL;
	
	return ret;
}

static void pc2400m_spi_irq_work(struct work_struct *work)
{
	struct net_local *nl = container_of(work, struct net_local, irq_work);
	spi_irq_cb cb = nl->irq_cb;
	unsigned long flags;

	spin_lock_irqsave(&(nl->irq_lock), flags);
	while(nl->irq_flags & PC2400M_IRQ_PENDING) {
		nl->irq_flags &= (~PC2400M_IRQ_PENDING);
		spin_unlock_irqrestore(&(nl->irq_lock), flags);
		
		/* BEWARE: entry into the single threaded driver 
		   library; must be protected. */
		if (cb) {
			pc2400m_com_lock(nl->net_dev);
			cb(nl->net_dev);
			pc2400m_com_unlock();
		}

		spin_lock_irqsave(&(nl->irq_lock), flags);
	}

	nl->irq_flags &= (~PC2400M_IRQ_EXECUTING);
	spin_unlock_irqrestore(&(nl->irq_lock), flags);


	return;
}


static irqreturn_t pc2400m_spi_irq(int irq, void *config)
{
        struct net_device *ndev = config;
        struct net_local *nl = netdev_priv(ndev);
	unsigned long flags;
	int schedule = 1;

	/* schedule work to call the driver interrupt cb */
	spin_lock_irqsave(&(nl->irq_lock), flags);
	nl->irq_flags |= PC2400M_IRQ_PENDING;

	/* COVERAGE: this scenario cannot be tested in the UT environment,
	   which is essentially single-thread. */
	if (nl->irq_flags & PC2400M_IRQ_EXECUTING)

		schedule = 0;
	spin_unlock_irqrestore(&(nl->irq_lock), flags);

	/* COVERAGE: due to the above, schedule will always be 1 in UT */
	if (schedule)
		schedule_work(&nl->irq_work);

	return IRQ_HANDLED;

}

/**
 * pc2400m_spi_enable_irq - register and enable interrupt handler
 * @ndev: pointer to network device 
 * @irq_cb: call back function to be called when interrupt occurs
 *
 * Register a callback for interrupt handling and enable interrupts
 */
static void pc2400m_spi_enable_irq(struct net_device* ndev, spi_irq_cb irq_cb)
{

	struct net_local *nl = netdev_priv(ndev);

	if (irq_cb) {
		if (nl->irq_cb) {
			nl->irq_cb = irq_cb;
		} else {
			nl->irq_cb = irq_cb;
			enable_irq(OMAP_GPIO_IRQ(nl->irq_gpio));
		}
	} else {
		if (nl->irq_cb) {
			disable_irq(OMAP_GPIO_IRQ(nl->irq_gpio));
			nl->irq_cb = NULL;
		}
	}

}

/**
 * pc2400m_spi_configure - change configured bus speed and parameters
 * @ndev: pointer to the device structure identifying the device
 * @freq: clock frequency to set for the bus
 * @flags: bus specific configuration flags
 *
 * Change the clock frequency and bus parameters for a specific spi device
 */
static void pc2400m_spi_configure(struct net_device *ndev, u32 freq, u32 flags)
{

	struct net_local *nl = netdev_priv(ndev);
	struct spi_device *spi = nl->spi_dev;

	DEBUG("SPI bus clock set to %d Hz, flags %8.8x.\n", freq, flags);
	if (flags & SPI_CONFIG_SAMPLE_FALLING) 
		spi->mode = SPI_MODE_0 | SPI_MOSI_HIGH;
	else
		spi->mode = SPI_MODE_1 | SPI_MOSI_HIGH;
	spi->bits_per_word = 8;
	spi->max_speed_hz = freq;
	spi_setup(spi);
	return;

}

/**
 * pc2400m_spi_set_power - power on and off the chipset
 * @ndev: pointer to the device structure identifying the device
 * @on: nonzero sets power on, zero sets power off
 *
 * Use the main power GPIO to set on and off power to the associated
 * chipset
 */
static void pc2400m_spi_set_power(struct net_device *ndev, u8 on)
{

	struct net_local *nl = netdev_priv(ndev);

	if (on) {
		DEBUG("Switching power on\n");
		omap_set_gpio_dataout(nl->power_gpio, 1);
		msleep(10);
		omap_set_gpio_dataout(nl->reset_gpio, 1);
		msleep(10);
	} else {
		DEBUG("Switching power off\n");
		omap_set_gpio_dataout(nl->reset_gpio, 0);
		omap_set_gpio_dataout(nl->power_gpio, 0);

		/* bus clock to slow after power down */
		pc2400m_spi_configure(ndev, 24000000, 0);

	}

	return;

}


static int pc2400m_spi_probe(struct spi_device *spi)
{
	
	struct net_device *ndev;
        struct net_local *lp;

	/* configure the SPI device */
	spi->mode = SPI_MODE_1 | SPI_MOSI_HIGH;
        spi->bits_per_word = 8;
	if (!spi->max_speed_hz || spi->max_speed_hz > 24000000)
		spi->max_speed_hz = 24000000;
        spi_setup(spi);

	/* create a network device to represent this device */
	ndev = pc2400m_if_new_device();
	if (!ndev) {
		ERROR("Failed to bind SPI bus %d, cs %d, mode %2.2x\n", 
		      spi->master->bus_num, spi->chip_select, spi->mode);
		return -ENOMEM;
	}

	/* store net netdevice to driver data */
	spi->dev.driver_data = (void*)ndev;

	lp = netdev_priv(ndev);

	/* initialize workqueues */
	INIT_WORK(&lp->irq_work, pc2400m_spi_irq_work);

	lp->spi_dev = spi;
	lp->irq_gpio = spi->irq;
	
	/* register interrupt handler for this device */
	spin_lock_init(&(lp->irq_lock));
	lp->irq_flags =  0;
	if (omap_request_gpio(lp->irq_gpio)) {
                ERROR("Cannot request IRQ GPIO %d\n", lp->irq_gpio);
                goto err2;
        }
	omap_set_gpio_direction(lp->irq_gpio, 1);

	if (request_irq(OMAP_GPIO_IRQ(lp->irq_gpio),
			pc2400m_spi_irq, IRQF_DISABLED | IRQF_TRIGGER_FALLING,
			DEVICE_NAME, (void*)ndev)) {
		ERROR("Cannot register IRQ on GPIO %d\n", lp->irq_gpio);
		goto err1;
	}
	disable_irq(OMAP_GPIO_IRQ(lp->irq_gpio));

	/* get power and reset GPIO's for this device instance */
#warning GPIO configuration issue still open! Hardcoded values.
	lp->power_gpio = 11;
	lp->reset_gpio = 47;

	if (omap_request_gpio(lp->power_gpio)) {
                ERROR("Cannot request Power GPIO %d\n", lp->power_gpio);
                goto err0;
        }
	omap_set_gpio_direction(lp->power_gpio, 0);

	if (omap_request_gpio(lp->reset_gpio)) {
                ERROR("Cannot request Reset GPIO %d\n", lp->reset_gpio);
                goto errz;
        }
	omap_set_gpio_direction(lp->reset_gpio, 0);

	/* COVERAGE: as of currently, the pc2400m_if_init_device is a linear
	   function always returning 0. */
	if (pc2400m_if_init_device(ndev)) {
		ERROR("Device init failed (%s)\n", ndev->name);
		goto erry;
	}

	INFO("Bound %s to SPI bus %d, cs %d, mode %2.2x, irq %2.2x\n", 
	     ndev->name, spi->master->bus_num, spi->chip_select, spi->mode,
	     lp->irq_gpio);
	goto out;

 erry:
	omap_free_gpio(lp->reset_gpio);
 errz:
	omap_free_gpio(lp->power_gpio);
 err0:
	free_irq(OMAP_GPIO_IRQ(lp->irq_gpio), ndev);
 err1:
	omap_free_gpio(lp->irq_gpio);
 err2:
	pc2400m_if_free_device(ndev);
	return -ENODEV;

 out:
	list_add(&(lp->listelem), &pc2400m_spi_wimax_device);
        return 0;
}

static int pc2400m_spi_remove(struct spi_device *spi)
{

	struct net_local *pos;
	int irq, power, reset;
	void *dev_id;

	/* need to get rid of the corresponding network device */
	list_for_each_entry(pos, &pc2400m_spi_wimax_device, listelem) {
		if (pos->spi_dev == spi) break;
	}

	BUG_ON(!pos);

	list_del(&(pos->listelem));
	INFO("Unbinding %s\n", pos->net_dev->name);
	
	irq = pos->irq_gpio;
	power = pos->power_gpio;
	reset = pos->reset_gpio;
	dev_id = (void*)pos->net_dev;
	
	/* this should take care of possible ongoing
	   transactions too */
	pc2400m_if_free_device(pos->net_dev);
	
	free_irq(OMAP_GPIO_IRQ(irq), dev_id); 
	/* this is used as identifier only, needs not be allocated*/
	omap_free_gpio(irq);
	omap_free_gpio(power);
	omap_free_gpio(reset);
	
        return 0;
}

static int pc2400m_spi_suspend(struct spi_device *spi, pm_message_t mesg)
{

	struct net_local *pos;

	list_for_each_entry(pos, &pc2400m_spi_wimax_device, listelem) {
		if (pos->spi_dev == spi) break;
	}

	BUG_ON(!pos);
	INFO("Suspending %s...\n", pos->net_dev->name);

	pc2400m_if_suspend_device(pos->net_dev);
	
	return 0;

}

static int pc2400m_spi_resume(struct spi_device *spi)
{

	struct net_local *pos;

	list_for_each_entry(pos, &pc2400m_spi_wimax_device, listelem) {
		if (pos->spi_dev == spi) break;
	}

	BUG_ON(!pos);
	INFO("Resuming %s...\n", pos->net_dev->name);

	pc2400m_if_resume_device(pos->net_dev);
	
	return 0;

}


static int __init pc2400m_spi_init_module(void)
{

	INFO("Intel PC2400M driver (c) Nokia 2007 loaded.\n");

	memset(&pc2400m_drv_adapt, 0, sizeof(pc2400m_drv_adapt));
	init_MUTEX(&pc2400m_com_mutex);
	pc2400m_drv_adapt.h2d = pc2400m_spi_h2d;
	pc2400m_drv_adapt.d2h = pc2400m_spi_d2h;
	pc2400m_drv_adapt.commit = pc2400m_spi_commit;
	pc2400m_drv_adapt.enable_irq = pc2400m_spi_enable_irq;
	pc2400m_drv_adapt.configure = pc2400m_spi_configure;
	pc2400m_drv_adapt.set_power = pc2400m_spi_set_power;

	if (spi_register_driver(&pc2400m_driver))
		return -ENODEV;

	return 0;
	
}

static void __exit pc2400m_spi_cleanup_module(void)
{

	/* the SPI framework will call .remove for each SPI device
	   connected to this driver */
	spi_unregister_driver(&pc2400m_driver);
}

module_init(pc2400m_spi_init_module);
module_exit(pc2400m_spi_cleanup_module);

MODULE_LICENSE("GPL v2");

#endif /* CONFIG_NET_PC2400M_SIM */
