/*
 * pc2400m_sim.c
 *
 * Copyright (C) 2007 Nokia Corporation
 * Author: Juuso Oikarinen <juuso.oikarinen@nokia.com>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */



#define __KERNEL_SYSCALLS__

#include <linux/version.h>
#ifdef MODULE
#ifdef MODVERSIONS
#include <linux/modversions.h>
#endif
#include <linux/module.h>
#else
#define MOD_INC_USE_COUNT
#define MOD_DEC_USE_COUNT
#endif

#include <linux/init.h>
#include <linux/netdevice.h>
#include <linux/inetdevice.h>
#include <linux/clk.h>
#include <linux/spi/spi.h>
#include <linux/platform_device.h>
#include <linux/string.h>
#include <linux/firmware.h>

#ifdef CONFIG_NET_PC2400M_SIM

#include <linux/moduleparam.h>
#include <linux/workqueue.h>
#include <net/tcp.h>
#include <linux/net.h>
#include <linux/in.h>
#include <linux/netfilter.h>
#include <linux/netfilter_ipv4.h>

#include <asm/checksum.h>

#if !defined(CONFIG_FW_LOADER) && !defined(CONFIG_FW_LOADER_MODULE)
#error No Firmware Loading configured in the kernel !
#endif

#include "pc2400m_if.h"
#include "pc2400m_drv_com.h"
#include "pc2400m_drv_hdi.h"
#include "pc2400m_al.h"
#include "pc2400m_spi.h"
#include "pc2400m_com.h"


/* simulator connectivity information */
static u32 sim_addr = 0x7f000001;
static u16 sim_port = 11045;
static u32 for_addr = 0x00000000;

/* hook parameters */
static unsigned int pc2400m_sim_hook(unsigned int, struct sk_buff**,
				   const struct net_device *, 
				   const struct net_device *,
				   int (*)(struct sk_buff *));

static struct nf_hook_ops pc2400m_sim_hook_ops_in = {
	{NULL, NULL},
	pc2400m_sim_hook,
	THIS_MODULE,
	PF_INET,
	NF_IP_PRE_ROUTING,
	NF_IP_PRI_FILTER-1
};

static struct nf_hook_ops pc2400m_sim_hook_ops_out = {
	{NULL, NULL},
	pc2400m_sim_hook,
	THIS_MODULE,
	PF_INET,
	NF_IP_LOCAL_OUT,
	NF_IP_PRI_FILTER-1
};


/* module options configuration */
module_param(sim_addr, int, 0);
MODULE_PARM_DESC(sim_addr, "IPv4 address of simulator in hex\n");
module_param(sim_port, short, 0);
MODULE_PARM_DESC(sim_port, "port number of simulator in hex\n");
module_param(for_addr, int, 0);
MODULE_PARM_DESC(sim_addr, "IPv4 address of foreign data plane interface\n");


/* list to maintain all pc2400m devices managed by this driver */
static struct net_device *pc2400m_spi_wimax_device = NULL;

/*
 * Global lock for the adaptation library. This is global, because the driver
 * lib needs to be concurrency protected from multiple accesses regardless
 * of the number of instances with one single lock. */
struct semaphore pc2400m_com_mutex;

/* function structure for adaptation layers */
struct adaptation_if pc2400m_drv_adapt;

/* internal read data buffer */
static u8 sim_spi_buf[PC2400M_MAX_TRANSACTION];
static u32 sim_spi_buf_len = 0;
static u8 sim_read_buf[PC2400M_MAX_TRANSACTION];
static u32 sim_read_buf_len = 0;

/* socket functions */
static void (*orig_data_ready)(struct sock*, int);
static void (*orig_state_change)(struct sock*);

/* spi queue */
#define SIM_QUEUE_MAX 5
struct sim_queue_elem {
	int read;   /* 1 = read operation, 0 = write operation */
	int len;
	u8 *ptr;
};
static struct sim_queue_elem sim_queue[SIM_QUEUE_MAX];
static int sim_queue_len = 0;

/* device registration information */
static void pc2400m_sim_device_release(struct device * dev);
struct platform_device pc2400m_sim_device = {
        .name           = "wimax-simu",
        .id             = -1,
        .dev            = {
                .release = pc2400m_sim_device_release,
        },
};


#define GENERIC_PAD(x,y) (((x) + (y-1)) & (~(y-1)))
#define PAYLOAD_PAD(x)  GENERIC_PAD(x, PC2400M_FRAME_PAD)
#define BLOCK_PAD(x)    GENERIC_PAD(x, PC2400M_BLOCK_SIZE)
#define PAD4(x)         GENERIC_PAD(x, 4)


#define EXTRACT_LE32(ptr) ((((u32)((u8*)ptr)[3])<<24) | \
                           (((u32)((u8*)ptr)[2])<<16) | \
		           (((u32)((u8*)ptr)[1])<<8) | (((u32)((u8*)ptr)[0])))

#define EXTRACT_LE16(ptr) ((((u16)((u8*)ptr)[1])<<8) | (((u16)((u8*)ptr)[0])))

#define INSERT_LE32(ptr, val) { ((u8*)(ptr))[0] = (u8)(((u32)val)); \
                                ((u8*)(ptr))[1] = (u8)(((u32)val)>>8); \
                                ((u8*)(ptr))[2] = (u8)(((u32)val)>>16); \
                                ((u8*)(ptr))[3] = (u8)(((u32)val)>>24); }

#define INSERT_LE16(ptr, val) { ((u8*)(ptr))[0] = (u8)(((u32)val)); \
                                ((u8*)(ptr))[1] = (u8)(((u32)val)>>8); }

static void pc2400m_sim_device_release(struct device * dev)
{

}

/**
 * pc2400m_spi_h2d - queue write to simulator socket
 * @ndev: pointer to network device 
 * @data: a buffer containing the data to write
 * @length: length of the buffer to write (in bytes)
 *
 * Queue an SPI write to be executed upon the next commit
 */
static s32 pc2400m_spi_h2d(struct net_device *ndev, u8* data, s32 length) 
{

	struct net_local *nl = netdev_priv(ndev);

	/* queue the data for later handling */
	sim_queue[sim_queue_len].read = 0;
	sim_queue[sim_queue_len].ptr = data;
	sim_queue[sim_queue_len++].len = length;

	nl->spi_total += length;

	BUG_ON(sim_queue_len == SIM_QUEUE_MAX);

	return length;
}

/**
 * pc2400m_spi_h2d - queue read from simulator data buffer
 * @ndev: pointer to network device 
 * @data: a buffer containing space for the read data
 * @length: length of the data to read (in bytes)
 *
 * Queue an SPI write to be executed upon the next commit
 */
static s32 pc2400m_spi_d2h(struct net_device *ndev, u8* buf, s32 length) 
{

	struct net_local *nl = netdev_priv(ndev);


	/* queue the data for later handling */
	sim_queue[sim_queue_len].read = 1;
	sim_queue[sim_queue_len].ptr = buf;
	sim_queue[sim_queue_len++].len = length;

	nl->spi_total += length;

	BUG_ON(sim_queue_len == SIM_QUEUE_MAX);

	return length;
	
}

static void pc2400m_sim_send_state_ind(struct net_device *ndev, u8 state)
{

	struct net_local *nl = netdev_priv(ndev);

	INSERT_LE32(sim_read_buf+sim_read_buf_len, 12+8+4);
	sim_read_buf_len += 4;
	INSERT_LE32(sim_read_buf+sim_read_buf_len, FRAME_TYPE_CONTROL);
	sim_read_buf_len += 4;

	/* main message */
	INSERT_LE16(sim_read_buf+sim_read_buf_len, L3_L4_OPCODE_REPORT_STATE);
	sim_read_buf_len += 2;
	INSERT_LE16(sim_read_buf+sim_read_buf_len, 8);
	sim_read_buf_len += 2;
	INSERT_LE16(sim_read_buf+sim_read_buf_len, 0);
	sim_read_buf_len += 2;
	INSERT_LE16(sim_read_buf+sim_read_buf_len, 0);
	sim_read_buf_len += 2;
	INSERT_LE16(sim_read_buf+sim_read_buf_len, 
		    L3L4_RESPONSE_STATUS_SUCCESS_IN_PROCESS);
	sim_read_buf_len += 2;
	INSERT_LE16(sim_read_buf+sim_read_buf_len, 0);
	sim_read_buf_len += 2;

	/* state TLV */
	INSERT_LE16(sim_read_buf+sim_read_buf_len, L3L4_TLV_TYPE_SYSTEM_STATE);
	sim_read_buf_len += 2;
	INSERT_LE16(sim_read_buf+sim_read_buf_len, 4);
	sim_read_buf_len += 2;
	INSERT_LE32(sim_read_buf+sim_read_buf_len, state);
	sim_read_buf_len += 4;
	
	/* signal interrupt for data */
	schedule_work(&nl->spi_work);

	return;

}

static void pc2400m_sim_send_init_resp(struct net_device *ndev)
{

	INSERT_LE32(sim_read_buf+sim_read_buf_len, 12+4);
	sim_read_buf_len += 4;
	INSERT_LE32(sim_read_buf+sim_read_buf_len, FRAME_TYPE_CONTROL);
	sim_read_buf_len += 4;

	INSERT_LE16(sim_read_buf+sim_read_buf_len, L4_L3_OPCODE_CMD_INIT);
	sim_read_buf_len += 2;
	INSERT_LE16(sim_read_buf+sim_read_buf_len, 0);
	sim_read_buf_len += 2;
	INSERT_LE16(sim_read_buf+sim_read_buf_len, 0);
	sim_read_buf_len += 2;
	INSERT_LE16(sim_read_buf+sim_read_buf_len, 0);
	sim_read_buf_len += 2;
	INSERT_LE16(sim_read_buf+sim_read_buf_len, 
		    L3L4_RESPONSE_STATUS_SUCCESS_IN_PROCESS);
	sim_read_buf_len += 2;
	INSERT_LE16(sim_read_buf+sim_read_buf_len, 0);
	sim_read_buf_len += 2;

	/* send ind (will also signal interrupt) */
	pc2400m_sim_send_state_ind(ndev, L3L4_SYSTEM_STATE_INIT);

	return;

}

#define OID_INTEL_80216_CUSTOM (0xff000000)

static int pc2400m_sim_local_resp(struct net_device *ndev, u8 *data_ptr, 
				  u32 size) 
{

	int ret = 0;
	u16 type;

	/* check if this message is something that must be locally handled
	   to get round DSim limitations */
	type = EXTRACT_LE16(data_ptr);
	switch(type) {
	case L4_L3_OPCODE_CMD_INIT:
		pc2400m_sim_send_init_resp(ndev);
		ret = 1;
		break;
	}

	return ret;

}

static void pc2400m_sim_send_to_socket(struct net_device *ndev, u8 *data_ptr, 
				       u32 size) 
{

	struct msghdr msg;
        struct kvec iov;
	struct net_local *nl = netdev_priv(ndev);
	u8 *message;
	int ret;

	if (nl->sock) {
		message = (u8*)kmalloc(size+8, GFP_KERNEL);
		memcpy(message+8, data_ptr, size);
		*((u32*)&message[0]) = cpu_to_le32(size+4);
		*((u32*)&message[4]) = cpu_to_le32(OID_INTEL_80216_CUSTOM);
		
		iov.iov_base = message;
		iov.iov_len = size + 8;
		msg.msg_name = NULL;
		msg.msg_control= NULL;
		msg.msg_controllen = 0;
		msg.msg_flags = /*0 MSG_DONTWAIT*/MSG_WAITALL;
		msg.msg_namelen = 0;
		
		ret = kernel_sendmsg(nl->sock, &msg, &iov, 1, size + 8);
		if (ret <= 0) {
			ERROR("SIM: Sending to socket failed (%d)\n", ret);
		}
		kfree(message);
	} else {
		ERROR("SIM: Simulator not bound for write.\n");
	}
	
}

static void pc2400m_sim_return_to_sender_work(struct work_struct *work)
{

	struct net_local *nl = container_of(work, struct net_local, spi_work);
	spi_irq_cb cb = nl->irq_cb;

	/* BEWARE: entry into the single threaded driver library; must
	   be protected. */
	if (cb) {
		pc2400m_com_lock();
		cb(nl->net_dev);
		pc2400m_com_unlock();
	} else {
		DEBUG("SIM: Data interrupt lost.\n");
	}

}


static void pc2400m_sim_return_to_sender(struct net_device *ndev, 
					 u8 *data_ptr, 
					 u32 size) 
{

	struct net_local *nl = netdev_priv(ndev);
	u32 src;
	u32 dst;
	u16 checksum;

	/* get rid of old header */
	data_ptr += 4;
	size -= 4;

	ERROR("SIM: Data from interface, assuming ping, generating reply\n");
	
	/* put in OID headers */
#if 0   /* MISSING: REMOVED FOR THE SAKE OF COMPLIANCE WITH FW2.0 */
	INSERT_LE32(sim_read_buf+sim_read_buf_len, size+4+8);
#else
	INSERT_LE32(sim_read_buf+sim_read_buf_len, size+4);
#endif
	sim_read_buf_len += 4;
	INSERT_LE32(sim_read_buf+sim_read_buf_len, FRAME_TYPE_DATA);
	sim_read_buf_len += 4;
#if 0   /* MISSING: REMOVED FOR THE SAKE OF COMPLIANCE WITH FW2.0 */
	*(sim_read_buf+sim_read_buf_len) = 1;  /* CS Type IPV4 */
	sim_read_buf_len += 8;
#endif

	memcpy(sim_read_buf+sim_read_buf_len, data_ptr, size);
	
	/* reverse the IP addresses */
	src = EXTRACT_LE32(sim_read_buf+sim_read_buf_len+12);
	dst = EXTRACT_LE32(sim_read_buf+sim_read_buf_len+16);
	INSERT_LE32(sim_read_buf+sim_read_buf_len+12, dst);
	INSERT_LE32(sim_read_buf+sim_read_buf_len+16, src);
	/* ping reply */
	if (*(sim_read_buf+sim_read_buf_len+20) == 8) {
		*(sim_read_buf+sim_read_buf_len+20) = 0;
		
		/* ICMP checksum */
		checksum = ntohs(*((u16*)(sim_read_buf+
					  sim_read_buf_len+22)));
		checksum = 0xffff-checksum;
		checksum -= 0x800;
		checksum = 0xffff-checksum;
		*((u16*)(sim_read_buf+sim_read_buf_len+22)) = 
			htons(checksum);
	}
	
	
	sim_read_buf_len += size;
	
	schedule_work(&nl->spi_work);

}


static u16 pc2400m_sim_calc_read_size(void) 
{

#define PC2400M_D2H_BARKER    0xbeefbabe
	
	int cnt = 0;
	int size;
	u16 type;
	static u32 seq = 0;

	u8 *ptr = sim_read_buf;
	u8 *spi_ptr = sim_spi_buf;

	u8 desc[60 * 4];
	u8 *desc_ptr = desc;

	if (!sim_read_buf_len) {
		sim_spi_buf_len = 0;
		return 0;
	}

	/* prepare the pre-read data for sending to the driver */
	while (sim_read_buf_len) {

		size = EXTRACT_LE32(ptr);
		type = (u16)EXTRACT_LE32(ptr+4);
		wimax_osal_assert(size+4 <= sim_read_buf_len);
		ptr+=8;
		sim_read_buf_len -= (size+4);
		size-=4;
		
		/* fill in the descriptor */
		*((u16*)desc_ptr) = cpu_to_le16(size);
		*((u16*)desc_ptr+1) = cpu_to_le16(type);
		desc_ptr += 4;

		/* data block */
		memcpy(spi_ptr, ptr, size);
		spi_ptr += PAYLOAD_PAD(size);
		ptr += (size);
		cnt++;
	}

	/* insert the headers before the data */
	sim_spi_buf_len = spi_ptr - sim_spi_buf;
	memmove(sim_spi_buf + 16 + PAYLOAD_PAD(cnt*4), sim_spi_buf,
		sim_spi_buf_len);
	sim_spi_buf_len += (16 + PAYLOAD_PAD(cnt*4));

	spi_ptr = sim_spi_buf;
	*((u32*)(&spi_ptr[0])) = cpu_to_le32(PC2400M_D2H_BARKER);
	*((u32*)(&spi_ptr[4])) = cpu_to_le32(seq);
	*((u32*)(&spi_ptr[8])) = cpu_to_le16(cnt);
	*((u32*)(&spi_ptr[12])) = cpu_to_le16(PC2400M_BLOCK_SIZE - 
					      (sim_spi_buf_len%256));
	seq++;
	spi_ptr+=16;
	memcpy(spi_ptr, desc, cnt * 4);
	sim_spi_buf_len = BLOCK_PAD(sim_spi_buf_len);

	/* insert SPI/SDIO transaction headers */
	for (size=sim_spi_buf_len/PC2400M_BLOCK_SIZE-1; size>=0; size--) {
		
		memmove(sim_spi_buf+size*(PC2400M_BLOCK_SIZE+8)+4,
			sim_spi_buf+size*PC2400M_BLOCK_SIZE,
			PC2400M_BLOCK_SIZE);
		cnt = size*(PC2400M_BLOCK_SIZE+8);
		sim_spi_buf[cnt+0] = 0xff;
		sim_spi_buf[cnt+1] = 0xff;
		sim_spi_buf[cnt+2] = 0xff;
		sim_spi_buf[cnt+3] = 0xfe;

		cnt += (PC2400M_BLOCK_SIZE + 4);
		sim_spi_buf[cnt+0] = 0xff;
		sim_spi_buf[cnt+1] = 0xff;
		sim_spi_buf[cnt+2] = 0xff;
		sim_spi_buf[cnt+3] = 0xff;

	}
	
	return sim_spi_buf_len;
}

#define SIM_BLOCK_FOOTER   4
#define SIM_BLOCK_HEADER   4
#define SIM_BLOCK_PAD      (SIM_BLOCK_HEADER + SIM_BLOCK_FOOTER)

static void pc2400m_sim_lose_spi_padding(u8 *buf, u32 len) {

	int i = 0;

	for (i=0; i<len; i+=(PC2400M_BLOCK_SIZE+SIM_BLOCK_PAD)) {
		memmove(buf+i*PC2400M_BLOCK_SIZE,
			buf+i*(PC2400M_BLOCK_SIZE+SIM_BLOCK_PAD)
			+SIM_BLOCK_HEADER,
			PC2400M_BLOCK_SIZE);
	}

	return;

}

static void sim_handle_write_data(struct net_device *ndev, u8 *buf, int length)
{

#define PC2400M_CMD_WRITE    0xcbbc0702
#define PC2400M_CMD_JUMP     0xcbbc0503
#define PC2400M_H2D_BARKER   0xcafe900d

	u8 *ptr;
	u8 *dataptr;
	u8 *descptr;
	u32 cnt;
	u32 val;
	u16 size;

	/* scan the data for supported blocks */
	pc2400m_sim_lose_spi_padding(buf, length);

	ptr = buf;
	val = EXTRACT_LE32(ptr);


	/* simulate state change to CONFIG (DSim won't send it) */
	if (val == PC2400M_CMD_JUMP)
		pc2400m_sim_send_state_ind(
			ndev, L3L4_SYSTEM_STATE_CONFIG);

 	switch (val&0xffff0000) {
	case PC2400M_CMD_WRITE&0xffff0000:
	  /*case PC2400M_CMD_JUMP&0xffff0000:*/
		break;
		
	case PC2400M_H2D_BARKER&0xffff0000:
		DEBUG("SIM: received H2D data.\n");
		
		cnt = EXTRACT_LE16(ptr+8);
		descptr = ptr + 16;
		dataptr = descptr + PAYLOAD_PAD(PC2400M_DESCRIPTOR_SIZE * cnt);

		while (cnt) {
			size = EXTRACT_LE16(descptr);
			switch (EXTRACT_LE16(descptr + 2)) {
			case FRAME_TYPE_DATA:
				/* data frames not supported */
				pc2400m_sim_return_to_sender(
					ndev, dataptr, size);
				break;
			case FRAME_TYPE_CONTROL:
				/* send to simulator socket */
				if (!pc2400m_sim_local_resp(
					    ndev, dataptr, size))
					pc2400m_sim_send_to_socket
						(ndev, dataptr, size);
				break;
			default:
				ERROR("SIM: unknown data frame type "
				      "received\n");
				break;

			}

			cnt--;
			descptr += PC2400M_DESCRIPTOR_SIZE;
			size = PAYLOAD_PAD(size);
			dataptr += size;

		}
		break;

	default:
		ERROR("SIM: Unknown data block ignored\n");
	
	}

	return;

}

		
static void pc2400m_sim_commit(struct net_device *ndev) 
{

	/* determine operation */
	static u32 readsize = 0;
	u32 cmd;

	if (sim_queue_len >= 2) {
		/* the first message will be the SDIO command */
		if (sim_queue[0].read) {
			ERROR("SIM transaction started with a read.\n");
			return;
		}
		
		switch ((*(sim_queue[0].ptr+1))&0x3f) {
		case 0:
			sim_queue[1].ptr[0] = 1;
			break;
		case 5:	      
			memset(sim_queue[1].ptr, 0, sim_queue[1].len);
			sim_queue[1].ptr[1] = 0x80;
			sim_queue[1].ptr[2] = 0xff;
			sim_queue[1].ptr[3] = 0x80;
			sim_queue[1].ptr[4] = 0x00;
			break;
		case 52:
			/* return the data for a write command */
			memset(sim_queue[1].ptr, 0, sim_queue[1].len);
			if (sim_queue[0].ptr[2] & 0x80) {
				sim_queue[1].ptr[1] = sim_queue[0].ptr[5];
			} else {
				sim_queue[1].ptr[1] = 0;
			}
			
			/* for certain registers specific behavior is needed */
			cmd = ((u32)sim_queue[0].ptr[2]) << 24 |
				((u32)sim_queue[0].ptr[3]) << 16 |
				((u32)sim_queue[0].ptr[4]) << 8 |
				sim_queue[0].ptr[5];
			cmd >>= 9;
			cmd &= 0x1ffff;
			if (cmd == 0x1c) { 
				readsize = pc2400m_sim_calc_read_size();
				sim_queue[1].ptr[1] = (readsize)&0xff;
			}
			if (cmd == 0x1d) {
				sim_queue[1].ptr[1] = 
					((readsize)>>8)&0xff;
			}
			if (cmd == 0x05) {
				sim_queue[1].ptr[1] = 0x02;
			}
			if (cmd == 0x04 && !(sim_queue[0].ptr[2] & 0x80)) {
				sim_queue[1].ptr[1] = 0x01;
			}
				
			break;
		case 53:
			if (sim_queue[0].ptr[2] & 0x80 &&
			    sim_queue_len >= 3) {
				/* a write command */

				cmd = ((u32)sim_queue[1].ptr[2]) << 24 |
					((u32)sim_queue[1].ptr[3]) << 16 |
					((u32)sim_queue[1].ptr[4]) << 8 |
					sim_queue[1].ptr[5];
				cmd &= 0x1f;

				/* check the data direction */
				/* a write command */
				if (!sim_queue[2].read)
				{
					sim_handle_write_data(
						ndev,
						sim_queue[2].ptr, 
						sim_queue[2].len);
				} else {
					ERROR("SIM: Assumed write, "
					      "got read\n");
				}
				
				memset(sim_queue[1].ptr, 0, sim_queue[1].len);
			} else if (!(sim_queue[0].ptr[2]&0x80) &&
				    sim_queue_len >= 2) {
				memcpy(sim_queue[1].ptr+2, sim_spi_buf,
				       (sim_spi_buf_len/PC2400M_BLOCK_SIZE) *
				       (PC2400M_BLOCK_SIZE+8));

				sim_queue[1].ptr[0] = 0x00;
				sim_queue[1].ptr[1] = 0x00; /* cmd53 resp */
				sim_spi_buf_len = 0;
			} else {
				ERROR("SIM: Unknown cmd53\n");
			}
		

			break;
		default:
			ERROR("Unknown command encountered.\n");
			break;
			
		}
		
	} else {
		ERROR("SIM not enough messages for transaction\n");
	}

	/* clear the queue */
	sim_queue_len = 0;
	return;

}

#if 0
static void pc2400m_spi_commit_work(void *data)
{

	struct net_local *nl = netdev_priv((struct net_device*)data);
	spi_commit_cb cb = nl->complete_cb;
	void *cbdata = nl->cb_data;
	s32 total = nl->spi_total;

	/* commit the data */
	pc2400m_sim_commit((struct net_device*)data);
	
	/* BEWARE: entry into the single threaded driver library; must
	   be protected. */
	pc2400m_com_lock();
	nl->complete_cb = NULL;
	nl->cb_data = NULL;	
	cb((struct net_device*)data, total, cbdata);
	pc2400m_com_unlock();

}
#endif

/**
 * pc2400m_spi_commit - commit queued simulator socket transactions
 * @ndev: pointer to network device 
 * @cb: call back function to be called when transactions complete
 * @data: a data pointer passed as is to the completion callback
 *
 * Perform all queued transactions on the SPI bus
 */
static int pc2400m_spi_commit(struct net_device *ndev, spi_commit_cb cb, 
			    void* data)
{

	struct net_local *nl = netdev_priv(ndev);
	int ret = 0;

	if (cb) {
		BUG_ON(0);
		nl->complete_cb = cb;
		nl->cb_data = data;
	} else {
		pc2400m_sim_commit(ndev);
	}

	return ret;
}

static void pc2400m_spi_irq_work(struct work_struct *work)
{
	struct net_local *nl = container_of(work, struct net_local, irq_work);
	struct socket *csocket = nl->sock;
	struct socket *cindsocket = nl->indsock;

	u32 start_length;
	u32 hdrsize;

  	char buffer[100];
	int ret = 0;
        struct msghdr msg;
        struct kvec iov;

	pc2400m_com_lock();
	read_lock_bh(csocket->sk->sk_callback_lock);

	/* read the data from the socket */
	start_length = sim_read_buf_len;
	while (1) {
		iov.iov_base = buffer;
		iov.iov_len = 100;
		msg.msg_name = NULL;
		msg.msg_control= NULL;
		msg.msg_controllen = 0;
		msg.msg_flags = MSG_DONTWAIT;
		msg.msg_namelen = 0;

		ret = kernel_recvmsg(
			csocket, &msg, &iov, 1, 100, msg.msg_flags); 
		if (ret == 0 || ret == -EAGAIN) break;
		if (ret < 0) {
			ERROR("SIM: Socket read error occurred (%d)\n", ret);
			goto err1;
		}

		if (sim_read_buf_len + ret >= PC2400M_MAX_TRANSACTION) {
			ERROR("SIM: Read buffer overflow\n");
			goto err1;
		}
		
		memcpy(sim_read_buf+sim_read_buf_len, buffer, ret);
		sim_read_buf_len += ret;
		
	}

	goto ind;
 err1:
	sim_read_buf_len = 0;

 ind:
	if (sim_read_buf_len > start_length) {
		hdrsize = EXTRACT_LE32(sim_read_buf+start_length);
		sim_read_buf_len = start_length + hdrsize + 4;
		INSERT_LE32(sim_read_buf+start_length+4, FRAME_TYPE_CONTROL);
	}

	read_unlock_bh(csocket->sk->sk_callback_lock);

	read_lock_bh(cindsocket->sk->sk_callback_lock);

	/* read the data from the indications socket */
	start_length = sim_read_buf_len;
	while (1) {
		iov.iov_base = buffer;
		iov.iov_len = 100;
		msg.msg_name = NULL;
		msg.msg_control= NULL;
		msg.msg_controllen = 0;
		msg.msg_flags = MSG_DONTWAIT;
		msg.msg_namelen = 0;

		ret = kernel_recvmsg(
			cindsocket, &msg, &iov, 1, 100, msg.msg_flags); 
		if (ret == 0 || ret == -EAGAIN) break;
		if (ret < 0) {
			ERROR("SIM: Socket read error occurred (%d)\n", ret);
			goto err2;
		}

		if (sim_read_buf_len + ret >= PC2400M_MAX_TRANSACTION) {
			ERROR("SIM: Read buffer overflow\n");
			goto err2;
		}

#if 0			
		/* simulate oid message header */
		INSERT_LE32(sim_read_buf+sim_read_buf_len, ret+4);
		sim_read_buf_len+=8;
#endif	
	
		memcpy(sim_read_buf+sim_read_buf_len, buffer, ret);
		sim_read_buf_len += ret;
	
	}

	goto out;
 err2:
	sim_read_buf_len = 0;
 out:
	if (sim_read_buf_len > start_length) {
		hdrsize = EXTRACT_LE32(sim_read_buf+start_length);
		sim_read_buf_len = start_length + hdrsize + 4;
		INSERT_LE32(sim_read_buf+start_length+4, FRAME_TYPE_CONTROL);
	}

	read_unlock_bh(cindsocket->sk->sk_callback_lock);
	pc2400m_com_unlock();

	/* BEWARE: entry into the single threaded driver library; must
	   be protected. */
	pc2400m_com_lock();
	if (nl->irq_cb) {
		nl->irq_cb(nl->net_dev);
	} else {
		DEBUG("SIM: Data interrupt lost.\n");
	}
	pc2400m_com_unlock();

}

/**
 * pc2400m_spi_enable_irq - register and enable interrupt handler
 * @ndev: pointer to network device 
 * @irq_cb: call back function to be called when interrupt occurs
 *
 * Register a callback for interrupt handling and enable interrupts
 */
static void pc2400m_spi_enable_irq(struct net_device* ndev, spi_irq_cb irq_cb)
{

	struct net_local *nl = netdev_priv(ndev);

	if (irq_cb) {
		nl->irq_cb = irq_cb;
	} else {
		nl->irq_cb = NULL;
	}

}


/**
 * pc2400m_spi_set_power - power on and off the chipset
 * @ndev: pointer to the device structure identifying the device
 * @on: nonzero sets power on, zero sets power off
 *
 * Use the main power GPIO to set on and off power to the associated
 * chipset
 */
static void pc2400m_spi_set_power(struct net_device *ndev, u8 on)
{

	if (on) {
		DEBUG("Switching power on\n");
	} else {
		DEBUG("Switching power off\n");
	}

	return;

}
/**
 * pc2400m_spi_configure - change configured bus speed and parameters
 * @ndev: pointer to the device structure identifying the device
 * @freq: clock frequency to set for the bus
 * @flags: bus specific configuration flags
 *
 * Change the clock frequency and bus parameters for a specific spi device
 */
static void pc2400m_spi_configure(struct net_device *ndev, u32 freq, u32 flags)
{

	INFO("SPI configuration change ignored\n");
	return;

}

static void pc2400m_sim_state_change(struct sock *sk) 
{

	/*struct net_device *ndev = (struct net_device *)sk->sk_user_data;*/

	/* then make our deductions on the info */
	read_lock(&sk->sk_callback_lock);
	switch(sk->sk_state) {
                /* ignore connecting sockets as they make progress */
	case TCP_SYN_SENT:
	case TCP_SYN_RECV:
		break;
	case TCP_ESTABLISHED:
		break;
	default:
		/* will be released upon module removal */

		ERROR("SIM: Simulator connection lost, reinsert of module "
		      "needed!\n");
		
		break;
        }
	
	read_unlock(&sk->sk_callback_lock);

	/* call the regular state change handler */
        orig_state_change(sk);

}


static void pc2400m_sim_data_ready(struct sock* sock, int bytes)
{
	struct net_local *nl = 
		netdev_priv((struct net_device*)sock->sk_user_data);

	/* indicate an interrupt */
	schedule_work(&nl->irq_work);

#if 0 /* this may cause troube */
	/* perform original handling */
	//orig_data_ready(sock, bytes);
#endif
	return;	
}


static unsigned int pc2400m_sim_hook(unsigned int hooknum,
				   struct sk_buff **skb,
				   const struct net_device *in,
				   const struct net_device *out,
				   int (*okfn)(struct sk_buff *))
{

	struct net_local *nl;
	struct in_device *idev;

	if (!pc2400m_spi_wimax_device) return NF_ACCEPT;
       
	nl = netdev_priv(pc2400m_spi_wimax_device);
	idev = (struct in_device*)(pc2400m_spi_wimax_device->ip_ptr);

	/* no mangling is done, until our interface has an address */
	if (!idev) return NF_ACCEPT;
	if (!idev->ifa_list) return NF_ACCEPT;

	if (*((u32*)(&((*skb)->data[12]))) == 
	    idev->ifa_list->ifa_address) {

		DEBUG("SIM: Outgoing: %d.%d.%d.%d => %d.%d.%d.%d\n",
		      (*skb)->data[12],
		      (*skb)->data[13], 
		      (*skb)->data[14], 
		      (*skb)->data[15],
		      (*skb)->data[16],
		      (*skb)->data[17], 
		      (*skb)->data[18], 
		      (*skb)->data[19]);
		
		/* replace the source address */
		*((u32*)(&((*skb)->data[12]))) =
			htonl(for_addr);
		
		/* replace the target address network part */
		(*skb)->data[16] = (u8)(for_addr>>24);
		(*skb)->data[17] = (u8)(for_addr>>16);
		(*skb)->data[18] = (u8)(for_addr>>8);

		DEBUG("SIM: Mangled:  %d.%d.%d.%d => %d.%d.%d.%d\n",
		      (*skb)->data[12],
		      (*skb)->data[13], 
		      (*skb)->data[14], 
		      (*skb)->data[15],
		      (*skb)->data[16],
		      (*skb)->data[17], 
		      (*skb)->data[18], 
		      (*skb)->data[19]);

		
		/* recalc the header checksum */
		*((u16*)(&((*skb)->data[10]))) = 0;
		*((u16*)(&((*skb)->data[10]))) = 
			ip_fast_csum((const void*)
				     ((*skb)->data),
				     ((*skb)->data[0])&0xf);

		ip_route_me_harder(skb, RTN_UNSPEC);
		
	} else if (ntohl(*((u32*)(&((*skb)->data[16])))) == for_addr) {
		
		DEBUG("SIM: Incoming: %d.%d.%d.%d => %d.%d.%d.%d\n",
		      (*skb)->data[12],
		      (*skb)->data[13], 
		      (*skb)->data[14], 
		      (*skb)->data[15],
		      (*skb)->data[16],
		      (*skb)->data[17], 
		      (*skb)->data[18], 
		      (*skb)->data[19]);

		/* replace the target address */
		*((u32*)(&((*skb)->data[16]))) =
			idev->ifa_list->ifa_address;
		
		/* replace the source address network part */
		(*skb)->data[12] = (u8)((idev->ifa_list->ifa_address));
		(*skb)->data[13] = (u8)((idev->ifa_list->ifa_address>>8));
		(*skb)->data[14] = (u8)((idev->ifa_list->ifa_address>>16));

		DEBUG("SIM: Mangled:  %d.%d.%d.%d => %d.%d.%d.%d\n",
		      (*skb)->data[12],
		      (*skb)->data[13], 
		      (*skb)->data[14], 
		      (*skb)->data[15],
		      (*skb)->data[16],
		      (*skb)->data[17], 
		      (*skb)->data[18], 
		      (*skb)->data[19]);

		/* recalc the header checksum */
		*((u16*)(&((*skb)->data[10]))) = 0;
		*((u16*)(&((*skb)->data[10]))) = 
			ip_fast_csum((const void*)
				     ((*skb)->data),
				     ((*skb)->data[0])&0xf);
				
	}
	
	return NF_ACCEPT;

}



static int pc2400m_sim_connect(void)
{
	
	struct net_device *ndev;
        struct net_local *lp;
	struct socket *csocket;
	struct socket *cindsocket;
	struct sockaddr_in addr;
	int rc;

	/* create a network device to represent this device */
	ndev = pc2400m_if_new_device();
	if (!ndev) {
		ERROR("Failed to bind %d.%d.%d.%d:%d\n",
		      (u8)(sim_addr>>24),
		      (u8)(sim_addr>>16),
		      (u8)(sim_addr>>8),
		      (u8)(sim_addr),
		      sim_port);
		return -ENOMEM;
	}
	lp = netdev_priv(ndev);
	
	rc = platform_device_register(&pc2400m_sim_device);
	if (rc) {
		ERROR("SIM: Could not register Pc2400m simulator dev\n");
		goto err3;
	}
	lp->spi_dev = &pc2400m_sim_device;
	lp->spi_dev->dev.driver_data = (void*)ndev;
	
	/* initialize workqueues */
	INIT_WORK(&(lp->irq_work), pc2400m_spi_irq_work);
	INIT_WORK(&(lp->spi_work), pc2400m_sim_return_to_sender_work);

	pc2400m_com_lock();
	rc = sock_create_kern(PF_INET, SOCK_STREAM, IPPROTO_TCP, &csocket);
	if (rc < 0) {
		ERROR("Error %d creating socket\n",rc);
		pc2400m_com_unlock();
		goto err2;
	} else {
		csocket->sk->sk_allocation = GFP_NOFS;
	}

	rc = sock_create_kern(PF_INET, SOCK_STREAM, IPPROTO_TCP, &cindsocket);
	if (rc < 0) {
		ERROR("Error %d creating socket\n",rc);
		pc2400m_com_unlock();
		goto err1;
	} else {
		cindsocket->sk->sk_allocation = GFP_NOFS;
	}
	
	/* register callbacks */
	write_lock_bh(csocket->sk->sk_callback_lock);
	csocket->sk->sk_user_data = (void*)ndev;
	orig_data_ready = csocket->sk->sk_data_ready;
	orig_state_change = csocket->sk->sk_state_change;
	csocket->sk->sk_data_ready = pc2400m_sim_data_ready;
	csocket->sk->sk_state_change = pc2400m_sim_state_change;
	write_unlock_bh(csocket->sk->sk_callback_lock);

	write_lock_bh(cindsocket->sk->sk_callback_lock);
	cindsocket->sk->sk_user_data = (void*)ndev;
	cindsocket->sk->sk_data_ready = pc2400m_sim_data_ready;
	cindsocket->sk->sk_state_change = pc2400m_sim_state_change;
	write_unlock_bh(cindsocket->sk->sk_callback_lock);

	memset(&addr, 0x00, sizeof(struct sockaddr_in));
        addr.sin_family = AF_INET;
	addr.sin_port = htons(sim_port);
	addr.sin_addr.s_addr = htonl(sim_addr);
	rc = csocket->ops->connect(csocket, (struct sockaddr*)&addr, 
				   sizeof(struct sockaddr_in), 0);
	if (rc < 0) {
		ERROR("Connect failure %d\n", rc);
		pc2400m_com_unlock();
		goto err0;
        }
	lp->sock = csocket;

	memset(&addr, 0x00, sizeof(struct sockaddr_in));
        addr.sin_family = AF_INET;
	addr.sin_port = htons(sim_port+1);
	addr.sin_addr.s_addr = htonl(sim_addr);
	rc = cindsocket->ops->connect(cindsocket, (struct sockaddr*)&addr, 
				      sizeof (struct sockaddr_in), 0);
	if (rc < 0) {
		ERROR("Connect (ind) failure %d\n", rc);
		pc2400m_com_unlock();
		goto err0;
        }
	lp->indsock = cindsocket;

	pc2400m_com_unlock();

	if (pc2400m_if_init_device(ndev)) {
		ERROR("Device init failed (%s)\n", ndev->name);
		goto err0;
	}

	/* register a netfilter hook to steal interface data */
	if (for_addr) {

		if (nf_register_hook(&pc2400m_sim_hook_ops_in)) {
			ERROR("Registering nf hook failed.\n");
		}

		if (nf_register_hook(&pc2400m_sim_hook_ops_out)) {
			ERROR("Registering nf hook failed.\n");
		}
		
	}

		
	ERROR("Bound %s to simulator @ %d.%d.%d.%d:%d (%d)\n",
	      ndev->name,
	      (u8)(sim_addr>>24),
	      (u8)(sim_addr>>16),
	      (u8)(sim_addr>>8),
	      (u8)(sim_addr),
	      sim_port, sim_port+1);

	goto out;

 err0:
	sock_release(cindsocket);

 err1:
	sock_release(csocket);

 err2:
	platform_device_unregister(&pc2400m_sim_device);

 err3:
	pc2400m_if_free_device(ndev);
	return -ENODEV;

 out:	 
	pc2400m_spi_wimax_device = ndev;
        return 0;
}

static int pc2400m_sim_remove(void)
{

	struct net_local *lp = NULL;
	struct socket *sock = NULL;
	struct socket *indsock = NULL;

	/* MISSING: be sure that there is no asynchronous operation ongoing
	   depending on the network interfaces */

	/* need to get rid of the corresponding network device */
	if (pc2400m_spi_wimax_device) {
		lp = netdev_priv(pc2400m_spi_wimax_device);
		sock = lp->sock;
		indsock = lp->indsock;

		if (for_addr) {
			nf_unregister_hook(&pc2400m_sim_hook_ops_in);
			nf_unregister_hook(&pc2400m_sim_hook_ops_out);
		}

		INFO("Unbinding %s\n", lp->net_dev->name);
		pc2400m_if_free_device(lp->net_dev);
		platform_device_unregister(&pc2400m_sim_device);
		if (sock) sock_release(sock);
		if (indsock) sock_release(indsock);

	}

        return 0;
}

void pc2400m_spi_stop(struct net_device *dev)
{
#if 0
	omap_free_gpio(wlan_config->power_gpio);
	omap_free_gpio(wlan_config->irq_gpio);
#endif
}


static int __init pc2400m_spi_init_module(void)
{

	INFO("Pc2400m driver (c) Nokia 2007 loaded.\n");

	memset(&pc2400m_drv_adapt, 0, sizeof(pc2400m_drv_adapt));
	init_MUTEX(&pc2400m_com_mutex);
	pc2400m_drv_adapt.h2d = pc2400m_spi_h2d;
	pc2400m_drv_adapt.d2h = pc2400m_spi_d2h;
	pc2400m_drv_adapt.commit = pc2400m_spi_commit;
	pc2400m_drv_adapt.enable_irq = pc2400m_spi_enable_irq;
	pc2400m_drv_adapt.configure = pc2400m_spi_configure;
	pc2400m_drv_adapt.set_power = pc2400m_spi_set_power;

	return pc2400m_sim_connect();
	
}

static void __exit pc2400m_spi_cleanup_module(void)
{

	/* MISSING: be sure that no transactions are ongoing when doing
	   this */

	pc2400m_sim_remove();

}

module_init(pc2400m_spi_init_module);
module_exit(pc2400m_spi_cleanup_module);

MODULE_LICENSE("GPL v2");


#endif /* CONFIG_NET_PC2400M_SIM */
