/*
 * pc2400m_if_netlink.c
 *
 *
 * Copyright (C) 2007 Nokia Corporation
 * Author: Jouni Lappi <jouni.lappi@nokia.com>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */


#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/init.h>
#include <linux/list.h>
#include <linux/firmware.h>
#include <linux/completion.h>
#include <linux/platform_device.h>
#include <linux/netlink.h>
#include <linux/socket.h>
#include <net/sock.h>

#include "pc2400m_com.h"
#include "pc2400m_if.h"
#include "pc2400m_al.h"
#include "pc2400m_drv.h"
#include "pc2400m_drv_if.h"
#include "pc2400m_if.h"
#include "pc2400m_if_netlink.h"

static struct sock *pc2400m_nl_sk1 = NULL;
static struct sock *pc2400m_nl_sk2 = NULL;
static struct sock *pc2400m_nl_sk3 = NULL;

/**
 * pc2400m_netlink_trace_pipe_ready - check whether trace pipe can send
 */
int pc2400m_netlink_trace_pipe_ready(struct net_device *ndev)
{

	struct net_local *nl = netdev_priv(ndev);
	return ((int)pc2400m_nl_sk3) && nl->tube3_pid;

}


/**
 * pc2400m_netlink_send_data - data from device to host
 * @skb:data buffer
 * @netlink:netlink number
 */
int pc2400m_netlink_send_data(struct net_device *ndev,
			      struct sk_buff *skb, 
			      char netlink)
{
	struct net_local *nl = netdev_priv(ndev);
	struct sock *pc2400m_nl_sk = NULL;
	int tube_pid = 0;

	BUG_ON(!skb);

	BUG_ON((netlink != NETLINK_PC2400M1) &&
	       (netlink != NETLINK_PC2400M2) &&
	       (netlink != NETLINK_PC2400M3));
	
	switch (netlink) {
	case NETLINK_PC2400M1:  /* Production testing */
		pc2400m_nl_sk = pc2400m_nl_sk1;
		BUG_ON(!pc2400m_nl_sk);
		tube_pid = nl->tube1_pid;
		break;
	case NETLINK_PC2400M2:  /* Diagnostics access */
		pc2400m_nl_sk = pc2400m_nl_sk2;
		BUG_ON(!pc2400m_nl_sk);
		tube_pid = nl->tube2_pid;
		break;
	case NETLINK_PC2400M3:  /* Traces / debug */
		pc2400m_nl_sk = pc2400m_nl_sk3;
		BUG_ON(!pc2400m_nl_sk);
		tube_pid = nl->tube3_pid;
		break;
	}
	
	/* we'll add ifname */
	skb_push(skb,IFNAMSIZ);
	memcpy(skb->data,ndev->name,IFNAMSIZ);

	/* we make room for nlmsghdr */
	skb_push(skb,sizeof(struct nlmsghdr));
	memset(skb->data,0,sizeof(struct nlmsghdr));

	NETLINK_CB(skb).pid = 0;
	NETLINK_CB(skb).dst_group = 0;

	if (tube_pid == 0) {
		netlink_broadcast(pc2400m_nl_sk, skb, 0, 0, GFP_ATOMIC);
	}
	else {
		netlink_unicast(pc2400m_nl_sk, skb, tube_pid, MSG_DONTWAIT);
	}
 
	return 0;

}

/**
 * nl_data_ready1 - handler for incoming netlink packets from host
 * @sk: socket
 * @len: data length
 */
static void nl_data_ready1(struct sock *sk, int len)
{

	struct net_device *ndev;
	struct net_local *nl;
	struct sk_buff *skb = NULL;
	struct nlmsghdr *nlh = NULL;
	struct drv_cmd cmd;	
	char *datptr;

	if (!pc2400m_nl_sk1) {
		ERROR("No netlink initialization done\n");
		goto out1;	
	}
	ndev = pc2400m_nl_sk1->sk_user_data;
	nl = netdev_priv(ndev);
	skb = skb_dequeue(&sk->sk_receive_queue);
	while(skb) {
		/* get netlink message header */
		nlh = (struct nlmsghdr *)skb->data;

		/* check that this message is for us */
		if(!strncmp((skb->data),ndev->name,IFNAMSIZ))goto out1;
		/* save pid */
		nl->tube1_pid = nlh->nlmsg_pid;

		/* remove netlink header */
		skb_pull(skb,sizeof(struct nlmsghdr));
		/* remove ifname */
		skb_pull(skb,IFNAMSIZ);

		if(skb->len > 1){
			datptr = kmalloc(skb->len,GFP_KERNEL);
			memcpy(datptr,skb->data,skb->len);

			cmd.cmd_id = E_PROD_TEST;	
			cmd.cmd.prod_test.data_length = skb->len;
			cmd.cmd.prod_test.data = datptr;

			WIMAX_NO_WAIT_CMD(pc2400m_nl_sk1->sk_user_data, &cmd);
		}
		kfree_skb(skb);
		skb = skb_dequeue(&sk->sk_receive_queue);
	}
 out1:
	return ;
}

/**
 * nl_data_ready2 - handler for incoming netlink packets from host
 * @sk: socket
 * @len: data length
 */
static void nl_data_ready2(struct sock *sk, int len)
{

	struct net_device *ndev;
	struct net_local *nl;
	struct sk_buff *skb = NULL;
	struct nlmsghdr *nlh = NULL;
	struct drv_cmd cmd;	
	char *datptr;

	if (!pc2400m_nl_sk2) {
		ERROR("No netlink initialization done\n");
		goto out1;	
	}
	ndev = pc2400m_nl_sk2->sk_user_data;
	nl = netdev_priv(ndev);
	skb = skb_dequeue(&sk->sk_receive_queue);
	while(skb) {
		/* get netlink message header */
		nlh = (struct nlmsghdr *)skb->data;

		/* check that this message is for us */	
		if(!strncmp((skb->data),ndev->name,IFNAMSIZ))goto out1;
		/* save pid */
		nl->tube2_pid = nlh->nlmsg_pid;

		/* remove netlink header */
		skb_pull(skb,sizeof(struct nlmsghdr));
		/* remove ifname */
		skb_pull(skb,IFNAMSIZ);
		if(skb->len > 1){
			datptr = kmalloc(skb->len,GFP_KERNEL);
			memcpy(datptr,skb->data,skb->len);

			cmd.cmd_id = E_STATISTICS_GET;	
			cmd.cmd.statistics_get.data_length = skb->len;
			cmd.cmd.statistics_get.data = datptr;

			WIMAX_NO_WAIT_CMD(pc2400m_nl_sk2->sk_user_data, &cmd);
		}
		kfree_skb(skb);
		skb = skb_dequeue(&sk->sk_receive_queue);
	}
 out1:
	return ;
}

/**
 * nl_data_ready3 - handler for incoming netlink packets from host
 * @sk: socket
 * @len: data length
 */
static void nl_data_ready3(struct sock *sk, int len)
{

	struct netlink_pipe_msg_str *pipehdr;
	struct net_device *ndev;
	struct net_local *nl;
	struct sk_buff *skb = NULL;
	struct nlmsghdr *nlh = NULL;
	struct drv_cmd cmd;	
	char *datptr;

	if (!pc2400m_nl_sk3) {
		ERROR("No netlink initialization done\n");
		goto out1;	
	}
	ndev = pc2400m_nl_sk3->sk_user_data;
	nl = netdev_priv(ndev);
	skb = skb_dequeue(&sk->sk_receive_queue);
	while(skb) {
		/* get netlink message header */
		nlh = (struct nlmsghdr *)skb->data;

		/* check that this message is for us */
		if(!strncmp((skb->data),ndev->name,IFNAMSIZ))goto out1;
		/* save pid */
		nl->tube3_pid = nlh->nlmsg_pid;

		/* remove netlink header */
		skb_pull(skb,sizeof(struct nlmsghdr));

		/* remove ifname */
		skb_pull(skb,IFNAMSIZ);

		/* check utilized version */
		pipehdr = (struct netlink_pipe_msg_str*)skb->data;
		if (pipehdr->version != NETLINK_PIPE_VERSION) {
			ERROR("Unsupported netlink multiplex version!\n");
			kfree_skb(skb);
			goto out1;
		}
		BUG_ON(pipehdr->type != NETLINK_PIPE_TYPE_MESSAGE);

		/* remove the pipe multiplex header */
		skb_pull(skb, sizeof(struct netlink_pipe_msg_str));

		if(skb->len > 1) {
			datptr = kmalloc(skb->len,GFP_KERNEL);
			memcpy(datptr,skb->data,skb->len);

			cmd.cmd_id = E_TRACE;	
			cmd.cmd.trace.data_length = skb->len;
			cmd.cmd.trace.data = datptr;

			WIMAX_NO_WAIT_CMD(pc2400m_nl_sk3->sk_user_data, &cmd);
		}
		kfree_skb(skb);
		skb = skb_dequeue(&sk->sk_receive_queue);	
	}
 out1:
	return ;
}

void pc2400m_netlink_init(struct net_device *ndev) 
{
	struct net_local *nl = netdev_priv(ndev);

	DEBUG("Creating netlinks\n");

	pc2400m_nl_sk1 = netlink_kernel_create(NETLINK_PC2400M1, 
				       0, 
                                       nl_data_ready1, 
				       THIS_MODULE);
	if(!pc2400m_nl_sk1){
		ERROR("Failed to create netlink1\n");
		goto out1;
	}
	pc2400m_nl_sk1->sk_user_data = ndev;
	nl->tube1_pid = 0;

	pc2400m_nl_sk2 = netlink_kernel_create(NETLINK_PC2400M2, 
					     0, 	
					     nl_data_ready2, 
					     THIS_MODULE);
	if(!pc2400m_nl_sk2){
		ERROR("Failed to create netlink2\n");
		goto out1;
	}
	pc2400m_nl_sk2->sk_user_data = ndev;
	nl->tube2_pid = 0;

	pc2400m_nl_sk3 = netlink_kernel_create(NETLINK_PC2400M3, 
					     0, 
					     nl_data_ready3, 
					     THIS_MODULE);
	if(!pc2400m_nl_sk3){
		ERROR("Failed to create netlink3\n");
		goto out1;
	}
	pc2400m_nl_sk3->sk_user_data = ndev;
	nl->tube3_pid = 0;

 out1:
	return ;

}

void pc2400m_netlink_close(void)
{

	if (pc2400m_nl_sk1)
		sock_release(pc2400m_nl_sk1->sk_socket);
	if (pc2400m_nl_sk2)
		sock_release(pc2400m_nl_sk2->sk_socket);
	if (pc2400m_nl_sk3)
		sock_release(pc2400m_nl_sk3->sk_socket);

	pc2400m_nl_sk1 = NULL;
	pc2400m_nl_sk2 = NULL;
	pc2400m_nl_sk3 = NULL;

} 
