/*
 * pc2400m_drv_spi.c
 *
 * Copyright (C) 2007 Nokia Corporation
 * Author: Juuso Oikarinen <juuso.oikarinen@nokia.com>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */


#include "pc2400m_al.h"
#include "pc2400m_drv_sdio.h"
#include "pc2400m_drv_com.h"
#include "pc2400m_drv_if.h"
#include "pc2400m_drv_hdi.h"
#include "pc2400m_drv_dm.h"
#include "pc2400m_drv_spi.h"

#define PC2400M_REBOOT_BARKER     0xdeadbeef
#define PC2400M_H2D_BARKER        0xcafe900d
#define PC2400M_D2H_BARKER        0xbeefbabe

#define PC2400M_BLOCK_HEADER_PAT  0xfffffffc
#define PC2400M_BLOCK_FOOTER_PAT  0xffffffff

#ifdef CONFIG_NET_PC2400M_SPI_48MHZ
#define PC2400M_USE_HIGH_SPEED
#endif /* CONFIG_NET_PC2400M_SPI_48MHZ */

#define GENERIC_PAD(x,y)          (((x) + (y-1)) & (~(y-1)))
#define PAYLOAD_PAD(x)            GENERIC_PAD(x, PC2400M_FRAME_PAD)
#define BLOCK_PAD(x)              GENERIC_PAD(x, PC2400M_BLOCK_SIZE)
#define XFER_PAD(x)               GENERIC_PAC(x, 4)

#define PC2400M_BLOCK_HEADER      4
#define PC2400M_BLOCK_FOOTER      4
#define PC2400M_BLOCK_PADDING     (PC2400M_BLOCK_HEADER + PC2400M_BLOCK_FOOTER)

/* SPI transfer message header */
struct pc2400m_transfer_msg_hdr {
	u32 barker;
	u32 seq;
	u16 num_desc;
	u16 pad1;
	u16 pad_cnt;
	u16 pad2;
};

/* SPI transfer descriptor */
struct pc2400m_transfer_desc {
	u16 size;
	u16 type;
};

/* ASIC command header */
struct pc2400m_asic_hdr {
	u32 command;
	u32 address;
	u32 size;
	u32 checksum;
};



/**
 * pc2400m_drv_spi_put - queue a frame for SPI transmission
 * @data: the packet frame to queue for transmission
 * @type: the packet type to be set into the header
 *
 * Queue the provided packet frame for SPI transmission and assume ownership
 * of the frame. Returns the frame, if it does not fit into the transmission
 * buffer, otherwise NULL.
 */
static wimax_osal_packet *pc2400m_drv_spi_put(
	struct pc2400m_drv_hdi_if *hnd, 
	wimax_osal_packet *data, 
	s32 type) 
{

	struct pc2400m_drv_hdi_spi_if *spif = 
	        (struct pc2400m_drv_hdi_spi_if*)hnd;
	s32 i;

	wimax_osal_assert(data != NULL &&
			  (type & 0xfffffff0) == 0 &&
			  (wimax_osal_packet_size(data) & 0xffffc000) == 0);

	if (Q_FULL(spif->tx_q, PC2400M_TRANS_Q_SIZE))
		return data;
	
	i = PAYLOAD_PAD(wimax_osal_packet_size(data));
	if ((PC2400M_MAX_PAYLOAD-spif->tx_q_len) < i)
		return data;
	
	spif->tx_q_len += i;
	i = Q_PUT(spif->tx_q, PC2400M_TRANS_Q_SIZE);
	spif->tx_q[i].packet = data;
	spif->tx_q[i].type = type;
	
	return NULL;

}


/**
 * pc2400m_drv_spi_get - extract one frame from SPI transmission buffer
 * @hnd: handle to the spi instance
 * @type: the packet type read from the header
 *
 * Read one packet frame from the SPI transmission buffers. The returned
 * packet will be the responsibility of the called once received.
 */
static wimax_osal_packet *pc2400m_drv_spi_get(
        struct pc2400m_drv_hdi_if *hnd,
	s32*type)
{

	struct pc2400m_drv_hdi_spi_if *spif = 
	        (struct pc2400m_drv_hdi_spi_if*)hnd;
	wimax_osal_packet *ret;
	s32 i;

	if (Q_EMPTY(spif->rx_q))
		return NULL;
	
	i = Q_GET(spif->rx_q, PC2400M_TRANS_Q_SIZE);
	*type = spif->rx_q[i].type;
	ret = spif->rx_q[i].packet;
	spif->rx_q[i].packet = NULL;

	return ret;

}


/**
 * pc2400m_drv_spi_write - write an SPI message to SPI 
 * @ctx: handle to identify the physical device for the driver
 *
 * Synchronously transmit the requested number of data blocks onto
 * the SPI bus to the device specified by ctx.
 */
static s32 pc2400m_drv_spi_write(struct pc2400m_drv_hdi_if *hnd) 
{
        
	struct pc2400m_drv_hdi_spi_if *spif = 
	        (struct pc2400m_drv_hdi_spi_if*)hnd;

	s32 num_desc;
	s32 total;
	s32 pad;
	s32 i, ctr, len, size, ret;
	u8* ptr;
	u8* desc_ptr; 
	u8* pkt_ptr;
	struct pc2400m_transfer_msg_hdr *xfer_hdr;
	struct pc2400m_transfer_desc *xfer_desc;
	u32 hdr;
	u32 ftr;

	/* calculate padding needed */
	num_desc = Q_LEN(spif->tx_q, PC2400M_TRANS_Q_SIZE);
	total = PC2400M_TRANSPORT_HDR_SIZE + 
		PAYLOAD_PAD(num_desc * PC2400M_DESCRIPTOR_SIZE) +
		spif->tx_q_len;
	pad = BLOCK_PAD(total) - total;
	total = total + pad;
	
	ptr = spif->spi_data;
	ptr += PC2400M_BLOCK_HEADER;

	/* the transfer header and descriptors will never exceed the 
	   block limit */

	/* transfer message header */
	xfer_hdr = (struct pc2400m_transfer_msg_hdr*)ptr;
	xfer_hdr->barker = WIMAX_OSAL_U32_TO_LE(PC2400M_H2D_BARKER);
	xfer_hdr->seq = WIMAX_OSAL_U32_TO_LE(spif->tx_seq++);
	xfer_hdr->num_desc = WIMAX_OSAL_U16_TO_LE(num_desc);
	xfer_hdr->pad1 = 0;
	xfer_hdr->pad_cnt = WIMAX_OSAL_U16_TO_LE(pad);
	xfer_hdr->pad2 = 0;
	ptr += sizeof(struct pc2400m_transfer_msg_hdr);

	/* transfer descriptors & payload */
	desc_ptr = ptr;
	pad = num_desc * PC2400M_DESCRIPTOR_SIZE;
	ptr = desc_ptr + PAYLOAD_PAD(pad);
	ctr = ptr - spif->spi_data - PC2400M_BLOCK_HEADER;

	/* fill excess descriptor space with 0xff */
	wimax_osal_mem_set(desc_ptr + pad, 0xff, PAYLOAD_PAD(pad)-pad);

	while (!Q_EMPTY(spif->tx_q)) {
		i = Q_GET(spif->tx_q, PC2400M_TRANS_Q_SIZE);

		/* transfer descriptor */
		size = wimax_osal_packet_size(spif->tx_q[i].packet);
		pkt_ptr = wimax_osal_packet_ptr(spif->tx_q[i].packet);
		xfer_desc = (struct pc2400m_transfer_desc*)desc_ptr;
		xfer_desc->size = WIMAX_OSAL_U16_TO_LE((u16)size);
		xfer_desc->type = 
		        WIMAX_OSAL_U16_TO_LE((u16)spif->tx_q[i].type);
		desc_ptr += sizeof(struct pc2400m_transfer_desc);

		/* payload */
		while (size) {
			if (!(ctr % PC2400M_BLOCK_SIZE)) 
				ptr += PC2400M_BLOCK_PADDING;
			len = PC2400M_BLOCK_SIZE - (ctr % PC2400M_BLOCK_SIZE);
			if (len > size) len = size;
			wimax_osal_mem_cpy(ptr, pkt_ptr, len);
			size -= len;
			pkt_ptr += len;
			ptr += len;
			ctr += len;
		}

		size = wimax_osal_packet_size(spif->tx_q[i].packet);
		size = PAYLOAD_PAD(size) - size;
		
		while (size) {
			if (!(ctr % PC2400M_BLOCK_SIZE)) 
				ptr += PC2400M_BLOCK_PADDING;
			len = PC2400M_BLOCK_SIZE - (ctr % PC2400M_BLOCK_SIZE);
			if (len > size) len = size;
			wimax_osal_mem_set(ptr, 0xff, len);
			size -= len;
			ptr += len;
			ctr += len;
		}
		
		wimax_osal_packet_free(&(spif->tx_q[i].packet));
		spif->tx_q[i].packet = NULL;
	}

	/* add block padding if necessary */
	if (ctr % PC2400M_BLOCK_SIZE) {
		len = PC2400M_BLOCK_SIZE - (ctr % PC2400M_BLOCK_SIZE);
		wimax_osal_mem_set(ptr, 0xff, len);
		ctr += len;
	}
	
	/* add block headers and footers */
#if (PC2400M_BLOCK_HEADER != 4 || PC2400M_BLOCK_FOOTER != 4)
#error Code relies on block header sizes 4!
#endif
	ptr = spif->spi_data;
	hdr = WIMAX_OSAL_U32_TO_BE(PC2400M_BLOCK_HEADER_PAT);
	ftr = WIMAX_OSAL_U32_TO_BE(PC2400M_BLOCK_FOOTER_PAT);
	for (i=0; i<ctr; i+=PC2400M_BLOCK_SIZE) {
		*((u32*)ptr) = hdr;
		*((u32*)(ptr+PC2400M_BLOCK_HEADER+PC2400M_BLOCK_SIZE)) = ftr;
		ptr += (PC2400M_BLOCK_SIZE + PC2400M_BLOCK_PADDING);
	}

        /* queue a write command */
	ret = pc2400m_drv_send_write_cmd53(hnd, total/PC2400M_BLOCK_SIZE, 0,
					 spif->spi_data, ptr-spif->spi_data);
	spif->tx_q_len = 0;


        return ret;

}


static s32 spi_raw_write(struct pc2400m_drv_hdi_if *hnd, s32 len)
{

	struct pc2400m_drv_hdi_spi_if *spif = 
	        (struct pc2400m_drv_hdi_spi_if*)hnd;
	u8 *spi_data_ptr = spif->spi_data + len;
	u8 *ptr;
        s32 i,ret;
	u32 hdr;
	u32 ftr;

	/* transfer occurs in full blocks, pad if necessary, add
	   padding data and ready signals */
	if (len % PC2400M_BLOCK_SIZE) {
		i = PC2400M_BLOCK_SIZE - (len % PC2400M_BLOCK_SIZE);
		wimax_osal_mem_set(spi_data_ptr, 0xff, i);
		spi_data_ptr += i;
		len += i;
	}
	
	/* add intermediate headers to the mix */
	hdr = WIMAX_OSAL_U32_TO_BE(PC2400M_BLOCK_HEADER_PAT);
	ftr = WIMAX_OSAL_U32_TO_BE(PC2400M_BLOCK_FOOTER_PAT);
	spi_data_ptr -= PC2400M_BLOCK_SIZE;
	for (i=len/PC2400M_BLOCK_SIZE-1; i>=0; i--) {
		ptr = spif->spi_data + i * (PC2400M_BLOCK_SIZE + 
					    PC2400M_BLOCK_PADDING) + 
			PC2400M_BLOCK_HEADER;
		wimax_osal_mem_move(ptr, spi_data_ptr, PC2400M_BLOCK_SIZE);

		*((u32*)(ptr-4)) = hdr;
		*((u32*)(ptr+PC2400M_BLOCK_SIZE)) = ftr;

		spi_data_ptr -= PC2400M_BLOCK_SIZE;

	}
	
        /* queue a write command */
	ret = pc2400m_drv_send_write_cmd53(hnd, len/PC2400M_BLOCK_SIZE, 0,
					 spif->spi_data, 
					 (len/PC2400M_BLOCK_SIZE)*
					 (PC2400M_BLOCK_SIZE+8));
	spif->tx_q_len = 0;

        return ret;

}


static s32 spi_raw_read(struct pc2400m_drv_hdi_if *hnd) 
{

	struct pc2400m_drv_hdi_spi_if *spif = 
	        (struct pc2400m_drv_hdi_spi_if*)hnd;
	u8 *spi_data_ptr;
	u16 len;
	s32 i, block, ret = ESABABA;

#define COUNT_REG_OFFSET  0x1c
	len = (u16)pc2400m_drv_send_cmd52(hnd, COUNT_REG_OFFSET, 0, d2h, 
					func1);
	len |= ((u16)pc2400m_drv_send_cmd52
		(hnd, COUNT_REG_OFFSET+1, 0, d2h, func1)) << 8;

	if (!len) goto out;

	if (0 != len % PC2400M_BLOCK_SIZE) {
		wimax_osal_trace_u16(PC2400M_GROUP_SPI_ERROR, 
				     PC2400M_TRACE_NOT_FULL_BLOCKS, 
				     WIMAX_OSAL_PRIORITY_ERROR, len);
		goto err2;
	}

	if (pc2400m_drv_send_read_cmd53(hnd, len/PC2400M_BLOCK_SIZE, 0,
				      spif->spi_data, 
				      len + (len/PC2400M_BLOCK_SIZE) *
				      PC2400M_BLOCK_EXCESS) < 0)
		goto err1;

	/* parse the data */
	ret = len;
	spi_data_ptr = spif->spi_data + R_5_SIZE;
	block = 0;
	do {
		/* look for a start code */
		i = 0;
		while (*(spi_data_ptr++) != 0xfe) {
			i++;
			if (i >= PC2400M_BLOCK_EXCESS) {
				wimax_osal_trace_byte(
					PC2400M_GROUP_SPI_ERROR, 
					PC2400M_TRACE_OUT_OF_DATA, 
					WIMAX_OSAL_PRIORITY_ERROR, 
					i);
				goto err2;
			}
			if (spi_data_ptr - spif->spi_data >=
			    PC2400M_DATA_SIZE - PC2400M_BLOCK_SIZE) {
				wimax_osal_trace(
					PC2400M_GROUP_SPI_ERROR, 
					PC2400M_TRACE_OUT_OF_DATA, 
					WIMAX_OSAL_PRIORITY_ERROR);
				goto err2;
			}
		}
		
		wimax_osal_mem_move(spif->spi_data + block, spi_data_ptr, 
				    PC2400M_BLOCK_SIZE);
		block += PC2400M_BLOCK_SIZE;
		spi_data_ptr += PC2400M_BLOCK_SIZE;
		len -= PC2400M_BLOCK_SIZE;

	} while (len);
	
	goto out;

 err1:
	ret = -EIO;
 err2:

 out:

	return ret;

}


static u8 *find_next_block(u8 *ptr, u8 *end) {

	s32 i = 0;
	while (*(ptr++) != 0xfe) {
		i++;
		if (i >= PC2400M_BLOCK_EXCESS) {
			wimax_osal_trace_byte(PC2400M_GROUP_SPI_ERROR, 
					      PC2400M_TRACE_OUT_OF_DATA, 
					      WIMAX_OSAL_PRIORITY_ERROR,
					      i);
			goto err1;
		}
		if (ptr + PC2400M_BLOCK_SIZE >= end) {
			wimax_osal_trace(PC2400M_GROUP_SPI_ERROR, 
					 PC2400M_TRACE_OUT_OF_DATA, 
					 WIMAX_OSAL_PRIORITY_ERROR);
			goto err1;
		}
	}

	goto out;
 err1:
	ptr = NULL;
 out:
	return ptr;
	
}

/**
 * pc2400m_drv_spi_read - read an SPI message from SPI
 * @ctx: handle to identify the physical device for the driver
 *
 * Synchronously receive the requested number of data blocks from
 * the SPI bus to the device specified by ctx.
 */
static s32 pc2400m_drv_spi_read(struct pc2400m_drv_hdi_if *hnd) 
{

	struct pc2400m_transfer_msg_hdr *xfer_hdr;
	struct pc2400m_transfer_desc *xfer_desc;
	struct pc2400m_drv_hdi_spi_if *spif = 
	        (struct pc2400m_drv_hdi_spi_if*)hnd;
	struct pc2400m_private *priv = wimax_osal_ctx_priv_get(hnd->ctx);
	u16 len;
	s32 ret = ESABABA;
	s32 total, i;
	u32 barker;
	u32 seq;
	u16 num_desc;
	u16 btype;
	u16 blength;
	u16 pad_blength;
	u32 value;
	u8 *start;
	u8 *ptr;
	u8 *desc_ptr;
	wimax_osal_packet *packet;

#define COUNT_REG_OFFSET  0x1c
	len = (u16)pc2400m_drv_send_cmd52(hnd, COUNT_REG_OFFSET, 0, d2h, 
					func1);
	len |= ((u16)pc2400m_drv_send_cmd52
		(hnd, COUNT_REG_OFFSET+1, 0, d2h, func1)) << 8;
	if (!len) goto out;

	if (0 != len % PC2400M_BLOCK_SIZE) {
		wimax_osal_trace_u16(PC2400M_GROUP_SPI_ERROR, 
				     PC2400M_TRACE_NOT_FULL_BLOCKS, 
				     WIMAX_OSAL_PRIORITY_ERROR, len);
		goto err1;
	}

	total = len + (len/PC2400M_BLOCK_SIZE) * PC2400M_BLOCK_EXCESS;
	if (pc2400m_drv_send_read_cmd53(hnd, len/PC2400M_BLOCK_SIZE, 0,
				      spif->spi_data, total) < 0)
		goto err1;

	ret = len;
	ptr = spif->spi_data + R_5_SIZE;

	/* parse the transfer header */
	if (!(start = find_next_block(ptr, spif->spi_data + total))) {
		wimax_osal_trace(PC2400M_GROUP_SPI_ERROR, 
				 PC2400M_TRACE_CORRUPT_XFER_HDR, 
				 WIMAX_OSAL_PRIORITY_ERROR);
		goto err1;
	}
	
	ptr = start;
	
	/* the assumption: the block will start at correct alignment */
	xfer_hdr = (struct pc2400m_transfer_msg_hdr*)ptr;
	barker = WIMAX_OSAL_LE_TO_U32(xfer_hdr->barker);
	seq = WIMAX_OSAL_LE_TO_U32(xfer_hdr->seq);
	num_desc = WIMAX_OSAL_LE_TO_U16(xfer_hdr->num_desc);
	ptr += sizeof(struct pc2400m_transfer_msg_hdr);

	if (barker == PC2400M_REBOOT_BARKER) {
		wimax_osal_trace(PC2400M_GROUP_SPI_ERROR, 
				 PC2400M_TRACE_DEVICE_REBOOT, 
				 WIMAX_OSAL_PRIORITY_ERROR);
		priv->dm->abort_mission(priv->dm);
		goto err1;
	}

	if (barker != PC2400M_D2H_BARKER) {
		wimax_osal_trace_u32(PC2400M_GROUP_SPI_ERROR, 
				     PC2400M_TRACE_CORRUPT_XFER_HDR, 
				     WIMAX_OSAL_PRIORITY_ERROR, 
				     barker);
		goto err1;
	}

	if (seq != spif->rx_seq++) {
		wimax_osal_trace_u32(PC2400M_GROUP_SPI_ERROR,
				     PC2400M_TRACE_RX_SYNC_LOST, 
				     WIMAX_OSAL_PRIORITY_ERROR, 
				     spif->rx_seq-1);
		spif->rx_seq = seq + 1;
	}

	if (num_desc == 0) {
		wimax_osal_trace(PC2400M_GROUP_SPI_ERROR, 
				 PC2400M_TRACE_RX_EMPTY,
				 WIMAX_OSAL_PRIORITY_ERROR);
	}

	/* start parsing the descriptors, and read the appropriate data */
	desc_ptr = ptr;
	ptr = ptr + PAYLOAD_PAD(num_desc * PC2400M_DESCRIPTOR_SIZE);

	for (i=0; i<num_desc; i++) {
		xfer_desc = (struct pc2400m_transfer_desc*)desc_ptr;
		pad_blength = WIMAX_OSAL_LE_TO_U16(xfer_desc->size);
		btype = WIMAX_OSAL_LE_TO_U16(xfer_desc->type);
		desc_ptr += sizeof(struct pc2400m_transfer_desc);

		blength = PAYLOAD_PAD(pad_blength);
		pad_blength = blength - pad_blength;
		packet = wimax_osal_packet_alloc(blength);

		while (blength) {
			len = PC2400M_BLOCK_SIZE - (ptr-start);
			if (len > blength) len = blength;
			wimax_osal_mem_cpy(wimax_osal_packet_put(packet, len), 
					   ptr, len);
			ptr += len;
			blength -= len;

			if (blength) {
				/* a block is always followed by four status
				   bytes (which could be misinterpreted as
				   the "ready" code for the next block) */
				ptr += 4;
				if (!(start = 
				      find_next_block(
					      ptr, spif->spi_data + total))) {
					goto err2;
				}
				ptr = start;
			} else {
				wimax_osal_packet_trim
					(packet, 
					 wimax_osal_packet_size(packet) - 
					 pad_blength);
			}
		}

		wimax_osal_assert(!Q_FULL(spif->rx_q, PC2400M_TRANS_Q_SIZE));
		value = Q_PUT(spif->rx_q, PC2400M_TRANS_Q_SIZE);
		spif->rx_q[value].packet = packet;
		spif->rx_q[value].type = btype;
		
	}

	goto out;

 err2:
	wimax_osal_packet_free(&packet);
 err1:
	ret = -EIO;

 out:

	return ret;

}

/**
 * pc2400m_drv_spi_init - send initialization commands to the chipset
 * @ctx: handle to identify the physical device for the driver
 *
 * Synchronously initialize the chipset and its SPI/SDIO controller
 */
static s32 pc2400m_drv_spi_init(struct pc2400m_drv_hdi_if *hnd) 
{
	
	struct R_4 r4;
	u32 ocr;

	/* perform a card reset */
	if (pc2400m_drv_send_cmd0(hnd))
		return -EIO;

	/* query operating conditions register */
	ocr = pc2400m_drv_send_cmd5(hnd, 0, &r4 );

	/* Pc2400m always returns this value */
#define PC2400M_FIXED_OCR   0x000FF8000
	wimax_osal_assert( PC2400M_FIXED_OCR == ocr);

	/* offer the same operating conditions to the card until it is
	   ready */
	do
	{
		pc2400m_drv_send_cmd5(hnd, ocr, &r4 );
	}
	while( !r4.u.bits.card_bit);

	/* enable interrupts */
	pc2400m_drv_send_cmd52(hnd, 4, 3, h2d, func0 );
	/* enable I/O */
	pc2400m_drv_send_cmd52(hnd, 2, 2, h2d, func0 );
	/* set I/O ready */
	pc2400m_drv_send_cmd52(hnd, 3, 0, h2d, func0 ); 
	/* card capability */
	pc2400m_drv_send_cmd52(hnd, 8, 0x20, h2d, func0);
#ifdef PC2400M_USE_HIGH_SPEED
	/* configure for full speed SPI transfers */
	pc2400m_drv_send_cmd52(hnd, 0x13, 3, h2d, func0);
	pc2400m_drv_adapt.configure(hnd->ctx, 48000000, 
				  SPI_CONFIG_SAMPLE_FALLING);
#endif /* PC2400M_USE_HIGH_SPEED */
	/* set cmd53 block size */
	pc2400m_drv_send_cmd52(hnd, 0x110, PC2400M_BLOCK_SIZE&0xFF, h2d, 
			     func0 );
	pc2400m_drv_send_cmd52(hnd, 0x111, (PC2400M_BLOCK_SIZE&0xFF00)>>8, 
			     h2d, func0 );

	return ESABABA;

}

/**
 * pc2400m_drv_spi_fw_start - start the firmware at the specified location
 * @ctx: handle to identify the physical device for the driver
 * @addr: address to start firmware at
 *
 * Issue a jump command to start the firmware at the specified address
 */
static s32 pc2400m_drv_spi_fw_start(struct pc2400m_drv_hdi_if *hnd, u32 addr)
{

#define PC2400M_CMD_JUMP     0xcbbc0503
#define PC2400M_FW_HEADER    16
#define PC2400M_FW_PAYLOAD   2048
#define PC2400M_FW_BLOCK_LEN (PC2400M_FW_PAYLOAD + PC2400M_FW_HEADER)

	struct pc2400m_drv_hdi_spi_if *spif = 
	        (struct pc2400m_drv_hdi_spi_if*)hnd;
	struct pc2400m_asic_hdr *asic_hdr;
	u32 command = PC2400M_CMD_JUMP;
	u32 length = 0;
	u32 checksum = command + addr + length;
	s32 ret = ESABABA;

	asic_hdr = (struct pc2400m_asic_hdr*)spif->spi_data;
	asic_hdr->command = WIMAX_OSAL_U32_TO_LE(command);
	asic_hdr->address = WIMAX_OSAL_U32_TO_LE(addr);
	asic_hdr->size = WIMAX_OSAL_U32_TO_LE(length);
	asic_hdr->checksum = WIMAX_OSAL_U32_TO_LE(checksum);
	
	/* off with the block! */
	if (spi_raw_write(hnd, PC2400M_FW_HEADER) < 0) goto err;

	goto out;
 err:
	ret = -EIO;

 out:
	return ret;

}

/**
 * pc2400m_drv_spi_fw_dl_block - download a contiguous firmware block
 * @ctx: handle to identify the physical device for the driver
 * @block: pointer to the contiguous firmware block
 * @addr: address to download the firmware to
 * @len: length of the block
 *
 * Upload the firmware block to the chipset to the specified address
 */
static s32 pc2400m_drv_spi_fw_dl_block(
        struct pc2400m_drv_hdi_if *hnd, 
	u8* block, 
	u32 addr, 
	u32 len,
	u32 no_checksum,
	u32 no_da)
{

#define PC2400M_CMD_WRITE         0xcbbc0702
#define PC2400M_CMD_CHECKSUM_BIT  0x00000100
#define PC2400M_CMD_DIRECTAC_BIT  0x00000400

#define PC2400M_FW_ACK_WAIT_MAX   1000

	struct pc2400m_drv_hdi_spi_if *spif = 
	        (struct pc2400m_drv_hdi_spi_if*)hnd;
	struct pc2400m_asic_hdr *asic_hdr;
	u32 command;
	u32 checksum;
	u32 length;
	s32 i = 0; 
	s32 ret = ESABABA;

	wimax_osal_trace_u32(PC2400M_GROUP_SPI_DEBUG, 
			     PC2400M_TRACE_FW_BLOCK_ADDRESS,
			     WIMAX_OSAL_PRIORITY_DEBUG,
			     addr);
	wimax_osal_trace_u32(PC2400M_GROUP_SPI_DEBUG, 
			     PC2400M_TRACE_FW_BLOCK_SIZE,
			     WIMAX_OSAL_PRIORITY_DEBUG,
			     len);
	
	while (len) {

		/* construct the firmware block */
		length = (len<PC2400M_FW_PAYLOAD?len:PC2400M_FW_PAYLOAD);
		wimax_osal_mem_cpy(
			spif->spi_data + PC2400M_FW_HEADER, block, length);
		block += length;
		len -= length;

		/* determine the command */
		command = PC2400M_CMD_WRITE;

		if ((length & 0xf) || no_da) {
			/* if not divisible by 16, disable direct access */
			command &= (~PC2400M_CMD_DIRECTAC_BIT);
		}		

		if ((length & 0x3) || no_checksum) {
			/* cannot calc checksum, not divisible by four */
			command &= (~PC2400M_CMD_CHECKSUM_BIT);
			checksum = 0;
		} else {
			/* calculate a checksum */
			checksum = command + addr + length;
			for (i=0; i<length; i+=4) {
				/* the checksum must be counted with the values
				   as would be seen by a little endian 
				   machine */
				checksum += WIMAX_OSAL_LE_TO_U32(
					*((u32*)(spif->spi_data+i+
						 PC2400M_FW_HEADER)));
			}
		}

		/* fill in the header */
		asic_hdr = (struct pc2400m_asic_hdr*)spif->spi_data;
		asic_hdr->command = WIMAX_OSAL_U32_TO_LE(command);
		asic_hdr->address = WIMAX_OSAL_U32_TO_LE(addr);
		asic_hdr->size = WIMAX_OSAL_U32_TO_LE(length);
		asic_hdr->checksum = WIMAX_OSAL_U32_TO_LE(checksum);

		addr += length;

		/* off with the block! */
		if (spi_raw_write(hnd, 
				  PC2400M_FW_HEADER + PAYLOAD_PAD(length)) < 0)
			goto err;

		/* wait for the response */
		i = 0;
		do {
			ret = pc2400m_drv_send_cmd52(hnd, 5, 0, d2h, func0);
			if (ret < 0) goto err;
			if (i++ > PC2400M_FW_ACK_WAIT_MAX)
				goto err;
		} while (!(ret & 0x02));
		
		/* clear the interrupt */
		ret = pc2400m_drv_send_cmd52(hnd, 4, 0, d2h, func1);
		if (ret < 0) goto err;

		if (spi_raw_read(hnd)<0) goto err;
		
		if (WIMAX_OSAL_LE_TO_U32(*((u32*)spif->spi_data)) ==
		    PC2400M_REBOOT_BARKER) {
			goto err;
		}

	}

	goto out;

 err:
	/* SPI synchronization has been lost */
	wimax_osal_trace(PC2400M_GROUP_SPI_ERROR, 
			 PC2400M_TRACE_DEVICE_REBOOT, 
			 WIMAX_OSAL_PRIORITY_ERROR);
	ret = -ENODEV;
 out:
	return ret;

}

/**
 * pc2400m_drv_spi_fw_dl - download firmware images to the chipset
 * @ctx: handle to identify the physical device for the driver
 * @fw: pointer to a buffer containing the firmware image script
 * @len: length of the firmware image script
 *
 * Upload the firmware image script consisting of multiple separate firmware
 * blocks to the chipset - requires that the chipset has been initialized 
 * first, and that no firmware is running.
 */
static s32 pc2400m_drv_spi_fw_dl(
        struct pc2400m_drv_hdi_if *hnd, 
	u8* fw, 
	u32 len) 
{

#define PC2400M_FW_FLAG_NO_CHECKSUM       0x80000000
#define PC2400M_FW_FLAG_NO_DIRECT_ACCESS  0x40000000
#define PC2400M_FW_FLAG_MASK              0xc0000000         

#define PC2400M_FW_COOKIE  0xbabeface	

	s32 ret = ESABABA;
	s32 cursize = 0;
	u32 length;
	u32 flags;
	u32 addr;
	u32 jump_addr = 0;

	/* check the validity of the firmware */
	if (WIMAX_OSAL_GET_BE32(fw) != PC2400M_FW_COOKIE) {
		wimax_osal_trace_data(PC2400M_GROUP_SPI_ERROR, 
				      PC2400M_TRACE_FW_INVALID,
				      WIMAX_OSAL_PRIORITY_ERROR,
				      fw, sizeof(u32));
		goto err1;
	}
	cursize+=sizeof(u32);

	/* start parsing the firmware */
	while(cursize < len) {
		addr = WIMAX_OSAL_GET_LE32(fw+cursize);
		flags = WIMAX_OSAL_GET_LE32(fw+cursize+sizeof(addr));
		length = flags & (~PC2400M_FW_FLAG_MASK);
		
		if (length) {
			ret = pc2400m_drv_spi_fw_dl_block(
				hnd, 
				fw+cursize+sizeof(length)+sizeof(addr),
				addr, 
				length,
				flags & PC2400M_FW_FLAG_NO_CHECKSUM,
				flags & PC2400M_FW_FLAG_NO_DIRECT_ACCESS);
			if (ret < 0) goto err2;
		} else {
			jump_addr = addr;
		}
		
		cursize+=(length + sizeof(length) + sizeof(addr));
	}

	/* start the firmware */
	wimax_osal_trace_u32(PC2400M_GROUP_SPI_DEBUG, 
			     PC2400M_TRACE_FW_JUMPING,
			     WIMAX_OSAL_PRIORITY_DEBUG,
			     jump_addr);
	
	if (jump_addr)
		ret = pc2400m_drv_spi_fw_start(hnd, jump_addr);
	else
		goto err1;

	goto out;

 err1:
	ret = -EINVAL;
 err2:
	
 out:
	return ret;

} 


static void pc2400m_drv_spi_irq(wimax_osal_context *ctx)
{

	struct pc2400m_private *priv = 
		(struct pc2400m_private*)wimax_osal_ctx_priv_get(ctx);
	struct pc2400m_drv_hdi_spi_if *spif = 
	        (struct pc2400m_drv_hdi_spi_if*)priv->dm->hdi;

	priv->ref(priv);

	/* check interrupt */
	if (!(pc2400m_drv_send_cmd52(priv->dm->hdi, 5, 0, d2h, func0) & 
	      0x02)) {
		wimax_osal_trace(PC2400M_GROUP_SPI_ERROR, 
				 PC2400M_TRACE_FAKE_IRQ,
				 WIMAX_OSAL_PRIORITY_ERROR);
		goto out;
	}
	
	/* acknowledge interrupt */
	if (!(pc2400m_drv_send_cmd52(priv->dm->hdi, 4, 0, d2h, func1) & 
	      0x01)) {
		wimax_osal_trace(PC2400M_GROUP_SPI_ERROR, 
				 PC2400M_TRACE_UNSET_IRQ_FLAG,
				 WIMAX_OSAL_PRIORITY_ERROR);
		goto out;
	}

	/* MISSING: how to avoid a flood of queued work? */
	spif->irq_cb(ctx);

 out:

	priv->unref(priv);
	return;

}


/**
 * pc2400m_drv_spi_initialize - perform full card initialization
 * @ctx: handle to identify the physical device for the driver
 * @fw: pointer to a buffer containing the firmware image
 * @len: length of the firmware image
 *
 * Perform full card initialization upon bootup, including card
 * initialization, and firmware download.
 */
static s32 pc2400m_drv_spi_initialize(
	struct pc2400m_drv_hdi_if *hnd, 
	u8* fw, 
	u32 len,
	pc2400m_hdi_int_cb callback)
{
	
#define PC2400M_FW_DL_RETRIES    5

	struct pc2400m_drv_hdi_spi_if *spif = 
	        (struct pc2400m_drv_hdi_spi_if*)hnd;
	s32 retries = PC2400M_FW_DL_RETRIES;
	s32 ret = ESABABA;

	wimax_osal_trace_u32(PC2400M_GROUP_SPI_INFO, 
			     PC2400M_TRACE_FW_DL, 
			     WIMAX_OSAL_PRIORITY_DEBUG, len);
	
	while (retries) {
		/* power on the chipset */
		pc2400m_drv_adapt.set_power(hnd->ctx, 1);

		/* perform SPI/SDIO init */
		ret = pc2400m_drv_spi_init(hnd);
		if (ret < 0) {
			wimax_osal_trace(PC2400M_GROUP_SPI_ERROR, 
					 PC2400M_TRACE_INIT_ERR, 
					 WIMAX_OSAL_PRIORITY_ERROR);
		} else {
			/* download firmware */
			if (!pc2400m_drv_spi_fw_dl(hnd, fw, len))
				break;
		}

		/* power of the chipset to reboot */
		pc2400m_drv_adapt.set_power(hnd->ctx, 0);
		msleep(10);

		wimax_osal_trace_byte(PC2400M_GROUP_SPI_ERROR, 
				      PC2400M_TRACE_FW_DL_RETRY,
				      WIMAX_OSAL_PRIORITY_DEBUG, retries);
		retries--;
	}

	if (!retries) {
		wimax_osal_trace(PC2400M_GROUP_SPI_ERROR, 
				 PC2400M_TRACE_FW_DL_FAIL,
				 WIMAX_OSAL_PRIORITY_DEBUG);
		ret = -EIO;
	} else {
		/* register the interrupt handler */
		if (callback) {
			spif->irq_cb = callback;
			pc2400m_drv_adapt.enable_irq(
				hnd->ctx, pc2400m_drv_spi_irq);
			ret = ESABABA;
		}
	}

	return ret;

}

static void pc2400m_drv_spi_reset(struct pc2400m_drv_hdi_if *hnd)
{

	struct pc2400m_drv_hdi_spi_if *spif = 
	        (struct pc2400m_drv_hdi_spi_if*)hnd;
	s32 i;

	/* disconnect interrupts */
	if (spif->irq_cb) {
	        spif->irq_cb = NULL;
		pc2400m_drv_adapt.enable_irq(hnd->ctx, NULL);
	}

	/* free queued blocks and packets */
	while (!Q_EMPTY(spif->tx_q)) {
		i = Q_GET(spif->tx_q, PC2400M_TRANS_Q_SIZE);
		wimax_osal_packet_free(&(spif->tx_q[i].packet));
	}

	while (!Q_EMPTY(spif->rx_q)) {
		i = Q_GET(spif->rx_q, PC2400M_TRANS_Q_SIZE);
		wimax_osal_packet_free(&(spif->rx_q[i].packet));
	}
		
	spif->tx_seq = 0;
	spif->rx_seq = 0;

	return;

}

static void pc2400m_drv_spi_cleanup(struct pc2400m_drv_hdi_if *hnd)
{

	/* free resources, cancel ongoing actions */
	pc2400m_drv_spi_reset(hnd);

	/* free the instance */
	wimax_osal_mem_free(&hnd);
	return;

}


struct pc2400m_drv_hdi_if *pc2400m_drv_hdi_if_new(wimax_osal_context* ctx)
{

	struct pc2400m_drv_hdi_spi_if *iface;

	iface = wimax_osal_mem_alloc_ex(sizeof(struct pc2400m_drv_hdi_spi_if),
					WIMAX_OSAL_MEM_TYPE_DMA);
	wimax_osal_mem_set(iface, 0x00, sizeof(struct pc2400m_drv_hdi_spi_if));

	/* HDI generic part */
	iface->hdi.ctx = ctx;
	iface->hdi.put = pc2400m_drv_spi_put;
	iface->hdi.get = pc2400m_drv_spi_get;
	iface->hdi.write = pc2400m_drv_spi_write;
	iface->hdi.read = pc2400m_drv_spi_read;
	iface->hdi.initialize = pc2400m_drv_spi_initialize;
	iface->hdi.cleanup = pc2400m_drv_spi_cleanup;
	iface->hdi.reset = pc2400m_drv_spi_reset;
	
	/* SPI part */
	/* SPI vars are currently initialized to zero */

	return (struct pc2400m_drv_hdi_if*)iface;

}
