/*
 * linux/arch/arm/plat-omap/sec.c
 *
 * Copyright (C) 2007 Nokia Corporation
 * Author: Sami Tolvanen <sami.tolvanen@nokia.com>
 *
 * OMAP HS secure mode driver
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/miscdevice.h>
#include <linux/mutex.h>
#include <linux/poll.h>
#include <linux/stat.h>
#include <asm/arch/sec.h>

MODULE_LICENSE("GPL v2");

#define SEC_NAME		"omap_sec"

#define SEC_MAX_DATA_SIZE	SECS_SIZE
#define SEC_MAX_OBUF_SIZE	16384

#define SECS_SLOTS		2	/* Max. simultaneous CMD_SECS */
#define SECS_SIZE		4096	/* >= 4096 bytes */

enum {
	CMD_NONE,
	CMD_INIT,
	CMD_SECS,
	CMD_RANDOM,
	CMD_PA_FIRST = 0x100,
};

typedef struct {
	u32 cmd;
	u32 length;
} cmd_param;

typedef struct {
	u32 length;
	u8 data[SECS_SIZE];
} cmd_op_secs;

enum {
	RPC_SETTINGS,
	RPC_ALLOC,
	RPC_DEALLOC,
	RPC_SECS_WRITE,
};

typedef struct {
	struct mutex lock;
	u32 cmd;		/* Command identifier */
	cmd_param param;
	u8 *ip;			/* Input pointer */
	u8 *data;		/* Parameter data */
	u8 *obuf;		/* Output buffer */
	u8 *rp;			/* Read position */
	u8 *wp;			/* Write position */
	u8 *end;
	wait_queue_head_t inq;
	wait_queue_head_t outq;
} sec_status;

typedef struct {
	struct mutex lock; /* For secsd */
	sec_status *secsd[SECS_SLOTS];
	struct device dev;
	struct miscdevice misc_dev;
} sec_device;

typedef struct {
	struct mutex lock;
	sec_hal_operations *ops;
} sec_hal;

static int sec_minor = MISC_DYNAMIC_MINOR;
static sec_device dev;
static sec_hal hal;
static void *storage_ptr = NULL;

module_param(sec_minor, uint, S_IRUGO);

static u32 rpc_handler(u32 service, u32 a1, u32 a2, u32 a3);

/*
 * Treats ss->obuf as a circular buffer and copies size bytes of data
 * to it starting from ss->wp. Doesn't care if ss->rp overflows.
 */
static u8 * obufcpy(sec_status *ss, u8 *src, size_t size)
{
	size_t count;

	if (ss->wp == ss->end)
		ss->wp = ss->obuf;

	count = min(size, (size_t)(ss->end - ss->wp));

	memcpy(ss->wp, src, count);
	ss->wp += count;

	if (count < size)
		obufcpy(ss, src + count, size - count);

	return ss->wp;
}

static void reset_status(sec_status *ss)
{
	if (ss->data) {
		kfree(ss->data);
		ss->data = NULL;
	}
	if (ss->obuf) {
		kfree(ss->obuf);
		ss->obuf = NULL;
	}
	ss->cmd = CMD_NONE;
	ss->ip   = (u8 *)&ss->param;
	ss->rp   = NULL;
	ss->wp   = NULL;
	ss->end  = NULL;
}

static void * set_obuf(sec_status *ss, size_t size)
{
	if (ss->obuf) {
		kfree(ss->obuf);
		ss->obuf = NULL;
		ss->rp   = NULL;
		ss->wp   = NULL;
		ss->end  = NULL;
	}

	if (size > SEC_MAX_OBUF_SIZE)
		return NULL;

	ss->obuf = kzalloc(size, GFP_KERNEL);

	if (ss->obuf) {
		ss->rp  = ss->obuf;
		ss->wp  = ss->obuf;
		ss->end = ss->obuf + size;
	}

	return ss->obuf;
}

/*
 * Copies ss->obuf contents to user space from ss->rp to ss->wp (at most). This
 * never blocks the reading process and assumes there is data to read. Once all
 * data has been read, the command for the descriptor is completed and if there
 * are any processes waiting to write another command, they are woken up.
 */
static ssize_t read_obuf(sec_status *ss, char __user *buf, size_t count)
{
	ssize_t rv = -EIO;

	if (ss->wp <= ss->rp)
		goto done;

	count = min(count, (size_t)(ss->wp - ss->rp));

	if (copy_to_user(buf, ss->rp, count)) {
		rv = -EFAULT;
		goto done;
	}

	ss->rp += count;
	rv = count;

	if (ss->wp == ss->rp)
		goto done;

	mutex_unlock(&ss->lock);
	return rv;

done:	/* Command completed, wake up any pending writers */
	reset_status(ss);
	wake_up_interruptible(&ss->inq);
	mutex_unlock(&ss->lock);
	return rv;
}

/*
 * Treats ss->obuf as a circular buffer, copies available contents to user
 * space and blocks if nothing is available. CMD_SECS never completes.
 */
static ssize_t read_secs(struct file *filp, sec_status *ss,
		char __user *buf, size_t count)
{
	/* Block until there is data in the output buffer */
	while (ss->wp == ss->rp) {
		mutex_unlock(&ss->lock);

		if (filp->f_flags & O_NONBLOCK)
			return -EAGAIN;

		if (wait_event_interruptible(ss->outq, (ss->wp != ss->rp)))
			return -ERESTARTSYS;

		if (mutex_lock_interruptible(&ss->lock))
			return -ERESTARTSYS;
	}

	if (ss->rp == ss->end)
		ss->rp = ss->obuf;

	if (ss->wp > ss->rp)
		count = min(count, (size_t)(ss->wp  - ss->rp));
	else
		count = min(count, (size_t)(ss->end - ss->rp));

	if (copy_to_user(buf, ss->rp, count)) {
		mutex_unlock(&ss->lock);
		return -EFAULT;
	}

	ss->rp += count;

	mutex_unlock(&ss->lock);
	return count;
}

/* Reads command-specific results from ss->obuf, blocks if nothing to read */
static ssize_t sec_read(struct file *filp, char __user *buf, size_t count,
		loff_t *f_pos)
{
	sec_status *ss = (sec_status *)filp->private_data;

	if (mutex_lock_interruptible(&ss->lock))
		return -ERESTARTSYS;

	/*
	 * There is always something to read when we have a command, or the
	 * command-specific read function does the blocking.
	 */
	while (ss->cmd == CMD_NONE) {
		mutex_unlock(&ss->lock);

		if (filp->f_flags & O_NONBLOCK)
			return -EAGAIN;

		if (wait_event_interruptible(ss->outq, (ss->cmd != CMD_NONE)))
			return -ERESTARTSYS;

		if (mutex_lock_interruptible(&ss->lock))
			return -ERESTARTSYS;
	}

	if (ss->cmd == CMD_SECS)
		return read_secs(filp, ss, buf, count);
	else
		return read_obuf(ss, buf, count);
}

/*
 * Writes secure storage buffer contents to the ss->obuf for those processes
 * who have requested it (CMD_SECS). If the ss->obuf buffer is full, simply
 * skip the process.
 */
static void write_secs(void)
{
	int i;
	size_t space;
	sec_status *ss;
	u32 size = SECS_SIZE;

	mutex_lock(&dev.lock);

	for (i = 0; i < SECS_SLOTS; ++i) {
 		ss = dev.secsd[i];

		if (!ss || !ss->obuf)
			continue;

		mutex_lock(&ss->lock);

		/* Rewind to replace the first unread block */
		if (ss->wp > ss->rp) {
			while ((ss->wp - ss->rp) > sizeof(cmd_op_secs))
				ss->wp -= sizeof(cmd_op_secs);

			space = (ss->end - ss->wp) + (ss->rp - ss->obuf);
		} else if (ss->wp < ss->rp) {
			while ((ss->wp - ss->obuf) > sizeof(cmd_op_secs))
				ss->wp -= sizeof(cmd_op_secs);

			space = ss->rp - ss->wp;
		} else {
			space = ss->end - ss->obuf;
		}

		if (space >= sizeof(cmd_op_secs)) {
			obufcpy(ss, (u8 *)&size, sizeof(size));
			obufcpy(ss, (u8 *)storage_ptr, size);
			wake_up_interruptible(&ss->outq);
		}

		mutex_unlock(&ss->lock);
	}

	mutex_unlock(&dev.lock);
}

static int process_cmd_init(sec_status *ss)
{
	sec_hal_par_secenv_init par;
	u32 *op;

	if (!hal.ops)
		return -ENODEV;

	if (ss->param.length > SECS_SIZE)
		return -EFBIG;

	op = set_obuf(ss, sizeof(*op));
	if (!op)
		return -ENOMEM;

	/* Initial secure storage */
	memcpy(storage_ptr, ss->data, ss->param.length);

	memset(&par, 0, sizeof(par));
	par.rpc_ptr = rpc_handler;
	par.storage_ptr	= storage_ptr;
	par.length = SECS_SIZE;

	*op = hal.ops->secenv_init(&par);
	ss->wp += sizeof(*op);

	wake_up_interruptible(&ss->outq);
	return 0;
}

static int process_cmd_secs(sec_status *ss)
{
	int rv, i;

	/*
	 * If >= 2 * sizeof(cmd_op_secs), there is always enough space
	 * for secure storage in the buffer.
	 */
	if (!set_obuf(ss, 2 * sizeof(cmd_op_secs)))
		return -ENOMEM;

	/* Set the descriptor to receive secure storage contents. */
	rv = -EBUSY;
	mutex_lock(&dev.lock);

	for (i = 0; i < SECS_SLOTS; ++i) {
		if (!dev.secsd[i]) {
			dev.secsd[i] = ss;
			rv = 0; /* Found a slot */
			break;
		}
	}

	mutex_unlock(&dev.lock);

	/*
	 * Release any pending writers that may have been competing for
	 * descriptor access.
	 */
	if (!rv)
		wake_up_interruptible(&ss->inq);

	return rv;
}

static int process_cmd_random(sec_status *ss)
{
	sec_hal_par_random_get par;
	u32 *length = (u32 *)ss->data;
	u32 *op;

	if (!hal.ops)
		return -ENODEV;

	if (ss->param.length != sizeof(*length))
		return -EFBIG;

	op = set_obuf(ss, sizeof(*op) + *length);
	if (!op)
		return -ENOMEM;

	memset(&par, 0, sizeof(par));
	par.data = (u8 *)(op + 1);
	par.length = *length;

	*op = hal.ops->random_get(&par);
	if (*op == SEC_HAL_OK)
		ss->wp += *length;

	ss->wp += sizeof(*op);

	wake_up_interruptible(&ss->outq);
	return 0;
}

static int process_cmd_pa(sec_status *ss)
{
	u32 *op;
	u32 cpar, cres;

	if (!hal.ops)
		return -ENODEV;

	if (hal.ops->pa_query(ss->cmd, ss->data, ss->param.length,
			&cpar, &cres) != SEC_HAL_OK)
		return -EACCES;

	if (ss->param.length != cpar)
		return -EFBIG;

	op = set_obuf(ss, sizeof(*op) + cres);
	if (!op)
		return -ENOMEM;

	*op = hal.ops->pa_service(ss->cmd, ss->data, ss->param.length,
			(u8 *)(op + 1), cres);
	if (*op == SEC_HAL_OK)
		ss->wp += cres;

	ss->wp += sizeof(*op);

	wake_up_interruptible(&ss->outq);
	return 0;
}

static inline int process_cmd(sec_status *ss)
{
	int rv = -EACCES;

	mutex_lock(&hal.lock);
	ss->cmd = ss->param.cmd;

	switch (ss->cmd) {
	case CMD_INIT:
		rv = process_cmd_init(ss);
		break;
	case CMD_SECS:
		rv = process_cmd_secs(ss);
		break;
	case CMD_RANDOM:
		rv = process_cmd_random(ss);
		break;
	default:
		if (ss->cmd >= CMD_PA_FIRST)
			rv = process_cmd_pa(ss);
		break;
	}

	mutex_unlock(&hal.lock);
	return rv;
}

static ssize_t sec_write(struct file *filp, const char __user *buf,
			size_t count, loff_t *f_pos)
{
	size_t size;
	ssize_t rv = 0;
	sec_status *ss = (sec_status *)filp->private_data;

	if (mutex_lock_interruptible(&ss->lock))
		return -ERESTARTSYS;

	while (ss->cmd != CMD_NONE) {
		if (ss->cmd == CMD_SECS) {
			/* This will never complete, no sense in waiting */
			mutex_unlock(&ss->lock);
			return -EBUSY;
		}

		mutex_unlock(&ss->lock);

		if (filp->f_flags & O_NONBLOCK)
			return -EAGAIN;

		if (wait_event_interruptible(ss->inq, (ss->cmd == CMD_NONE)))
			return -ERESTARTSYS;

		if (mutex_lock_interruptible(&ss->lock))
			return -ERESTARTSYS;
	}

	/*
	 * The amount of data we are still expecting to receive before a
	 * command can be processed.
	 */
	if (ss->data)
		size = ss->param.length  - (size_t)(ss->ip - ss->data);
	else
		size = sizeof(ss->param) - (size_t)(ss->ip - (u8 *)&ss->param);

	count = min(count, size);

	if (copy_from_user(ss->ip, buf, count)) {
		rv = -EFAULT;
		goto done;
	}

	ss->ip += count;

	if (size > count)
		goto done;

	/* Process the command or continue waiting for parameter data */
	size = ss->param.length;

	if (size > 0 && !ss->data) {
		if (size > SEC_MAX_DATA_SIZE) {
			rv = -EFBIG;
		} else {
			ss->data = kmalloc(size, GFP_KERNEL);

			if (!ss->data)
				rv = -ENOMEM;

			/* Continue with the parameter data */
			ss->ip = ss->data;
		}
		goto done;
	}

	/*
	 * This blocks the parent process until the secure mode operation
	 * completes.
	 */
	rv = process_cmd(ss);

done:
	if (rv < 0) {
		/* Command failed, wake up any pending writers */
		reset_status(ss);
		wake_up_interruptible(&ss->inq);
	} else
		rv = count;

	mutex_unlock(&ss->lock);
	return rv;
}

static unsigned int sec_poll(struct file *filp, struct poll_table_struct *wait)
{
	sec_status *ss = (sec_status *)filp->private_data;
	unsigned int mask = 0;

	mutex_lock(&ss->lock);

	poll_wait(filp, &ss->inq,  wait);
	poll_wait(filp, &ss->outq, wait);

	if (ss->cmd == CMD_NONE)
		mask |= POLLOUT | POLLWRNORM;
	else if (ss->obuf && ss->rp != ss->wp)
		mask |= POLLIN | POLLRDNORM;

	mutex_unlock(&ss->lock);
	return mask;
}

static int sec_open(struct inode *inode, struct file *filp)
{
	sec_status *ss;

	if (!capable(CAP_SYS_RAWIO))
		return -EPERM;

	ss = kzalloc(sizeof(*ss), GFP_KERNEL);

	if (!ss)
		return -ENOMEM;

	mutex_init(&ss->lock);
	init_waitqueue_head(&ss->inq);
	init_waitqueue_head(&ss->outq);
	reset_status(ss);

	filp->private_data = ss;
	return nonseekable_open(inode, filp);
}

static int sec_release(struct inode *inode, struct file *filp)
{
	int i;
	sec_status *ss = (sec_status *)filp->private_data;

	mutex_lock(&ss->lock);

	if (ss->cmd == CMD_SECS) {
		/* Release the command slot */
		mutex_lock(&dev.lock);

		for (i = 0; i < SECS_SLOTS; ++i) {
			if (dev.secsd[i] == ss) {
				dev.secsd[i] = NULL;
				break;
			}
		}

		mutex_unlock(&dev.lock);
	}

	reset_status(ss);
	kfree(ss);

	return 0;
}

static u32 rpc_handler(u32 service, u32 a1, u32 a2, u32 a3)
{
	u32 rv = 0;

	switch (service) {
	case RPC_SETTINGS:
		break;
	case RPC_ALLOC:
		if (hal.ops)
			rv = hal.ops->rpc_alloc(a1);
		break;
	case RPC_DEALLOC:
		if (hal.ops)
			hal.ops->rpc_dealloc(a1);
		break;
	case RPC_SECS_WRITE:
		write_secs();
		break;
	}

	return rv;
}

int sec_hal_register(sec_hal_operations *ops)
{
	int rv = -ENODEV;
	sec_hal_init init = {
		.dev	= &dev.dev,
	};

	mutex_lock(&hal.lock);

	if (ops && !hal.ops) {
		rv = ops->hal_init(&init);
		if (!rv)
			hal.ops = ops;
	}

	mutex_unlock(&hal.lock);
	return rv;
}
EXPORT_SYMBOL(sec_hal_register);

int sec_hal_deregister(sec_hal_operations *ops)
{
	int rv = -ENODEV;

	mutex_lock(&hal.lock);

	if (ops && hal.ops == ops) {
		hal.ops->hal_exit();
		hal.ops = NULL;
		rv = 0;
	}

	mutex_unlock(&hal.lock);
	return rv;
}
EXPORT_SYMBOL(sec_hal_deregister);

static void sec_sysfs_release(struct device *dev)
{
}

struct file_operations sec_fops = {
	.owner		= THIS_MODULE,
	.llseek		= no_llseek,
	.read		= sec_read,
	.write		= sec_write,
	.poll		= sec_poll,
	.open		= sec_open,
	.release	= sec_release,
};

static int __init sec_init(void)
{
	int rv = -ENODEV;

	printk(KERN_INFO SEC_NAME ": OMAP HS Secure Mode Driver\n");

	memset(&dev, 0, sizeof(dev));
	memset(&hal, 0, sizeof(hal));
	mutex_init(&dev.lock);
	mutex_init(&hal.lock);

#ifdef SEC_HACK_SYMBIAN_WRAPPER
	storage_ptr = kzalloc(2 * SECS_SIZE, GFP_KERNEL);
#else
	storage_ptr = kzalloc(SECS_SIZE, GFP_KERNEL);
#endif

	if (!storage_ptr) {
		rv = -ENOMEM;
		goto errmsg;
	}

	strncpy(dev.dev.bus_id, SEC_NAME, BUS_ID_SIZE);
	dev.dev.release = sec_sysfs_release;

	rv = device_register(&dev.dev);

	if (rv < 0) {
		printk(KERN_WARNING SEC_NAME
			": Error registering sysfs device\n");
		goto errptr;
	}

	dev.misc_dev.minor = sec_minor;
	dev.misc_dev.name  = SEC_NAME;
	dev.misc_dev.fops  = &sec_fops;

	rv = misc_register(&dev.misc_dev);

	if (rv < 0) {
		printk(KERN_WARNING SEC_NAME
			": Error registering misc device\n");
		goto errdev;
	}

	return 0;

errdev:
	device_unregister(&dev.dev);
errptr:
	kfree(storage_ptr);
errmsg:
	printk(KERN_WARNING SEC_NAME
		": Error %d initializing device\n", rv);
	return rv;
}

static void __exit sec_exit(void)
{
	misc_deregister(&dev.misc_dev);
	sec_hal_deregister(hal.ops);
	device_unregister(&dev.dev);
	kfree(storage_ptr);
}

module_init(sec_init);
module_exit(sec_exit);
