/*
 * linux/arch/arm/mach-omap2/snapshot.c
 *
 * OMAP2 Debug Snapshot
 *
 * Copyright (C) 2006 Nokia Corporation
 * Igor Stoppa <igor.stoppa@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <linux/pm.h>
#include <linux/sched.h>
#include <linux/proc_fs.h>
#include <linux/pm.h>
#include <linux/interrupt.h>
#include <linux/sysfs.h>
#include <linux/module.h>
#include <linux/debugfs.h>

#include <asm/io.h>
#include <asm/irq.h>
#include <asm/atomic.h>
#include <asm/uaccess.h>
#include <asm/mach/time.h>
#include <asm/mach/irq.h>
#include <asm/mach-types.h>

#include <asm/arch/irqs.h>
#include <asm/arch/clock.h>
#include <asm/arch/sram.h>
#include <asm/arch/mux.h>
#include <asm/arch/pm.h>
#include <asm/arch/gpio.h>
#include <asm/arch/sram.h>

#include <asm/arch/snapshot.h>

/* Accepts a set of phisical memory locations and takes a snapshot of
 * their values, then returns them */

static struct dentry *snapshot_root;
static struct dentry *snapshot_f;
static struct dentry *take_snapshot_f;

/* The code and address / value pairs are stored in SRAM to allow
 * triggering the snapshot even when executing from SRAM */

static snapshot_t *omap_snapshot;

static unsigned *omap_index = NULL;

static void omap_snap_shot(void)
{
	unsigned i;
	static unsigned *local_index = NULL;
	static snapshot_t *local_snapshot = NULL;
	if (local_index == NULL) {
		local_index = omap_index;
		local_snapshot = omap_snapshot;
	}
	else {
		for (i = 0; i < *local_index; i++)
			local_snapshot[i][1] = omap_readl(local_snapshot[i][0]);
	}
}

void (*omap_snap_shot_ptr)(void);
EXPORT_SYMBOL(omap_snap_shot_ptr);

int omap_snapshot_pop_data(snapshot_t *read)
{
	if ((*omap_index) != 0) {
		(*omap_index)--;

		(*read)[0] = omap_snapshot[*omap_index][0];
		(*read)[1] = omap_snapshot[*omap_index][1];

		omap_snapshot[*omap_index][0] = 0;
		omap_snapshot[*omap_index][1] = 0;

		return 1;
	}
	return 0;
}
EXPORT_SYMBOL(omap_snapshot_pop_data);

static struct dentry *snapshot_f;

static ssize_t omap_snapshot_show(struct file *file, char __user *user_buf,
				  size_t count, loff_t *ppos)
{
	char buf[23]; /* "0xaaaaaaaa 0xdddddddd\n" */
	snapshot_t read;
	int ret = 0;
	static int i = 0;

/*This is a temporary workaround: the show method is modeled after the bool
 * handling code in fs/debugfs/file.c, which is broken; for some reason this
 * method is called twice and only the first time the buffer is shown.
 * Generally it is not harmful but if the showing actually affects the internal
 * status, like doing a pop ... only one entry every 2 is shown :-P */
	if (i) {
		i = 0;
		return 0;
	}
	i++;
	if (omap_snapshot_pop_data(&read))
		ret = sprintf(buf, "0x%08x 0x%08x\n", read[0], read[1]);


	return simple_read_from_buffer(user_buf, count, ppos, buf, ret);
}


void omap_snapshot_push_address(u32 address)
{
	if (*omap_index == (OMAP_SNAPSHOT_SIZE - 2)) {
		printk(KERN_ERR "List full\n");
		return;
	}
	omap_snapshot[*omap_index][0] = address;
	(*omap_index)++;
	omap_snapshot[*omap_index][0] = omap_snapshot[*omap_index][1] = 0;
}
EXPORT_SYMBOL(omap_snapshot_push_address);

static ssize_t
omap_snapshot_store(struct file *file, const char __user *user_buf,
		   size_t count, loff_t *ppos)
{
	char buf[12];
	int buf_size;
	unsigned address;

	buf_size = min(count, (sizeof(buf)-1));
	if (copy_from_user(buf, user_buf, buf_size))
		return -EFAULT;

	if (sscanf(buf, "0x%08X", &address) != 1) {
		printk(KERN_ERR "Invalid address\n");
		return -EINVAL;
	}

	omap_snapshot_push_address(address);

	return count;
}

static const struct file_operations fops_snapshot = {
	.read =		omap_snapshot_show,
	.write =	omap_snapshot_store,
};

static ssize_t
omap_take_snapshot_store(struct file *file, const char __user *user_buf,
			 size_t count, loff_t *ppos)
{
	(*omap_snap_shot_ptr)();
	return count;
}

static const struct file_operations fops_take_snapshot = {
	.write =	omap_take_snapshot_store,
};

int __init omap2_snapshot_init(void)
{
	snapshot_t dummy_snapshot[OMAP_SNAPSHOT_SIZE] = {{0, 0}, };
	unsigned dummy_index = 0;

	printk("HW Debug interface for TI OMAP.\n");

#ifdef MODULE
/*unsafe to remove because it allocates sram and it's not possible to free it*/
	__unsafe(THIS_MODULE);
#endif
	omap_index = omap_sram_push(&dummy_index, sizeof(dummy_index));
	omap_snapshot = omap_sram_push(dummy_snapshot, sizeof(dummy_snapshot));
	omap_snap_shot_ptr = omap_sram_push(omap_snap_shot, 1024);
	(*omap_snap_shot_ptr)();

	snapshot_root = debugfs_create_dir("snapshot", NULL);
	snapshot_f = debugfs_create_file("snapshot", 0644, snapshot_root,
					   NULL, &fops_snapshot);

	take_snapshot_f =
		debugfs_create_file("take_snapshot", 0644, snapshot_root,
				    NULL, &fops_take_snapshot);

	return 0;
}

#ifndef MODULE
late_initcall(omap2_snapshot_init);
#else
module_init(omap2_snapshot_init);
MODULE_DESCRIPTION("Takes a snapshot of the values of memory locations");
MODULE_LICENSE("GPL");
#endif
