/*
 * (C) Copyright 2004
 * Texas Instruments, <www.ti.com>
 * Richard Woodruff <r-woodruff2@ti.com>
 *
 * (C) Copyright 2007 Nokia Corporation
 * Contact: Igor Stoppa <igor.stoppa@nokia.com>
 * Rewritten, adding selective scaling
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR /PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */
#include <linux/linkage.h>
#include <asm/assembler.h>
#include <asm/arch/io.h>
#include <asm/arch/pm.h>
#include "scale_freq.h"

#define DPLL_RELOCK_LOOPS	1000
#define VALIDATE_RELOCK_LOOPS	100
#define DLL_RELOCK_LOOPS	0x800	/* min 0x400 L3 clocks */
		.text


/*
 * omap24xx_scale_freq() - set dividers and pll in the safety of SRAM.
 *  A positive side effect of this is the DLL value for DDR and unlock
 *  mode will be recalculated.
 *
 *
 *  Inputs: R0 = bit31: DPLL transition first pass
 		 bit30: Doubler inversion in first pass
		 bit29: Verify and apply new scalings in first pass
 *  		 bit28: DPLL transition second pass
 		 bit27: Doubler inversion in second pass
		 bit26: Verify and apply new scalings in second pass
 		 bit25-0: CM_CLKSEL1_PLL (the 6 highest bits are reserved)
 *          R1 = sdram refresh timing (SRDC_RFR_CTRL_0)
 *
 * First perform all the transitions that scale down the frequency.
 *
 * This one will be touchy.  We need to:
 *      - irs's off on entry.
 *      - do the icache block preload
 *      - go to bypass
 *      - wait till there
 *      - validate the new dividers.
 *      - set the new dpll values.
        - if target mode is NOT bypass -
 *          - set source back to dpll
 *          - wait for a lock.
 *      - dump in the new sdrc reg setting.
 *      - unlock the dll so it adjusts.
 *      - give it time to lock.
 *      - head out.
 */
ENTRY(omap24xx_scale_freq)
	stmfd	sp!, {r4-r12, lr}	@ regs to stack

	ldr	r10, prcm_base		@ base for all the prcm offsets
	ldr	r11, cm_base		@ base for all the cm offsets
	ldr	r12, sdrc_base		@ base for all the sdrc offsets

#ifndef CONFIG_CPU_ICACHE_DISABLE
	mov	r4,#0
	mcr	p15,0,r4,c7,c10,4	@ explicit memory barrier
	adr	r4, pbegin		@ addr of preload start
	adr	r5, pend		@ addr of preload end
	mcrr	p15, 1, r5, r4, c12	@ preload into icache.
.L1:	mrc	p15,0,r4,c7,c12,4	@ wait for preload to finish
	cmp	R4,#0
	bne	.L1
#endif
pbegin:
	/* Separate DPLL M,N from scaling sequence*/
	mov	r3, #0x3f		@ generate mask for lower 6 bits
	and	r2, r0, r3, lsl #26	@ r2:31-26 contains the scaling sequence
	bic	r0, r0, r3, lsl #26	@ r0 contains the new cm_clksel1_pll

	mov	r3, #FREQ_SCALING_DOWN
scaling_sequence:

	lsls	r2, r2, #1
	bcs	handle_dpll
end_dpll:

	lsls	r2, r2, #1
	bcs	handle_quick_divisor
end_quick_divisor:

	lsls	r2, r2, #1
	bcs	handle_divisors
end_divisors:

	cmp	r3, #FREQ_SCALING_DOWN
	bne	pend
	mov	r3, #FREQ_SCALING_UP
	b	scaling_sequence

	/* DPLL Divisor and Multiplier */
handle_dpll:
	/* Move into fast relock bypass by setting core_clk to ref_clk */

	add	r4, r11, #CM_CLKEN_PLL_OFFS	@ get addr
	ldr	r5, [r4]			@ get val
	bic	r5, r5, #0x3			@ clear field EN_DPLL
	orr	r5, r5, #0x2			@ fast relock val
	str	r5, [r4]			@ go to fast relock.

	add	r6, r11, #CM_IDLEST_CKGEN_OFFS	@ get addr
wait_for_bypass:
	ldr	r7, [r6]			@ state value
	and	r7, r7, #0x3			@ mask for stat ST_CORE_CLK
	cmp	r7, #0x1			@ is core clock == ref clock?
	bne	wait_for_bypass			@ loop if not
	/* ok, core_clk is set to ref_clk */

	/* set new dpll dividers _after_ in bypass */
	add	r7, r11, #CM_CLKSEL1_PLL_OFFS	@ get addr
	str	r0, [r7]			@ set dpll ctrl val from r0

	/* relock DPLL with new vals */
	orr	r5, r5, #0x3			@ val for lock dpll
	str	r5, [r4]			@ set val
	mov	r4, #DPLL_RELOCK_LOOPS		@ dead spin a bit
dead_spin_dpll:
	subs	r4, r4, #0x1			@ dec loop
	bne	dead_spin_dpll			@ delay done?

wait_for_dpll_lock:
	ldr	r4, [r6]			@ state value
	and	r4, r4, #0x3			@ mask for stat ST_CORE_CLK
	cmp	r4, #0x2			@ core_clk == DPLL freq? locked
	bne	wait_for_dpll_lock		@ wait if not
	b	end_dpll

	/* Quick scaler cascaded to the DPLL */
handle_quick_divisor:
	add	r4, r11, #CM_CLKSEL2_PLL_OFFS	@ quick scaler
	cmp	r3, #FREQ_SCALING_DOWN
	moveq	r5, #QUICK_PRESCALER_x1
	movne	r5, #QUICK_PRESCALER_x2
	str	r5, [r4]
	b	end_quick_divisor

	/* Validation of clock divisors */
handle_divisors:

	/* Validate new configuration */
	add	r4, r10, #PRCM_CLKCFG_CTRL_OFFS	@ get addr
	mov	r5, #1				@ vaild cfg msk
	str	r5, [r4]			@ activate dividers

	mov	r4, #VALIDATE_RELOCK_LOOPS	@ dead spin a bit
dead_spin_validate:
	subs	r4, r4, #1			@ dec loop
	bne	dead_spin_validate		@ delay done?
	b	end_divisors

	/* update memory timings & briefly lock dll */
	add	r4, r12, #SDRC_RFR_CTRL0_OFFS	@ get addr sdrc_rfr_ctrl0
	str	r1, [r4]			@ update refresh timing
	add	r4, r12, #SDRC_RFR_CTRL1_OFFS	@ get addr sdrc_rfr_ctrl1
	str	r1, [r4]			@ update refresh timing

	add	r4, r12, #SDRC_DLLA_CTRL_OFFS	@ get addr sdrc_dlla_ctrl
	ldr	r5, [r4]			@ get current sdrc_dlla_ctrl
	bic	r5, r5, #0x4			@ clear bit2 for lock mode.
	orr	r5, r5, #0x8			@ make sure DLL on (es2 bit pos)
	str	r5, [r4]			@ commit to DLLA_CTRL

	add	r4, r12, #SDRC_DLLB_CTRL_OFFS	@ get addr sdrc_dllb_ctrl
	ldr	r5, [r4]			@ get current sdrc_dllb_ctrl
	bic	r5, r5, #0x4			@ clear bit2 for lock mode.
	orr	r5, r5, #0x8			@ make sure DLL on (es2 bit pos)
	str	r5, [r4]			@ commit to DLLB_CTRL

	mov	r4, #DLL_RELOCK_LOOPS		@ relock time
wait_for_dll_lock:
	subs	r4, r4, #0x1
	bne	wait_for_dll_lock
pend:
	nop
	ldmfd	sp!, {r4-r12, pc}		@ restore regs and return


prcm_base:
	.word A_OMAP24XX_PRCM_BASE
cm_base:
	.word A_OMAP24XX_CM_BASE
sdrc_base:
	.word A_OMAP24XX_SDRC_BASE

ENTRY(omap24xx_scale_freq_sz)
	.word	. - omap24xx_scale_freq


