/*
 * linux/arch/arm/mach-omap2/observability.c
 *
 * OMAP2 Debug Interface Routines
 *
 * Copyright (C) 2007 Nokia Corporation
 * Igor Stoppa <igor.stoppa@nokia.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <linux/pm.h>
#include <linux/sched.h>
#include <linux/proc_fs.h>
#include <linux/pm.h>
#include <linux/interrupt.h>
#include <linux/sysfs.h>
#include <linux/module.h>
#include <linux/debugfs.h>

#include <asm/io.h>
#include <asm/irq.h>
#include <asm/atomic.h>
#include <asm/uaccess.h>
#include <asm/mach/time.h>
#include <asm/mach/irq.h>
#include <asm/mach-types.h>

#include <asm/arch/irqs.h>
#include <asm/arch/clock.h>
#include <asm/arch/sram.h>
#include <asm/arch/mux.h>
#include <asm/arch/pm.h>
#include <asm/arch/gpio.h>

#include <asm/arch/observability.h>


#define CONTROL_PADCONF_CAM_D8		0x480000d0
#define CONTROL_PADCONF_CAM_D4		0x480000d4
#define CONTROL_PADCONF_CAM_D0		0x480000d8
#define CONTROL_PADCONF_CAM_XCLK	0x480000dc
#define CONTROL_DEBOBS			0x48000270

static struct dentry *observability_root;

/* OBSMUX handling: insert and remove OBSMUX entry */
static struct dentry *obs_mux_f;

static ssize_t omap_obs_mux_show(struct file *file, char __user *user_buf,
				 size_t count, loff_t *ppos)
{
	char buf[6];

	sprintf(buf, "0x%02X\n", omap_readl(CONTROL_DEBOBS) & 0x7f);
	return simple_read_from_buffer(user_buf, count, ppos, buf, 5);
}

static ssize_t
omap_obs_mux_store(struct file *file, const char __user *user_buf,
		   size_t count, loff_t *ppos)
{
	char buf[6];
	int buf_size;
	u32 val;
	u32 old;

	buf_size = min(count, (sizeof(buf)-1));
	if (copy_from_user(buf, user_buf, buf_size))
		return -EFAULT;

	if (sscanf(buf, "%i", &val) != 1) {
		printk(KERN_ERR "HW Debug: Invalid OBSMUX\n");
		return -EINVAL;
	}

	old = omap_readl(CONTROL_DEBOBS) & ~0x7f;
	omap_writel(old | val, CONTROL_DEBOBS);

	return count;
}

static const struct file_operations fops_obs_mux = {
	.read =		omap_obs_mux_show,
	.write =	omap_obs_mux_store,
};

static inline void enable_dbg_obs(void)
{
	obs_mux_f = debugfs_create_file("obs_mux", 0644, observability_root,
				      NULL, &fops_obs_mux);
}

static inline void disable_dbg_obs(void)
{
	debugfs_remove(obs_mux_f);
}

/* GPIO handling: insert and remove GPIOs entries */

struct dbg_gpio {
#ifdef CONFIG_OMAP_MUX
	unsigned	index; /* index has to be sorted ascending*/
#endif
	u8		num;
	u8		dir;
};

static u8 dbg_gpios_val[OMAP2420_HW_DBG_GPIOS_NUM];

#ifdef CONFIG_OMAP_MUX
static const struct dbg_gpio dbg_gpios_def[] = {
	{
		.index	= V4_242X_GPIO49,
		.num	= 49,
		.dir	= 0
	},
	{
		.index	= W2_242X_GPIO50,
		.num	= 50,
		.dir	= 0
	},
	{
		.index	= U4_242X_GPIO51,
		.num	= 51,
		.dir	= 0
	},
	{
		.index	= V3_242X_GPIO52,
		.num	= 52,
		.dir	= 0
	},
	{
		.index	= V2_242X_GPIO53,
		.num	= 53,
		.dir	= 0
	},
	{
		.index	= T4_242X_GPIO54,
		.num	= 54,
		.dir	= 0
	},
	{
		.index	= T3_242X_GPIO55,
		.num	= 55,
		.dir	= 0
	},
	{
		.index	= U2_242X_GPIO56,
		.num	= 56,
		.dir	= 0
	}
};

#else

static const struct dbg_gpio dbg_gpios_def[] = {
	{
		.num	= 49,
		.dir	= 0
	},
	{
		.num	= 50,
		.dir	= 0
	},
	{
		.num	= 51,
		.dir	= 0
	},
	{
		.num	= 52,
		.dir	= 0
	},
	{
		.num	= 53,
		.dir	= 0
	},
	{
		.num	= 54,
		.dir	= 0
	},
	{
		.num	= 55,
		.dir	= 0
	},
	{
		.num	= 56,
		.dir	= 0
	}
};
#endif

static enum dbg_gpios current_dbg_gpio;

/* Handle dbg_gpio selection */
static struct dentry *available_gpios_f;

static ssize_t
omap_available_gpios_show(struct file *file, char __user *user_buf,
			  size_t count, loff_t *ppos)
{
	unsigned gpio;

/* 4 -> 3 digits for gpio number + 1 separator, 1 -> terminator */
	char buf[sizeof(dbg_gpios_def) / sizeof(struct dbg_gpio) * 4 + 1];
	char * start = buf;
	char * cursor = buf;

	for (gpio = 0; gpio < OMAP2420_HW_DBG_GPIOS_NUM; gpio++) {
		cursor += sprintf(cursor,"%d ", dbg_gpios_def[gpio].num);
	}
	strcat(cursor, "\n");

	return simple_read_from_buffer(user_buf, count, ppos, buf,
				       cursor - start);
}

static const struct file_operations fops_available_gpios = {
	.read =		omap_available_gpios_show,
};



static struct dentry *gpio_num_f;

static ssize_t omap_gpio_num_show(struct file *file, char __user *user_buf,
			      size_t count, loff_t *ppos)
{
	char buf[5];
	int len;

	len = sprintf(buf, "%d\n", dbg_gpios_def[current_dbg_gpio].num);
	return simple_read_from_buffer(user_buf, count, ppos, buf, len);
}

static ssize_t
omap_gpio_num_store(struct file *file, const char __user *user_buf,
		    size_t count, loff_t *ppos)
{
	char buf[5];
	int buf_size;
	unsigned num;
	unsigned gpio;

	buf_size = min(count, (sizeof(buf)-1));

	if (copy_from_user(buf, user_buf, buf_size))
		return -EFAULT;

	if (sscanf(buf, "%u", &num) != 1) {
		printk(KERN_ERR "HW Debug: Invalid gpio number\n");
		return -EINVAL;
	}

	for (gpio = 0; gpio < OMAP2420_HW_DBG_GPIOS_NUM; gpio++) {
		if (dbg_gpios_def[gpio].num == num) {
			current_dbg_gpio = gpio;
			return count;
		}
	}

	printk(KERN_ERR "HW Debug: Invalid gpio number\n");
	return -EINVAL;
}

static const struct file_operations fops_gpio_num = {
	.read =		omap_gpio_num_show,
	.write =	omap_gpio_num_store,
};

/* Handle dbg_gpio setting/resetting */
int omap_set_dbg_gpio_val(enum dbg_gpios gpio, u8 val)
{
	if (val != 0 && val != 1)
		return -EINVAL;

	dbg_gpios_val[gpio] = val;
	omap_set_gpio_dataout(dbg_gpios_def[gpio].num, val);

	return 0;
}
EXPORT_SYMBOL(omap_set_dbg_gpio_val);



static struct dentry *gpio_val_f;

static ssize_t omap_gpio_val_show(struct file *file, char __user *user_buf,
				  size_t count, loff_t *ppos)
{
	char buf[3];

	sprintf(buf, "%d\n", dbg_gpios_val[current_dbg_gpio]);
	return simple_read_from_buffer(user_buf, count, ppos, buf, 3);
}

static ssize_t
omap_gpio_val_store(struct file *file, const char __user *user_buf,
		    size_t count, loff_t *ppos)
{
	char buf[3];
	int buf_size;
	unsigned val;

	buf_size = min(count, (sizeof(buf)-1));

	if (copy_from_user(buf, user_buf, buf_size))
		return -EFAULT;

	if (sscanf(buf, "%u", &val) != 1 ||
	    !(val == 0 || val == 1)) {
		printk(KERN_ERR "HW Debug: Invalid gpio value\n");
		return -EINVAL;
	}

	omap_set_dbg_gpio_val(current_dbg_gpio, val);
	return count;

}

static const struct file_operations fops_gpio_val = {
	.read =		omap_gpio_val_show,
	.write =	omap_gpio_val_store,
};


/* Handle dbg_gpio setting/resetting as a 8 bit word */

int omap_get_dbg_gpio_word(void)
{
	unsigned gpio;
	int val = 0;

	for (gpio = 0; gpio < OMAP2420_HW_DBG_GPIOS_NUM; gpio++) {
		val |= dbg_gpios_val[gpio] << gpio;
	}

	return val;
}
EXPORT_SYMBOL(omap_get_dbg_gpio_word);

int omap_set_dbg_gpio_word(u8 val)
{
	unsigned gpio;
	u8 v;

	for (gpio = 0; gpio < OMAP2420_HW_DBG_GPIOS_NUM; gpio++) {
		v = (val >> gpio) & 1;
		dbg_gpios_val[gpio] = v;
		omap_set_gpio_dataout(dbg_gpios_def[gpio].num, v);
	}

	return 0;
}
EXPORT_SYMBOL(omap_set_dbg_gpio_word);


static struct dentry *gpio_word_f;

static ssize_t omap_gpio_word_show(struct file *file, char __user *user_buf,
				   size_t count, loff_t *ppos)
{
	char buf[6];

	sprintf(buf, "0x%02X\n", omap_get_dbg_gpio_word());
	return simple_read_from_buffer(user_buf, count, ppos, buf, 6);
}

static ssize_t
omap_gpio_word_store(struct file *file, const char __user *user_buf,
		   size_t count, loff_t *ppos)
{
	char buf[6];
	int buf_size;
	unsigned val;

	buf_size = min(count, (sizeof(buf)-1));

	if (copy_from_user(buf, user_buf, buf_size))
		return -EFAULT;

	if (sscanf(buf, "%i", &val) != 1) {
		printk(KERN_ERR "HW Debug: Invalid gpio value\n");
		return -EINVAL;
	}

	omap_set_dbg_gpio_word(val);

	return count;

}

static const struct file_operations fops_gpio_word = {
	.read =		omap_gpio_word_show,
	.write =	omap_gpio_word_store,
};


/* Actual handling of entries */
static inline void enable_dbg_gpio(void)
{
	available_gpios_f = debugfs_create_file("available_gpios", 0444,
						observability_root, NULL,
						&fops_available_gpios);
	gpio_num_f = debugfs_create_file("gpio_num", 0644, observability_root,
					 NULL, &fops_gpio_num);
	gpio_val_f = debugfs_create_file("gpio_val", 0644, observability_root,
					 NULL, &fops_gpio_val);
	gpio_word_f = debugfs_create_file("gpio_word", 0644, observability_root,
					  NULL, &fops_gpio_word);
}

static inline void disable_dbg_gpio(void)
{
	debugfs_remove(available_gpios_f);
	debugfs_remove(gpio_num_f);
	debugfs_remove(gpio_val_f);
	debugfs_remove(gpio_word_f);
}

static int omap_dbg_gpio_init(void)
{
	int ret = 0;
	enum dbg_gpios gpio;

	for (gpio = 0; gpio < OMAP2420_HW_DBG_GPIOS_NUM && !ret; gpio++) {
#ifdef CONFIG_OMAP_MUX
		omap_cfg_reg(dbg_gpios_def[gpio].index);
#endif
		ret = omap_request_gpio(dbg_gpios_def[gpio].num);
		omap_set_gpio_direction(dbg_gpios_def[gpio].num,
					dbg_gpios_def[gpio].dir);
		omap_set_dbg_gpio_val(gpio, 0);
	}

	for (; ret && gpio >= 0; gpio--)
		omap_free_gpio(dbg_gpios_def[gpio].num);
	return ret;
}

static void omap_dbg_gpio_exit(void)
{
	enum dbg_gpios gpio;

	for (gpio = 0; gpio < OMAP2420_HW_DBG_GPIOS_NUM; gpio++)
		omap_free_gpio(dbg_gpios_def[gpio].num);
}

/* Peeking: exposure of 1 byte from a selectable memory location */

static const unsigned register_sizes[] = {1, 2, 4};

static enum omap_dbg_register_sizes register_size;

static struct dentry *available_register_sizes_f;

static ssize_t
omap_available_register_sizes_show(struct file *file, char __user *user_buf,
				   size_t count, loff_t *ppos)
{
	char buf[6];
	char * start = buf;
	char * cursor = buf;
	int i;

	for (i = 0; i < OMAP2420_HW_DBG_REG_SIZES_NUM; i++)
		cursor += sprintf(cursor, "%d ", register_sizes[i]);
	cursor += sprintf(buf, "\n");

	return simple_read_from_buffer(user_buf, count, ppos, buf,
				       cursor - start);
}

static const struct file_operations fops_available_register_sizes = {
	.read =		omap_available_register_sizes_show,
};


void omap_dbg_set_register_size(const enum omap_dbg_register_sizes new_size)
{
	register_size = new_size;
	if (omap_dbg_get_position() >= register_sizes[register_size]) {
		omap_dbg_set_position(register_sizes[register_size] - 1);
	}
	omap_dbg_update_visible_byte();
}
EXPORT_SYMBOL(omap_dbg_set_register_size);

enum omap_dbg_register_sizes omap_dbg_get_register_size(void)
{
	return register_sizes[register_size];
}
EXPORT_SYMBOL(omap_dbg_get_register_size);


static struct dentry *register_size_f;

static ssize_t omap_register_size_show(struct file *file, char __user *user_buf,
				       size_t count, loff_t *ppos)
{
	char buf[3];

	sprintf(buf, "%u\n", omap_dbg_get_register_size());
	return simple_read_from_buffer(user_buf, count, ppos, buf, 3);
}

static ssize_t
omap_register_size_store(struct file *file, const char __user *user_buf,
		   size_t count, loff_t *ppos)
{
	char buf[3];
	int buf_size;
	unsigned val;
	int i;

	buf_size = min(count, (sizeof(buf)-1));

	if (copy_from_user(buf, user_buf, buf_size))
		return -EFAULT;

	if (sscanf(buf, "%u", &val) != 1) {
		printk(KERN_ERR "HW Debug: Invalid word size value\n");
		return -EINVAL;
	}

	for (i = 0; i < OMAP2420_HW_DBG_REG_SIZES_NUM; i++)
		if (val == register_sizes[i])
			break;
	if (i == OMAP2420_HW_DBG_REG_SIZES_NUM) {
		printk(KERN_ERR "HW Debug: Invalid word size value\n");
		return -EINVAL;
	}

	omap_dbg_set_register_size(i);

	return count;

}

static const struct file_operations fops_register_size = {
	.read =		omap_register_size_show,
	.write =	omap_register_size_store,
};

static unsigned *address;

void omap_dbg_set_address(unsigned * new_address)
{
	address = new_address;
	omap_dbg_set_position(0);
	omap_dbg_set_register_size(register_sizes[0]);
}
EXPORT_SYMBOL(omap_dbg_set_address);

unsigned * omap_dbg_get_address(void)
{
	return address;
}
EXPORT_SYMBOL(omap_dbg_get_address);


static struct dentry *address_f;

static ssize_t omap_dbg_address_show(struct file *file, char __user *user_buf,
				     size_t count, loff_t *ppos)
{
	char buf[12];

	sprintf(buf, "0x%08x\n", (unsigned)omap_dbg_get_address());
	return simple_read_from_buffer(user_buf, count, ppos, buf, 12);
}

static ssize_t
omap_dbg_address_store(struct file *file, const char __user *user_buf,
		       size_t count, loff_t *ppos)
{
	char buf[12];
	int buf_size;
	unsigned val;

	buf_size = min(count, sizeof(buf));

	if (copy_from_user(buf, user_buf, buf_size))
		return -EFAULT;

	buf[buf_size - 1] = '\0';

	if (sscanf(buf, "%i", &val) != 1) {
		printk(KERN_ERR "HW Debug: Invalid address\n");
		return -EINVAL;
	} else if (val % register_size) {
		printk(KERN_ERR "HW Debug: Address must be"
				"aligned to max word size\n");
		return -EINVAL;
	}

	omap_dbg_set_address((unsigned *)val);

	return count;

}

static const struct file_operations fops_address = {
	.read =		omap_dbg_address_show,
	.write =	omap_dbg_address_store,
};

static unsigned position;

void omap_dbg_set_position(unsigned new_position)
{
	position = new_position;
	omap_dbg_update_visible_byte();
}
EXPORT_SYMBOL(omap_dbg_set_position);

unsigned omap_dbg_get_position(void)
{
	return position;
}
EXPORT_SYMBOL(omap_dbg_get_position);

static struct dentry *position_f;

static ssize_t omap_dbg_position_show(struct file *file, char __user *user_buf,
				      size_t count, loff_t *ppos)
{
	char buf[3];

	sprintf(buf, "%u\n", (unsigned)omap_dbg_get_position());
	return simple_read_from_buffer(user_buf, count, ppos, buf, 3);
}

static ssize_t 
omap_dbg_position_store(struct file *file, const char __user *user_buf,
			size_t count, loff_t *ppos)
{
	char buf[6];
	int buf_size;
	unsigned val;

	buf_size = min(count, (sizeof(buf)-1));

	if (copy_from_user(buf, user_buf, buf_size))
		return -EFAULT;

	if (sscanf(buf, "%u", &val) != 1 ||
	    val >= register_sizes[register_size]) {
		printk(KERN_ERR "HW Debug: Invalid position\n");
		return -EINVAL;
	}

	omap_dbg_set_position(val);

	return count;

}

static const struct file_operations fops_position = {
	.read =		omap_dbg_position_show,
	.write =	omap_dbg_position_store,
};


void omap_dbg_update_visible_byte(void)
{
	omap_set_dbg_gpio_word(omap_readb((u32)position + (u32)address));
}
EXPORT_SYMBOL(omap_dbg_update_visible_byte);

void omap_dbg_set_visible_byte(u8 new_visible_byte)
{
	u32 tmp;
	u32 mask = ((u32)0xff) << (8 * position);
	u32 new_val = ((u32)new_visible_byte) << (8 * position);

	switch (register_size) {
	case OMAP2420_HW_DBG_REG_SIZE_4:
		tmp = omap_readl((u32)address);
		tmp = (tmp & (~mask)) | new_val;
		omap_writel((u32)tmp, (u32)address);
		break;
	case OMAP2420_HW_DBG_REG_SIZE_2:
		tmp = omap_readw((u32)address);
		tmp = (tmp & (~mask)) | new_val;
		omap_writew((u16)tmp, (u32)address);
		break;
	case OMAP2420_HW_DBG_REG_SIZE_1:
		tmp = omap_readb((u32)address);
		tmp = (tmp & (~mask)) | new_val;
		omap_writeb((u8)tmp, (u32)address);
		break;
	default:
		break;
	}

	omap_dbg_update_visible_byte();
}
EXPORT_SYMBOL(omap_dbg_set_visible_byte);

u8 omap_dbg_get_visible_byte(void)
{
	return omap_readb(position + (u8*)address);
}
EXPORT_SYMBOL(omap_dbg_get_visible_byte);


static struct dentry *visible_byte_f;

static ssize_t
omap_dbg_visible_byte_show(struct file *file, char __user *user_buf,
			   size_t count, loff_t *ppos)
{
	char buf[6];

	sprintf(buf, "0x%x\n", (unsigned)omap_dbg_get_visible_byte());
	return simple_read_from_buffer(user_buf, count, ppos, buf, 6);
}

static ssize_t 
omap_dbg_visible_byte_store(struct file *file, const char __user *user_buf,
			    size_t count, loff_t *ppos)
{
	char buf[6];
	int buf_size;
	unsigned val;

	buf_size = min(count, (sizeof(buf)-1));

	if (copy_from_user(buf, user_buf, buf_size))
		return -EFAULT;

	if (sscanf(buf, "%i", &val) != 1 || val >= 256) {
		printk(KERN_ERR "HW Debug: Invalid value\n");
		return -EINVAL;
	}

	omap_dbg_set_visible_byte((u8)val);

	return count;

}

static const struct file_operations fops_visible_byte = {
	.read =		omap_dbg_visible_byte_show,
	.write =	omap_dbg_visible_byte_store,
};



static struct dentry *update_f;

static ssize_t
omap_dbg_update_store(struct file *file, const char __user *user_buf,
		      size_t count, loff_t *ppos)
{
	char buf[6];
	int buf_size;

	buf_size = min(count, (sizeof(buf)-1));

	if (copy_from_user(buf, user_buf, buf_size))
		return -EFAULT;

	omap_dbg_update_visible_byte();

	return count;
}

static const struct file_operations fops_update = {
	.write =	omap_dbg_update_store,
};

static inline void enable_dbg_peek(void)
{
	omap_dbg_set_address((unsigned*)0x48008000);
	available_register_sizes_f =
		debugfs_create_file("available_register_sizes", 0444,
				    observability_root, NULL,
				    &fops_available_register_sizes);

	register_size_f =
		debugfs_create_file("register_size", 0644, observability_root,
				    NULL, &fops_register_size);

	address_f = debugfs_create_file("address", 0644, observability_root,
					NULL, &fops_address);

	position_f = debugfs_create_file("position", 0644, observability_root,
					 NULL, &fops_position);

	visible_byte_f =
		debugfs_create_file("visible_byte", 0644, observability_root,
				    NULL, &fops_visible_byte);

	update_f = debugfs_create_file("update", 0200, observability_root,
					 NULL, &fops_update);
}

static inline void disable_dbg_peek(void)
{
	debugfs_remove(available_register_sizes_f);
	debugfs_remove(register_size_f);
	debugfs_remove(address_f);
	debugfs_remove(position_f);
	debugfs_remove(visible_byte_f);
	debugfs_remove(update_f);
}


/*
 *	Debug modes handling: insert and remove mode-specific
 *	entries	upon selection/deselection of debug mode
 */

#define DBG_MODE_NAME_MAX_LEN	10

struct dbg_mode {
	char	name[10];
	u32	pattern;
};

static const struct dbg_mode dbg_modes_def[] = {
	{
		.name		= "camera",
		.pattern	= 0x00000000
	},
	{
		.name		= "obs",
		.pattern	= 0x01010101
	},
	{
		.name		= "gpio",
		.pattern	= 0x03030303
	},
	{
		.name		= "peek",
		.pattern	= 0x03030303
	}
};

static enum omap_dbg_modes current_dbg_mode;

void omap_set_dbg_mode(const enum omap_dbg_modes mode)
{
	if (mode == current_dbg_mode)
		return;
	switch (current_dbg_mode) {
	case OMAP2420_HW_DBG_MODE_OBS:
		disable_dbg_obs();
		break;

	case OMAP2420_HW_DBG_MODE_GPIO:
		disable_dbg_gpio();
		break;

	case OMAP2420_HW_DBG_MODE_PEEK:
		disable_dbg_peek();
		break;

	case OMAP2420_HW_DBG_MODE_CAMERA:
	default:
		break;
	}

	omap_writel(dbg_modes_def[mode].pattern, CONTROL_PADCONF_CAM_D8);
	omap_writel(dbg_modes_def[mode].pattern, CONTROL_PADCONF_CAM_D4);
	omap_writel(dbg_modes_def[mode].pattern, CONTROL_PADCONF_CAM_D0);
	omap_writel(dbg_modes_def[mode].pattern, CONTROL_PADCONF_CAM_XCLK);

	switch (mode) {
	case OMAP2420_HW_DBG_MODE_OBS:
		enable_dbg_obs();
		break;

	case OMAP2420_HW_DBG_MODE_GPIO:
		enable_dbg_gpio();
		break;

	case OMAP2420_HW_DBG_MODE_PEEK:
		enable_dbg_peek();
		break;

	case OMAP2420_HW_DBG_MODE_CAMERA:
	default:
		break;
	}
	current_dbg_mode = mode;
}
EXPORT_SYMBOL(omap_set_dbg_mode);

static struct dentry *available_debug_modes_f;

static ssize_t 
omap_available_debug_modes_show(struct file *file, char __user *user_buf,
			  size_t count, loff_t *ppos)
{
	unsigned mode;
	char buf[sizeof(dbg_modes_def[0].name) * OMAP2420_HW_DBG_MODES_NUM + 2];
	char * start = buf;
	char * cursor = buf;

	for (mode = 0; mode < OMAP2420_HW_DBG_MODES_NUM; mode++) {
		cursor += sprintf(cursor, "%s ", dbg_modes_def[mode].name);
	}
	strcat(cursor - 1, "\n");

	return simple_read_from_buffer(user_buf, count, ppos, buf,
				       cursor - start);
}

static const struct file_operations fops_available_debug_modes = {
	.read =		omap_available_debug_modes_show,
};

static struct dentry *debug_mode_f;

static ssize_t omap_dbg_debug_mode_show(struct file *file, char __user *user_buf,
					size_t count, loff_t *ppos)
{
	char buf[sizeof(dbg_modes_def[0].name) + 2];
	unsigned buf_size;

	buf_size = sprintf(buf, "%s\n", dbg_modes_def[current_dbg_mode].name);
	return simple_read_from_buffer(user_buf, count, ppos, buf, buf_size);
}

static ssize_t 
omap_dbg_debug_mode_store(struct file *file, const char __user *user_buf,
			  size_t count, loff_t *ppos)
{
	char buf[sizeof(dbg_modes_def[0].name) + 1];
	int buf_size;
	unsigned mode;

	buf_size = min(count, sizeof(buf));

	if (copy_from_user(buf, user_buf, buf_size))
		return -EFAULT;

	buf[buf_size - 1] = '\0';

	for (mode = 0; mode < OMAP2420_HW_DBG_MODES_NUM; mode++) {
		if (strncmp(buf, dbg_modes_def[mode].name, buf_size) == 0) {
			omap_set_dbg_mode(mode);
			return count;
		}
	}

	printk(KERN_ERR "HW Debug: Invalid mode\n");
	return -EINVAL;

}

static const struct file_operations fops_debug_mode = {
	.read =		omap_dbg_debug_mode_show,
	.write =	omap_dbg_debug_mode_store,
};



int __init omap2_hw_debug_init(void)
{
	printk("HW Debug interface for TI OMAP.\n");

	current_dbg_mode = OMAP2420_HW_DBG_MODE_CAMERA;

	if (omap_dbg_gpio_init()) {
		printk("Failed to initialise gpio.\n");
		return -EINVAL;
	}

	observability_root = debugfs_create_dir("observability", NULL);

	debug_mode_f = debugfs_create_file("debug_mode", 0644,
					   observability_root, NULL,
					   &fops_debug_mode);

	available_debug_modes_f =
		debugfs_create_file("available_debug_modes", 0644,
				    observability_root, NULL,
				    &fops_available_debug_modes);

	return 0;
}

static void __exit omap2_hw_debug_exit(void)
{
	omap_set_dbg_mode(OMAP2420_HW_DBG_MODE_CAMERA);
	debugfs_remove(debug_mode_f);
	debugfs_remove(available_debug_modes_f);
	debugfs_remove(observability_root);
	omap_dbg_gpio_exit();
}

#ifndef MODULE
late_initcall(omap2_hw_debug_init);
#else
module_init(omap2_hw_debug_init);
module_exit(omap2_hw_debug_exit);

MODULE_DESCRIPTION("OMAP2 HW Observability");
MODULE_LICENSE("GPL");
#endif

