/*
 * arch/arm/mach-omap2/dvs.c
 *
 * OMAP2 Dynamic Voltage Scaling
 *
 * Copyright (C) 2006 Nokia Corporation
 * Amit Kucheria <amit.kucheria@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <linux/module.h>
#include <linux/delay.h>
#include <linux/clk.h>

#include <asm/io.h>

#include <asm/arch/menelaus.h>

#define PRCM_BASE		0x48008000
#define PRCM_VOLTCTRL		0x050
#define		AUTO_EXTVOLT	(1 << 15)
#define		FORCE_EXTVOLT	(1 << 14)
#define		SETOFF_LEVEL(x)	(((x) & 0x3) << 12)
#define		MEMRETCTRL	(1 << 8)
#define		SETRET_LEVEL(x)	(((x) & 0x3) << 6)
#define		VOLT_LEVEL(x)	(((x) & 0x3) << 0)

static struct clk *l3_ck;

static u32 prcm_base = IO_ADDRESS(PRCM_BASE);

static inline void prcm_write_reg(int idx, u32 val)
{
	__raw_writel(val, prcm_base + idx);
}

static inline u32 prcm_read_reg(int idx)
{
	return __raw_readl(prcm_base + idx);
}

static void omap2_toggle_vmode(void)
{
	u32 saved, l;

	/* There is a bug in OMAP that makes the state of VMODE line
	 * inconsistent when Voltage scaling mode is toggled between
	 * AUTO and FORCE mode. We work around it by first FORCE scaling
	 * to FLOOR and then scaling to ROOF.
	 */
	saved = prcm_read_reg(PRCM_VOLTCTRL);

	/* FORCE voltage to FLOOR */
	l = (saved & ~AUTO_EXTVOLT) | FORCE_EXTVOLT | VOLT_LEVEL(1);
	prcm_write_reg(PRCM_VOLTCTRL, l);

	/* Wait for 960 L3 clock cycles + VOLTSETUP time */
	msleep(5); /* Fixme: Use l3_ck->rate, clk_get_rate(l3_ck) */

	/* FORCE voltage to ROOF */
	l = (saved & ~AUTO_EXTVOLT) | FORCE_EXTVOLT | VOLT_LEVEL(0);
	prcm_write_reg(PRCM_VOLTCTRL, l);

        /* Back to AUTO mode */
	prcm_write_reg(PRCM_VOLTCTRL, saved);

	/* FIXME: Try to use Voltage transition interrupt,
	 * MPU and Core need to be ON
	 */

}

void omap2_scale_voltage(unsigned int roof_mV, unsigned int floor_mV)
{
	menelaus_set_vcore_hw(roof_mV, roof_mV);
	omap2_toggle_vmode();
	menelaus_set_vcore_hw(roof_mV, floor_mV);
}
EXPORT_SYMBOL(omap2_scale_voltage);

int __init omap2_dvs_init(void)
{
	printk(KERN_INFO "DVS for OMAP2 initializing\n");

	l3_ck = clk_get(NULL, "core_l3_ck");

	return 0;
}

late_initcall(omap2_dvs_init);
