/*
 * arch/arm/mach-omap2/dvfs.c
 *
 * Copyright (C) 2007 Nokia Corporation.
 *
 * Contact: Igor Stoppa <igor.stoppa@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/list.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/clk.h>
#include <linux/cpufreq.h>
#include <asm/mach-types.h>
#include <asm/arch/dvfs_notif.h>

#include "scale_freq.h"
#include "dvfs.h"

/* A HW bug on 2420 prevents dividers greater than 8 */
#define MAX_DPLL_DIV		8
#define MAX_DPLL_MULT		((1 << 10) - 1)

#define CORE_VTG_FLOOR		1050

#define DPLL_SCALES_DOWN	0x80000000
#define DOUBLER_SCALES_DOWN	0x40000000
#define DOMAINS_SCALE_DOWN	0x20000000
#define DPLL_SCALES_UP		0x10000000
#define DOUBLER_SCALES_UP	0x08000000
#define DOMAINS_SCALE_UP	0x04000000

enum omap242x_packed_op_parms {
	CM_CLKSEL_MPU = 0,
	CM_CLKSEL_GFX,
	CM_CLKSEL_DSP,
	CM_CLKSEL1_CORE,
	NUM_MMAPPED_PARMS,
	OTHER_PARMS = NUM_MMAPPED_PARMS - 1,
	CM_CLKSEL1_PLL,
	CM_CLKSEL2_PLL,
	VCORE_ROOF,
	NUM_PACKED_OP_PARMS,
};

enum omap242x_bitfields {
	CLKSEL_MPU = 0,
	CLKSEL_GFX,
	CLKSEL_DSP,
	CLKSEL_DSP_IF,
	CLKSEL_IVA,
	CLKSEL_L3,
	CLKSEL_L4,
	CLKSEL_DSS1,
	CLKSEL_DSS2,
	CLKSEL_VLYNQ,
	CLKSEL_SSI,
	CLKSEL_USB,
	NUM_DIVIDERS,
	OTHER_BITFIELDS = NUM_DIVIDERS - 1,
	_48M_SOURCE,
	_54M_SOURCE,
	DPLL_DIV,
	DPLL_MULT,
	APLLS_CLKIN,
	CORE_CLK_SRC,
	SYNC_DSP,
	SYNC_IVA,
	NUM_BITFIELDS,
};

enum omap242x_bitfield_parms{
	REG = 0,
	OFFS,
	LEN,
	NUM_BITFIELD_PARMS,
};

static const u32 mmapped_parms_addr[NUM_MMAPPED_PARMS] = {
	[CM_CLKSEL_MPU]		= 0x48008140,
	[CM_CLKSEL_DSP]		= 0x48008840,
	[CM_CLKSEL_GFX]		= 0x48008340,
	[CM_CLKSEL1_CORE]	= 0x48008240,
};

static const u8 bitfields[NUM_BITFIELDS][NUM_BITFIELD_PARMS] = {
	[CLKSEL_MPU]	= {[REG] = CM_CLKSEL_MPU,   [OFFS] =  0, [LEN] =  5,},
	[CLKSEL_GFX]	= {[REG] = CM_CLKSEL_GFX,   [OFFS] =  0, [LEN] =  5,},
	[CLKSEL_DSP]	= {[REG] = CM_CLKSEL_DSP,   [OFFS] =  0, [LEN] =  5,},
	[CLKSEL_DSP_IF]	= {[REG] = CM_CLKSEL_DSP,   [OFFS] =  5, [LEN] =  2,},
	[SYNC_DSP]	= {[REG] = CM_CLKSEL_DSP,   [OFFS] =  7, [LEN] =  1,},
	[CLKSEL_IVA]	= {[REG] = CM_CLKSEL_DSP,   [OFFS] =  8, [LEN] =  5,},
	[SYNC_IVA]	= {[REG] = CM_CLKSEL_DSP,   [OFFS] = 13, [LEN] =  1,},
	[CLKSEL_L3]	= {[REG] = CM_CLKSEL1_CORE, [OFFS] =  0, [LEN] =  5,},
	[CLKSEL_L4]	= {[REG] = CM_CLKSEL1_CORE, [OFFS] =  5, [LEN] =  2,},
	[CLKSEL_DSS1]	= {[REG] = CM_CLKSEL1_CORE, [OFFS] =  8, [LEN] =  5,},
	[CLKSEL_DSS2]	= {[REG] = CM_CLKSEL1_CORE, [OFFS] = 13, [LEN] =  1,},
	[CLKSEL_VLYNQ]	= {[REG] = CM_CLKSEL1_CORE, [OFFS] = 15, [LEN] =  5,},
	[CLKSEL_SSI]	= {[REG] = CM_CLKSEL1_CORE, [OFFS] = 20, [LEN] =  5,},
	[CLKSEL_USB]	= {[REG] = CM_CLKSEL1_CORE, [OFFS] = 25, [LEN] =  3,},
	[_48M_SOURCE]	= {[REG] = CM_CLKSEL1_PLL,  [OFFS] =  3, [LEN] =  1,},
	[_54M_SOURCE]	= {[REG] = CM_CLKSEL1_PLL,  [OFFS] =  5, [LEN] =  1,},
	[DPLL_DIV]	= {[REG] = CM_CLKSEL1_PLL,  [OFFS] =  8, [LEN] =  4,},
	[DPLL_MULT]	= {[REG] = CM_CLKSEL1_PLL,  [OFFS] = 12, [LEN] = 10,},
	[APLLS_CLKIN]	= {[REG] = CM_CLKSEL1_PLL,  [OFFS] = 23, [LEN] =  3,},
	[CORE_CLK_SRC]	= {[REG] = CM_CLKSEL2_PLL,  [OFFS] =  0, [LEN] =  2,},
};

static const unsigned int (*op_list)[NUM_OP_PARMS];
static unsigned int op_num;
static unsigned cur_op;
static u32 (*packed_op_list)[NUM_PACKED_OP_PARMS];
static u32 *trends;
static struct cpufreq_frequency_table *freq_tables[NUM_TABLES];

static u32 (*op_packer[NUM_PACKED_OP_PARMS])(const unsigned int op);

void omap2_clk_recalc_tree(void);

void omap2_scale_voltage(unsigned int roof_mV, unsigned int floor_mV);

void omap24xx_scale_freq(u32 dpll_ctl, u32 sdram_refresh);

extern u32 omap24xx_scale_freq_sz;

static void (*omap2_scale_freq)(u32 dpll_ctl, u32 sdram_refresh);

static DEFINE_MUTEX(dvfs_mutex);

static inline unsigned int get_op_parm(unsigned int op, int parm)
{
	return op_list[op][parm];
}

static inline u32 bitfield(u32 val, int id)
{
	u32 offs, mask;

	offs = bitfields[id][OFFS];
	mask = (1 << bitfields[id][LEN]) - 1;

	return (val& mask) << offs;
}

static inline u32 get_packed_op_bitfield(unsigned int packed_op, int id)
{
	u32 val, offs, mask;

	val  = packed_op_list[packed_op][bitfields[id][REG]];
	offs = bitfields[id][OFFS];
	mask = (1 << bitfields[id][LEN]) - 1;

	return (val >> offs) & mask;
}

static inline u32 get_packed_op_parm(unsigned int packed_op, int id)
{
	return packed_op_list[packed_op][id];
}

static inline u32 get_packed_op_parm_addr(int id)
{
	return mmapped_parms_addr[id];
}

static inline u32 get_trend(unsigned int src, unsigned int dst)
{
	return trends[src * op_num + dst];
}

static inline void set_trend(u32 val, unsigned int src, unsigned int dst)
{
	 trends[src * op_num + dst] = val;
}

static u32 omap242x_pack_cm_clksel1_pll(unsigned int op)
{
	u32 ret = 0;
	u32 m = 0, n = 0, delta, apll;
	int tmp, i;

	tmp = get_op_parm(op, _48M_SRC_CLK_KHZ) == 96000 ? 0 : 1;
	ret |= bitfield(tmp, _48M_SOURCE);

	tmp = get_op_parm(op, _54M_SRC_CLK_KHZ) == 54000 ? 0 : 1;
	ret |= bitfield(tmp, _54M_SOURCE);

	apll = 19200;				/* FIXME: add autodetection */
	ret |= bitfield(0, APLLS_CLKIN);	/* FIXME: add autodetection */

	/* Find the best pair (multiplier, divider) that gives a freq smaller
	 * or equal than the one requested */
	delta = apll;
	for (i = 1; i < MAX_DPLL_DIV; i++) {
		u32 tmp_m, tmp_delta;

		tmp_m = get_op_parm(op, DPLL_CLK_KHZ) * i / apll;
		if (tmp_m <= MAX_DPLL_MULT) {
			tmp_delta = get_op_parm(op, DPLL_CLK_KHZ) * i % apll;
			if (tmp_delta < delta) {
				n = i - 1;
				delta = tmp_delta;
				m = tmp_m;
			}
		}
	}
	ret |= bitfield(m, DPLL_MULT);
	ret |= bitfield(n, DPLL_DIV);
	return ret;
}

static u32 omap242x_pack_cm_clksel2_pll(const unsigned int op)
{
	u32 clk_src;

	if (get_op_parm(op, CORE_CLK_KHZ) == 32000)
		clk_src = 0;
	else
		clk_src =  get_op_parm(op, CORE_CLK_KHZ) /
			   get_op_parm(op, DPLL_CLK_KHZ);

	return bitfield(clk_src, CORE_CLK_SRC);
}

static u32 omap242x_pack_cm_clksel_mpu(const unsigned int op)
{
	u32 div;

	div =  get_op_parm(op, CORE_CLK_KHZ) /
	       get_op_parm(op, MPU_CLK_KHZ);
	return bitfield(div, CLKSEL_MPU);
}

static u32 omap242x_pack_cm_clksel_dsp(const unsigned int op)
{
	u32 ret = 0, val;

	val =  get_op_parm(op, CORE_CLK_KHZ) /
	       get_op_parm(op, DSP_CLK_KHZ);
	ret |= bitfield(val, CLKSEL_DSP);

	val =  get_op_parm(op, DSP_CLK_KHZ) /
	       get_op_parm(op, DSP_IF_CLK_KHZ);
	ret |= bitfield(val, CLKSEL_DSP_IF);

	ret |= bitfield(get_op_parm(op, DSP_SYNC), SYNC_DSP);

	val =  get_op_parm(op, CORE_CLK_KHZ) /
	       get_op_parm(op, IVA_CLK_KHZ);
	ret |= bitfield(val, CLKSEL_IVA);

	ret |= bitfield(get_op_parm(op, IVA_SYNC), SYNC_IVA);

	return ret;
}

static u32 omap242x_pack_cm_clksel_gfx(const unsigned int op)
{
	u32 div;

	div =  get_op_parm(op, L3_CLK_KHZ) / get_op_parm(op, GFX_CLK_KHZ);
	return bitfield(div, CLKSEL_GFX);
}

static u32 omap242x_pack_cm_clksel1_core(const unsigned int op)
{
	u32 ret = 0, val;

	val =  get_op_parm(op, CORE_CLK_KHZ) /
	       get_op_parm(op, L3_CLK_KHZ);
	ret |= bitfield(val, CLKSEL_L3);

	val =  get_op_parm(op, L3_CLK_KHZ) /
	       get_op_parm(op, L4_CLK_KHZ);
	ret |= bitfield(val, CLKSEL_L4);

	val =  get_op_parm(op, CORE_CLK_KHZ) /
	       get_op_parm(op, DSS1_CLK_KHZ);
	ret |= bitfield(val, CLKSEL_DSS1);

	val =  get_op_parm(op, CORE_CLK_KHZ) /
	       get_op_parm(op, VLYNQ_CLK_KHZ);
	ret |= bitfield(val, CLKSEL_VLYNQ);

	val =  get_op_parm(op, DSS1_CLK_KHZ) == 48000 ? 1 : 0;
	ret |= bitfield(val, CLKSEL_DSS2);

	val =  get_op_parm(op, CORE_CLK_KHZ) /
	       get_op_parm(op, SSI_CLK_KHZ);
	ret |= bitfield(val, CLKSEL_SSI);

	val =  get_op_parm(op, L3_CLK_KHZ) /
	       get_op_parm(op, USB_CLK_KHZ);
	ret |= bitfield(val, CLKSEL_USB);

	return ret;
}

static u32 omap242x_pack_vcore_roof(const unsigned int op)
{
	return get_op_parm(op, VCORE_ROOF_MV);
}

static int pack_op_list(void)
{
	unsigned int i, j;

	packed_op_list = kmalloc(sizeof(u32) * NUM_PACKED_OP_PARMS * op_num,
				 GFP_KERNEL);
	if (!packed_op_list)
		return -ENOMEM;

	for (i = 0; i < op_num; i++)
		for (j = 0; j < NUM_PACKED_OP_PARMS; j++)
			if (op_packer[j])
				packed_op_list[i][j] = op_packer[j](i);
	return 0;
}

static u32 dvfs_calc_trend(unsigned int src, unsigned int dst)
{
	u32 ret = 0;
	u32 src_dblr, dst_dblr;
	u32 src_div, dst_div;
	unsigned int i, found_divider_growing = 0;

	if (op_list[src][DPLL_CLK_KHZ] > op_list[dst][DPLL_CLK_KHZ])
		ret |= DPLL_SCALES_DOWN;
	else if (op_list[src][DPLL_CLK_KHZ] < op_list[dst][DPLL_CLK_KHZ])
		ret |= DPLL_SCALES_UP;

	src_dblr = get_packed_op_bitfield(src, CORE_CLK_SRC);
	dst_dblr = get_packed_op_bitfield(dst, CORE_CLK_SRC);
	if (src_dblr > dst_dblr)
		ret |= DOUBLER_SCALES_DOWN;
	else if (src_dblr < dst_dblr)
		ret |= DOUBLER_SCALES_UP;

	/* Check the trend of every divider; even a single one scaling up is
	 * enough to set the trend, otherwise check for decreasing cases */
	for (i = 0; i < NUM_DIVIDERS; i++) {
		src_div = get_packed_op_bitfield(src, i);
		dst_div = get_packed_op_bitfield(dst, i);
		if (src_div > dst_div)
			return ret | DOMAINS_SCALE_UP;
		else if ((!found_divider_growing) && (src_div > dst_div))
			found_divider_growing = 1;
	}

	if (found_divider_growing)
		return ret | DOMAINS_SCALE_DOWN;

	/* No divider is shrinking */
	if (found_divider_growing)
		return ret | DOMAINS_SCALE_DOWN;

	return ret;

}

static int create_trends_table(void)
{
	unsigned int src, dst;

	trends = kmalloc(sizeof(u32) * op_num * op_num, GFP_KERNEL);
	if (!trends)
		return -ENOMEM;

	for (src = 0; src < op_num; src++)
		for (dst = 0; dst < op_num; dst++)
			set_trend(dvfs_calc_trend(src, dst), src, dst);
	return 0;
}

static int create_freq_tables(void)
{
	unsigned int i, j, k;

	for (i = 0; i < NUM_TABLES; i++) {
		freq_tables[i] =
			kmalloc(sizeof(struct cpufreq_frequency_table) *
				(op_num + 1), GFP_KERNEL);

		if (!freq_tables[i]) {
			for (; i > 0; i--)
				kfree(freq_tables[i - 1]);
			return -ENOMEM;
		}

		for (j = 0; j < op_num; j++) {
			freq_tables[i][j].frequency = op_list[j][i];
			freq_tables[i][j].index = j;
		}

		for (j = op_num - 1; j; j--) /* inplace bubblesort */
			for (k = 0; k < j; k++)
				if (freq_tables[i][k].frequency <
				    freq_tables[i][k + 1].frequency) {
					struct cpufreq_frequency_table tmp;

					tmp = freq_tables[i][k];
					freq_tables[i][k] = freq_tables[i][k + 1];
					freq_tables[i][k + 1] = tmp;
				}
		freq_tables[i][op_num].frequency = CPUFREQ_TABLE_END;
	}
	return 0;
}

int omap242x_load_op_list(const unsigned int list[][NUM_OP_PARMS], unsigned int num,
			  unsigned int active)
{
	if (!(list && num && (active < num)))
		return -EINVAL;

	op_list = list;
	op_num = num;
	cur_op = active;

	if (pack_op_list())
		return -EINVAL;

	if (create_trends_table())
		goto trend_error;

	if (create_freq_tables())
		goto freq_tables_error;

	return 0;

freq_tables_error:
	kfree(trends);
trend_error:
	kfree(packed_op_list);
	return -EINVAL;
}
EXPORT_SYMBOL(omap242x_load_op_list);

unsigned int omap242x_dvfs_get_op(void)
{
	unsigned int tmp;

	mutex_lock(&dvfs_mutex);
	tmp = cur_op;
	mutex_unlock(&dvfs_mutex);

	return tmp;
}
EXPORT_SYMBOL(omap242x_dvfs_get_op);

int omap242x_dvfs_set_op(unsigned dst)
{
	unsigned i, src = cur_op;
	unsigned rfr;
	int ret;

	mutex_lock(&dvfs_mutex);

	if (op_num <= dst) {
		mutex_unlock(&dvfs_mutex);
		return -EINVAL;
	}

	if (src == dst) {
		mutex_unlock(&dvfs_mutex);
		return 0;
	}

	ret = dvfs_run_notifications(DVFS_PRE_NOTIFICATION);
	if (ret)
		goto out;

	if (get_packed_op_parm(dst, VCORE_ROOF) >
	    get_packed_op_parm(src, VCORE_ROOF))
		omap2_scale_voltage(get_packed_op_parm(dst, VCORE_ROOF),
				    CORE_VTG_FLOOR);

	for (i = 0; i < NUM_MMAPPED_PARMS; i++)
		omap_writel(get_packed_op_parm(dst, i),
			    get_packed_op_parm_addr(i));

	local_irq_disable();
	local_fiq_disable();

/* FIXME:
 * -adjust the memory refresh period to the L3 frequency
 * -make the memory refresh period part of the parameters to be written
 * -convert the asm function to C */
	rfr = 1 | ((64 * get_op_parm(dst, L3_CLK_KHZ) / 8192 - 50) <<8);
	omap2_scale_freq(get_packed_op_parm(dst, CM_CLKSEL1_PLL) |
			 get_trend(src, dst), rfr);

	local_fiq_enable();
	local_irq_enable();

	omap2_clk_recalc_tree();

	if (get_packed_op_parm(dst, VCORE_ROOF) <
	    get_packed_op_parm(src, VCORE_ROOF))
		omap2_scale_voltage(get_packed_op_parm(dst, VCORE_ROOF),
				    CORE_VTG_FLOOR);
	cur_op = dst;
out:
	/* FIXME: check return value*/
	dvfs_run_notifications(DVFS_POST_NOTIFICATION);

	mutex_unlock(&dvfs_mutex);

	return ret;
}
EXPORT_SYMBOL(omap242x_dvfs_set_op);

static ssize_t op_list_show(struct subsystem * subsys, char *buf)
{
	unsigned int i, j;

	j = 0;
	for (i = 0; i < op_num; i++)
		j += sprintf(buf + j, "%u ", i);
	sprintf(buf + j - 1, "\n");

	return j;
}

static struct subsys_attribute op_list_attr = {
	.attr   = {
		.name = __stringify(op_list),
		.mode = 0444,
	},
	.show   = op_list_show,
};

static ssize_t op_active_show(struct subsystem * subsys, char *buf)
{
	return sprintf(buf, "%u\n", cur_op);
}

static ssize_t op_active_store(struct subsystem * subsys,
			      const char * buf, size_t n)
{
	unsigned value;

	if (sscanf(buf, "%u", &value) != 1 || (value >= op_num)) {
		printk(KERN_ERR "OP: Invalid index\n");
		return -EINVAL;
	}

	omap242x_dvfs_set_op(value);
	return n;
}

static struct subsys_attribute op_active_attr = {
	.attr   = {
		.name = __stringify(op_active),
		.mode = 0644,
	},
	.show   = op_active_show,
	.store  = op_active_store,
};

extern struct subsystem power_subsys;

int __init dvfs_init(void)
{
	int ret;

	printk(KERN_INFO "DVFS for OMAP2 initializing\n");

	op_packer[CM_CLKSEL1_PLL]  = omap242x_pack_cm_clksel1_pll;
	op_packer[CM_CLKSEL2_PLL]  = omap242x_pack_cm_clksel2_pll;
	op_packer[CM_CLKSEL_MPU]   = omap242x_pack_cm_clksel_mpu;
	op_packer[CM_CLKSEL_DSP]   = omap242x_pack_cm_clksel_dsp;
	op_packer[CM_CLKSEL_GFX]   = omap242x_pack_cm_clksel_gfx;
	op_packer[CM_CLKSEL1_CORE] = omap242x_pack_cm_clksel1_core;
	op_packer[VCORE_ROOF]	   = omap242x_pack_vcore_roof;

	ret = subsys_create_file(&power_subsys, &op_list_attr);
	if (ret)
		printk(KERN_ERR "subsys_create_file failed: %d\n", ret);

	ret = subsys_create_file(&power_subsys, &op_active_attr);
	if (ret)
		printk(KERN_ERR "subsys_create_file failed: %d\n", ret);

#ifdef CONFIG_CPU_ICACHE_DISABLE
	omap2_scale_freq = omap_sram_push(omap24xx_scale_freq,
					  omap24xx_scale_freq_sz);
#else
	omap2_scale_freq = omap24xx_scale_freq;
#endif
	return ret;
}
