/*
 * arch/arm/mach-omap2/board-n800-dvfs.c
 *
 * Copyright (C) 2007 Nokia Corporation.
 *
 * Contact: Igor Stoppa <igor.stoppa@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <linux/kernel.h>
#include <linux/device.h>
#include <linux/list.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/clk.h>
#include <asm/mach-types.h>
#include <asm/arch/board.h>
#include <asm/arch/dsp_common.h>
#include <linux/cpufreq.h>
#include "dvfs.h"

enum op_indexes {
	OP_0 = 0,
	OP_1,
	OP_2,
	OP_3,
	NUM_OP,
	DEFAULT_OP = OP_1,
	OP_DSP_H   = OP_1,
};

static unsigned int op_list[NUM_OP][NUM_OP_PARMS] = {
	[OP_0] = {
		[_54M_SRC_CLK_KHZ]	=  54000,
		[_48M_SRC_CLK_KHZ]	=  96000,
		[DPLL_CLK_KHZ]		= 400000,
		[CORE_CLK_KHZ]		= 800000,
		[MPU_CLK_KHZ]		= 400000,
		[DSP_CLK_KHZ]		= 133000,
		[DSP_IF_CLK_KHZ]	= 133000,
		[DSP_SYNC]		= SYNC_DISABLED,
		[IVA_CLK_KHZ]		= 200000,
		[IVA_SYNC]		= SYNC_DISABLED,
		[GFX_CLK_KHZ]		=  66000,
		[L3_CLK_KHZ]		= 133000,
		[L4_CLK_KHZ]		= 133000,
		[DSS1_CLK_KHZ]		= 133000,
		[DSS2_CLK_KHZ]		= SYS_CLK,
		[VLYNQ_CLK_KHZ]		=  66000,
		[SSI_CLK_KHZ]		= 266000,
		[USB_CLK_KHZ]		=  66000,
		[VCORE_ROOF_MV]		=   1375,
	},
	[OP_1] = {
		[_54M_SRC_CLK_KHZ]	=  54000,
		[_48M_SRC_CLK_KHZ]	=  96000,
		[DPLL_CLK_KHZ]		= 330000,
		[CORE_CLK_KHZ]		= 660000,
		[MPU_CLK_KHZ]		= 330000,
		[DSP_CLK_KHZ]		= 220000,
		[DSP_IF_CLK_KHZ]	= 110000,
		[DSP_SYNC]		= SYNC_DISABLED,
		[IVA_CLK_KHZ]		= 165000,
		[IVA_SYNC]		= SYNC_DISABLED,
		[GFX_CLK_KHZ]		=  55000,
		[L3_CLK_KHZ]		= 110000,
		[L4_CLK_KHZ]		= 110000,
		[DSS1_CLK_KHZ]		= 110000,
		[DSS2_CLK_KHZ]		= SYS_CLK,
		[VLYNQ_CLK_KHZ]		=  55000,
		[SSI_CLK_KHZ]		= 220000,
		[USB_CLK_KHZ]		=  55000,
		[VCORE_ROOF_MV]		=   1375,
	},
	[OP_2] = {
		[_54M_SRC_CLK_KHZ]	=  54000,
		[_48M_SRC_CLK_KHZ]	=  96000,
		[DPLL_CLK_KHZ]		= 266000,
		[CORE_CLK_KHZ]		= 532000,
		[MPU_CLK_KHZ]		= 266000,
		[DSP_CLK_KHZ]		= 177000,
		[DSP_IF_CLK_KHZ]	=  88000,
		[DSP_SYNC]		= SYNC_DISABLED,
		[IVA_CLK_KHZ]		= 133000,
		[IVA_SYNC]		= SYNC_DISABLED,
		[GFX_CLK_KHZ]		=  88000,
		[L3_CLK_KHZ]		=  88000,
		[L4_CLK_KHZ]		=  44000,
		[DSS1_CLK_KHZ]		=  88000,
		[DSS2_CLK_KHZ]		= SYS_CLK,
		[VLYNQ_CLK_KHZ]		=  44000,
		[SSI_CLK_KHZ]		= 177000,
		[USB_CLK_KHZ]		=  44000,
		[VCORE_ROOF_MV]		=   1300,
	},
	[OP_3] = {
		[_54M_SRC_CLK_KHZ]	=  54000,
		[_48M_SRC_CLK_KHZ]	=  96000,
		[DPLL_CLK_KHZ]		= 330000,
		[CORE_CLK_KHZ]		= 330000,
		[MPU_CLK_KHZ]		= 165000,
		[DSP_CLK_KHZ]		= 110000,
		[DSP_IF_CLK_KHZ]	=  55000,
		[DSP_SYNC]		= SYNC_DISABLED,
		[IVA_CLK_KHZ]		=  82000,
		[IVA_SYNC]		= SYNC_DISABLED,
		[GFX_CLK_KHZ]		=  55000,
		[L3_CLK_KHZ]		=  55000,
		[L4_CLK_KHZ]		=  27500,
		[DSS1_CLK_KHZ]		= 110000,
		[DSS2_CLK_KHZ]		= SYS_CLK,
		[VLYNQ_CLK_KHZ]		=  27500,
		[SSI_CLK_KHZ]		= 110000,
		[USB_CLK_KHZ]		=  27500,
		[VCORE_ROOF_MV]		=   1225,
	},
};

static DEFINE_MUTEX(dvfs_mutex);
extern struct dvfs_cfg_data dvfs_cfg;

extern void omap2_clk_recalc_tree(void);

/**
 * dsp_force_high - DSP need full performance
 *
 * Set when the DSP-GW indicates that the DSP need full performance,
 * and used by the CPUFREQ_POLICY_NOTIFIER chain to constrain the
 * available frequencies.
 */
static int dsp_force_high;

/**
 * dsp_state_updated - called by the DSP-GW when some internal state changes
 *
 * @type: type of information
 * @d: data associated
 *
 * Currently this function is called when a dynamic task is loaded
 * or unloaded on the DSP.
 * The problem addressed here is that if the DSP has at least one
 * dynamic task loaded then we should use the DSP-HI operating point,
 * or cpufreq-speak: apply a policy to limit the frequency to this.
 */
void dsp_state_updated(enum dsp_state_change type, void *d)
{
	int n_tasks = *(int *)d;
	int new_dfh;

	switch (type) {
	case DSP_DYNAMIC_TASK_CHANGED:
		new_dfh = (n_tasks > 0);
		if (new_dfh != dsp_force_high) {
			dsp_force_high = new_dfh;
			cpufreq_update_policy(0);
		}
		break;
	default:
		break;
	}
}
EXPORT_SYMBOL(dsp_state_updated);

/**
 * dsp_state_change_notifier - called by CPUFREQ_POLICY_NOTIFIER chain
 *
 * @nb: calling notifier_block
 * @event: reason we are called
 * @data:  associated data
 *
 * This function is called in response to a call to cpufreq_update_policy.
 * If "dsp_force_high" is active, then we force DSP-HI operation-point.
 */
static int dsp_state_change_notifier(struct notifier_block *nb,
				     unsigned long event, void *data)
{
	struct cpufreq_policy *policy = data;
	if (event != CPUFREQ_INCOMPATIBLE)
		return 0;

	if (dsp_force_high) {
		cpufreq_verify_within_limits(policy,
			     op_list[OP_DSP_H][MPU_CLK_KHZ],
			     op_list[OP_DSP_H][MPU_CLK_KHZ]);
	}
	return 0;
}

static struct notifier_block dsp_state_change_block = {
       .notifier_call = dsp_state_change_notifier,
};

int __init n800_dvfs_init(void)
{
	omap242x_load_op_list((const unsigned int (*)[NUM_OP_PARMS])op_list,
			      NUM_OP, DEFAULT_OP);
	cpufreq_register_notifier(&dsp_state_change_block,
				  CPUFREQ_POLICY_NOTIFIER);
	return 0;
}
