/*
 * board-n800-camera.c
 *
 * Copyright (C) 2007 Nokia Corporation
 *
 * Contact: Sakari Ailus <sakari.ailus@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <linux/clk.h>
#include <linux/platform_device.h>

#include <linux/delay.h>
#include <asm/mach-types.h>                                  
#include <asm/arch/menelaus.h>
#include <asm/arch/gpio.h>
#include <asm/arch/board.h>

#include <../drivers/cbus/retu.h>

#include "../../../drivers/media/video/omap/tcm825x.h"

#if defined(CONFIG_VIDEO_CAMERA_SENSOR_TCM825X)			\
	|| defined(CONFIG_VIDEO_CAMERA_SENSOR_TCM825X_MODULE)

#if !defined(CONFIG_CBUS_RETU) && !defined(CONFIG_CBUS_RETU_MODULE)
#error Please select CONFIG_CBUS_RETU
#endif

#if !defined(CONFIG_MENELAUS) && !defined(CONFIG_MENELAUS_MODULE)
#error Please select CONFIG_MENELAUS
#endif

#define N800_CAM_SENSOR_RESET_GPIO	53

static int sensor_okay;

/* common TCM825X register initialization for all image sizes, pixel
 * formats, and frame rates
 */

#ifdef CONFIG_MACH_NOKIA_N800
const static struct tcm825x_reg tcm825x_regs_n800[] = {
	/* initial settings for 2.5 V */
	{0x00, 0x03}, {0x03, 0x29}, {0xaa, 0x2a}, {0xc0, 0x2b},
	{0x10, 0x2c}, {0x4c, 0x2d}, {0x9c, 0x3f},

	/* main settings */
	{0x00, 0x00}, {0x30, 0x01}, {0x0e, 0x02}, /* initial */
	{0x0f, 0x04}, {0x02, 0x05}, {0x0d, 0x06}, {0xc0, 0x07},
	{0x38, 0x08}, {0x50, 0x09}, {0x80, 0x0a}, {0x40, 0x0b},
	{0x40, 0x0c}, {0x00, 0x0d}, {0x04, 0x0e}, {0x04, 0x0f},
	{0x22, 0x10}, {0x96, 0x11}, {0xf0, 0x12}, {0x08, 0x13},
	{0x08, 0x14}, {0x30, 0x15}, {0x30, 0x16}, {0x01, 0x17},
	{0x40, 0x18}, {0x87, 0x19}, {0x2b, 0x1a}, {0x44, 0x1b},
	{0x52, 0x1c}, {0x44, 0x1d}, {0x68, 0x1e}, {0x00, 0x1f},
	{0x00, 0x20}, {0x01, 0x21}, {0x27, 0x22}, {0x40, 0x23},
	{0x27, 0x24}, {0x5f, 0x25}, {0x00, 0x26}, {0x16, 0x27},
	{0x23, 0x28}, /* initial */ /* initial */ /* initial */
	/* initial */ /* initial */ {0x00, 0x2e}, {0x00, 0x2f},
	{0x00, 0x30}, {0x00, 0x31}, {0x00, 0x32}, {0x00, 0x33},
	{0x00, 0x34}, {0x00, 0x35}, {0x00, 0x36}, {0x00, 0x37},
	{0x00, 0x38}, {0x8c, 0x39}, {0xc8, 0x3A}, {0x80, 0x3b},
	{0x00, 0x3c}, {0x17, 0x3d}, {0x85, 0x3e}, /* initial */
	{0xa0, 0x40}, {0x00, 0x41}, {0x00, 0x42}, {0x00, 0x43},
	{0x08, 0x44}, {0x12, 0x45}, {0x00, 0x46}, {0x20, 0x47},
	{0x30, 0x48}, {0x18, 0x49}, {0x20, 0x4a}, {0x4d, 0x4b},
	{0x0c, 0x4c}, {0xe0, 0x4d}, {0x20, 0x4e}, {0x89, 0x4f},
	{0x21, 0x50}, {0x80, 0x51}, {0x02, 0x52}, {0x00, 0x53},
	{0x30, 0x54}, {0x90, 0x55}, {0x40, 0x56}, {0x06, 0x57},
	{0x0f, 0x58}, {0x23, 0x59}, {0x08, 0x5A}, {0x04, 0x5b},
	{0x08, 0x5c}, {0x08, 0x5d}, {0x08, 0x5e}, {0x08, 0x5f},
	{TCM825X_VAL_TERM, TCM825X_REG_TERM}
};
#endif

#ifdef CONFIG_MACH_NOKIA_RX44
const static struct tcm825x_reg tcm825x_regs_rx44[] = {
	/* initial settings for 2.5 V */
	{0x00, 0x03}, {0x03, 0x29}, {0xaa, 0x2a}, {0xc0, 0x2b},
	{0x10, 0x2c}, {0x4c, 0x2d}, {0x9c, 0x3f},

	/* main settings */
	{0x00, 0x00}, {0x30, 0x01}, {0x0e, 0x02}, /* initial */
	{0xcf, 0x04}, {0x02, 0x05}, {0x0d, 0x06}, {0xc0, 0x07},
	{0x38, 0x08}, {0x50, 0x09}, {0x80, 0x0a}, {0x40, 0x0b},
	{0x40, 0x0c}, {0x00, 0x0d}, {0x04, 0x0e}, {0x04, 0x0f},
	{0x22, 0x10}, {0x96, 0x11}, {0xf0, 0x12}, {0x08, 0x13},
	{0x08, 0x14}, {0x30, 0x15}, {0x30, 0x16}, {0x01, 0x17},
	{0x40, 0x18}, {0x87, 0x19}, {0x2b, 0x1a}, {0x44, 0x1b},
	{0x52, 0x1c}, {0x44, 0x1d}, {0x68, 0x1e}, {0x00, 0x1f},
	{0x00, 0x20}, {0x01, 0x21}, {0x27, 0x22}, {0x40, 0x23},
	{0x27, 0x24}, {0x5f, 0x25}, {0x00, 0x26}, {0x16, 0x27},
	{0x23, 0x28}, /* initial */ /* initial */ /* initial */
	/* initial */ /* initial */ {0x00, 0x2e}, {0x00, 0x2f},
	{0x00, 0x30}, {0x00, 0x31}, {0x00, 0x32}, {0x00, 0x33},
	{0x00, 0x34}, {0x00, 0x35}, {0x00, 0x36}, {0x00, 0x37},
	{0x00, 0x38}, {0x8c, 0x39}, {0xc8, 0x3A}, {0x80, 0x3b},
	{0x00, 0x3c}, {0x17, 0x3d}, {0x85, 0x3e}, /* initial */
	{0xa0, 0x40}, {0x00, 0x41}, {0x00, 0x42}, {0x00, 0x43},
	{0x08, 0x44}, {0x12, 0x45}, {0x00, 0x46}, {0x20, 0x47},
	{0x30, 0x48}, {0x18, 0x49}, {0x20, 0x4a}, {0x4d, 0x4b},
	{0x0c, 0x4c}, {0xe0, 0x4d}, {0x20, 0x4e}, {0x89, 0x4f},
	{0x21, 0x50}, {0x80, 0x51}, {0x02, 0x52}, {0x00, 0x53},
	{0x30, 0x54}, {0x90, 0x55}, {0x40, 0x56}, {0x06, 0x57},
	{0x0f, 0x58}, {0x23, 0x59}, {0x08, 0x5A}, {0x04, 0x5b},
	{0x08, 0x5c}, {0x08, 0x5d}, {0x08, 0x5e}, {0x08, 0x5f},
	{TCM825X_VAL_TERM, TCM825X_REG_TERM}
};
#endif

/*
 * VSIM1	--> CAM_IOVDD	--> IOVDD (1.8 V)
 */
static int tcm825x_sensor_power_on(void)
{
	int ret;

	/* Set VMEM to 1.5V and VIO to 2.5V */
	ret = menelaus_set_vmem(1500);
	if (ret < 0) {
		/* Try once more, it seems the sensor power up causes
		 * some problems on the I2C bus. */
		ret = menelaus_set_vmem(1500);
		if (ret < 0)
			return ret;
	}

	ret = menelaus_set_vio(2500);
	if (ret < 0)
		return ret;

	/* Disable automatic VSim1 powersaving */
	ret = retu_read_reg(RETU_REG_CC2);
	retu_write_reg(RETU_REG_CC2, ret | 0x11);

	/* Set VSim1 on */
	retu_write_reg(RETU_REG_CTRL_SET, 0x0080);
	msleep(1);

	omap_set_gpio_dataout(N800_CAM_SENSOR_RESET_GPIO, 1);
	msleep(1);

	return 0;
}

static int tcm825x_sensor_power_off(void)
{
	int ret;

	omap_set_gpio_dataout(N800_CAM_SENSOR_RESET_GPIO, 0);
	msleep(1);

	/* Set VSim1 off */
	retu_write_reg(RETU_REG_CTRL_CLR, 0x0080);
	msleep(1);

	/* Set VIO_MODE to off */
	ret = menelaus_set_vio(0);
	if (ret < 0)
		return ret;
	msleep(1);

	/* Set VMEM_MODE to off */
	ret = menelaus_set_vmem(0);
	if (ret < 0)
		return ret;
	msleep(1);

	return 0;
}

static int tcm825x_sensor_power_set(int power) {
	BUG_ON(!sensor_okay);
	
	if (power)
		return tcm825x_sensor_power_on();
	else
		return tcm825x_sensor_power_off();
}

static int tcm825x_sensor_is_okay(void) {
	return sensor_okay;
}

static const struct tcm825x_reg *tcm825x_default_regs(void) {
	if (machine_is_nokia_n800())
		return tcm825x_regs_n800;
	if (machine_is_nokia_rx44())
		return tcm825x_regs_rx44;

	BUG();

	return NULL;
}

static int tcm825x_is_upside_down(void)
{
	if (machine_is_nokia_rx44())
		return 1;

	return 0;
}

struct omap_camera_sensor_config n800_camera_sensor_config = {
	.is_okay	= &tcm825x_sensor_is_okay,
	.power_set	= &tcm825x_sensor_power_set,
	.default_regs	= &tcm825x_default_regs,
	.is_upside_down = &tcm825x_is_upside_down,
};

void __init n800_cam_init(void)
{
	int r;

	r = omap_request_gpio(N800_CAM_SENSOR_RESET_GPIO);
	if (r < 0) {
		printk(KERN_WARNING "%s: failed to request gpio\n",
			__FUNCTION__);
		return;
	}

	omap_set_gpio_dataout(N800_CAM_SENSOR_RESET_GPIO, 0);
	omap_set_gpio_direction(N800_CAM_SENSOR_RESET_GPIO, 0);

	sensor_okay = 1;
}

#else
void __init n800_cam_init(void)
{
}

#endif
