/*
 * linux/arch/arm/mach-omap2/board-n800-audio.c
 *
 * Copyright (C) 2006 Nokia Corporation
 * Contact: Juha Yrjola
 *          Jarkko Nikula <jarkko.nikula@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <linux/delay.h>
#include <linux/err.h>
#include <linux/clk.h>
#include <linux/platform_device.h>
#include <linux/spi/tsc2301.h>
#include <linux/tlv320aic33.h>

#include <asm/io.h>
#include <asm/mach-types.h>
#include <asm/arch/eac.h>
#include <asm/arch/gpio.h>

#include "../plat-omap/dsp/dsp_common.h"

#if (defined(CONFIG_SPI_TSC2301_AUDIO) || defined(CONFIG_SND_AIC33)) && \
     defined(CONFIG_SND_OMAP24XX_EAC)
#define AUDIO_ENABLED

static struct clk *sys_clkout2;
static struct clk *func96m_clk;
static struct device *eac_device;
static struct device *codec_device;

static int enable_audio;
static int audio_ok;
static spinlock_t audio_lock;

/*
 * Leaving EAC and sys_clkout2 pins multiplexed to those subsystems results
 * in about 2 mA extra current leak when audios are powered down. The
 * workaround is to multiplex them to protected mode (with pull-ups enabled)
 * whenever audio is not being used.
 */
static int eac_mux_disabled = 0;
static int clkout2_mux_disabled = 0;
static u32 saved_mux[2];

static void n800_enable_eac_mux(void)
{
	if (!eac_mux_disabled)
		return;
	__raw_writel(saved_mux[1], IO_ADDRESS(0x48000124));
	eac_mux_disabled = 0;
}

static void n800_disable_eac_mux(void)
{
	if (eac_mux_disabled) {
		WARN_ON(eac_mux_disabled);
		return;
	}
	saved_mux[1] = __raw_readl(IO_ADDRESS(0x48000124));
	__raw_writel(0x1f1f1f1f, IO_ADDRESS(0x48000124));
	eac_mux_disabled = 1;
}

static void n800_enable_clkout2_mux(void)
{
	if (!clkout2_mux_disabled)
		return;
	__raw_writel(saved_mux[0], IO_ADDRESS(0x480000e8));
	clkout2_mux_disabled = 0;
}

static void n800_disable_clkout2_mux(void)
{
	u32 l;

	if (clkout2_mux_disabled) {
		WARN_ON(clkout2_mux_disabled);
		return;
	}
	saved_mux[0] = __raw_readl(IO_ADDRESS(0x480000e8));
	l = saved_mux[0] & ~0xff;
	l |= 0x1f;
	__raw_writel(l, IO_ADDRESS(0x480000e8));
	clkout2_mux_disabled = 1;
}

static int n800_eac_enable_ext_clocks(struct device *dev)
{
	BUG_ON(codec_device == NULL);
	tsc2301_mixer_enable_mclk(codec_device);

	return 0;
}

static void n800_eac_disable_ext_clocks(struct device *dev)
{
	BUG_ON(codec_device == NULL);
	tsc2301_mixer_disable_mclk(codec_device);
}

static int n800_audio_set_power(void *pdata, int dac, int adc)
{
	int enable = (dac || adc);

	BUG_ON(codec_device == NULL);
	if (enable)
		n800_enable_eac_mux();

	tsc2301_mixer_set_power(codec_device, dac, adc);

	if (!enable)
		n800_disable_eac_mux();

	return 0;
}

static int rx44_audio_set_power(void *pdata, int dac, int adc)
{
	int enable = (dac || adc);

	BUG_ON(codec_device == NULL);
	if (enable) {
		n800_enable_eac_mux();
		aic33_enable_mclk(codec_device);
	}

	aic33_mixer_set_power(codec_device, dac, adc);

	if (!enable) {
		aic33_disable_mclk(codec_device);
		n800_disable_eac_mux();
	}

	return 0;
}

static int n800_audio_register_controls(void *pdata, struct snd_card *card)
{
	BUG_ON(codec_device == NULL);
	if (machine_is_nokia_n800())
		return tsc2301_mixer_register_controls(codec_device, card);
	else if (machine_is_nokia_rx44())
		return aic33_mixer_register_controls(codec_device,
							   card);

	return 0;
}

static struct eac_codec n800_eac_codec = {
	.mclk_src = EAC_MCLK_EXT_2x12288000,
	.codec_mode = EAC_CODEC_I2S_MASTER,
	.codec_conf.i2s.polarity_changed_mode = 1,
	.codec_conf.i2s.sync_delay_enable = 0,
	.default_rate = 48000,
	.set_power = n800_audio_set_power,
	.register_controls = n800_audio_register_controls,
	.short_name = "TSC2301",
};

static struct eac_codec rx44_eac_codec = {
	.mclk_src = EAC_MCLK_INT_11290000,
	.codec_mode = EAC_CODEC_I2S_SLAVE,
	.codec_conf.i2s.polarity_changed_mode = 0,
	.codec_conf.i2s.sync_delay_enable = 1,
	.default_rate = 48000,
	.set_power = rx44_audio_set_power,
	.register_controls = n800_audio_register_controls,
	.short_name = "TLV320AIC33",
};

static int n800_register_codec(void)
{
	int r, do_enable = 0;
	unsigned long flags;

	if (machine_is_nokia_n800()) {
		n800_eac_codec.private_data = codec_device;
		r = eac_register_codec(eac_device, &n800_eac_codec);
	} else if (machine_is_nokia_rx44()) {
		rx44_eac_codec.private_data = codec_device;
		r = eac_register_codec(eac_device, &rx44_eac_codec);
	} else
		return 0;

	if (r < 0)
		return r;
	spin_lock_irqsave(&audio_lock, flags);
	audio_ok = 1;
	if (enable_audio)
		do_enable = 1;
	spin_unlock_irqrestore(&audio_lock, flags);
	if (do_enable)
		eac_set_mode(eac_device, 1, 1);
	return 0;
}

static void n800_unregister_codec(void)
{
	audio_ok = 0;
	eac_unregister_codec(eac_device);
	eac_set_mode(eac_device, 0, 0);
}

static int n800_eac_init(struct device *dev)
{
	int r;

	BUG_ON(eac_device != NULL);
	eac_device = dev;
	if (codec_device != NULL) {
		r = n800_register_codec();
		if (r < 0)
			return r;
	}

	return 0;
}

static void n800_eac_cleanup(struct device *dev)
{
	eac_device = NULL;
	if (codec_device != NULL)
		n800_unregister_codec();
}

static int n800_codec_get_clocks(struct device *dev)
{
	sys_clkout2 = clk_get(dev, "sys_clkout2");
	if (IS_ERR(sys_clkout2)) {
		dev_err(dev, "Could not get sys_clkout2\n");
		return -ENODEV;
	}
	/* configure 12 MHz output on SYS_CLKOUT2. Therefore we must use
	 * 96 MHz as its parent in order to get 12 MHz */
	func96m_clk = clk_get(dev, "func_96m_ck");
	if (IS_ERR(func96m_clk)) {
		dev_err(dev, "Could not get func 96M clock\n");
		clk_put(sys_clkout2);
		return -ENODEV;
	}

	clk_set_parent(sys_clkout2, func96m_clk);
	clk_set_rate(sys_clkout2, 12000000);

	return 0;
}

static void n800_codec_put_clocks(struct device *dev)
{
	clk_put(func96m_clk);
	clk_put(sys_clkout2);
}

static int n800_codec_enable_clock(struct device *dev)
{
	n800_enable_clkout2_mux();
	return clk_enable(sys_clkout2);
}

static void n800_codec_disable_clock(struct device *dev)
{
	clk_disable(sys_clkout2);
	n800_disable_clkout2_mux();
}

static int n800_codec_init(struct device *dev)
{
	int r;

	BUG_ON(codec_device != NULL);
	codec_device = dev;
	if ((r = n800_codec_get_clocks(dev)) < 0)
		return r;
	if (eac_device != NULL) {
		r = n800_register_codec();
		if (r < 0) {
			n800_codec_put_clocks(dev);
			return r;
		}
	}
	return 0;
}

static void n800_codec_cleanup(struct device *dev)
{
	codec_device = NULL;
	if (eac_device != NULL)
		n800_unregister_codec();
	n800_codec_put_clocks(dev);
}

static struct eac_platform_data n800_eac_data = {
	.init = n800_eac_init,
	.cleanup = n800_eac_cleanup,
	.enable_ext_clocks = n800_eac_enable_ext_clocks,
	.disable_ext_clocks = n800_eac_disable_ext_clocks,
};

static struct eac_platform_data rx44_eac_data = {
	.init = n800_eac_init,
	.cleanup = n800_eac_cleanup,
};

static const struct tsc2301_mixer_gpio n800_mixer_gpios[] = {
	{
		.name			= "Headset Amplifier",
		.gpio			= 1,
		.deactivate_on_pd	= 1,
	}, {
		.name			= "Speaker Amplifier",
		.gpio			= 2,
		.def_enable		= 1,
		.deactivate_on_pd	= 1,
	}, {
		.name			= "Headset Mic Select",
		.gpio			= 3,
	}
};

static struct platform_device retu_headset_device = {
	.name		= "retu-headset",
	.id		= -1,
	.dev		= {
		.release	= NULL,
	},
};

void __init n800_audio_init(struct tsc2301_platform_data *tc)
{
	spin_lock_init(&audio_lock);

	if (platform_device_register(&retu_headset_device) < 0)
		return;
	omap_init_eac(&n800_eac_data);

	tc->pll_pdc = 7;
	tc->pll_a = 7;
	tc->pll_n = 9;
	tc->pll_output = 1;
	tc->mclk_ratio = TSC2301_MCLK_256xFS;
	tc->i2s_sample_rate = TSC2301_I2S_SR_48000;
	tc->i2s_format = TSC2301_I2S_FORMAT0;
	tc->power_down_blocks = TSC2301_REG_PD_MISC_MOPD;
	tc->mixer_gpios = n800_mixer_gpios;
	tc->n_mixer_gpios = ARRAY_SIZE(n800_mixer_gpios);
	tc->codec_init = n800_codec_init;
	tc->codec_cleanup = n800_codec_cleanup;
	tc->enable_clock = n800_codec_enable_clock;
	tc->disable_clock = n800_codec_disable_clock;
}

#define RX44_AIC33_RESET_GPIO	118
#define RX44_HEADSET_AMP_GPIO	10
#define RX44_SPEAKER_AMP_GPIO	101

static void rx44_aic33_reset(int reset_active)
{
	omap_set_gpio_dataout(RX44_AIC33_RESET_GPIO, !reset_active);
}

static void rx44_headset_amplifier_ctrl(struct device *dev, int on)
{
	/*
	 * Wait until AIC33 HP output to amplifier is biased before activating
	 * the amplifier. This is approximately 100 ms in our HW
	 */
	if (on)
		msleep(100);
	omap_set_gpio_dataout(RX44_HEADSET_AMP_GPIO, on);
}

static void rx44_speaker_amplifier_ctrl(struct device *dev, int on)
{
	omap_set_gpio_dataout(RX44_SPEAKER_AMP_GPIO, on);
}

static struct aic33_dmic_data rx44_dmic_config = {
	.dmic_rate		= AIC33_DMIC_OVERSAMPLING_64,
	.needs_bias		= 1,
	.dmic_bias		= AIC33_MICBIAS_2V,
};

static struct aic33_hp_data rx44_hp_config = {
	.ac_coupled		= 1,
	.pd_tri_stated		= 0,
};

static struct aic33_platform_data aic33_config = {
	.codec_reset		= rx44_aic33_reset,
	.codec_init		= n800_codec_init,
	.codec_cleanup		= n800_codec_cleanup,
	.enable_clock		= n800_codec_enable_clock,
	.disable_clock		= n800_codec_disable_clock,
	.ext_hp_amplifier_ctrl	= rx44_headset_amplifier_ctrl,
	.ext_lo_amplifier_ctrl	= rx44_speaker_amplifier_ctrl,
	.bclk_output		= 1,
	.wclk_output		= 1,
	.pll_p			= 1,
	.pll_r			= 1,
	.pll_j			= 8,
	.pll_d			= 1920,
	.adc_hp_filter		= AIC33_ADC_HP_0_0045xFS,
	.dmic_data		= &rx44_dmic_config,
	.hp_data		= &rx44_hp_config,
};

static struct platform_device rx44_aic33_device = {
	.name		= "aic33-driver",
	.id		= -1,
	.dev		= {
		.platform_data	= &aic33_config,
	},
};

void __init rx44_audio_init(void)
{
	spin_lock_init(&audio_lock);

	if (omap_request_gpio(RX44_AIC33_RESET_GPIO) < 0)
		BUG();
	if (omap_request_gpio(RX44_HEADSET_AMP_GPIO) < 0)
		BUG();
	if (omap_request_gpio(RX44_SPEAKER_AMP_GPIO) < 0)
		BUG();
	omap_set_gpio_direction(RX44_AIC33_RESET_GPIO, 0);
	omap_set_gpio_direction(RX44_HEADSET_AMP_GPIO, 0);
	omap_set_gpio_direction(RX44_SPEAKER_AMP_GPIO, 0);
	if (platform_device_register(&retu_headset_device) < 0)
		BUG();
	if (platform_device_register(&rx44_aic33_device) < 0)
		BUG();

	omap_init_eac(&rx44_eac_data);
}

#else

void __init n800_audio_init(void)
{
}
void __init rx44_audio_init(void)
{
}

#endif

#ifdef CONFIG_OMAP_DSP

int n800_audio_enable(struct dsp_kfunc_device *kdev, int stage)
{
#ifdef AUDIO_ENABLED
	unsigned long flags;
	int do_enable = 0;

	spin_lock_irqsave(&audio_lock, flags);

	pr_debug("DSP power up request (audio codec %sinitialized)\n",
		 audio_ok ? "" : "not ");

	if (enable_audio)
		goto out;
	enable_audio = 1;
	if (audio_ok)
		do_enable = 1;
out:
	spin_unlock_irqrestore(&audio_lock, flags);
	if (do_enable)
		eac_set_mode(eac_device, 1, 1);
#endif
	return 0;
}

int n800_audio_disable(struct dsp_kfunc_device *kdev, int stage)
{
#ifdef AUDIO_ENABLED
	unsigned long flags;
	int do_disable = 0;

	spin_lock_irqsave(&audio_lock, flags);

	pr_debug("DSP power down request (audio codec %sinitialized)\n",
		audio_ok ? "" : "not ");

	if (!enable_audio)
		goto out;
	enable_audio = 0;
	if (audio_ok)
		do_disable = 1;
out:
	spin_unlock_irqrestore(&audio_lock, flags);
	if (do_disable)
		eac_set_mode(eac_device, 0, 0);
#endif
	return 0;
}

#endif /* CONFIG_OMAP_DSP */
