/*
 * This file is part of functracer-postproc.
 *
 * Copyright (C) 2008 by Nokia Corporation
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <config.h>
#include <errno.h>
#include <regex.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "list.h"
#include "maps.h"
#include "trace_file.h"

static int parse_header_version(const char *version, int *maj, int *min,
	int *rev)
{
	int ret;

	ret = sscanf(version, "%d.%d.%d", maj, min, rev);
	if (ret != 3) {
		ret = sscanf(version, "%d.%d", maj, min);
		if (ret != 2) {
			fprintf(stderr, "invalid header version format\n");
			return -EINVAL;
		}
		*rev = 0;
	}

	return 0;
}

static int check_header_version(const char *version)
{
	int ret, maj1, min1, rev1, maj2, min2, rev2;

	ret = parse_header_version(version, &maj1, &min1, &rev1);
	if (ret < 0)
		return ret;
	ret = parse_header_version(FT_VERSION, &maj2, &min2, &rev2);
	if (ret < 0)
		return ret;

	if (maj1 < maj2 || min1 < min2 || rev1 < rev2) {
		fprintf(stderr, "error: trace header version (%s) not "
			"supported\n", version);
		return -EINVAL;
	}

	return 0;
}

/* Reads header information from .trace file */
static int read_header_info (t_trace_file *trace_data, char line[LINE_MAX])
{
	t_header *header = &trace_data->header;
	int ret;

	header->is_resolved = 0;

	ret = sscanf(line, "generated by functracer %10[^,], with resolved names, %s",
		header->version, header->arch);
	if (header->version[sizeof(header->version) - 1] != '\0') {
		fprintf(stderr, "error: not enough space to hold version string\n");
		return -EINVAL;
	} else if (ret != 2) {
		ret = sscanf(line, "generated by functracer %10[^,], %s",
			header->version, header->arch);
		if (header->version[sizeof(header->version) - 1] != '\0') {
			fprintf(stderr, "error: not enough space to hold version string\n");
			return -EINVAL;
		} else if (ret != 2) {
			fprintf(stderr, "error: invalid header format\n");
			return -EINVAL;
		}
	} else header->is_resolved = 1;

	if (strcmp(header->arch, BUILD_ARCH)) {
		fprintf(stderr, "error: trace architecture (%s) not supported\n",
			header->arch);
		return -EINVAL;
	}

	ret = check_header_version(header->version);
	if (ret < 0)
		return ret;

	return 0;
}

/* Read .trace file and store entries on trace (t_list) list */
int read_trace_file (t_trace_file *trace_data)
{
	regex_t r;
	char line[LINE_MAX], fcn_arg[LINE_MAX], func[LINE_MAX];
	static t_line *line_info;
	t_list *list = &trace_data->list;
	FILE *trace_file = trace_data->trace_file;
	int ret;
	char *ptr;


	line_info = (t_line *) malloc(sizeof(t_line));
	if (!line_info) {
		perror("malloc");
		return -ENOMEM;
	}

	/* initialize regex structure */
	if ((regcomp (&r, "^[0-9]+\\.", REG_EXTENDED|REG_NOSUB)) != 0 ) {
		fprintf (stderr, "error: could not initialize regex_t structure\n");
		return -EINVAL;
	}

	/* read header info */
	fgets(line, LINE_MAX, trace_file);
	if (line[strlen(line)-1] == '\n')
		line[strlen(line)-1] = '\0';

	if ((ret = read_header_info(trace_data, line)))
		return ret;

	/* read traces (incl. backtraces) */
	while (fgets(line, LINE_MAX, trace_file)) {

		if (line[strlen(line)-1] == '\n')
			line[strlen(line)-1] = '\0';

		if ((regexec(&r, line,  0, (regmatch_t *) NULL, 0)) != 1) {

			/* reset line_info structure */
			memset(line_info, 0, sizeof(line_info));

			/* get line offset from .trace file */
			line_info->offset = ftell(trace_file);

			/* capture trace id and type from line */
			ret = sscanf(line, "%d. %s", &line_info->identifier, fcn_arg);
			if (ret != 2) {
				fprintf(stderr, "error: invalid trace line format: %s\n", line);
				return -EINVAL;
			}

			ptr = strrchr(fcn_arg, '(');
			memset(func, 0, sizeof(func));
			strncpy(func, fcn_arg, ptr - fcn_arg);

			if (strcmp(func, "free") == 0) {

				line_info->type = TYPE_FREE;

				ret = sscanf(ptr, "(%lx)", &line_info->address);
				if (ret != 1) {
					fprintf(stderr, "error: invalid free address\n");
					return -EINVAL;
				}

			} else if (strcmp(func, "malloc") == 0) {

				line_info->type = TYPE_MALLOC;

				ret = sscanf(ptr, "(%d)", &line_info->size);
				if (ret != 1) {
					fprintf(stderr, "error: invalid malloc size\n");
					return -EINVAL;
				}
				ret = sscanf(line, "%*d. %*s = %lx", &line_info->address);
				if (ret != 1) {
					fprintf(stderr, "error: invalid malloc address\n");
					return -EINVAL;
				}

				/* remove spurious malloc() */
				if ((list->last->type == TYPE_MALLOC) &&
					 (list->last->address == line_info->address) &&
					 (list->last->size == line_info->size))
					if ((ret = list_remove_last(list)))
						return ret;

			} else if (strcmp(func, "calloc") == 0) {

				line_info->type = TYPE_CALLOC;

				ret = sscanf(ptr, "(%d,", &line_info->nmemb);
				if (ret != 1) {
					fprintf(stderr, "error: invalid calloc array of elements (nmemb)\n");
					return -EINVAL;
				}
				ret = sscanf(line, "%*d. %*s%d) = %lx", &line_info->size,
					&line_info->address);
				if (ret != 2) {
					fprintf(stderr, "error: invalid calloc size/address\n");
					return -EINVAL;
				}

				/* remove spurious malloc() */
				if ((list->last->type == TYPE_MALLOC) &&
					 (list->last->address == line_info->address) &&
					 (list->last->size == line_info->size))
					if ((ret = list_remove_last(list)))
						return ret;

			} else if (strcmp(func, "realloc") == 0) {

				line_info->type = TYPE_REALLOC;

				ret = sscanf(ptr, "(%lx,", &line_info->address);
				if (ret != 1) {
					fprintf(stderr, "error: invalid realloc address\n");
					return -EINVAL;
				}
				ret = sscanf(line, "%*d. %*s%d) = %lx", &line_info->size,
					&line_info->address_new);
				if (ret != 2) {
					fprintf(stderr, "error: invalid realloc size/new address\n");
					return -EINVAL;
				}

				/* remove spurious realloc() */
				if ((list->last->type == TYPE_REALLOC) &&
					 (list->last->size == line_info->size) &&
					 (list->last->address == line_info->address) &&
					 (list->last->address_new == line_info->address_new))
					if ((ret = list_remove_last(list)))
						return ret;

				/* remove spurious malloc() - x86 arch only */
				if ((list->last->type == TYPE_MALLOC) &&
					(!line_info->address) &&
					(line_info->address_new == list->last->address) &&
					(line_info->size == list->last->size))
					if ((ret = list_remove_last(list)))
						return ret;

				/* remove spurious free() */
				if ((list->last->type == TYPE_FREE) &&
					(list->last->address == line_info->address) &&
					(line_info->size == 0))
					if ((ret = list_remove_last(list)))
						return ret;
			} else {
				fprintf(stderr, "error: invalid trace function name\n");
				return -EINVAL;
			}

			if ((ret = list_add(list, line_info)))
				return ret;
		}
	}

	regfree(&r);

	free(line_info);

	return 0;
}
