/*
 * This file is part of functracer-postproc.
 *
 * Copyright (C) 2008 by Nokia Corporation
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include <errno.h>
#include <config.h>

#include "options.h"

struct t_arguments arguments;

/* print usage information */
static int print_usage (FILE *stream, int exit_code)
{
	fprintf(stream, "Functracer-postproc version %s\n", PACKAGE_VERSION);
	fprintf(stream, "Supports functracer trace header version %s "
		"or greater\n\n", FT_VERSION);
	fprintf(stream, "Usage: %s options [ inputfile ]\n", PACKAGE_NAME);
	fprintf(stream,
		"   -c   --compress    Join allocations with same backtrace (use with -f or -l).\n"
		"   -h   --help        Display this usage information.\n"
		"   -v   --verbose     Print verbose messages.\n"
		"   -f   --free        List freed allocations.\n"
		"   -l   --leak        List non-freed allocations (possible leaks).\n"
		"   -V   --version     Print program version.\n");

	exit(exit_code);
}

/* process options from command line parameters */
int process_options (int argc, char *argv[])
{
	int next_option;

	/* a string listing valid short option letters. */
	const char* short_options = "chvflV";

	/* an array describing valid long options. */
	const struct option long_options[] = {
		{	"compress",	0, NULL,	'c' },
		{	"help",	0,	NULL,	'h' },
		{	"verbose",	0, NULL,	'v' },
		{	"free",	0,	NULL,	'f' },
		{	"leak",	0,	NULL,	'l' },
		{	"version",	0,	NULL,	'V' },
		{	NULL,	0,	NULL,	0 }
	};

	/* set arguments values to 0 */
	memset(&arguments, 0, sizeof(t_arguments));

	while (1)	{
		next_option = getopt_long (argc, argv, short_options, long_options, NULL);

		if (next_option == -1) break;

		switch (next_option) {
		case 'c': /* -c or --compress */
			arguments.compress = 1;
			break;
		case 'h': /* -h or --help */
			print_usage (stdout, 0);

		case 'v': /* -v or --verbose */
			/* FIXME: verbose level */
			arguments.verbose++;
			break;

		case 'f': /* -f or -free */
			arguments.free = 1;
			break;

		case 'l': /* -l or --leak */
			arguments.leak = 1;
			break;
		case 'V': /* -V or --version */
			fprintf(stdout, "%s\n", PACKAGE_STRING);
			exit(0);
			break;

		default:
			print_usage(stdout, -EINVAL);
		}
	}

	if (argv[optind])
		/* copy inputfile (if any) to arguments.trace_file */
		strcpy(arguments.trace_file, argv[optind]);
	else {
		fprintf(stderr, "error: no input file\n");
		return -EINVAL;
	}

	return 0;
}
