/*
 * This file is part of functracer-postproc.
 *
 * Copyright (C) 2008 by Nokia Corporation
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "list.h"
#include "maps.h"
#include "options.h"
#include "resolve.h"

/* initialize map list */
int maplist_init (t_maplist *maplist)
{
	maplist->first = (t_map *) malloc(sizeof(t_map));
	if (!maplist->first) {
		perror("malloc");
		return -ENOMEM;
	}

	maplist->last = maplist->first;
	maplist->first->next = NULL;

	return 0;
}

/* Verify if the map list is empty */
int maplist_is_empty (t_maplist *maplist)
{
	return (maplist->first == maplist->last);
}

/* Free map list memory allocations */
int maplist_free (t_maplist *maplist)
{
	t_map *map_current, *map_aux;

	map_current = maplist->first->next; /* skip sentinel */

	free(maplist->first); /* free sentinel */

	while (map_current)
	{
		map_aux = map_current;
		map_current = map_current->next;

		free(map_aux);
	}

	return 0;
}

/* Add a member to map list */
int maplist_add (t_maplist *maplist,
	t_address dyn_addr_init,
	t_address dyn_addr_end,
	char pathname[LINE_MAX]) {
	int ret;

	maplist->last->next = (t_map *) malloc(sizeof(t_map));
	if (!maplist->last->next) {
		perror("malloc");
		return -ENOMEM;
	}

	maplist->last = maplist->last->next;
	maplist->last->next = NULL;

	maplist->last->dyn_addr_init = dyn_addr_init;
	maplist->last->dyn_addr_end = dyn_addr_end;
	strcpy(maplist->last->pathname, pathname);

	ret = is_absolute(maplist->last->pathname);
	if (ret == 0 || ret == 1)
		maplist->last->is_absolute = ret;
	else return ret;

	return 0;
}

/* Read .map file and insert entries on map list */
int read_map_file (t_maplist *maplist)
{
	FILE *map_file;
	t_address dyn_addr_init, dyn_addr_end, offset;
	char line[LINE_MAX], map_fname[LINE_MAX], *aux;
	char perms[LINE_MAX], dev[LINE_MAX], pathname[LINE_MAX];
	int inode, ret;

	strcpy(map_fname, arguments.trace_file);
	if ((aux = strrchr(map_fname, '.')))
		strcpy(aux, ".map");
	else strcat(map_fname, ".map");

	if (!(map_file = fopen(map_fname, "r"))) {
		fprintf(stderr, "error: could not open map file %s\n", map_fname);
		return -ENOENT;
	}

	while (fgets(line, LINE_MAX, map_file)) {

		if (line[strlen(line)-1] == '\n')
			line[strlen(line)-1] = '\0';

		ret = sscanf(line, "%lx-%lx %s", &dyn_addr_init, &dyn_addr_end, perms);
		if (ret != 3) {
			fprintf(stderr, "error: invalid map file format\n");
			return -EINVAL;
		}

		if (!strcmp(perms, "r-xp")) { /* grab only executable files */
			ret = sscanf(line, "%*x-%*x %*s %lx %s %d    %s",
				&offset, dev, &inode, pathname);
			if (ret < 3) { /* pathname not shown in some cases */
				fprintf(stderr, "error: invalid map file format\n");
				return -EINVAL;
			}

			/* skip map entry w/o pathname or [vdso] */
			if ((ret == 4) && strcmp(pathname, "[vdso]"))
				if ((ret = maplist_add(maplist, dyn_addr_init, dyn_addr_end, pathname)))
					return ret;
		}
	}

	fclose(map_file);

	return 0;
}


