/*
 * This file is part of functracer-postproc.
 *
 * Copyright (C) 2008 by Nokia Corporation
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <bfd.h>
#include <config.h>
#include <errno.h>
#include <regex.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "compress.h"
#include "list.h"
#include "maps.h"
#include "options.h"
#include "resolve.h"
#include "trace_file.h"

/* Initialize trace list */
static int list_init (t_list *list)
{
	list->first = (t_line *) malloc(sizeof(t_line));
	if (!list->first) {
		perror("malloc");
		return -ENOMEM;
	}

	list->last = list->first;
	list->first->next = NULL;
	list->first->prev = NULL;

	/* sanity check */
	list->last->type = TYPE_INVALID;

	return 0;
}

/* Verify if the list is empty */
static int list_is_empty (t_list *list)
{
	return (list->first == list->last);
}

/* Print backtrace list contents */
int btlist_print (t_trace_file *trace_data, long offset) {

	regex_t r;
	static t_address address, cut_addr = 0;
	static t_address_info *addr_info;
	static t_map *map;
	t_maplist *maplist = &trace_data->maplist;
	FILE *trace_file = trace_data->trace_file;
	t_header header = trace_data->header;
	char line[LINE_MAX];
	int ret;

	addr_info = (t_address_info *) malloc (sizeof(t_address_info));
	if (!addr_info) {
		perror("malloc");
		return -ENOMEM;
	}

	/* initialize regex structure */
	if ((regcomp (&r, "^[0-9]+\\.", REG_EXTENDED|REG_NOSUB)) != 0 ) {
		fprintf (stderr, "error: could not initialize regex_t structure\n");
		return -EINVAL;
	}

	if (maplist_is_empty(maplist)) {
		fprintf(stderr, "error: map list is empty.\n");
		return -EINVAL;
	}

	fseek(trace_file, offset, SEEK_SET);

	while (fgets(line, LINE_MAX, trace_file)) {

		if (line[strlen(line)-1] == '\n')
			line[strlen(line)-1] = '\0';

		if ((regexec(&r, line, 0, (regmatch_t *) NULL, 0)) != 1)
			break;

		if (header.is_resolved) {
			ret = sscanf(line, "   %*s [%lx]", &address);
			if (ret != 1) {
				fprintf(stderr, "error: invalid backtrace line format\n");
				return -EINVAL;
			}
		} else {
			ret = sscanf(line, "   [%lx]", &address);
			if (ret != 1) {
				fprintf(stderr, "error: invalid backtrace line format\n");
				return -EINVAL;
			}
		}

		map = find_map_entry_from_address(maplist, address);
		if (map == NULL) {
			printf("   0x%08lx: <address unknown>\n", address);
			continue;
		}

		if ((ret = resolve_address(map, address, addr_info))) {
			fprintf(stderr, "error: invalid backtrace address\n");
			return ret;
		}

		/* don't print backtraces below main() */
		if ((!cut_addr) && !strcmp(addr_info->function, "__libc_start_main")) {
			cut_addr = address;
			break;
		} else if (cut_addr == address)
			break;

		if (strcmp(addr_info->filename, "<undefined>")) {
			if (addr_info->line == 0)
				printf ("   0x%08lx: %s (%s in %s)\n",
					address,
					addr_info->function,
					basename(addr_info->filename),
					basename(map->pathname));
			else printf ("   0x%08lx: %s (%s:%d in %s)\n",
					address,
					addr_info->function,
					basename(addr_info->filename),
					addr_info->line,
					basename(map->pathname));
		} else {
			 printf ("   0x%08lx: %s (in %s)\n",
				address,
				addr_info->function,
				basename(map->pathname));
		}
	}

	regfree(&r);

	free(addr_info);

	return 0;
}

/* Print list contents */
static int list_print (t_trace_file *trace_data) {
	t_line *aux;
	int ret;

	if (list_is_empty(&trace_data->list))
		return 0;

	aux = trace_data->list.first->next; /* skip sentinel */

	while (aux)
	{
		switch (aux->type) {
			case TYPE_FREE:
				printf ("%d. free(0x%08lx) = <void>\n",
					aux->identifier, aux->address);
				break;
			case TYPE_MALLOC:
				printf ("%d. malloc(%d) = 0x%08lx\n",
					aux->identifier, aux->size, aux->address);
				break;
			case TYPE_CALLOC:
				printf ("%d. calloc(%d, %d) = 0x%08lx\n",
					aux->identifier, aux->nmemb, aux->size, aux->address);
				break;
			case TYPE_REALLOC:
				printf ("%d. realloc(0x%08lx, %d) = 0x%08lx\n",
					aux->identifier, aux->address, aux->size, aux->address_new);
				break;
			case TYPE_INVALID:
			default:
				break;
		}

		if (aux->another_stack)
			if ((ret = btlist_print(trace_data, aux->offset)))
				return ret;

		aux = aux->next;
	}

	return 0;
}

/* Free list memory allocations */
static int list_free (t_list *list)
{

	t_line *current, *aux;

	current = list->first->next; /* skip sentinel */

	free(list->first); /* free sentinel */

	while (current)
	{
		aux = current;
		current = current->next;

		free(aux);
	}

	return 0;
}

/* Add a member to trace list */
int list_add (t_list *list,
	t_line *line_info) {

	t_line *aux;

	list->last->next = (t_line *) malloc(sizeof(t_line));
	if (!list->last->next) {
		perror("malloc");
		return -ENOMEM;
	}

	aux = list->last;
	list->last = list->last->next;
	list->last->prev = aux;
	list->last->next = NULL;

	list->last->identifier = line_info->identifier;
	list->last->type = line_info->type;
	list->last->address = line_info->address;
	list->last->address_new = line_info->address_new;
	list->last->size = line_info->size;
	list->last->nmemb = line_info->nmemb;
	list->last->another_stack = 1;

	list->last->offset = line_info->offset;

	return 0;
}

/* Remove last member from trace list */
int list_remove_last (t_list *list)
{
	t_line *aux;

	aux = list->last;
	list->last = list->last->prev;

	free(aux);

	return 0;
}

/* Remove one member from trace list */
int list_remove_item (t_list *list, t_line *item)
{
	if (item == NULL)
		return 0;
	if (list->last == item) /* last item */
		list->last = item->prev;

	if (list->first->next == item) /* first item, skip sentinel */
		list->first->next = item->next;

	item->prev->next = item->next;
	
	if (item->next != NULL)
		item->next->prev = item->prev;

	free(item);

	return 0;
}

/* Filter trace information (freed allocations and leaks detection) */
static int list_filter (t_trace_file *trace_data) {

	t_line *current, *aux, *old_current;
	int found = 0, leaks_count, freed_count;
	t_address free_addr;
	t_list *list = &trace_data->list;

	if (list_is_empty(list)) {
		fprintf(stderr, "error: list is empty.\n");
		return -EINVAL;
	}

	leaks_count = freed_count = 0;

	if (arguments.leak && !arguments.free)
		printf("\nleaks detection:\n");
	else if (!arguments.leak && arguments.free)
		printf("\nfreed allocations detection:\n");
	else printf("\nfreed allocations and leaks detection:\n");

	current = list->first->next; /* skip sentinel */

	while (current)
	{
		aux = NULL;
		if (current->type == TYPE_INVALID)
			break;

		if (current->type != TYPE_FREE || (current->type == TYPE_REALLOC && !current->address_new)) { /* {m|c|re}alloc functions only */
			aux = current->next;
			found = 0;

			if (current->type == TYPE_REALLOC) {
				if (!current->address) { /* realloc acting as malloc */
					free_addr = current->address_new;
				} else if (current->address != current->address_new) { /* realloc allocated another memory address */
					free_addr = current->address_new;
				} else { /* realloc used same memory address */
					free_addr = current->address_new;
				}
			} else free_addr = current->address;

			while (aux)
			{
				if ((current->type == TYPE_REALLOC) && (!current->address_new)) {
					found = 1;
					break;
				}

				if (((aux->type == TYPE_FREE) || (aux->type == TYPE_REALLOC)) && (free_addr == aux->address)) {
					if (arguments.free)
						freed_count++;
					found = 1;
					break;
				}
				aux = aux->next;
			}

			if (!found && arguments.leak)
				leaks_count++;
		} else {
			current = current->next;
			continue;
		}
		old_current = current;
		current = current->next;
		if (found) {
			if (arguments.leak)
				list_remove_item(list, old_current);
			if (aux->type == TYPE_FREE || (aux->type == TYPE_REALLOC && !current->address_new)) {
				if (current == aux && current != NULL)
					current = current->next;
				list_remove_item(list, aux);
			}
		} else if (arguments.free) /* remove leaks if we are detecting freed allocations */
			list_remove_item(list, old_current);
	}

	if (arguments.leak && !arguments.free)
		printf("\n%d leaks found\n", leaks_count);
	else if (!arguments.leak && arguments.free)
		printf("\n%d freed allocations found\n", freed_count);
	else printf("\n%d leaks found\n%d freed allocations found\n", leaks_count, freed_count);

	return 0;
}

void list_sort(t_trace_file *trace_data, int (*cmp)(t_trace_file *trace_data,
	       struct t_line* item1, struct t_line* item2))
{
	t_line *p, *q, *e, *list, *tail;
	int insize, nmerges, psize, qsize, i;
	t_list *list_orig = &trace_data->list;

	if (list_is_empty(list_orig))
		return;
	/* remove sentinel from sort */
	list = list_orig->first->next;
	list->prev = list_orig->first->prev;
	
	insize = 1;

	for (;;) {
		p = list;
		list = tail = NULL;
		nmerges = 0;

		while (p) {
			nmerges++;
			q = p;
			psize = 0;
			for (i = 0; i < insize; i++) {
				psize++;
				q = q->next;
				if (!q)
					break;
			}

			qsize = insize;

			while (psize > 0 || (qsize > 0 && q)) {

				if (!psize) {
					e = q;
					q = q->next;
					qsize--;
				} else if (!qsize || !q) {
					e = p;
					p = p->next;
					psize--;
				} else if (cmp(trace_data, p, q) <= 0) {
					e = p;
					p = p->next;
					psize--;
				} else {
					e = q;
					q = q->next;
					qsize--;
				}
				if (tail)
					tail->next = e;
				else
					list = e;
				e->prev = tail;
				tail = e;
			}
			p = q;
		}

		tail->next = NULL;

		if (nmerges <= 1)
			break;

		insize *= 2;
	}

	list_orig->first->next = list;
	list_orig->first->prev = list->prev;
	list->prev = list_orig->first; /* sentinel */
	list_orig->last = tail;
}

int process_data ()
{
	t_trace_file *trace_data;
	int ret;

	trace_data = calloc(1, sizeof(t_trace_file));
	if (trace_data == NULL) {
		fprintf(stderr, "error: could not allocate trace_data\n");
		return -ENOMEM;
	}
	/* open trace file for reading */
	if (!(trace_data->trace_file = fopen(arguments.trace_file, "r"))) {
		fprintf(stderr, "error: could not open trace file %s\n",
			arguments.trace_file);
		ret = -ENOENT;
		goto error;
	}

	/* initialize BFD library */
	bfd_init();

	/* initialize trace list */
	if ((ret = list_init(&trace_data->list)))
		goto error;

	/* initialize map list */
	if ((ret = maplist_init(&trace_data->maplist)))
		goto error;

	/* read data from map file */
	if ((ret = read_map_file(&trace_data->maplist)))
		goto error;

	/* read data from input file */
	if ((ret = read_trace_file(trace_data)))
		goto error;

#if 0
	/* FIXME: implement verbose option properly */
	if ((arguments.verbose) &&
		(ret = list_print(trace_data)))
		goto error;
#endif

	/* filter freed allocations and/or leaks info */
	if ((arguments.leak || arguments.free) &&
		(ret = list_filter(trace_data)))
		goto error;

	if (arguments.compress)
		compact(trace_data);
	
	list_print(trace_data);

	/* close trace file */
	fclose(trace_data->trace_file);

	/* free map list allocations */
	if ((ret = maplist_free(&trace_data->maplist)))
		goto error;

	/* free list memory allocations */
	if ((ret = list_free(&trace_data->list)))
		goto error;

error:
	free(trace_data);

	return ret;
		
}
