/*
 * This file is part of functracer-postproc.
 *
 * Copyright (C) 2008 by Nokia Corporation
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 * Based on backtrace code from libleaks.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <regex.h>

#include "list.h"
#include "maps.h"
#include "trace_file.h"

t_address read_backtrace_lines(t_trace_file *trace_data, long* offset)
{
	regex_t r;
	static t_address address;
	char line[LINE_MAX];
	t_header header = trace_data->header;
	FILE *trace_file = trace_data->trace_file;

	int ret;

	/* initialize regex structure */
	if ((regcomp (&r, "^[0-9]+\\.", REG_EXTENDED|REG_NOSUB)) != 0 ) {
		fprintf (stderr, "error: could not initialize regex_t structure\n");
		return -EINVAL;
	}

	if ((ret = fseek(trace_file, *offset, SEEK_SET) < 0)) {
		fprintf(stderr, "error: could not set the file position indicator\n");
		return -EINVAL;
	}

	if (fgets(line, LINE_MAX, trace_file) == NULL) {
		regfree(&r);
		return 0;
	}
	*offset = ftell(trace_file);

	if (line[strlen(line)-1] == '\n')
		line[strlen(line)-1] = '\0';
 
	if ((regexec(&r, line, 0, (regmatch_t *) NULL, 0)) == 0) {
		regfree(&r);
		return 0;
	}

	if (header.is_resolved) {
		ret = sscanf(line, "   %*s [%lx]", &address);
		if (ret != 1) {
			fprintf(stderr, "error: invalid backtrace line format\n");
			regfree(&r);
			return -EINVAL;
		}
	} else {
		ret = sscanf(line, "   [%lx]", &address);
		if (ret != 1) {
			fprintf(stderr, "error: invalid backtrace line format\n");
			regfree(&r);
			return -EINVAL;
		}
	}

	regfree(&r);
	return address;
}

int backtrace_compare(t_trace_file *trace_data, struct t_line* item1,
		      struct t_line* item2)
{
	long offset1 = item1->offset;
	long offset2 = item2->offset;

	while (1) {
		t_address a = read_backtrace_lines(trace_data, &offset1);
		t_address b = read_backtrace_lines(trace_data, &offset2);

		/* Some sopthisticated operations */
		if (a < b)
			return -1;
		if (a > b)
			return 1;
		if ( !a )
			break;
	}
	
	/* Well, they are equal - so return 0 */
	return 0;
}

void compact(t_trace_file *trace_data)
{
	t_line *current;
	t_list *list = &trace_data->list;

	list_sort(trace_data, backtrace_compare);

	current = list->first->next;
	while (current && current->next) {
        	/* Set flag if that is the last or next item has another stack
		 * trace */
		if (backtrace_compare(trace_data, current, current->next) == 0)
			current->another_stack = 0;
		current = current->next;
	}
}
