/*
 * This file is part of eds-sync
 *
 * Copyright (C) 2007 Nokia Corporation. All rights reserved.
 *
 * Author: Ross Burton <ross@openedhand.com>
 * Author: Onne Gorter <onne.gorter@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#ifndef _XOPEN_SOURCE
#define _XOPEN_SOURCE
#define _XOPEN_SOURCE_EXTENDED
#endif

#include <string.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>

#include <glib/gstdio.h>

#include <libtelepathy/tp-conn.h>

#include "util.h"
#include "eds-sync.h"

#if ! HAVE_E_VCARD_ATTRIBUTE_REMOVE_PARAM
/* Nasty nasty nasty. */
void
e_vcard_attribute_remove_param (EVCardAttribute *attr, const char *param_name)
{
  GList *params = NULL, *l;

  g_return_if_fail (attr);
  g_return_if_fail (param_name);

  /* Copy all parameters which are not the one we are looking for */
  for (l = e_vcard_attribute_get_params (attr); l ; l = l->next) {
    EVCardAttributeParam *param = l->data;
    
    if (g_ascii_strcasecmp (e_vcard_attribute_param_get_name (param), param_name) != 0) {
      params = g_list_prepend (params, e_vcard_attribute_param_copy (param));
    }
  }

  /* Remove all parameters */
  e_vcard_attribute_remove_params (attr);

  /* Now add our copies back */
  for (l = params; l ; l = l->next) {
    e_vcard_attribute_add_param (attr, l->data);
  }
}
#endif

/* TODO: inspect_handles varient */

/**
 * inspect_handle:
 * @conn: a #TpConn
 * @handle: a handle obtained from @conn
 * @username: a location to set the name of @handle. Do not free this string.
 * @error: a #GError to set if there is an error
 *
 * Inspect the #handle on #conn (currently hard-coded to contact handles), and
 * caches the result in the connection object.  If the handle is inspected for a
 * second time, the cached response is returned.  Do not free #username when you
 * are done with it, as it is owned by #conn.
 *
 * Returns: #TRUE if the handle could be inspected, #FALSE otherwise.
 */
gboolean
inspect_handle (TpConn *conn, const guint handle, const char **username, GError **error)
{
  const guint handle_type = TP_CONN_HANDLE_TYPE_CONTACT; /* TODO: make parameter */
  GHashTable *hash;
  char *s = NULL;

  g_return_val_if_fail (TELEPATHY_IS_CONN (conn), FALSE);

  /* Try and get the handle->username hash from the connection, and if it
     doesn't exist create it. */
  hash = g_object_get_data (G_OBJECT (conn), "tp-handle-cache");
  if (hash == NULL) {
    hash = g_hash_table_new_full (NULL, NULL, NULL, g_free);
    g_object_set_data_full (G_OBJECT (conn), "tp-handle-cache",
                            hash, (GDestroyNotify)g_hash_table_destroy);
  }

  /* Look up the handle in the hash, and if there is an entry, return it */
  s = g_hash_table_lookup (hash, GINT_TO_POINTER (handle));
  if (s) {
    *username = s;
    return TRUE;
  }
  
  /* Inspect the handle */
  if (!eds_tp_conn_inspect_handle (DBUS_G_PROXY (conn), handle_type,
                                   handle, &s, error)) {
    /* Got an error, return NULL and let the error percolate up */
    return FALSE;
  }

  /* If we got a username, store it in the hash and return it */
  if (s && s[0] != '\0') {
    g_hash_table_replace (hash, GINT_TO_POINTER (handle), s);
    *username = s;
    return TRUE;
  } else {
    return FALSE;
  }
}

/**
 * remove_handle_cache:
 * @conn: A #TpConn
 * @handle: a Telepathy handle
 *
 * Evict the cached username for the specified handle from the cache.
 */
void
remove_handle_cache (TpConn *conn, const guint handle)
{
  GHashTable *hash;

  g_return_if_fail (TELEPATHY_IS_CONN (conn));
  
  hash = g_object_get_data (G_OBJECT (conn), "tp-handle-cache");
  g_hash_table_remove (hash, GINT_TO_POINTER (handle));
}


gboolean
eds_tp_conn_inspect_handle (DBusGProxy *proxy, const guint IN_handle_type, const guint IN_handle, char ** OUT_arg2, GError **error)
{
  GArray *arr;
  char **names;
  gboolean ret;

  arr = g_array_new (FALSE, FALSE, sizeof(guint));
  g_array_append_val (arr, IN_handle);

  ret = tp_conn_inspect_handles (proxy, IN_handle_type, arr, &names, error);
  if (ret) {
    *OUT_arg2 = names[0];
    g_free (names);
  }
  g_array_free (arr, TRUE);
  return ret;
}

gboolean
eds_tp_conn_request_handle (DBusGProxy *proxy, const guint IN_handle_type, const char * IN_name, guint *OUT_arg2, GError **error)
{
  const char *names[2] = { NULL, NULL };
  GArray *arr;

  names[0] = IN_name;

  if (tp_conn_request_handles (proxy, IN_handle_type, names, &arr, error)) {
    *OUT_arg2 = g_array_index (arr, guint, 0);
    g_array_free (arr, TRUE);
    return TRUE;
  } else {
    return FALSE;
  }
}

void
print_handle (guint handle, TpConn *conn)
{
  GError *error = NULL;
  const char *s;

  g_assert (conn);

  if (!inspect_handle (conn, handle, &s, &error)) {
    g_warning ("Cannot inspect handle %d: %s", handle, _ERROR_MSG (error));
    g_clear_error (&error);
    return;
  }

  g_debug ("%3d -> %s", handle, s);
}


static const char *
contact_state_name (const ContactState state)
{
  switch (state) {
  case CONTACT_STATE_AUTOMATIC:
    return "AUTOMATIC";
  case CONTACT_STATE_ADDED:
    return "ADDED";
  case CONTACT_STATE_BLOCKED:
    return "BLOCKED";
  case CONTACT_STATE_DELETED:
    return "DELETED";
  case CONTACT_STATE_MANUAL:
    return "MANUAL";
  }
  /* Do this here so the compiler can warn about missing cases in the switch */
  g_assert_not_reached();
  return NULL;
}

gboolean
contact_state_is (EContact *contact, const ContactState state)
{
  EVCardAttribute *attr;
  GList *vals;
  const char *name;

  g_assert (contact);

  name = contact_state_name (state);

  attr = e_vcard_get_attribute (E_VCARD (contact), EVC_X_OSSO_CONTACT_STATE);
  if (attr == NULL)
    return FALSE;

  vals = e_vcard_attribute_get_values (attr);
  for (; vals ; vals = g_list_next (vals)) {
    if (g_ascii_strcasecmp (vals->data, name) == 0) {
      return TRUE;
    }
  }
  return FALSE;
}

void
contact_state_remove (EContact *contact, const ContactState state)
{
  EVCardAttribute *attr;
  const char *name;

  g_assert (contact);

  name = contact_state_name (state);

  attr = e_vcard_get_attribute (E_VCARD (contact), EVC_X_OSSO_CONTACT_STATE);
  if (attr == NULL) {
    return;
  }

  e_vcard_attribute_remove_value (attr, name);

  /* If there are no values left in the contact state, remove it */
  if (e_vcard_attribute_get_values (attr) == NULL) {
    e_vcard_remove_attribute (E_VCARD (contact), attr);
  }
}


static const char *
field_state_name (const FieldState state)
{
  switch (state) {
  case FIELD_STATE_TOADD:
    return "TOADD";
  case FIELD_STATE_TODELETE:
    return "TODELETE";
  case FIELD_STATE_TOBLOCK:
    return "TOBLOCK";
  case FIELD_STATE_TOUNBLOCK:
    return "TOUNBLOCK";
  case FIELD_STATE_BLOCKED:
    return "BLOCKED";
  }
  /* Do this here so the compiler can warn about missing cases in the switch */
  g_assert_not_reached();
  return NULL;
}

gboolean
field_state_is (EVCardAttribute *attr, const FieldState state)
{
  GList *vals;
  const char *name;

  g_assert (attr);

  name = field_state_name (state);

  vals = e_vcard_attribute_get_param (attr, EVC_X_OSSO_FIELD_STATE);

  for (; vals ; vals = g_list_next (vals)) {

    if (g_ascii_strcasecmp (vals->data, name) == 0) {
      return TRUE;
    }
  }

  return FALSE;
}

void
field_state_remove (EVCardAttribute *attr, const FieldState state)
{
  const char *name;

  g_assert (attr);

  name = field_state_name (state);

  e_vcard_attribute_remove_param_value (attr, EVC_X_OSSO_FIELD_STATE, name);
}

void
field_state_set (EVCardAttribute *attr, const FieldState state)
{
  const char *name;

  g_assert (attr);

  name = field_state_name (state);

  e_vcard_attribute_remove_param (attr, EVC_X_OSSO_FIELD_STATE);
  e_vcard_attribute_add_param_with_value (attr, e_vcard_attribute_param_new (EVC_X_OSSO_FIELD_STATE), name);
}


gboolean
attribute_is_bound_to (EVCardAttribute *attr, const char *bound)
{
  GList *params;

  g_assert (attr);

  params = e_vcard_attribute_get_param (attr, EVC_X_OSSO_BOUND);
  if (params == NULL) {
    return bound == NULL;
  }

  if (bound == NULL) return FALSE;

  return strcmp (params->data, bound) == 0;
}

void
set_contact_flag (EContact   *contact,
                  const char *flag,
                  gboolean    flag_set)
{
        EVCardAttribute *attr;
        GList *values, *new_values, *l;
        gboolean have_flag_set;

        attr = e_vcard_get_attribute (E_VCARD (contact),
                                      EVC_X_OSSO_CONTACT_STATE);
        if (!attr) {
                attr = e_vcard_attribute_new (NULL, EVC_X_OSSO_CONTACT_STATE);
                e_vcard_add_attribute (E_VCARD (contact), attr);
        }

        have_flag_set = FALSE;
        new_values = NULL;

        values = e_vcard_attribute_get_values (attr);
        for (l = values; l; l = l->next) {
                if (!strcmp (l->data, flag)) {
                        if (flag_set)
                                have_flag_set = TRUE;
                        else
                                continue;
                }

                new_values = g_list_prepend (new_values,
                                             g_strdup (l->data));
        }

        if (flag_set && !have_flag_set) {
                new_values = g_list_prepend (new_values,
                                             g_strdup (flag));
        }

        e_vcard_attribute_remove_values (attr);

        while (new_values) {
                e_vcard_attribute_add_value (attr, new_values->data);

                g_free (new_values->data);
                new_values = g_list_delete_link (new_values, new_values);
        }

        /* We explicitly dont cache here, as the contact will be
         * invalidated */
}

static void
expire_avatar_cache (GSList *tokens)
{
  GKeyFile *index;
  gchar *index_filename;
  GError *index_error = NULL;
  gchar *index_data;
  gsize index_len;
  GSList *l;

  g_debug (G_STRLOC ": expiring %d tokens", g_slist_length (tokens));
  
  index = g_key_file_new ();
  index_filename = g_build_filename (g_get_home_dir (),
                                     ".osso-abook",
                                     "avatars",
                                     "avatars.index",
                                     NULL);
  
  g_key_file_load_from_file (index, index_filename,
                             G_KEY_FILE_NONE,
                             &index_error);
  if (index_error) {
    if (index_error->domain != G_FILE_ERROR ||
        index_error->code != G_FILE_ERROR_NOENT)
      {
        g_warning ("Unable to load avatar index file from %s: %s\n"
                   "Forcing an overwrite of the index",
                   index_filename,
                   _ERROR_MSG (index_error));
      }

    g_clear_error (&index_error);
    g_key_file_free (index);
    g_free (index_filename);
    return;
  }

  for (l = tokens; l; l = l->next) {
    gchar *token = l->data;

    if (g_key_file_has_key (index, "Usernames", token, NULL)) {
      gchar *username;

      username = g_key_file_get_string (index, "Usernames", token, NULL);

      g_key_file_remove_key (index, "Avatars", username, NULL);
      g_key_file_remove_key (index, "Usernames", token, NULL);
    }
  }

  index_data = g_key_file_to_data (index, &index_len, &index_error);
  if (index_error) {
    g_warning ("Unable to save avatar index file: %s",
               _ERROR_MSG (index_error));

    g_clear_error (&index_error);
    g_key_file_free (index);
    g_free (index_filename);
    return;
  }
        
  g_file_set_contents (index_filename,
                       index_data, index_len,
                       &index_error);
  if (index_error) {
    g_warning ("Unable to save avatar index file: %s",
               _ERROR_MSG (index_error));
    g_clear_error (&index_error);
  }

  g_key_file_free (index);
  g_free (index_data);
  g_free (index_filename);

  return;
}

typedef struct {
        GDir *base_dir;
        GDir *current_dir;

        gchar *base_path;
        gchar *current_path;
        
        gpointer data;
        GDestroyNotify destroy;

        GSList *expired_tokens;
} ExpireData;

static void
expire_data_free (gpointer data)
{
        if (G_LIKELY (data)) {
                ExpireData *expire = data;

                if (expire->expired_tokens) {
                        expire_avatar_cache (expire->expired_tokens);

                        g_slist_foreach (expire->expired_tokens,
                                         (GFunc) g_free,
                                         NULL);
                        g_slist_free (expire->expired_tokens);
                }

                if (expire->base_dir)
                        g_dir_close (expire->base_dir);

                if (expire->current_dir)
                        g_dir_close (expire->current_dir);

                g_free (expire->base_path);
                g_free (expire->current_path);
                
                if (expire->destroy)
                        expire->destroy (expire->data);

                expire->destroy = NULL;
                expire->data = NULL;

                g_free (expire);
        }
}

/* 30 days without touching an avatar should be enough to mark it
 * for expiration
 */
#define CACHE_EXPIRATION_DELTA  (60 * 60 * 24 * 30)

static gboolean
avatar_cache_expire (gpointer data)
{
        ExpireData *expire = data;
        const gchar *filename;
        gchar *path = NULL;
        struct stat stat_buf;

        if (!expire->base_dir) {
                GError *error = NULL;
                
                path = g_build_filename (g_get_home_dir (),
                                         ".osso-abook",
                                         "avatars",
                                         NULL);
                
                expire->base_dir = g_dir_open (path, 0, &error);
                if (error) {
                        g_clear_error (&error);
                        g_free (path);
                        return FALSE;
                }

                expire->base_path = g_strdup (path);
                g_free (path);
                
                return TRUE;
        }

        if (!expire->current_dir) {
                GError *error;
                
                /* read the next account; if none found, we have finished */
                filename = g_dir_read_name (expire->base_dir);
                if (!filename) {
                        return FALSE;
                }
                
                /* skip the cache file */
                if (strcmp (filename, "avatars.index") == 0) {
                        return TRUE;
                }
                
                path = g_build_filename (expire->base_path,
                                         filename,
                                         NULL);
                
                error = NULL;
                expire->current_dir = g_dir_open (path, 0, &error);
                if (error) {
                        if (error->domain == G_FILE_ERROR &&
                            error->domain == G_FILE_ERROR_NOTDIR) {
                                g_clear_error (&error);
                                g_free (path);
                                return TRUE;
                        }

                        g_warning ("Unable to open `%s': %s",
                                   path,
                                   _ERROR_MSG (error));
                        g_clear_error (&error);
                        g_free (path);
                        return FALSE;
                }

                expire->current_path = g_strdup (path);
                g_free (path);
                
                return TRUE;
        }
        
        filename = g_dir_read_name (expire->current_dir);
        if (!filename) {
                g_dir_close (expire->current_dir);
                g_free (expire->current_path);
                expire->current_dir = NULL;
                expire->current_path = NULL;
                return TRUE;
        }
        
        path = g_build_filename (expire->current_path, filename, NULL);
        if (g_stat (path, &stat_buf) == -1) {
                g_warning ("Unable to stat() `%s': %s",
                           path,
                           g_strerror (errno));
                g_free (path);
                return TRUE;
        }

        if (time (NULL) > CACHE_EXPIRATION_DELTA + stat_buf.st_atime) {
                a_debug ("%s has been untouched for more than 30 days... expiring",
                         path);
                if (g_unlink (path) == -1) {
                        g_warning ("Unable to unlink() `%s': %s",
                                   path,
                                   g_strerror (errno));
                }

                expire->expired_tokens =
                        g_slist_prepend (expire->expired_tokens,
                                         g_strdup (filename));
        }

        g_free (path);

        return TRUE;
}

guint
queue_avatar_cache_expire (gpointer       data,
                           GDestroyNotify destroy)
{
        ExpireData *expire_data;

        expire_data = g_new (ExpireData, 1);
        expire_data->base_dir = NULL;
        expire_data->current_dir = NULL;
        expire_data->base_path = NULL;
        expire_data->current_path = NULL;
        expire_data->expired_tokens = NULL;
        expire_data->data = data;
        expire_data->destroy = destroy;

        return g_idle_add_full (G_PRIORITY_LOW,
                                avatar_cache_expire,
                                expire_data,
                                expire_data_free);
}

time_t
get_last_used (EContact *contact)
{
  EVCardAttribute *attr;
  char *time_str = NULL;
  time_t retval = 0;

  g_return_val_if_fail (E_IS_CONTACT (contact), 0);

  attr = e_vcard_get_attribute (E_VCARD (contact), EVC_X_OSSO_LAST_USED);
  if (attr)
    time_str = e_vcard_attribute_get_value (attr);

  if (time_str && time_str[0] != '\0') {
    GTimeVal time_val;

    if (g_time_val_from_iso8601 (time_str, &time_val))
      retval = (time_t) time_val.tv_sec;
    else
      g_warning (G_STRLOC ": not a valid time format: %s", time_str);

    g_free (time_str);
  }

  return retval;
}
