/*
 * This file is part of eds-sync
 *
 * Copyright (C) 2007 Nokia Corporation. All rights reserved.
 *
 * Author: Ross Burton <ross@openedhand.com>
 * Author: Onne Gorter <onne.gorter@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include <glib.h>
#include <glib/gstdio.h>
#include <dbus/dbus-glib-bindings.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <libtelepathy/tp-conn.h>
#include <libtelepathy/tp-conn-iface-aliasing-gen.h>
#include <libtelepathy/tp-conn-iface-avatars-gen.h>
#include <libtelepathy/tp-chan.h>
#include <libtelepathy/tp-chan-iface-group-gen.h>

#include "context.h"
#include "eds-sync.h"
#include "notification.h"
#include "eds.h"
#include "sync.h"
#include "avatars.h"
#include "util.h"

#include "eds-sync-private.h"

/*** avatars ***/

/*
 * Notify other applications that avatars have been updated, and tell Mission
 * Control if the self avatar has changed.
 */
static void
inform_avatar_change (TelepathyContext *context, const char *username, guint handle, const char *token)
{
  GError *error = NULL;

  g_assert (context);
  g_assert (handle);

  if (username == NULL) {
    if (!inspect_handle (context->conn, handle, &username, &error)) {
      g_warning (G_STRLOC ": cannot inspect handle: %s", _ERROR_MSG (error));
      g_clear_error (&error);
      return;
    }
  }

  eds_sync_signal_avatar_updated (context->sync, context->account, username);
  
  if (handle == context->self_handle) {
    if (!mission_control_remote_avatar_changed (context->mc, context->conn,
                                                handle, token, &error)) {
      g_warning ("Cannot call mission-control: %s", _ERROR_MSG (error));
    }
  }
}

/*
 * Callback when an avatar is updated. If the new token is empty remove the
 * cached avatar, otherwise if we don't have the avatar already, request it.
 */
static void
avatar_updated_cb (DBusGProxy *proxy,
                   guint       handle,
                   char       *token,
                   gpointer    user_data)
{
  TelepathyContext *context = user_data;
  GArray *arr;

  a_debug ("Avatar for handle %d updated (new token %s)", handle, token);
  
  if (token && token[0] != '\0') {
    if (!avatar_have_for_token (context->conn, handle, context->vcard, token)) {
      arr = g_array_new (FALSE, FALSE, sizeof (guint));
      g_array_append_val (arr, handle);
      
      tp_conn_iface_avatars_request_avatars (context->avatar_proxy, arr, NULL);
      g_array_free (arr, TRUE);
    }
  } else {
    a_debug ("Removing avatar for handle %d", handle);
    avatar_remove (context->conn, context->vcard, handle);
    inform_avatar_change (context, NULL, handle, token);
  }
}

/*
 * Callback for when we request multiple avatars
 */
static void
avatar_retrieved_cb (DBusGProxy *proxy,
                     guint       handle,
                     const char *token,
                     GArray     *data,
                     const char *type,
                     gpointer    user_data)
{
  TelepathyContext *context = user_data;

  a_debug ("Got avatar for handle %d (token %s)", handle, token);

  if (avatar_save (context->conn, context->vcard, handle, token, data))
    inform_avatar_change (context, NULL, handle, token);
}
  
typedef struct {
    TelepathyContext *context;
    GArray *handles;
} HandleKnownTokensData;

static void
handle_known_tokens (gpointer key,
                     gpointer value,
                     gpointer user_data)
{
  HandleKnownTokensData *data = user_data;
  uint handle = GPOINTER_TO_INT (key);
  char *token = (char*)value;

  if (token && token[0] != '\0') {
    a_debug (" token for handle %d: >%s<", handle, token);

    /* request the avatar for the handle if we don't have it
     * in our on-disk cache already
     */
    if (!avatar_have_for_token (data->context->conn,
                                handle,
                                data->context->vcard,
                                token))
      g_array_append_val (data->handles, handle);
  } else {
    a_debug (" no token for handle %d", handle);

    /* remove the known handle from the list of handles we've got */
    avatar_remove (data->context->conn,
                   data->context->vcard,
                   handle);
    inform_avatar_change (data->context,
                          NULL,
                          handle,
                          NULL);
  }
}

static void
get_known_avatar_tokens_cb (DBusGProxy  *proxy,
                            GHashTable  *known_tokens,
                            GError      *error,
                            gpointer     user_data)
{
  TelepathyContext *context = user_data;
  GArray *handles;
  HandleKnownTokensData handle_known_tokens_data;
  
  if (error) {
    g_warning ("Unable to get the known avatar tokens: %s", _ERROR_MSG (error));
    g_clear_error (&error);
    return;
  }

  a_debug ("Got %d known avatar tokens:", g_hash_table_size (known_tokens));

  handles = g_array_new (FALSE, FALSE, sizeof (guint));
  handle_known_tokens_data.context = context;
  handle_known_tokens_data.handles = handles;
  g_hash_table_foreach (known_tokens,
                        handle_known_tokens,
                        &handle_known_tokens_data);
  /* We have to free the data passed to a telepathy-async-cb
   */
  g_hash_table_destroy (known_tokens);

  if (handles->len > 0) {
      a_debug ("Requesting %d avatars", handles->len);
      tp_conn_iface_avatars_request_avatars (context->avatar_proxy, handles, NULL);
  }
  g_array_free (handles, TRUE);
}

/*
 * For every handle in the array, fetch avatar tokens and see if they need
 * fetching.
 */
static void
get_avatars (TelepathyContext *context,
             GArray           *handles)
{
  /* TODO: is this is called too often? */

  if (!context->avatar_proxy)
    return;

  a_debug ("Requesting tokens for %d handles", handles->len);

  tp_conn_iface_avatars_get_known_avatar_tokens_async (context->avatar_proxy,
                                                       handles,
                                                       get_known_avatar_tokens_cb,
                                                       context);
}

/*** subscribe ***/

static void
get_known_members_cb (DBusGProxy *proxy,
                      GArray *members,
                      GArray *local_pending,
                      GArray *remote_pending,
                      GError *error,
                      gpointer user_data)
{
  TelepathyContext * context = user_data;
  int i;

  if (error) {
      g_warning ("error getting known members: %s", _ERROR_MSG (error));
      g_clear_error (&error);
      return;
  }

  e_debug ("get_known_members_cb with %d handles", members->len);

  for (i = 0; i < members->len; i++) {
    g_intset_add (context->known_handles, g_array_index (members, guint, i));
  }

  get_avatars (context, members);

  g_array_free (members, TRUE);
  g_array_free (local_pending, TRUE);
  g_array_free (remote_pending, TRUE);

  context->known_list_done = TRUE;

  do_sync (context);
}

/* get all the members of the subscribe group */
static void
get_subscribe_members_cb (DBusGProxy *proxy,
                          GArray *members,
                          GArray *local_pending,
                          GArray *remote_pending,
                          GError *error,
                          gpointer user_data)
{
  TelepathyContext *context = user_data;
  int i;

  if (error) {
      g_warning ("error getting subscribe members: %s", _ERROR_MSG (error));
      g_clear_error (&error);
      return;
  }

  g_assert (context);
  g_assert (members);

  e_debug ("get_subscribe_members_cb with %d handles", members->len);

  for (i = 0; i < members->len; i++) {
    g_intset_add (context->remote_handles, g_array_index (members, guint, i));
  }

  for (i = 0; i < remote_pending->len; i++) {
    g_intset_add (context->remote_handles, g_array_index (remote_pending, guint, i));
  }

  g_intset_clear (context->pending_auth_handles);
  for (i = 0; i < remote_pending->len; i++) {
    g_intset_add (context->pending_auth_handles, g_array_index(remote_pending, guint, i));
  }


  g_array_free (members, TRUE);
  g_array_free (local_pending, TRUE);
  g_array_free (remote_pending, TRUE);

  context->remote_subscribe_list_done = TRUE;

  do_sync (context);
}

/* signal callback for when the known members set changes */
static void
known_members_changed_cb (DBusGProxy *proxy,
                          char *message,
                          GArray *added,
                          GArray *removed,
                          GArray *local_pending,
                          GArray *remote_pending,
                          guint actor,
                          guint reason,
                          gpointer user_data)
{
  TelepathyContext *context = user_data;
  int i;

  e_debug ("known_members_changed_cb: %d added; %d removed; %d local_pending; %d remote_pending",
      added->len, removed->len, local_pending->len, remote_pending->len);

  for (i = 0; i < removed->len; i++) {
    remove_handle_cache (context->conn, g_array_index (removed, guint, i));
    g_intset_remove (context->known_handles, g_array_index (removed, guint, i));
  }

  for (i = 0; i < added->len; i++) {
    g_intset_add (context->known_handles, g_array_index (added, guint, i));
  }

  do_sync (context);

  /* resync the avatars of the newly added members */
  get_avatars (context, added);
}

/* signal callback for when the subscribe members set changes. */
static void
subscribe_members_changed_cb (DBusGProxy *proxy,
                              char *message,
                              GArray *added,
                              GArray *removed,
                              GArray *local_pending,
                              GArray *remote_pending,
                              guint actor,
                              guint reason,
                              gpointer user_data)
{
  TelepathyContext *context = user_data;
  int i;

  GError *error = NULL;
  guint handle;
  const char *address = NULL;

  g_assert (context);

  e_debug ("subscribe_members_changed_cb: %d added; %d removed; %d local_pending; %d remote_pending",
      added->len, removed->len, local_pending->len, remote_pending->len);

  for (i = 0; i < removed->len; i++) {
    remove_handle_cache (context->conn, g_array_index (removed, guint, i));
    g_intset_remove (context->remote_handles, g_array_index (removed, guint, i));
  }

  for (i = 0; i < added->len; i++) {
    g_intset_add (context->remote_handles, g_array_index (added, guint, i));
  }

  for (i = 0; i < remote_pending->len; i++) {
    g_intset_add (context->remote_handles, g_array_index (remote_pending, guint, i));
  }

  /* Look for handles that have moved from remote_pending to added, they have been authorised */
  for (i = 0; i < added->len; i++) {
    handle = g_array_index (added, guint, i);
    if (g_intset_is_member (context->pending_auth_handles, handle)) {
      GArray *array;

      e_debug ("handle %d authorised us", handle);

      if (!inspect_handle (context->conn, handle, &address, &error)) {
        g_warning ("cannot inspect handle %u: %s", handle, _ERROR_MSG (error));
        g_clear_error (&error);

        continue;
      }
      
      eds_sync_notify_auth_response (context->notify, handle,
                                     context->account, address, message,
                                     TRUE);
      
      if (context->publish_group_proxy) {
        // also add them to see our presence
        v_debug ("moving %d to publish group, so they can see our presence", handle);

        array = g_array_new (FALSE, FALSE, sizeof (guint));
        g_array_append_val (array, handle);
        tp_chan_iface_group_add_members (context->publish_group_proxy, array, "", &error);
        if (error) {
          g_warning ("cannot add members %u: %s", handle, _ERROR_MSG (error));
          g_clear_error (&error);
        }
        g_array_free(array, TRUE);
      } else {
        g_warning (G_STRLOC": No 'publish'-group");
      }
    }
  }

  /* Look for handles that have moved from remote_pending to remove,
   * they have been rejected, however check the known list, they might
   * also have been deleted
   *
   * ALERT: without known list, there is no difference between delete
   *        and de-authorize by remote user
   */
  for (i = 0; i < removed->len; i++) {
    handle = g_array_index (removed, guint, i);
    if (g_intset_is_member (context->pending_auth_handles, handle)
        && g_intset_is_member (context->known_handles, handle)) {
      GArray *array;

      e_debug ("handle %d rejected us", handle);

      if (!inspect_handle (context->conn, handle, &address, &error)) {
        g_warning ("cannot inspect handle %u: %s", handle, _ERROR_MSG (error));
        g_clear_error (&error);
        continue;
      }

      eds_sync_notify_auth_response (context->notify,
                                     handle,
                                     context->account, address, message,
                                     FALSE);

      if (context->publish_group_proxy) {
        /* also remove them from seeing our presence */
        v_debug ("removing %d from publish group, so they can no longer see our presence", handle);

        array = g_array_new (FALSE, FALSE, sizeof (guint));
        g_array_append_val (array, handle);
        tp_chan_iface_group_remove_members (context->publish_group_proxy, array, "", NULL);
        g_array_free(array, TRUE);
      } else {
        g_warning (G_STRLOC": No 'publish'-group");
      }
    }
  }

  /* reset pending authorization handles */
  g_intset_clear (context->pending_auth_handles);
  for (i = 0; i < remote_pending->len; i++) {
    g_intset_add (context->pending_auth_handles, g_array_index(remote_pending, guint, i));
  }

  do_sync (context);

  /* Now sync the avatars */
  get_avatars (context, added);
}

/*** blocked methods ***/

/* callback when we receive the full list of blocked contacts */
static void
get_blocked_members_cb (DBusGProxy *proxy,
                        GArray *members,
                        GArray *local_pending,
                        GArray *remote_pending,
                        GError *error,
                        gpointer user_data)
{
  TelepathyContext * context = user_data;
  int i;

  if (error) {
      g_warning ("error getting blocked members: %s", _ERROR_MSG (error));
      g_clear_error (&error);
      return;
  }

  e_debug ("get_blocked_members_cb with %d handles", members->len);

  for (i = 0; i < members->len; i++) {
    g_intset_add (context->blocked_handles, g_array_index (members, guint, i));
  }

  g_array_free (members, TRUE);
  g_array_free (local_pending, TRUE);
  g_array_free (remote_pending, TRUE);

  context->blocked_list_done = TRUE;

  do_sync (context);
}

/* signal callback for when the known members set changes */
static void
blocked_members_changed_cb (DBusGProxy *proxy,
                            char *message,
                            GArray *added,
                            GArray *removed,
                            GArray *local_pending,
                            GArray *remote_pending,
                            guint actor,
                            guint reason,
                            gpointer user_data)
{
  TelepathyContext *context = user_data;
  int i;

  e_debug ("blocked_members_changed_cb: %d added; %d removed; %d local_pending; %d remote_pending",
      added->len, removed->len, local_pending->len, remote_pending->len);

  for (i = 0; i < removed->len; i++) {
    remove_handle_cache (context->conn, g_array_index (removed, guint, i));
    g_intset_remove (context->blocked_handles, g_array_index (removed, guint, i));
  }

  for (i = 0; i < added->len; i++) {
    g_intset_add (context->blocked_handles, g_array_index (added, guint, i));
  }

  do_sync (context);

  /* refresh the avatars */
  get_avatars (context, added);
}

static void
on_notify_context_reply (EdsNotifyContext *notify,
                         guint             handle,
                         gboolean          response,
                         gpointer          user_data)
{
  TelepathyContext *context = user_data;
  GArray *array;

  array = g_array_new (FALSE, FALSE, sizeof (guint));
  g_array_append_val (array, handle);

  if (!response)
    {
      /* reject authorization if response was no OK */
      v_debug ("deleting local pending published members");

      if (context->publish_group_proxy) {
        tp_chan_iface_group_remove_members (context->publish_group_proxy,
                                            array, "", NULL);
      } else {
        g_warning (G_STRLOC": No 'publish'-group");
      }

      tp_chan_iface_group_remove_members (context->subscribe_group_proxy,
                                          array, "", NULL);
      if (context->known_group_proxy)
        tp_chan_iface_group_remove_members (context->known_group_proxy,
                                            array, "", NULL);
    }
  else
    {
      /* accept authorization and add the contact to publish list */
      v_debug ("moving local pending published members to published list");

      if (context->publish_group_proxy) {
        tp_chan_iface_group_add_members (context->publish_group_proxy,
                                         array, "", NULL);
      } else {
        g_warning (G_STRLOC": No 'publish'-group");
      }

      /* add the contacts to the subscribe list, effectively
       * asking their authorization
       */
      e_debug ("adding published members to subscribe list: len: %d",
               array->len);

      /* NOTICE set a message or not? but what language? */
      tp_chan_iface_group_add_members (context->subscribe_group_proxy,
                                       array, "", NULL);
    }

  g_array_free (array, TRUE);
}

/* process the publish pending members */
static void
handle_publish_pending_members (TelepathyContext *context,
                                GArray *handles,
                                char *message)
{
  GError *error = NULL;
  const char *address = NULL;
  char **names = NULL;
  guint i, handle;
  DBusGProxy *aliasing = NULL;

  aliasing = tp_conn_get_interface(context->conn, TELEPATHY_CONN_IFACE_ALIASING_QUARK);
  if (aliasing) {
    if (!tp_conn_iface_aliasing_request_aliases (aliasing, handles, &names, &error)) {
      g_warning ("cannot get aliases: %s", _ERROR_MSG (error));
      g_clear_error (&error);
    }
  }

  for (i = 0; i < handles->len; i++) {
    handle = g_array_index(handles, guint, i);

    if (!inspect_handle (context->conn, handle, &address, &error)) {
      g_warning ("cannot inspect handle %u: %s", handle, _ERROR_MSG (error));

      g_clear_error (&error);

      continue;
    }

    if (!message) {
      v_debug ("received an NULL authorization message");
    }

    e_debug ("got handle %d (%s) to authorise", handle, address);

    if (g_intset_is_member(context->remote_handles, handle)) {
      GArray * to_add_array;

      v_debug ("handle already gives us their presence, implicitly authorizing");

      to_add_array = g_array_new (FALSE, FALSE, sizeof (guint));
      g_array_append_val (to_add_array, handle);
      tp_chan_iface_group_add_members (context->publish_group_proxy, to_add_array, "", NULL);
      g_array_free(to_add_array, TRUE);
    }
    else {
      guint res;

      res = eds_sync_notify_auth_request (context->notify,
                                          handle,
                                          context->account,
                                          names ? names[i] : NULL,
                                          address,
                                          message);
    }
  }

  g_strfreev (names);
}

/* get all initial local pending members */
static void
get_publish_members_cb (DBusGProxy *proxy,
                        GArray *members,
                        GArray *local_pending,
                        GArray *remote_pending,
                        GError *error,
                        gpointer user_data)
{
  e_debug ("get_publish_local_pending_members_cb with %d handles",
           members->len);

  handle_publish_pending_members (user_data, local_pending, "");

  g_array_free (members, TRUE);
  g_array_free (local_pending, TRUE);
  g_array_free (remote_pending, TRUE);
}

/* signal callback for when the publish members set changes */
static void
publish_members_changed_cb (DBusGProxy *proxy,
                            char *message,
                            GArray *added,
                            GArray *removed,
                            GArray *local_pending,
                            GArray *remote_pending,
                            guint actor,
                            guint reason,
                            gpointer user_data)
{
  e_debug ("publish_members_changed_cb: %d added; %d removed; %d local_pending; %d remote_pending",
      added->len, removed->len, local_pending->len, remote_pending->len);

  handle_publish_pending_members (user_data, local_pending, message);
}


/*** management ***/

#define FREE(field, freer) \
  if (context->field) { \
    freer (context->field); \
    context->field = NULL; \
  }

void
context_free (TelepathyContext *context)
{
  v_debug("context_free");
  g_return_if_fail (context != NULL);

  eds_sync_channel_closed (context->sync);

  FREE (mc, g_object_unref);

  FREE (view, g_object_unref);
  FREE (book, g_object_unref);

  FREE (publish_chan, g_object_unref);
  FREE (known_chan, g_object_unref);
  FREE (blocked_chan, g_object_unref);

  FREE (conn, g_object_unref);
  
  FREE (notify, g_object_unref);

  FREE (account, g_free);
  FREE (vcard, g_free);

  FREE (remote_handles, g_intset_destroy);
  FREE (local_handles, g_intset_destroy);
  FREE (pending_auth_handles, g_intset_destroy);
  FREE (blocked_handles, g_intset_destroy);
  FREE (local_blocked_handles, g_intset_destroy);

  FREE (handle_to_contact_hash, g_hash_table_destroy);
  FREE (local_contacts_todo, g_hash_table_destroy);

  g_free (context);
  v_debug("context_free: done");
}

/* closing channel */
static void
on_subscribe_channel_close_cb (DBusGProxy *proxy, gpointer user_data)
{
  TelepathyContext *context = user_data;

  e_debug("channel closed");

  /* when gabble dies, or has exited, this is bad; but if gabble simply
   * disconnects, this is needed
   *
   * TODO - how to inspect the proxy before disconnecting? and don't do
   * it if it is not needed
   */
  dbus_g_proxy_disconnect_signal (DBUS_G_PROXY (context->subscribe_chan),
                                  "Closed",
                                  G_CALLBACK (on_subscribe_channel_close_cb),
                                  context);

  if (context->subscribe_group_proxy) {
    dbus_g_proxy_disconnect_signal (context->subscribe_group_proxy,
                                    "MembersChanged",
                                    G_CALLBACK (subscribe_members_changed_cb),
                                    context);
  }

  context_free (context);
}

typedef struct {
  GList *uids;
} HandleCacheEntry;

static HandleCacheEntry *
handle_cache_entry_new (void)
{
  HandleCacheEntry *retval;

  retval = g_slice_new (HandleCacheEntry);
  retval->uids = NULL;

  return retval;
}

static void
handle_cache_entry_free (gpointer data)
{
  if (G_LIKELY (data))
    {
      HandleCacheEntry *entry = data;

      while (entry->uids) {
        g_free (entry->uids->data);
        entry->uids = g_list_delete_link (entry->uids,
                                          entry->uids);
      }
      g_slice_free (HandleCacheEntry, entry);
    }
}

static void
handle_cache_entry_add (HandleCacheEntry *entry,
                        const gchar      *uid)
{
  g_assert (entry != NULL);
  g_assert (uid != NULL);
  GList *l;

  /* don't add dups! */
  for (l = entry->uids; l; l = l->next) {
    if (strcmp (l->data, uid) == 0)
      return;
  }
  entry->uids = g_list_append (entry->uids,
                               g_strdup (uid));
}

static gboolean
handle_cache_entry_remove (HandleCacheEntry *entry,
                           const gchar      *uid)
{
  GList *l;
  g_assert (entry != NULL);
  g_assert (uid != NULL);

  for (l = entry->uids; l; l = l->next) {
    if (strcmp (l->data, uid) == 0) {
      g_free (l->data);
      entry->uids = g_list_delete_link (entry->uids, l);
      return TRUE;
    }
  }
  return FALSE;
}

void
eds_sync_telepathy_add_uid (TelepathyContext *context,
                            const guint       handle,
                            const gchar      *uid)
{
  HandleCacheEntry *entry;

  g_return_if_fail (context->handle_to_contact_hash != NULL);
  g_return_if_fail (uid != NULL);

  entry = g_hash_table_lookup (context->handle_to_contact_hash,
                               GUINT_TO_POINTER (handle));
  if (entry)
    handle_cache_entry_add (entry, uid);
  else
    {
      entry = handle_cache_entry_new ();
      handle_cache_entry_add (entry, uid);
      g_hash_table_insert (context->handle_to_contact_hash,
                           GUINT_TO_POINTER (handle),
                           entry);
    }
}

static void
remove_uid (gpointer key,
            gpointer value,
            gpointer data)
{
  HandleCacheEntry *entry = value;
  gchar *uid = data;

  handle_cache_entry_remove (entry, uid);
}

void
eds_sync_telepathy_remove_uid (TelepathyContext *context,
                               const gchar      *uid)
{
  g_return_if_fail (context->handle_to_contact_hash != NULL);

  g_hash_table_foreach (context->handle_to_contact_hash,
                        remove_uid,
                        (void*)uid);
}

/* Don't free or edit the returned list, it's not yours */
GList *
eds_sync_telepathy_lookup_uids (TelepathyContext *context,
                               const guint       handle)
{
  HandleCacheEntry *entry;

  g_return_val_if_fail (context->handle_to_contact_hash != NULL, NULL);

  entry = g_hash_table_lookup (context->handle_to_contact_hash,
                               GUINT_TO_POINTER (handle));
  if (!entry)
    return NULL;

  return entry->uids;
}

#define EDS_SYNC_MC_ERROR g_quark_from_static_string ("eds-sync MC-error")

static MissionControl*
get_mission_control_and_credentials (DBusGConnection *bus,
                                     TpConn          *conn,
                                     char           **vcard,
                                     char           **account_param,
                                     char           **normalized_account,
                                     GError         **error)
{
  MissionControl *mc = NULL;
  McAccount *account = NULL;
  McProfile *profile = NULL;
  const char *na = NULL;
  const char *vc = NULL;
  char *ra = NULL;

  *vcard = *account_param = *normalized_account = NULL;

  /* First create the MC object, we need it before we do anything else.
   * This will be assigned into the context later.
   */
  mc = mission_control_new (bus);
  if (!mc) {
    *error = g_error_new (EDS_SYNC_MC_ERROR, 0, "Could not connect to mission-control");
    goto out;
  }

  /* Set the account name in the context */
  account = mission_control_get_account_for_connection (mc, conn, error);
  if (NULL == account) {
    if (*error == NULL) {
      *error = g_error_new (EDS_SYNC_MC_ERROR, 0, "Could not get account from mission-control");
    }
    goto out;
  }
  
  mc_account_get_param_string (account, "account", &ra);
  if (NULL == ra) {
    *error = g_error_new (EDS_SYNC_MC_ERROR, 0, "McAccount '%s' does not have an \"account\"-parameter",
                       mc_account_get_unique_name (account));
    goto out;
  }
  na = mc_account_get_normalized_name (account);

  profile = mc_account_get_profile (account);
  if (NULL == profile) {
    *error = g_error_new (EDS_SYNC_MC_ERROR, 0, "Could not get McProfile for McAccount '%s'",
                       mc_account_get_unique_name (account));
    goto out;
  }

  vc = mc_profile_get_vcard_field (profile);
  if (NULL == vc) {
    *error = g_error_new (EDS_SYNC_MC_ERROR, 0, "Could not get vcard for McAccount %s",
                       mc_account_get_unique_name (account));
    goto out;
  }

out:
  if (account) {
    g_object_unref (account);
  }
  if (profile) {
    g_object_unref (profile);
  }
  if (*error) {
    if (mc)
      g_object_unref (mc);
    if (ra)
      g_free (ra);
    return NULL;
  }

  *vcard = g_strdup (vc);
  *account_param = ra;
  *normalized_account = g_strdup (na);

  return mc;
}

gboolean
eds_sync_telepathy_new (EdsSync *sync,
                        TpConn *conn,
                        const gchar *bus_name,
                        const gchar *object_path,
                        const gchar *channel_type,
                        const gchar *channel,
                        guint handle_type,
                        guint handle,
                        GError **error)
{
    TelepathyContext *context;
    MissionControl *mc = NULL;
    char *vcard, *account_param, *normalized_account;
    guint known_chan_handle, blocked_chan_handle, publish_chan_handle;
    GError *err = NULL;

    e_debug ("new Telepathy Connection, initializing a context");

    g_return_val_if_fail (EDS_IS_SYNC (sync), FALSE);
    g_return_val_if_fail (TELEPATHY_IS_CONN (conn), FALSE);
    g_return_val_if_fail (bus_name != NULL, FALSE);
    g_return_val_if_fail (object_path != NULL, FALSE);
    g_return_val_if_fail (channel_type != NULL, FALSE);
    g_return_val_if_fail (channel != NULL, FALSE);

    mc = get_mission_control_and_credentials (sync->priv->connection,
                                              conn,
                                              &vcard,
                                              &account_param,
                                              &normalized_account,
                                              &err);
    if (NULL == mc) {
      g_warning ("Context initialization failed: %s\n",
                 _ERROR_MSG (err));
      g_clear_error (&err);
      return FALSE;
    }
    
    context = g_new0 (TelepathyContext, 1);
    context->sync = sync;
    context->conn = conn;
    context->mc = mc;
    context->vcard = vcard;
    context->raw_account = account_param;
    context->account = normalized_account;

    if (!context->account) {
      g_warning ("Normalized account not yet available for \"%s\"",
                 context->raw_account);
      /* Fallback to not normalized account id */
      context->account = context->raw_account;
      context->raw_account = NULL;
    }

    if (context->raw_account && context->account &&
        (strcmp (context->raw_account, context->account) == 0)) {
      g_free (context->raw_account);
      context->raw_account = NULL;
    }

    
    /* Initialise this to true so that the first run with no contacts works */
    context->local_contacts_changed = TRUE;

    context->local_contacts_done = FALSE;
    context->remote_subscribe_list_done = FALSE;
    context->known_list_done = FALSE;
    context->blocked_list_done = FALSE;
    context->first_run = TRUE;

    /* get subscribe list */
    context->subscribe_chan = tp_chan_new (sync->priv->connection, bus_name, channel, channel_type, handle_type, handle);
    g_assert (context->subscribe_chan);

    context->subscribe_group_proxy = tp_chan_get_interface (context->subscribe_chan, TELEPATHY_CHAN_IFACE_GROUP_QUARK);
    g_assert (context->subscribe_group_proxy);

#if 0
    g_object_add_weak_pointer (G_OBJECT (context->subscribe_group_proxy),
                               (gpointer *) &context->subscribe_group_proxy);
#endif

    dbus_g_proxy_connect_signal (DBUS_G_PROXY (context->subscribe_chan),
                                 "Closed",
                                 G_CALLBACK (on_subscribe_channel_close_cb),
                                 context, NULL);

    /* get publish list */
    eds_tp_conn_request_handle (DBUS_G_PROXY (conn),
                                TP_CONN_HANDLE_TYPE_LIST, "publish",
                                &publish_chan_handle, NULL);

    context->publish_chan = tp_conn_new_channel (sync->priv->connection,
                                                 context->conn,
                                                 bus_name,
                                                 TP_IFACE_CHANNEL_TYPE_CONTACT_LIST,
                                                 TP_CONN_HANDLE_TYPE_LIST,
                                                 publish_chan_handle, FALSE);
    if (context->publish_chan) {
      context->publish_group_proxy = tp_chan_get_interface (context->publish_chan, TELEPATHY_CHAN_IFACE_GROUP_QUARK);
      g_assert (context->publish_group_proxy);
    } else {
      g_warning ("Connection does not provide a 'publish'-channel.");
      context->publish_group_proxy = NULL;
    }

    /* get known list */
    eds_tp_conn_request_handle (DBUS_G_PROXY (conn), TP_CONN_HANDLE_TYPE_LIST, "known", &known_chan_handle, &err);
    if (err) {

      g_warning ("no handle for known channel: %s", _ERROR_MSG (err));
      g_clear_error (&err);

    } else {

      v_debug ("found and connecting to 'known' channel");

      context->known_chan = tp_conn_new_channel (sync->priv->connection, context->conn,
         bus_name, TP_IFACE_CHANNEL_TYPE_CONTACT_LIST, TP_CONN_HANDLE_TYPE_LIST, known_chan_handle, FALSE);

      if (context->known_chan) {
        context->known_group_proxy = tp_chan_get_interface (context->known_chan, TELEPATHY_CHAN_IFACE_GROUP_QUARK);
        g_assert (context->known_group_proxy);
      } else {
        v_debug("'known' channel is not really there ...");
      }
    }

    /* get blocked list */
    eds_tp_conn_request_handle (DBUS_G_PROXY (conn), TP_CONN_HANDLE_TYPE_LIST, "deny", &blocked_chan_handle, &err);
    if (err) {

      g_warning ("no handle for deny channel: %s", _ERROR_MSG (err));
      g_clear_error (&err);

    } else {

      v_debug ("found and connecting to 'deny' channel");

      context->blocked_chan = tp_conn_new_channel (sync->priv->connection, context->conn,
         bus_name, TP_IFACE_CHANNEL_TYPE_CONTACT_LIST, TP_CONN_HANDLE_TYPE_LIST, blocked_chan_handle, FALSE);

      if (context->blocked_chan) {

        context->blocked_group_proxy = tp_chan_get_interface (context->blocked_chan, TELEPATHY_CHAN_IFACE_GROUP_QUARK);
        g_assert (context->blocked_group_proxy);

      } else {
        v_debug("'deny' channel is not really there ...");
      }
    }

    context->avatar_proxy = tp_conn_get_interface (conn, TELEPATHY_CONN_IFACE_AVATARS_QUARK);
    if (context->avatar_proxy == NULL) {
      v_debug ("No avatar interface available");
    }

    context->aliasing_proxy = tp_conn_get_interface (conn, TELEPATHY_CONN_IFACE_ALIASING_QUARK);
    if (context->aliasing_proxy == NULL) {
      v_debug ("No aliasing interface available");
    }

    /* initialize */
    context->remote_handles = g_intset_new ();
    context->local_handles = g_intset_new ();
    context->pending_auth_handles = g_intset_new ();
    context->known_handles = g_intset_new ();
    context->blocked_handles = g_intset_new ();
    context->local_blocked_handles = g_intset_new ();

    context->local_contacts_todo = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);
    context->handle_to_contact_hash =
      g_hash_table_new_full (NULL, NULL, NULL, handle_cache_entry_free);

    context->notify = eds_sync_get_notify_context ();
    g_assert (EDS_IS_NOTIFY_CONTEXT (context->notify));
    v_debug ("Notify context type: %s", G_OBJECT_TYPE_NAME (context->notify));
    g_signal_connect (context->notify, "reply",
                      G_CALLBACK (on_notify_context_reply),
                      context);

    /* connect callbacks */
    dbus_g_proxy_connect_signal (context->subscribe_group_proxy, "MembersChanged", G_CALLBACK (subscribe_members_changed_cb), context, NULL);
    if (context->publish_group_proxy) {
      dbus_g_proxy_connect_signal (context->publish_group_proxy, "MembersChanged", G_CALLBACK (publish_members_changed_cb), context, NULL);
    }
    if (context->known_group_proxy) {
      dbus_g_proxy_connect_signal (context->known_group_proxy, "MembersChanged", G_CALLBACK (known_members_changed_cb), context, NULL);
    }
    if (context->blocked_group_proxy) {
      dbus_g_proxy_connect_signal (context->blocked_group_proxy, "MembersChanged", G_CALLBACK (blocked_members_changed_cb), context, NULL);
    }
    if (context->avatar_proxy) {
      dbus_g_proxy_connect_signal (context->avatar_proxy, "AvatarUpdated", G_CALLBACK (avatar_updated_cb), context, NULL);
      dbus_g_proxy_connect_signal (context->avatar_proxy, "AvatarRetrieved", G_CALLBACK (avatar_retrieved_cb), context, NULL);
    }

    /* initialize context */
    eds_populate_handles (context);

    tp_chan_iface_group_get_all_members_async (context->subscribe_group_proxy, get_subscribe_members_cb, context);
    if (context->publish_group_proxy) {
      tp_chan_iface_group_get_all_members_async (context->publish_group_proxy, get_publish_members_cb, context);
    }
    if (context->known_group_proxy) {
      tp_chan_iface_group_get_all_members_async (context->known_group_proxy, get_known_members_cb, context);
    } else {
      context->known_list_done = TRUE;
    }
    if (context->blocked_group_proxy) {
      tp_chan_iface_group_get_all_members_async (context->blocked_group_proxy, get_blocked_members_cb, context);
    } else {
      context->blocked_list_done = TRUE;
    }

    /* Get the self handle to monitor the local avatar */
    if (!tp_conn_get_self_handle (DBUS_G_PROXY (conn), &context->self_handle, &err)) {
      g_warning ("Cannot get self handle: %s", _ERROR_MSG (err));
      g_clear_error (&err);
    }

    if (context->avatar_proxy && context->self_handle) {
      GArray *handles;
      
      handles = g_array_new (FALSE, FALSE, sizeof (guint));
      g_array_append_val (handles, context->self_handle);
      
      get_avatars (context, handles);
      
      g_array_free (handles, TRUE);
    }

    return TRUE;
}
