/*
 * This file is part of eds-sync
 *
 * Copyright (C) 2007 Nokia Corporation. All rights reserved.
 *
 * Author: Ross Burton <ross@openedhand.com>
 * Author: Onne Gorter <onne.gorter@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <glib.h>

#include <osso-contact-plugin/rtcom-cp-interface.h>

#include "notification.h"
#include "notification-rtcom.h"

static gint
eds_notify_context_rtcom_auth_request (EdsNotifyContext *context,
                                       gint              handle,
                                       const gchar      *account,
                                       const gchar      *name,
                                       const gchar      *address,
                                       const gchar      *message)
{
  EdsNotifyContextRtcom *rtcom = EDS_NOTIFY_CONTEXT_RTCOM (context);
  gint op_id;
  gboolean res;

  res = rtcom_cp_interface_authorization_request (rtcom->rtcom,
                                                  account,
                                                  name,
                                                  address,
                                                  message,
                                                  &op_id);
  if (!res)
    return -1;

  g_hash_table_insert (rtcom->op_hash,
                       GINT_TO_POINTER (op_id),
                       GINT_TO_POINTER (handle));

  return op_id;
}

static gint
eds_notify_context_rtcom_auth_response (EdsNotifyContext *context,
                                        gint              handle,
                                        const gchar      *account,
                                        const gchar      *address,
                                        const gchar      *message,
                                        gboolean          is_accepted)
{
  EdsNotifyContextRtcom *rtcom = EDS_NOTIFY_CONTEXT_RTCOM (context);
  gint op_id;
  gboolean res;

  res = rtcom_cp_interface_authorization_response (rtcom->rtcom,
                                                   account,
                                                   address,
                                                   message,
                                                   is_accepted,
                                                   &op_id);
  if (!res)
    return -1;

#if 0
  /* we don't need to do this for authorisation response */
  g_hash_table_insert (rtcom->op_hash,
                       GINT_TO_POINTER (op_id),
                       GINT_TO_POINTER (handle));
#endif

  return op_id;
}

static void
eds_notify_context_rtcom_cancel_operations (EdsNotifyContext *context,
                                            GArray           *ids)
{
  EdsNotifyContextRtcom *notify_rtcom = EDS_NOTIFY_CONTEXT_RTCOM (context);

  if (!rtcom_cp_interface_cancel_requests (notify_rtcom->rtcom, ids))
    g_warning ("failed clearing the requests");
}

static void
eds_notify_context_iface_init (EdsNotifyContextIface *iface)
{
  iface->auth_request = eds_notify_context_rtcom_auth_request;
  iface->auth_response = eds_notify_context_rtcom_auth_response;
  iface->cancel_operations = eds_notify_context_rtcom_cancel_operations;
}

G_DEFINE_TYPE_WITH_CODE (EdsNotifyContextRtcom,
                         eds_notify_context_rtcom,
                         G_TYPE_OBJECT,
                         G_IMPLEMENT_INTERFACE (EDS_TYPE_NOTIFY_CONTEXT,
                                                eds_notify_context_iface_init));

static void
on_rtcom_operation_reply (RtcomCPInterface *rtcom,
                          guint             op_id,
                          gboolean          response,
                          gpointer          user_data)
{
  EdsNotifyContextRtcom *notify_rtcom = user_data;
  guint handle;

  handle = GPOINTER_TO_INT (g_hash_table_lookup (notify_rtcom->op_hash,
                                                 GINT_TO_POINTER (op_id)));
  if (handle == 0)
    {
      g_debug ("Unexpect operation id %u, probably for another "
               "eds-sync context",
               op_id);
      return;
    }

  g_hash_table_remove (notify_rtcom->op_hash, GINT_TO_POINTER (op_id));
  
  g_signal_emit_by_name (notify_rtcom, "reply", handle, response);
}

static void
add_to_array (gpointer key, gpointer value, gpointer user_data)
{
  GArray *ops = user_data;

  g_assert (ops);
  g_assert (key);

  g_array_append_val (ops, key);
}

/* clear out task navigators events from us */
static void
clear_rtcom_operations (EdsNotifyContextRtcom *notify_rtcom)
{
  GArray *ops;

  ops = g_array_new (FALSE, TRUE, sizeof (guint));
  g_hash_table_foreach (notify_rtcom->op_hash, add_to_array, ops);

  if (ops->len == 0)
    {
      g_array_free(ops, TRUE);
      return;
    }

  if (!rtcom_cp_interface_cancel_requests (notify_rtcom->rtcom, ops))
    g_warning ("failed clearing the requests");

  g_array_free (ops, TRUE);
}

static void
eds_notify_context_rtcom_dispose (GObject *gobject)
{
  EdsNotifyContextRtcom *notify_rtcom = EDS_NOTIFY_CONTEXT_RTCOM (gobject);

  if (notify_rtcom->op_hash)
    {
      clear_rtcom_operations (notify_rtcom);

      g_hash_table_destroy (notify_rtcom->op_hash);
      notify_rtcom->op_hash = NULL;
    }

  if (notify_rtcom->rtcom)
    {
      g_object_unref (notify_rtcom->rtcom);
      notify_rtcom->rtcom = NULL;
    }

  G_OBJECT_CLASS (eds_notify_context_rtcom_parent_class)->dispose (gobject);
}

static void
eds_notify_context_rtcom_class_init (EdsNotifyContextRtcomClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

  gobject_class->dispose = eds_notify_context_rtcom_dispose;
}

static void
eds_notify_context_rtcom_init (EdsNotifyContextRtcom *rtcom)
{
  rtcom->rtcom = RTCOM_CP_INTERFACE (rtcom_cp_interface_new ());
  g_assert (rtcom->rtcom);

  g_signal_connect (rtcom->rtcom, "operation-reply",
                    G_CALLBACK (on_rtcom_operation_reply),
                    rtcom);

  rtcom->op_hash = g_hash_table_new (NULL, NULL);
}

EdsNotifyContext *
eds_sync_get_notify_context (void)
{
  return g_object_new (EDS_TYPE_NOTIFY_CONTEXT_RTCOM, NULL);
}
