/*
 * This file is part of eds-sync
 *
 * Copyright (C) 2007 Nokia Corporation. All rights reserved.
 *
 * Author: Ross Burton <ross@openedhand.com>
 * Author: Onne Gorter <onne.gorter@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <libebook/e-book.h>

#include "gintset.h"
#include "context.h"
#include "eds.h"
#include "sync.h"
#include "util.h"

#include "eds-sync-private.h"

static void
on_local_contacts_added (EBookView *view, GList *contacts, gpointer user_data)
{
  GError *error = NULL;
  TelepathyContext *context = user_data;
  GList *l;

  g_assert (user_data != NULL);

  e_debug("on_local_contacts_added");

  for (l = contacts; l ; l = g_list_next (l)) {
    EContact *contact = l->data;
    const char *uid = e_contact_get_const (contact, E_CONTACT_UID);
    GList *l;

    e_debug ("contact added or changed, uid: %s", uid);

    /* Manual contacts are ignored by eds-sync as they don't really exist, but
       we'll take this opportunity to purge old manual contacts that haven't
       been used for a while. */
    if (contact_state_is (contact, CONTACT_STATE_MANUAL)) {
      time_t now, then;

      time (&now);
      then = get_last_used (contact);

      /* 30 days -> hours -> minutes -> seconds */
      if ((FALSE == contact_state_is (contact, CONTACT_STATE_BLOCKED)) &&
         ((now - then) > (30 * 24 * 60 * 60))) {
        /* Old manual contact, delete it */
        e_book_async_remove_contact (context->book, contact, NULL, NULL);
      }

      continue;
    }

    /* Note any pending state */
    if (e_vcard_get_attribute (E_VCARD (contact), EVC_X_OSSO_CONTACT_STATE)) {
      g_hash_table_insert (context->local_contacts_todo, g_strdup (uid), NULL);
      context->local_contacts_changed = TRUE;
    }

    l = e_vcard_get_attributes (E_VCARD (contact));
    for (; l ; l = l->next) {
      EVCardAttribute *attr;
      char *address;
      guint handle;

      attr = l->data;

      /* If its not the relevant vcard field, go to the next one */
      if (strcmp (e_vcard_attribute_get_name (attr), context->vcard) != 0)
        continue;

      /* Get the IM address value.  If we get NULL its probably a weird multi-valued
         field, so go to next */
      address = e_vcard_attribute_get_value (attr);
      if (address == NULL)
        continue;


      /* find the address in telepathy, if exists */
      if (!eds_tp_conn_request_handle (DBUS_G_PROXY (context->conn), TP_CONN_HANDLE_TYPE_CONTACT, address, &handle, &error)) {
        g_warning ("unable to get handle for address: %s: %s", address, _ERROR_MSG (error));
        g_clear_error (&error);
        g_free(address);

        continue; // return
      }

      v_debug("contact with address: %s is handle: %d", address, handle);

      /* Store a mapping from this IM address to the contact UID in
       * the global state, but only if we haven't already a contact
       * UID bound to it.
       */
      eds_sync_register_account (context->sync, context->vcard, address, uid);
      
      if (attribute_is_bound_to (attr, context->account)) {
        /* NOTICE: cards with multiple addresses will be added to todo list
         * more then once */

        /* Store a mapping from the handle to the contacts UID */
        eds_sync_telepathy_add_uid (context, handle, uid);
        
        /* Note any pending state */
        if (field_state_is (attr, FIELD_STATE_TODELETE) ||
            field_state_is (attr, FIELD_STATE_TOADD) ||
            field_state_is (attr, FIELD_STATE_TOBLOCK) ||
            field_state_is (attr, FIELD_STATE_TOUNBLOCK) ||
            field_state_is (attr, FIELD_STATE_BLOCKED)) {
          g_hash_table_insert (context->local_contacts_todo, g_strdup (uid), NULL);
          context->local_contacts_changed = TRUE;
        } else {
          /* Add the IM address to the local handles */
          if (!g_intset_is_member (context->local_handles, handle)) {
            g_intset_add (context->local_handles, handle);
            context->local_contacts_changed = TRUE;
          }
        }
      }

      g_free (address);
    }
  }
}

static void
on_local_contacts_removed (EBookView *view,
                           GList *contacts,
                           gpointer user_data)
{
  EdsSync *sync;
  TelepathyContext *context;
  GList *l;

  g_assert (user_data);

  context = user_data;
  sync = context->sync;

  e_debug ("purging deleted contacts from hash tables");

  for (l = contacts; l; l = l->next) {
    char *uid = l->data;

    eds_sync_telepathy_remove_uid (context, uid);
    eds_sync_remove_uid (sync, uid);
  }
}

static void
on_sequence_complete (EBookView *view,
                      gint unused,
                      gpointer user_data)
{
  TelepathyContext *context = user_data;

  g_assert (context);

  e_debug ("book view complete");

  context->local_contacts_done = TRUE;

  if (context->local_contacts_changed) {
    context->local_contacts_changed = FALSE;
    do_sync (context);
  }
}


void
eds_populate_handles (TelepathyContext *context)
{
  GError *error = NULL;
  EBookQuery *query;

  g_return_if_fail (context != NULL);

  context->book = e_book_new_system_addressbook (&error);
  if (context->book == NULL)
    g_error ("Cannot get addressbook: %s", _ERROR_MSG (error));

  if (!e_book_open (context->book, FALSE, &error))
    g_error ("Cannot open book: %s", _ERROR_MSG (error));

  query = e_book_query_orv (
                            e_book_query_vcard_field_exists (context->vcard),
                            e_book_query_vcard_field_exists (EVC_X_OSSO_CONTACT_STATE),
                            NULL);

  if (!e_book_get_book_view (context->book, query, NULL, -1, &context->view, &error))
    g_error ("Cannot get book view: %s", _ERROR_MSG (error));
  e_book_query_unref (query);
  g_assert (context->view);

  g_signal_connect (context->view, "contacts-added", G_CALLBACK (on_local_contacts_added), context);
  /* We can re-use the handler as the behaviour is almost identical */
  g_signal_connect (context->view, "contacts-changed", G_CALLBACK (on_local_contacts_added), context);
  g_signal_connect (context->view, "contacts-removed", G_CALLBACK (on_local_contacts_removed), context);
  g_signal_connect (context->view, "sequence-complete", G_CALLBACK (on_sequence_complete), context);

  e_book_view_start (context->view);
}
