/*
 * This file is part of eds-sync
 *
 * Copyright (C) 2007 Nokia Corporation. All rights reserved.
 *
 * Author: Ross Burton <ross@openedhand.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <errno.h>
#include <string.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <libtelepathy/tp-conn.h>

#include "util.h"
#include "avatars.h"

/* Any function that accesses this should call INIT_CACHE first */
static GKeyFile *cache = NULL;

#define INIT_CACHE { if (G_UNLIKELY (!cache)) init_cache (); }

static const char *
get_cache_filename (void)
{
  static char *filename = NULL;

  if (G_UNLIKELY (!filename)) {
    filename = g_build_filename (g_get_home_dir (),
                                 ".osso-abook", "avatars", "avatars.index",
                                 NULL);
  }

  return filename;
}

static void
init_cache (void)
{
  /* Note that this is not thread-safe */
  GError *error = NULL;

  if (G_LIKELY (cache))
    return;

  cache = g_key_file_new ();
  
  if (g_file_test (get_cache_filename (), G_FILE_TEST_EXISTS)) {
    if (!g_key_file_load_from_file (cache, get_cache_filename (), G_KEY_FILE_NONE, &error)) {
      g_warning (G_STRLOC ": cannot load avatar index: %s", _ERROR_MSG (error));
      g_clear_error (&error);
    }
  }
}

/**
 * Look up the cached token for the specified username
 */
static char *
get_token (const char *username)
{
  g_return_val_if_fail (username, NULL);

  INIT_CACHE;
  
  if (g_key_file_has_key (cache, "Avatars", username, NULL)) {
    return g_key_file_get_string (cache, "Avatars", username, NULL);
  } else {
    return NULL;
  }
}

static void
update_cache (const char *username, const char *token, gboolean add)
{
  GError *error = NULL;
  char *data, *old_token;
  gsize length;

  g_return_if_fail (username);
  g_return_if_fail (token);
  
  INIT_CACHE;
  
  /* Update the key file */
  if (add) {
    /* Remove the state token->username entry */
    old_token = g_key_file_get_string (cache, "Avatars", username, NULL);
    if (old_token)
      g_key_file_remove_key (cache, "Usernames", old_token, NULL);

    g_key_file_set_string (cache, "Avatars", username, token);
    g_key_file_set_string (cache, "Usernames", token, username);
  } else {
    g_key_file_remove_key (cache, "Avatars", username, NULL);
    g_key_file_remove_key (cache, "Usernames", token, NULL);
  }

  data = g_key_file_to_data (cache, &length, &error);
  if  (error) {
    g_warning (G_STRLOC ": cannot write cache: %s", _ERROR_MSG (error));
    g_clear_error (&error);
    return;
  }

  if (!g_file_set_contents (get_cache_filename (),
                            data, length, &error)) {
    g_warning (G_STRLOC ": cannot write cache: %s", _ERROR_MSG (error));
    g_clear_error (&error);
  }

  g_free (data);
}

/* Return true if we already have the avatar represented by this token, false otherwise */
gboolean
avatar_have_for_token (TpConn *conn, guint handle, const char *vcard, const char *token)
{
  GError *error = NULL;
  const char *username = NULL;
  char *existing_token;
  gboolean have_token = FALSE;

  g_return_val_if_fail (TELEPATHY_IS_CONN (conn), FALSE);
  g_return_val_if_fail (handle, FALSE);
  g_return_val_if_fail (vcard, FALSE);

  if (token == NULL || token[0] == '\0')
    return FALSE;

  if (!inspect_handle (conn, handle, &username, &error)) {
    g_warning (G_STRLOC ": cannot inspect handle %d: %s", handle, _ERROR_MSG (error));
    g_clear_error (&error);
    return FALSE;
  }

  existing_token = get_token (username);
  
  if (existing_token && strcmp (existing_token, token) == 0) {
    char *filename;
    
    filename = g_build_filename (g_get_home_dir (),
                                 ".osso-abook", "avatars",
                                 vcard, existing_token,
                                 NULL);

    have_token = g_file_test (filename, G_FILE_TEST_IS_REGULAR |  G_FILE_TEST_EXISTS);

    g_free (filename);
  }

  g_free (existing_token);

  return have_token;
}

/* Save the avatar @data for @handle with token @token, removing any old avatars
   for this user.  Returns TRUE on success, FALSE on failure. */
gboolean
avatar_save (TpConn *conn, const char *vcard, guint handle, const char *token, GArray *data)
{
  GError *error = NULL;
  char *path, *existing_token = NULL, *filename;
  const char *username = NULL;
  gboolean saved = FALSE;

  g_return_val_if_fail (TELEPATHY_IS_CONN (conn), FALSE);
  g_return_val_if_fail (vcard, FALSE);
  g_return_val_if_fail (handle, FALSE);
  g_return_val_if_fail (token, FALSE);
  g_return_val_if_fail (data, FALSE);
  
  /* Check that the avatar doesn't already exist */
  if (avatar_have_for_token (conn, handle, vcard, token))
    return TRUE;

  path = g_build_filename (g_get_home_dir (),
                           ".osso-abook", "avatars",
                           vcard, NULL);
  
  /* Always do this in case it gets wiped during the run */  
  g_mkdir_with_parents (path, 0755);

  if (!inspect_handle (conn, handle, &username, &error)) {
    g_warning (G_STRLOC ": cannot inspect handle %d: %s", handle, _ERROR_MSG (error));
    g_clear_error (&error);
    goto done;
  }

  /* Remove the existing avatar for this username, if it exists */
  /* TODO: call avatar_remove */
  existing_token = get_token (username);
  if (existing_token) {
    filename = g_build_filename (path, existing_token, NULL);
    if (g_unlink (filename) == -1) {
      if (errno != ENOENT) {
        g_warning (G_STRLOC ": unable to unlink the old avatar for %s/%s: %s",
                   username, existing_token, g_strerror (errno));
      }
    }
    g_free (filename);
  }

  /* Save the new avatar */
  filename = g_build_filename (path, token, NULL);
  if (!g_file_set_contents (filename, data->data, data->len, &error)) {
    g_warning (G_STRLOC ": cannot save avatar for %d: %s", handle, _ERROR_MSG (error));
    g_clear_error (&error);
    goto done;
  }
  g_free (filename);

  update_cache (username, token, TRUE);

  saved = TRUE;
  
 done:
  g_free (existing_token);
  g_free (path);

  return TRUE;
}

void
avatar_remove (TpConn *conn, const char *vcard, guint handle)
{
  GError *error = NULL;
  const char *username = NULL;
  char *token, *filename;

  g_return_if_fail (TELEPATHY_IS_CONN (conn));
  g_return_if_fail (vcard);
  g_return_if_fail (handle);

  if (!inspect_handle (conn, handle, &username, &error)) {
    g_warning (G_STRLOC ": cannot inspect handle %d: %s", handle, _ERROR_MSG (error));
    g_clear_error (&error);
    return;
  }

  token = get_token (username);

  /* If the token doesn't exist, it's already been deleted or never existed in
     the first place. */
  if (!token)
    return;

  /* Remove the cache entries */
  update_cache (username, token, FALSE);
  
  filename = g_build_filename (g_get_home_dir (),
                               ".osso-abook", "avatars",
                               vcard, token, NULL);

  if (g_unlink (filename) == -1) {
    if (errno != ENOENT) {
      g_warning (G_STRLOC ": unable to unlink old avatar %s: %s",
                 filename, g_strerror (errno));
    }
  }
  
  g_free (filename);
  g_free (token);
}
