/*
 * This file is part of cx3110x
 *
 * Copyright (C) 2004, 2005, 2006 Nokia Corporation
 *
 * Contact: Kalle Valo <Kalle.Valo@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */
#include <linux/platform_device.h>

#include "sm_drv.h"
#include "sm_drv_ioctl.h"
#include "sm_drv_pda.h"
#include "sm_drv_sysfs.h"

unsigned int driver_type = SM_DRIVER_TYPE_UMAC;

static ssize_t sm_drv_show_signal_quality(struct device *dev,
					  struct device_attribute *attr,
					  char *buf)
{
	struct net_device * net_dev;
	struct net_local *lp;
	uint32_t quality, noise_floor;
	struct obj_bss bss;
	
	net_dev = (struct net_device *)dev_get_drvdata(dev);
	lp = net_dev->priv;
	
	sm_drv_oid_get(net_dev, DOT11_OID_NOISEFLOOR,
		       (void*)&noise_floor, sizeof(uint32_t));

	spin_lock_bh(&lp->lock);

	/* copy this MAC to the bss */
	bss.ext = lp->ext;
	memcpy(bss.address, lp->ap_mac_address, ETH_ALEN);

	spin_unlock_bh(&lp->lock);
	
	/* now ask for the corresponding bss */
	sm_drv_oid_get(net_dev, DOT11_OID_BSSFIND,
		       (void *)&bss, sizeof(struct obj_bss));
	
	/* report the rssi and use it to calculate
	 *  link quality through a signal-noise
	 *  ratio 
	 */
	quality = bss.rssi - noise_floor;
	
	cx3110x_info("quality: %d", quality);
	return sprintf(buf, "%d\n", quality);
}

static ssize_t sm_drv_show_rssi(struct device *dev,
				struct device_attribute *attr, char *buf)
{
	struct net_device * net_dev;
	struct net_local *lp;
	uint32_t rssi, noise_floor;
	struct obj_bss bss;
	
	net_dev = (struct net_device *)dev_get_drvdata(dev);
	lp = net_dev->priv;
	
	sm_drv_oid_get(net_dev, DOT11_OID_NOISEFLOOR,
		       (void*)&noise_floor, sizeof(uint32_t));

	spin_lock_bh(&lp->lock);
	
	/* copy this MAC to the bss */
	bss.ext = lp->ext;
	memcpy(bss.address, lp->ap_mac_address, ETH_ALEN);

	spin_unlock_bh(&lp->lock);
	
	/* now ask for the corresponding bss */
	sm_drv_oid_get(net_dev, DOT11_OID_BSSFIND,
		       (void *)&bss, sizeof(struct obj_bss));
	
	/* report the rssi and use it to calculate
	 *  link quality through a signal-noise
	 *  ratio 
	 */
	rssi = bss.rssi;
	
	return sprintf(buf, "%d\n", rssi);
}


static ssize_t sm_drv_store_psm(struct device *dev,
				struct device_attribute *attr,
				const char *buf, size_t count)
{
	uint32_t psm = simple_strtol(buf, NULL, 0);
	
	if (psm < DOT11_PSM_ACTIVE || 
	    psm > DOT11_PSM_POWERSAVE)
		return 0;
	
	if (sm_drv_oid_set((struct net_device *)dev_get_drvdata(dev), 
			   DOT11_OID_PSM, (void *)&psm, sizeof(uint32_t)) < 0)
		return 0;
	
	return count;
}

static ssize_t sm_drv_show_psm(struct device *dev,
			       struct device_attribute *attr, char *buf)
{
	uint32_t psm;

	if (sm_drv_oid_get((struct net_device *)dev_get_drvdata(dev), 
			   DOT11_OID_PSM, (void *)&psm, sizeof(uint32_t)) < 0)
		return 0;
			
	return sprintf(buf, "%d\n", psm);
}

static ssize_t sm_drv_store_apsd(struct device *dev,
				 struct device_attribute *attr,
				 const char *buf, size_t count)
{
	int ret;
	uint32_t apsd = simple_strtoul(buf, NULL, 0);
	
	ret = sm_drv_oid_set((struct net_device *)dev_get_drvdata(dev), 
			     DOT11_OID_UAPSD, (void *)&apsd, sizeof(uint32_t));
	if (ret < 0)
		return 0;
	
	return count;
}

static ssize_t sm_drv_show_apsd(struct device *dev,
				struct device_attribute *attr, char *buf)
{
	int ret;
	uint32_t apsd;

	ret = sm_drv_oid_get((struct net_device *)dev_get_drvdata(dev), 
			     DOT11_OID_UAPSD, (void *)&apsd, sizeof(uint32_t));
	if (ret < 0)
		return 0;
			
	return sprintf(buf, "0x%x\n", apsd);
}

static DEVICE_ATTR(psm, S_IRUGO | S_IWUGO, sm_drv_show_psm, sm_drv_store_psm);
static DEVICE_ATTR(signal_quality, S_IRUGO | S_IWUSR,
		   sm_drv_show_signal_quality, NULL);
static DEVICE_ATTR(rssi, S_IRUGO | S_IWUSR, sm_drv_show_rssi, NULL);
static DEVICE_ATTR(uapsd, S_IRUGO | S_IWUSR, sm_drv_show_apsd,
		   sm_drv_store_apsd);

extern struct platform_device wlan_omap_device;
int sm_drv_sysfs_umac_create_files(void)
{
	struct device *dev = &(wlan_omap_device.dev);
	int ret;
	
	ret = device_create_file(dev, &dev_attr_psm);
	if (ret < 0) {
		cx3110x_error("failed to create dev_attr_.");
		goto error;
	}

	ret = device_create_file(dev, &dev_attr_signal_quality);
	if (ret < 0) {
		cx3110x_error("failed to create dev_attr_signal_quality.");
		goto error;
	}

	ret = device_create_file(dev, &dev_attr_rssi);	
	if (ret < 0) {
		cx3110x_error("failed to create dev_attr_rssi.");
		goto error;
	}

	ret = device_create_file(dev, &dev_attr_uapsd);
	if (ret < 0) {
		cx3110x_error("failed to create dev_attr_uapsd.");
		goto error;
	}
	
	return 0;

error:
	device_remove_file(dev, &dev_attr_psm);
	device_remove_file(dev, &dev_attr_signal_quality);
	device_remove_file(dev, &dev_attr_rssi);
	device_remove_file(dev, &dev_attr_uapsd);

	return ret;
}
