/* Certificate Management library
 * 
 * Copyright (C) 2005 Nokia. All rights reserved.
 * Author: Ed Bartosh <Eduard.Bartosh@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/**
   @file cst_helper.h

   Helper functions and structures
*/

#ifndef CST_HELPER_H_
#define CST_HELPER_H_

#include "cst_const.h"
#include "cst_key.h"
#include "cst_cert.h"
#include <glib.h>

#ifdef __cplusplus
extern "C" {
#endif

/* Helper function used as param in foreach functions */
    typedef struct HELPER_PARAM_st {
        CST *st;
        GSList *list;
        t_cert_folder folder;
        t_cert_purpose purpose;
    } HELPER_PARAM;

/* Helper functions and structure for parse on reading data from file */
    typedef struct PARSE_RES_st {
        t_rtype type;
        t_rsize size;
        unsigned char *data;
    } PARSE_RES;

    GSList *parse_buffer(unsigned char *buffer, t_rsize len);
    void parse_res_free(GSList * list);

/* Some common helper functions */
    int cst_lock(CST * st, int operation);
    int cst_unlock(CST * st, int operation);
    int cst_sync(DB * st);
    guint name_hash(gconstpointer key);
    gint helper_name_cmp(gconstpointer akey, gconstpointer bkey);
    gint helper_str_cmp(gconstpointer akey, gconstpointer bkey);

    t_seqnum extract_id(DBT * key);
    int cst_check_suffix(DBT * key, const char * suffix);
    void write_mem(void ** dist, void * src, t_rsize sz);
    void write_header_mem(void ** dist, t_rtype tp, t_rsize sz);
    void write_header(CRYPT_FILE * cf, t_rtype tp, t_rsize sz);
    void cert_free_stack(STACK_OF(X509) * certs);

/* Crypt functions */    
    unsigned char * cst_decrypt_buffer(unsigned char * buffer, 
            int *length, unsigned char *pass);
    unsigned char * cst_encrypt_buffer(unsigned char * buffer, 
            int *length, unsigned char *pass);
    int cst_bio_read(BIO * bio, void *buffer, int len);
    int cst_bio_write(BIO * bio, void *buffer, int len);
    
/* Make key */
    void make_dbkey(DBT * dbkey, const char *suffix, const t_seqnum id);    

/* Get and put data to storage */    
    void cst_mcount_update(CST * st);
    t_mcount cst_mcount_get(CST * st);
    int cst_data_put(CST * st, const char *suffix, const t_seqnum id, DBT * data);
    int cst_data_get(CST * st, const char *suffix, const t_seqnum id, DBT * data);

    int cst_refresh_if_need(CST * st);

    int cst_bdb_error_translate(const int bdberr);

/* Lock macros */    
#define CST_LOCK_ERROR  cst_local_error_812736891
    
#define CST_LOCK_BEGIN_S(lk)                         \
    int CST_LOCK_ERROR = cst_lock(st, lk);           \
    if (CST_ERROR_OK == CST_LOCK_ERROR)              \
    {                                                \
        int cst_local_lock_934753957392 = lk;
        
        
#define CST_LOCK_BEGIN(lk)                           \
    CST_LOCK_BEGIN_S(lk);                            \
        CST_LOCK_ERROR = cst_refresh_if_need(st);
        
#define CST_LOCK_END                                 \
        cst_unlock(st, cst_local_lock_934753957392); \
    }                                                \
    if (CST_ERROR_OK != CST_LOCK_ERROR)              \
    {                                                \
        CST_error(CST_LOCK_ERROR);                   \
    }

#define CST_LOCK_SET_ERROR(var)                      \
    if (CST_ERROR_OK != CST_LOCK_ERROR)              \
    {                                                \
        var = CST_LOCK_ERROR;                        \
    }
    
    
/* Some macros for error handling purpose */
#define ERR_CHECK_NULL_P(pointer, errcode) if(NULL == pointer) {CST_error(errcode); return NULL;}
#define ERR_CHECK_NULL_I(pointer, errcode) if(NULL == pointer) {return CST_error(errcode);}

#if 0
    #define NEED_TRACE 
#endif
    
/* Some macros for debug purposes */
#ifdef NEED_TRACE
#define TRACE(msg) printf("--> %s (%s: %s: %d)\n", msg, __FUNCTION__, __FILE__, __LINE__)
#define TRACE_I(i) printf("  >   %s = %i (%s: %s: %d)\n", #i, i, __FUNCTION__, __FILE__, __LINE__)
#define TRACE_U(i) printf("  >   %s = %u (%s: %s: %d)\n", #i, i, __FUNCTION__, __FILE__, __LINE__)    
#define TRACE_S(s) printf("  >   %s = %s (%s: %s: %d)\n", #s, s, __FUNCTION__, __FILE__, __LINE__)
#define TRACE_LIST(list) printListUINT(list, #list);
#else
#define TRACE(msg)
#define TRACE_I(i)
#define TRACE_U(s)
#define TRACE_S(s)    
#define TRACE_LIST(list)  
#endif

#define NAME_IS_NOT_EMPTY(name) ((NULL != name) && (0 != name[0]))     
    
/* Compare with UID */
    gboolean equal_X509_UID(X509 * x, X509_NAME * issuer,
                            ASN1_INTEGER * serial);
    gboolean equal_CERT_UID(CERT * c, X509_NAME * issuer,
                            ASN1_INTEGER * serial);

/* Convert buffer to hexstring */
    unsigned char *buffer_to_hex(unsigned char *buffer, int size);

/* Print GSList (of GUINT_TO_POINTER) */    
    void printListUINT(GSList * list, char *name);

    int cst_db_delete(CST * st, const char *suffix, const t_seqnum ID);
    
    /* GError */
    GQuark cst_error_quark(void);
    #define CST_ERROR_DOMAIN cst_error_quark()
    
/* Format coes for import/export */
#define IMPORT_FORMAT_PEM 0
#define IMPORT_FORMAT_DER 1    

#ifdef __cplusplus
}
#endif
#endif                          /* CST_HELPER_H_ */
