/**
    @file ui_feed_directory.c

    Feed directory functionality and dialogs.

    Copyright (c) 2004-2006 Nokia Corporation. All rights reserved.
	
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
    02111-1307 USA
*/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <libxml/parser.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <hildon/hildon-banner.h>
#include <hildon/hildon-help.h>
#include <osso-log.h>

#define DBUS_API_SUBJECT_TO_CHANGE
#include "ui_itemlist.h"
#include "ui_feed_directory.h"
#include "update.h"
#include "conf.h"

#define FEED_DIRECTORY_DIALOG_X_SIZE 500
#define FEED_DIRECTORY_DIALOG_Y_SIZE 300

GtkWidget *fd_dialog;           /* Feed directory dialog                       */
GtkWidget *dialog_vbox2;        /* Toplevel vbox for the feed directory dialog */
GtkWidget *dialog_vbox3;        /* Vbox for the checkboxes                     */
GtkWidget *combobox;            /* Combobox for selecting the category         */
GtkWidget *label;               /* The label for the category list             */
GtkWidget *dialog_scrolled;

gchar *fd_source;
folderPtr fd_folderPtr;
GSList *category_list;          /* List of categories                          */

gboolean can_quit = TRUE;

extern AppData *app_data;
gboolean adding_feeds_from_opml = FALSE;

/************************************************************************/
/*                        PRIVATE FUNCTIONS                             */
/************************************************************************/

/** Collects the selected checkbox values to the list of feeds
  * @param widget the checkbox for collecting the data
  * @param user_data NULL
  */
static void ui_feed_directory_collect_selected(GtkWidget * widget,
                                               gpointer user_data)
{
    fdNodePtr node = NULL;

    node = (fdNodePtr) gtk_object_get_user_data(GTK_OBJECT(widget));

    if (node)
        node->selected = GTK_TOGGLE_BUTTON(widget)->active;

    gtk_widget_hide(widget);
}

static gint viewport_get_height(GtkViewport * viewport)
{
    GtkWidget *widget = GTK_WIDGET(viewport);
    GtkAllocation *allocation = &widget->allocation;
    gint border_width = GTK_CONTAINER(viewport)->border_width;
    gint result = 0;

    if (viewport->shadow_type != GTK_SHADOW_NONE)
        result = widget->style->ythickness;

    return MAX(1, allocation->height - result * 2 - border_width * 2);
}

static gboolean
cb_focus_in(GtkWidget * widget, GdkEventFocus * event, gpointer user_data)
{
    GtkScrolledWindow *w = GTK_SCROLLED_WINDOW(dialog_scrolled);
    GtkViewport *vp = GTK_VIEWPORT(gtk_bin_get_child(GTK_BIN(w)));
    GtkAdjustment *adj = gtk_scrolled_window_get_vadjustment(w);
    gint top = (gint) gtk_adjustment_get_value(adj),
        height = viewport_get_height(vp);
    if (widget->allocation.y < top)
        gtk_adjustment_set_value(adj, widget->allocation.y);
    else if (widget->allocation.y + widget->allocation.height > top + height)
        gtk_adjustment_set_value(adj, widget->allocation.y - height +
                                 widget->allocation.height);
    return FALSE;
}

/** Fills the feedlist according to the category selection so that
  * there is a checkbox for each feed
  *
  * @param category the selected category
  */
static void ui_feed_directory_fill_feedlist(gchar * category)
{
    GSList *iter;

    iter = category_list;
    if (category)
        while (iter) {
            fdNodePtr node;

            node = (fdNodePtr) iter->data;
            iter = iter->next;

            if (node->url == NULL && !strcmp(category, node->title))
                break;
        }

    gtk_container_foreach(GTK_CONTAINER(dialog_vbox3),
                          (GtkCallback) gtk_widget_hide, NULL);

    while (iter) {
        fdNodePtr node;

        node = (fdNodePtr) iter->data;
        iter = iter->next;

        if (node->url == NULL)
            break;

        if (node->title == NULL)
            node->title = g_strdup("");

        if (node->checkbutton == NULL) {
            GtkWidget *checkbutton = node->checkbutton =
                gtk_check_button_new_with_label(node->title);
            gtk_object_set_user_data(GTK_OBJECT(checkbutton), node);
            gtk_widget_show(checkbutton);
            g_signal_connect(G_OBJECT(checkbutton), "focus-in-event",
                             G_CALLBACK(cb_focus_in), NULL);
            gtk_box_pack_start(GTK_BOX(dialog_vbox3), checkbutton, FALSE,
                               FALSE, 0);
        } else {
            gtk_widget_show(node->checkbutton);
        }
    }
}

/** Callback listening the combobox changed signal
  *
  * @param widget the combobox to listen
  * @param user_data NULL
  */
static void ui_feed_directory_combobox_changed(GtkWidget * widget,
                                               gpointer user_data)
{
    gchar *category = gtk_combo_box_get_active_text(GTK_COMBO_BOX(combobox));
    ui_feed_directory_fill_feedlist(category);
    g_free(category);
}

/** Fills the combobox with the category list provided by an external server
*	@return 0 if the list is empty (because we get a wrong file)
  */
static int ui_feed_directory_fill_categories()
{
    int categories_found = 0;
    GSList *iter;

    if (category_list == NULL)
        return 0;

    iter = category_list;
    while (iter) {
        fdNodePtr node;

        node = (fdNodePtr) iter->data;
        iter = iter->next;

        if (node->url != NULL || node->title == NULL)
            continue;

        categories_found++;
        if (GTK_IS_COMBO_BOX(combobox)) {
            gtk_combo_box_append_text(GTK_COMBO_BOX(combobox), node->title);
        }
    }

    if (categories_found == 0) {
        ui_feed_directory_fill_feedlist(NULL);
        gtk_widget_set_sensitive(combobox, FALSE);
        gtk_widget_set_sensitive(label, FALSE);
    } else
        gtk_combo_box_set_active(GTK_COMBO_BOX(combobox), 0);
    return 1;
}

/** Frees allocated memory
  */
static void ui_feed_directory_free()
{
    GSList *iter;

    if (category_list != NULL) {
        iter = category_list;
        while (iter) {
            fdNodePtr node;

            node = (fdNodePtr) iter->data;
            iter = iter->next;

            if (node != NULL) {
                if (node->title != NULL)
                    g_free(node->title);

                if (node->url != NULL)
                    g_free(node->url);

            }
            //check button will be destroyed when parent dialog is destroyed
            g_free(node);
        }
        g_slist_free(category_list);
    }
}

/** Called when the dialog buttons are being pressed
  *
  * @param widget the dialog
  * @param user_data NULL
  */
static void ui_feed_directory_dialog_response(GtkWidget * widget,
                                              gint response,
                                              gpointer user_data)
{
    GSList *iter;

    adding_feeds_from_opml = TRUE;

    folderPtr folder = (folderPtr) user_data;

    if (!can_quit)
        return;

    if (response == GTK_RESPONSE_OK) {
        gtk_container_foreach(GTK_CONTAINER(dialog_vbox3),
                              ui_feed_directory_collect_selected, NULL);

        iter = category_list;
        while (iter) {
            fdNodePtr node;

            node = (fdNodePtr) iter->data;
            iter = iter->next;
            if (node->url != NULL && node->selected)
                /* Also pass 'folder' as the param */
                ui_feedlist_new_subscription_for_fd(node->url, NULL,
                                                    FEED_REQ_RESET_TITLE |
                                                    FEED_REQ_RESET_UPDATE_INT
                                                    | FEED_REQ_AUTO_DISCOVER,
                                                    folder, TRUE, FALSE);
        }
        if (app_data->app_ui_data->connect_at_the_end) {
            request_iap_connect();
        }
        ui_feedlist_update();
    }
    ui_feed_directory_free();

    gtk_widget_destroy(fd_dialog);
    app_data->app_ui_data->feed_directory_dialog = NULL;
}

/** Appends a node to the category list
  *
  * @param title the title of the feed or category to add
  * @param url the url of the feed or NULL for a category
  */
static void ui_feed_directory_append_node_to_category_list(gchar * title,
                                                           gchar * url)
{
    fdNodePtr node = NULL;

    node = g_new0(struct fd_node_ptr, 1);

    node->title = title;
    node->url = url;
    node->selected = FALSE;
    node->checkbutton = NULL;

    category_list = g_slist_prepend(category_list, node);
}

/** Process outline tags of type rss and collapse the hieararchical structure
  * into one level so that the category list can be built into a single combobox
  *
  * @param cur the current node
  */
static void ui_feed_directory_process_outline_tags(xmlNodePtr cur)
{
    gchar *title = NULL;
    gchar *url = NULL;
    xmlNodePtr child = NULL;
    static int depth = 0;

    /* Do not iterate the opml tree infinitely */
    if (depth >= FEED_DIRECTORY_MAX_DEPTH)
        return;

    depth++;

    g_assert(cur != NULL);

    /* process all <outline> tags */
    child = cur->xmlChildrenNode;
    while (child != NULL) {
        /* If the outline is of type rss, save the necessary data */
        if (!xmlStrcmp(child->name, BAD_CAST "outline")) {
            if (!child->xmlChildrenNode || depth < 2) {
                if (!(title = utf8_fix(xmlGetProp(child, BAD_CAST "text"))))
                    title = utf8_fix(xmlGetProp(child, BAD_CAST "title"));

                url = utf8_fix(xmlGetProp(child, BAD_CAST "xmlUrl"));

                if (title)
                    ui_feed_directory_append_node_to_category_list(title,
                                                                   url);
            }

            if (child->xmlChildrenNode)
                ui_feed_directory_process_outline_tags(child);
        }

        child = child->next;
    }
    depth--;
}

/** Parses the feed directory contents provided in OPML format
  *
  * @param doc the document to parse
  * @param cur the current node
  */
static xmlDocPtr ui_feed_directory_opml_parse(xmlDocPtr doc, xmlNodePtr cur)
{
    g_assert(doc != NULL);
    g_assert(cur != NULL);

    if (xmlStrcmp(cur->name, BAD_CAST "opml") &&
        xmlStrcmp(cur->name, BAD_CAST "oml") &&
        xmlStrcmp(cur->name, BAD_CAST "outlineDocument")) {
        xmlFreeDoc(doc);
        doc = NULL;
    }

    cur = cur->xmlChildrenNode;

    while (cur && xmlIsBlankNode(cur)) {
        cur = cur->next;
    }

    while (cur != NULL) {
        if (!xmlStrcmp(cur->name, BAD_CAST "body"))
            ui_feed_directory_process_outline_tags(cur);
        cur = cur->next;
    }
    return doc;
}

/** Start parsing the feed directory contents
  *
  * @param data the feed directory contents
  * @param length the length of the contents
  */
static void ui_feed_directory_parse(gchar * data, size_t length)
{
    gchar *errors;
    xmlDocPtr doc = NULL;
    xmlNodePtr cur = NULL;

    /* try to parse buffer with XML and to create a DOM tree */
    if (NULL == (doc = parseBuffer(data, length, &errors))) {
        g_debug("XML error occurred. Unable to parse feed directory. ");
        return;
    }
    if (NULL == (cur = xmlDocGetRootElement(doc))) {
        g_debug("Empty document. Unable to parse feed directory. ");

        if (doc != NULL)
            xmlFreeDoc(doc);
        return;
    }
    while (cur && xmlIsBlankNode(cur)) {
        cur = cur->next;
    }
    if (NULL == cur->name) {
        g_debug("Invalid XML. Unable to parse feed directory. ");

        if (doc != NULL)
            xmlFreeDoc(doc);
        return;
    }

    doc = ui_feed_directory_opml_parse(doc, cur);
    if (doc != NULL) {
        xmlFreeDoc(doc);
    }
}

/** Creates the feed directory dialog with its widgets
  * 
  * @return the dialog
  */
//GtkWidget *ui_feed_directory_create_dialog(struct request *request)
GtkWidget *ui_feed_directory_create_dialog(folderPtr folder)
{
    GtkWidget *dialog_vbox;
    GtkWidget *subscribe_button;
    GtkWidget *close_button;

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    fd_dialog = gtk_dialog_new();

    hildon_help_dialog_help_enable(GTK_DIALOG(fd_dialog),
                                   RSS_NEWS_READER_HELP_DIRECTORY,
                                   app_data->osso);

    gtk_window_set_title(GTK_WINDOW(fd_dialog), _("rss_ti_feed_directory"));
    gtk_window_set_default_size(GTK_WINDOW(fd_dialog),
                                FEED_DIRECTORY_DIALOG_X_SIZE,
                                FEED_DIRECTORY_DIALOG_Y_SIZE);
    gtk_window_set_type_hint(GTK_WINDOW(fd_dialog),
                             GDK_WINDOW_TYPE_HINT_DIALOG);
    gtk_window_set_transient_for(GTK_WINDOW(fd_dialog),
                                 GTK_WINDOW(app_data->app_ui_data->
                                            main_view));
    gtk_window_set_modal(GTK_WINDOW(fd_dialog), TRUE);
    gtk_dialog_set_has_separator(GTK_DIALOG(fd_dialog), FALSE);

    dialog_vbox = GTK_DIALOG(fd_dialog)->vbox;
    gtk_widget_show(dialog_vbox);

    dialog_scrolled = gtk_scrolled_window_new(NULL, NULL);
    gtk_widget_show(dialog_scrolled);
    gtk_container_set_border_width(GTK_CONTAINER(dialog_scrolled),
                                   HILDON_MARGIN_DEFAULT);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(dialog_scrolled),
                                   GTK_POLICY_AUTOMATIC,
                                   GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW
                                        (dialog_scrolled), GTK_SHADOW_NONE);
    dialog_vbox2 = gtk_hbox_new(FALSE, 0);
    gtk_widget_show(dialog_vbox2);

    gtk_box_pack_start(GTK_BOX(dialog_vbox), dialog_vbox2, FALSE, FALSE, 0);
    gtk_box_pack_end(GTK_BOX(dialog_vbox), dialog_scrolled, TRUE, TRUE, 0);

    combobox = gtk_combo_box_new_text();
    gtk_widget_show(combobox);
    label =
        hildon_caption_new(NULL, _("rss_ia_category"), combobox, NULL,
                           HILDON_CAPTION_OPTIONAL);
    gtk_widget_show(label);

    gtk_box_pack_start(GTK_BOX(dialog_vbox2), label, TRUE, TRUE, 10);

    dialog_vbox3 = gtk_vbox_new(FALSE, 0);
    gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW
                                          (dialog_scrolled), dialog_vbox3);

    gtk_widget_show(dialog_vbox3);

    subscribe_button = gtk_button_new_with_label(_("rss_bd_subscribe"));
    close_button = gtk_button_new_with_label(_("rss_bd_close"));

    gtk_widget_show(subscribe_button);
    gtk_widget_show(close_button);

    gtk_dialog_add_action_widget(GTK_DIALOG(fd_dialog), subscribe_button,
                                 GTK_RESPONSE_OK);
    gtk_dialog_add_action_widget(GTK_DIALOG(fd_dialog), close_button,
                                 GTK_RESPONSE_CANCEL);

    GTK_WIDGET_SET_FLAGS(subscribe_button, GTK_CAN_DEFAULT);

    g_signal_connect(G_OBJECT(fd_dialog), "response",
                     G_CALLBACK(ui_feed_directory_dialog_response),
                     (gpointer) folder);

    g_signal_connect(G_OBJECT(combobox), "changed",
                     G_CALLBACK(ui_feed_directory_combobox_changed), NULL);

    return fd_dialog;
}

/** Checks the status of downloading the contents of the feed directory
  * and starts parsing it on success
  * @param request the download request
  */
void ui_feed_directory_process(struct request *request)
{
    GtkWidget *fd_dialog;

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    if (!app_data->app_ui_data->feed_directory_loading)
        return;

    app_data->app_ui_data->feed_directory_loading = FALSE;

    g_assert(NULL != request);

    if (401 == request->httpstatus) {
        hildon_banner_show_information(GTK_WIDGET
                                       (app_data->app_ui_data->main_view),
                                       NULL,
                                       _("rss_ni_errors_loading_directory"));
        g_debug("Download status: 401");
    } else if (410 == request->httpstatus) {
        hildon_banner_show_information(GTK_WIDGET
                                       (app_data->app_ui_data->main_view),
                                       NULL,
                                       _("rss_ni_errors_loading_directory"));
        g_debug("Download status: 410");
    } else if (304 == request->httpstatus) {
        g_debug("Download status: 304");
    } else if (NULL != request->data) {
        g_debug("Download status: OK");
	
        app_data->app_ui_data->feed_directory_dialog = fd_dialog =
            ui_feed_directory_create_dialog(request->folder);

        category_list = NULL;
        ui_feed_directory_parse(request->data, request->size);
        category_list = g_slist_reverse(category_list);
        if (ui_feed_directory_fill_categories()) {
            gtk_widget_show(GTK_WIDGET(fd_dialog));
            //now set the feed directory 
            setStringConfValue(FEED_DIRECTORY_SOURCE, fd_source);
        } else {
            gtk_widget_destroy(GTK_WIDGET(fd_dialog));
            ui_feed_directory_free();
            hildon_banner_show_information(GTK_WIDGET
                                           (app_data->app_ui_data->main_view),
                                           NULL,
                                           _("rss_ni_errors_while_parsing_feed"));
        }
    } else {
        hildon_banner_show_information(GTK_WIDGET
                                       (app_data->app_ui_data->main_view),
                                       NULL,
                                       _("rss_ni_errors_loading_directory"));
        g_debug("Feed directory not available. HTTP status: %d",
                   request->httpstatus);
    }
}

/** Checks if the type of the given outline is rss
  *
  * @param cur the current node
  * @return TRUE if the given outline is of type rss, otherwise FALSE
  */
gboolean ui_feed_directory_outline_type_is_rss(xmlNodePtr cur)
{
    gchar *value = NULL;

    if (NULL != (value = utf8_fix(xmlGetProp(cur, BAD_CAST "type")))) {
        if (!strcmp(value, "rss")) {
            g_free(value);
            return TRUE;
        }
    }

    if (value)
        g_free(value);

    return FALSE;
}

void on_menuitem_feed_directory_clicked(GtkWidget * menuitem,
                                        gpointer user_data)
{

    folderPtr ptr = NULL;

    /* While searching, no feed should be added */
    if (!GTK_WIDGET_SENSITIVE(app_data->app_ui_data->menuitem_feed_directory)) {
        return;
    }

    /* 28-06-2006 Added low mem check */
    if (app_data->app_ui_data->low_mem) {
        hildon_banner_show_information(GTK_WIDGET
                                       (app_data->app_ui_data->main_view),
                                       NULL, dgettext("ke-recv",
                                                      "memr_ib_operation_disabled"));
        return;
    }

    /* This value is either the default or the one set by the last
     * opml_import action */
    gchar *feed_dir_source = getStringConfValue(FEED_DIRECTORY_SOURCE);
    //If it's never been touched before
    if (strlen(feed_dir_source) == 0) {
        /* spec changed so that there should be no default feed directory.
         * Due to the permission from feed server
         */
        g_free(feed_dir_source);
        return;
    }
    //now continue with the feed directory once we've fot the feed dir source
    ptr = ui_feedlist_get_target_folder();
    ui_feed_directory_button_clicked(NULL, NULL, feed_dir_source, ptr);
    g_free(feed_dir_source);
}

void ui_feed_directory_button_clicked(GtkWidget * widget,
                                      gpointer user_data, gchar * fdsource,
                                      folderPtr folder)
{
    AppUIData *app_ui_data;

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    g_debug("ui_feed_directory_button_clicked: with source = %s",
               fdsource);

    if (fdsource)
        fd_source = g_strdup(fdsource);
    else                        //also needs to reset fd_source
    {
        if (fd_source)
            g_free(fd_source);
        fd_source = NULL;
    }

    if (folder)
        fd_folderPtr = folder;
    else
        fd_folderPtr = NULL;

    if (app_data->app_ui_data->feed_directory_loading)
        return;

    app_ui_data = app_data->app_ui_data;

    app_ui_data->iap_action = OSSO_IAP_REFRESH_FEED_DIRECTORY;
    request_iap_connect();
}


void ui_feed_directory_get()
{
    struct request *request = NULL;
    AppUIData *app_ui_data;

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);
    g_assert(app_data->app_ui_data->search_button != NULL);
    g_assert(app_data->app_ui_data->button_image_stop != NULL);

    if (!fd_source)
        fd_source = getStringConfValue(FEED_DIRECTORY_SOURCE);

    if (strlen(fd_source) == 0) {
        g_free(fd_source);
        hildon_banner_show_information(GTK_WIDGET
                                       (app_data->app_ui_data->main_view),
                                       NULL,
                                       _("rss_ni_errors_loading_directory"));
        return;
    }

    app_ui_data = app_data->app_ui_data;

    app_ui_data->feed_directory_loading = TRUE;

    switch_progressbar(PROGRESSBAR_RECEIVING_MODE);

    request = download_request_new();

    request->callback = ui_feed_directory_process;
    request->source = fd_source;
    request->user_data = NULL;

    /* Add a parent folder as well */
    if (fd_folderPtr != NULL)
        request->folder = fd_folderPtr;
    else
        request->folder = NULL;
    request->priority = 1;

    download_queue(request);
}

/*tvh: 2006May31: This is to create a feed directory when adding an opml file from
 * a add-feed dialog. In stead of the old confirmation dialog: "Add all xx feeds"
 * This feed directory is created.
 * Note : used in the feedParser func of the opml feed handler pointer : opml_parse()
 * in opml.c
 * couldn't think of a better name for the func :(
 */
void ui_feed_directory_from_add_feed_dialog(xmlDocPtr doc,
                                            xmlNodePtr cur, folderPtr folder,
                                            gchar * fd_source)
{
    app_data->app_ui_data->feed_directory_dialog = fd_dialog =
        ui_feed_directory_create_dialog(folder);

    category_list = NULL;
    ui_feed_directory_opml_parse(doc, cur);
    category_list = g_slist_reverse(category_list);
    if (ui_feed_directory_fill_categories()) {
        setStringConfValue(FEED_DIRECTORY_SOURCE, fd_source);
        gtk_widget_show(GTK_WIDGET(fd_dialog));
    } else {
        gtk_widget_destroy(GTK_WIDGET(fd_dialog));
        ui_feed_directory_free();
        hildon_banner_show_information(GTK_WIDGET
                                       (app_data->app_ui_data->main_view),
                                       NULL,
                                       _("rss_ni_errors_while_parsing_feed"));
    }
}
