/**
 * @file ui_feed.c	UI actions concerning a single feed
 *
 * Copyright (C) 2004 Lars Lindner <lars.lindner@gmx.net>
 * Copyright (C) 2004 Nathan J. Conrad <t98502@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <gtk/gtk.h>
#include <libxml/uri.h>
#include <hildon/hildon-banner.h>
#include <hildon/hildon-helper.h>
#include <strings.h>
#include "debug.h"
#include "support.h"
#include "feed.h"
#include "conf.h"
#include "callbacks.h"
#include "update.h"
#include "htmlview.h"
#include "interface.h"
#include "ui_feed.h"
#include "favicon.h"
#include "ui_notification.h"
#include "ui_feed_directory.h"
#include "debug_new.h"
#include <osso-rss-feed-reader/cache_handling.h>

#include <osso-log.h>

extern guint rss_max_feed;
extern AppData *app_data;
extern gboolean adding_feeds_from_opml;

/**
 * Creates a new error description according to the passed
 * HTTP status and the feeds parser errors. If the HTTP
 * status is a success status and no parser errors occured
 * no error messages is created. The created error message 
 * can be queried with feed_get_error_description().
 *
 * @param fp		feed
 * @param httpstatus	HTTP status
 * @param resultcode the update code's return code (see update.h)
 */
static void
ui_feed_set_error_description(feedPtr fp, gint httpstatus, gint resultcode)
{
    gchar *tmp1 = NULL, *tmp2 = NULL, *buffer = NULL;
    gboolean errorFound = FALSE;

    g_assert(NULL != fp);
    g_free(fp->errorDescription);
    fp->errorDescription = NULL;

    if (((httpstatus >= 200) && (httpstatus < 400)) &&  /* HTTP codes
                                                         * starting with 2
                                                         * and 3 mean no
                                                         * error */
        (NULL == fp->parseErrors))
        return;

    if ((200 != httpstatus) || (resultcode != NET_ERR_OK)) {
        /* first specific codes */
        switch (httpstatus) {
            case 401:
                tmp2 =
                    g_strdup
                    ("You are unauthorized to download this feed. Please update your username and "
                     "password in the feed properties dialog box.");
                break;
            case 402:
                tmp2 = g_strdup("Payment Required");
                break;
            case 403:
                tmp2 = g_strdup("Access Forbidden");
                break;
            case 404:
                tmp2 = g_strdup("Resource Not Found");
                break;
            case 405:
                tmp2 = g_strdup("Method Not Allowed");
                break;
            case 406:
                tmp2 = g_strdup("Not Acceptable");
                break;
            case 407:
                tmp2 = g_strdup("Proxy Authentication Required");
                break;
            case 408:
                tmp2 = g_strdup("Request Time-Out");
                break;
            case 410:
                tmp2 =
                    g_strdup
                    ("Gone. Resource doesn't exist. Please unsubscribe!");
                break;
        }
        /* Then, netio errors */
        if (tmp2 == NULL) {
            switch (resultcode) {
                case NET_ERR_URL_INVALID:
                    tmp2 = g_strdup("URL is invalid");
                    break;
                case NET_ERR_UNKNOWN:
                case NET_ERR_CONN_FAILED:
                case NET_ERR_SOCK_ERR:
                    tmp2 = g_strdup("Error connecting to remote host");
                    break;
                case NET_ERR_HOST_NOT_FOUND:
                    tmp2 = g_strdup("Hostname could not be found");
                    break;
                case NET_ERR_CONN_REFUSED:
                    tmp2 =
                        g_strdup
                        ("Network connection was refused by the remote host");
                    break;
                case NET_ERR_TIMEOUT:
                    tmp2 =
                        g_strdup("Remote host did not finish sending data");
                    break;
                    /* Transfer errors */
                case NET_ERR_REDIRECT_COUNT_ERR:
                    tmp2 =
                        g_strdup("Too many HTTP redirects were encountered");
                    break;
                case NET_ERR_REDIRECT_ERR:
                case NET_ERR_HTTP_PROTO_ERR:
                case NET_ERR_GZIP_ERR:
                    tmp2 = g_strdup("Remote host sent an invalid response");
                    break;
                    /* These are handled above case NET_ERR_HTTP_410: case
                     * NET_ERR_HTTP_404: case NET_ERR_HTTP_NON_200: */
                case NET_ERR_AUTH_FAILED:
                case NET_ERR_AUTH_NO_AUTHINFO:
                    tmp2 = g_strdup("Authentication failed");
                    break;
                case NET_ERR_AUTH_GEN_AUTH_ERR:
                case NET_ERR_AUTH_UNSUPPORTED:
                    tmp2 =
                        g_strdup
                        ("Webserver's authentication method incompatible with Liferea");
                    break;
            }
        }
        /* And generic messages in the unlikely event that the above didn't
         * work */
        if (NULL == tmp2) {
            switch (httpstatus / 100) {
                case 3:
                    tmp2 =
                        g_strdup
                        ("Feed not available: Server requested unsupported redirection!");
                    break;
                case 4:
                    tmp2 = g_strdup("Client Error");
                    break;
                case 5:
                    tmp2 = g_strdup("Server Error");
                    break;
                default:
                    tmp2 = g_strdup("(unknown networking error happened)");
                    break;
            }
        }
        errorFound = TRUE;
        tmp1 = g_strdup_printf(HTTP_ERROR_TEXT, httpstatus, tmp2);
        addToHTMLBuffer(&buffer, tmp1);
        g_free(tmp1);
        g_free(tmp2);
    }

    /* add parsing error messages */
    if (NULL != fp->parseErrors) {
        errorFound = TRUE;
    }

    /* if none of the above error descriptions matched... */
    if (!errorFound) {
    }

    fp->errorDescription = buffer;
}

static gboolean do_unsubscribe(struct fp_prop_ui_data *ui_data)
{
    if ((nodePtr) ui_data->fp != ui_feedlist_get_selected())
        ui_feedlist_select((nodePtr) ui_data->fp);

    if (SFM_SEARCH == app_data->app_ui_data->search_mode) {

        return FALSE;
    }

    /* TODO: downloading images also uses SFM_REFRESH mode */
    if (SFM_REFRESH == app_data->app_ui_data->search_mode) {

        return FALSE;
    }

    nodePtr ptr = (nodePtr) ui_feedlist_get_selected();

    if (!ptr || ptr->type != FST_FEED) {

        return FALSE;
    }

    if (ui_feedlist_check_factory_delete(TRUE))
        return FALSE;

    return TRUE;
}

// static void on_insensitive_unsubscribe(GtkButton * button, gpointer user_data)
// {
//     do_unsubscribe((struct fp_prop_ui_data *) user_data);
// }

/** Decodes source
 *
 * @param ui_data pointer to an ui data structure
 * @return the decoded source
 */
static gchar *ui_feed_dialog_decode_source(struct fp_prop_ui_data *ui_data)
{
    gchar *source = NULL;

    gchar *str = NULL, *tmp2 = NULL;
    /* First, strip leading and trailing whitespace */
    if (ui_data->sourceURI) {
        str = g_strstrip(g_strdup(ui_data->sourceURI));
    } else {
        str =
            g_strstrip(g_strdup
                       (gtk_entry_get_text(GTK_ENTRY(ui_data->sourceEntry))));
    }

    /* Add http:// if needed */
    if (strstr(str, "://") == NULL) {
        tmp2 = g_strdup_printf("http://%s", str);
        g_free(str);
        str = tmp2;
    }

    /* Add trailing / if needed */
    if (strstr(strstr(str, "://") + 3, "/") == NULL) {
        tmp2 = g_strdup_printf("%s/", str);
        g_free(str);
        str = tmp2;
    }

    /* Changing feed:// ->http:// */
    if (g_str_has_prefix(str, "feed:")) {
        str[0] = 'h';
        str[1] = 't';
        str[2] = 't';
        str[3] = 'p';
    }
    /* Use the values in the textboxes if also specified in the URL! */
    if ((NULL != ui_data->authcheckbox) &&
        gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON
                                     (ui_data->authcheckbox))) {
        xmlURIPtr uri = xmlParseURI(BAD_CAST str);
        if (uri != NULL) {
            xmlChar *sourceUrl;
            xmlFree(uri->user);
            uri->user = g_strdup_printf("%s:%s",
                                        gtk_entry_get_text(GTK_ENTRY
                                                           (ui_data->
                                                            username)),
                                        gtk_entry_get_text(GTK_ENTRY
                                                           (ui_data->
                                                            password)));
            sourceUrl = xmlSaveUri(uri);
            source = g_strdup(BAD_CAST sourceUrl);
            g_free(uri->user);
            uri->user = NULL;
            xmlFree(sourceUrl);
            xmlFreeURI(uri);
        } else
            source = g_strdup(str);
    } else {
        source = g_strdup(str);
    }
    g_free(str);
    str = NULL;

    return source;
}

static gboolean
ui_feed_newdialog_can_close(struct fp_prop_ui_data * ui_data, gboolean msg,
                            gchar ** result)
{
    gchar *source = ui_feed_dialog_decode_source(ui_data);

    if (result)
        *result = NULL;

    if (strcmp(source, "") == 0) {
        if (msg)
            hildon_banner_show_information(GTK_WIDGET
                                           (app_data->app_ui_data->main_view),
                                           NULL,
                                           _("rss_ni_no_web_address_for_feed"));
        g_free(source);
        return FALSE;
    } else if (strcasecmp(source, "http:///") == 0) {
        if (msg)
            hildon_banner_show_information(GTK_WIDGET
                                           (app_data->app_ui_data->main_view),
                                           NULL,
                                           _
                                           ("rss_ni_no_web_address_for_feed"));
        g_free(source);
        return FALSE;
    } else if ((strncasecmp(source, "http://", 7) != 0) &&
               (strncasecmp(source, "https://", 8) != 0) &&
               (strncasecmp(source, "file://", 7) != 0) &&
               (strncasecmp(source, "obex://", 7) != 0) &&
               (strncasecmp(source, "feed://", 7) != 0)) {
        if (msg)
            hildon_banner_show_information(GTK_WIDGET
                                           (app_data->app_ui_data->main_view),
                                           NULL, _("rss_ni_adding_failed"));
        g_free(source);
        return FALSE;
    } else if (strchr(source, '.') == NULL) {
        if (msg)
            hildon_banner_show_information(GTK_WIDGET
                                           (app_data->app_ui_data->main_view),
                                           NULL, _("rss_ni_adding_failed"));
        g_free(source);
        return FALSE;
    } else {
        // something like this http://qewr./ is not valid. Should be
        // http://abc.xyz/
        char *tmp = strchr(source, '.');
        if (strncmp(tmp, "./", 2) == 0) {
            if (msg)
                hildon_banner_show_information(GTK_WIDGET
                                               (app_data->app_ui_data->
                                                main_view), NULL,
                                               _("rss_ni_adding_failed"));
            g_free(source);
            return FALSE;
        }
    }

    if (result) {
        *result = source;
    } else {
        g_free(source);
    }

    return TRUE;
}

/** After closing the add new feed dialog this function is called
 *
 * @param dialog the new feed dialog
 * @param response_id the id sent by hitting one of the dialog buttons
 * @param user_data pointer to an ui data structure
 */
static void
on_newdialog_response(GtkDialog * dialog, gint response_id,
                      gpointer user_data)
{
    extern gint addfeedcounter;
    struct fp_prop_ui_data *ui_data = (struct fp_prop_ui_data *) user_data;
    AppUIData *app_ui_data;
    g_assert(app_data != NULL);

    app_ui_data = app_data->app_ui_data;

    gchar *source = NULL;

    g_assert(app_ui_data != NULL);

    if (app_ui_data->source != NULL) {
        g_free(app_ui_data->source);
        app_ui_data->source = NULL;
    }

    if (app_ui_data->filter != NULL) {
        g_free((gchar *) app_ui_data->filter);
        app_ui_data->filter = NULL;
    }

    switch (response_id) {
        case GTK_RESPONSE_OK:
            addfeedcounter = 0; 
            if (!ui_feed_newdialog_can_close(ui_data, FALSE, &source)) {
                gtk_widget_grab_focus(ui_data->sourceEntry);
            } else {
                gtk_window_set_modal(GTK_WINDOW(app_ui_data->add_feed_dialog),
                                     TRUE);
                gtk_widget_set_sensitive(app_data->app_ui_data->newfeedbtn,
                                         FALSE);
                gtk_widget_set_sensitive(app_data->app_ui_data->
                                         changefolderbtn, FALSE);
                gtk_widget_set_sensitive(app_data->app_ui_data->browsebtn,
                                         FALSE);
                gtk_widget_set_sensitive(ui_data->sourceEntry, FALSE);
                hildon_helper_set_insensitive_message(GTK_WIDGET
                                                      (ui_data->newfeedbtn),
                                                      _("rss_ib_subcribe_already_started"));
                ui_feedlist_new_subscription(source, NULL,
                                             FEED_REQ_RESET_TITLE |
                                             FEED_REQ_RESET_UPDATE_INT |
                                             FEED_REQ_AUTO_DISCOVER |
                                             FEED_REQ_DOWNLOAD_FAVICON,
                                             ui_data->parent_folder);
            }
            break;

        case GTK_RESPONSE_CANCEL:
        case GTK_RESPONSE_DELETE_EVENT:

            if (SFM_REFRESH == app_ui_data->search_mode) {
                app_ui_data->search_mode = SFM_INTERRUPTED;
                download_cancel_all(TRUE);
            }
            newdialog_destroy();
            break;
    }
}

static gboolean
feed_prop_can_close(struct fp_prop_ui_data * ui_data, gboolean show,
                    gchar ** ptitle, gchar ** psource)
{
    if (feed_get_nonremovable(ui_data->fp))
        return TRUE;

    const gchar *title =
        gtk_entry_get_text(GTK_ENTRY(ui_data->feedNameEntry));
    if (title) {
        gchar *tmp = g_strstrip(g_strdup(title));
        if (!*tmp)
            title = NULL;
        g_free(tmp);
    }

    if (!title) {
        if (show) {
            hildon_banner_show_information(GTK_WIDGET
                                           (app_data->app_ui_data->main_view),
                                           NULL,
                                           _("rss_ni_no_title_for_feed"));
        

            gtk_widget_grab_focus(ui_data->feedNameEntry);
        }
        return FALSE;
    }

    const gchar *source =
        gtk_entry_get_text(GTK_ENTRY(ui_data->feedLocationEntry));

    if (source) {
        gchar *tmp = g_strstrip(g_strdup(title));
        if (!*tmp)
            source = NULL;
        g_free(tmp);
    }

    if ((!source) || (strcmp(source, "") == 0)) {
        if (show) {
            hildon_banner_show_information(GTK_WIDGET
                                           (app_data->app_ui_data->main_view),
                                           NULL,
                                           _("rss_ni_no_web_address_for_feed"));
        

            gtk_widget_grab_focus(ui_data->feedLocationEntry);
        }
        return FALSE;
    }

    if (ptitle)
        *ptitle = (gchar *) title;
    if (psource)
        *psource = (gchar *) source;

    return TRUE;
}

/** By clicking on one of the buttons to close the feed details dialog
 * this function is called
 *
 * @param dialog the feed details dialog
 * @param response_id  the id sent by hitting one of the dialog buttons
 * @param user_data pointer to an ui data structure
 */
static void
on_propdialog_response(GtkDialog * dialog, gint response_id,
                       gpointer user_data)
{
    struct fp_prop_ui_data *ui_data = (struct fp_prop_ui_data *) user_data;
    AppUIData *app_ui_data;
    gboolean close_dialog = TRUE;
    gchar *title = NULL;
    gchar *source = NULL;

    g_assert(app_data != NULL);

    app_ui_data = app_data->app_ui_data;

    g_assert(app_ui_data != NULL);

    switch (response_id) {
        case GTK_RESPONSE_OK:
            /* General */
            g_assert(ui_data != NULL);
            g_assert(ui_data->feedNameEntry != NULL);

            if (feed_prop_can_close(ui_data, TRUE, &title, &source)) {
                if (ui_data->parent_folder != NULL) {
                    ui_feedlist_remove((nodePtr) ui_data->fp);
                    g_assert(ui_data->fp->ui_data == NULL);
                    ui_feedlist_add(ui_data->parent_folder,
                                    (nodePtr) ui_data->fp, -1);
                }

                if (!feed_get_nonremovable(ui_data->fp)) {
                    feed_set_title(ui_data->fp, title);
                    feed_set_source(ui_data->fp, source);
                    if (displayed_node == (nodePtr) ui_data->fp) {
                        displayed_node = NULL;
                        ui_feedlist_load_selected((nodePtr) ui_data->fp);
                    }
                }

                gboolean dn =
                    gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON
                                                 (ui_data->
                                                  downloadImagesEntry));
                if (dn != feed_get_download_images(ui_data->fp))
                    feed_set_download_images(ui_data->fp, dn);

                ui_feedlist_update();
                conf_feedlist_save_config();
            } else
                close_dialog = FALSE;

            break;
        case RSS_RESPONSE_FEED_PROP_UNSUBSCRIBE:

            g_assert(ui_data != NULL);
            g_assert(ui_data->fp != NULL);

            close_dialog = do_unsubscribe(ui_data)
                && ui_feedlist_delete((nodePtr) ui_data->fp);
            break;
    }

    if (close_dialog) {
        app_ui_data->dlg_btn_unsubscribe = NULL;
        g_free(ui_data);
        app_ui_data->dialog = NULL;
        gtk_widget_destroy(GTK_WIDGET(dialog));
    }
}

/** Determines the feeds favicon or default icon 
 *
 * @param fp the feed to get the icon for
 * @return the icon
 */
static GdkPixbuf *ui_feed_get_icon(feedPtr fp)
{
    gpointer favicon;

    g_assert(FST_FOLDER != fp->type);

    if ((NULL != (favicon = feed_get_favicon(fp))) && feed_get_available(fp))
        return favicon;
    if (feed_get_available(fp))
        return icons[ICON_RSS_NEWS_FEED];
    else
        return icons[ICON_RSS_NEWS_INVALID_FEED];
}


/* Remove the hidden and new attribute from that items, what it has these */
void ui_feed_remove_hidden_attribute(feedPtr fp)
{
    GSList *item = NULL;
    item = fp->items;
    int i = 0;
    /* We have to analyse, if we have hidden items. If we don't, we don't do
     * anything. */
    while ((NULL != item) && !i) {
        if (item_get_hidden((itemPtr) item->data)) {
            i++;
        }
        item = g_slist_next(item);
    }
    item = fp->items;
    /* If we don't have items, what have hidden status, than there are only
     * new items, so we don't need to do anything */
    if (i) {
        while (NULL != item) {
            if (item_get_new_status((itemPtr) item->data)) {
                if (item_get_hidden((itemPtr) item->data)) {
                    item_set_hidden((itemPtr) item->data, FALSE);
                    item_set_new_status((itemPtr) item->data, FALSE);
                }
            }
            item = g_slist_next(item);
        }
    }
}

/** Deletes feed items not present on the server or not marked as keep-for-later.
 *
 * @param feed feed to delete from
 */
static void ui_feed_delete_offline_items(feedPtr feed)
{
    /* copy the list for safety */
    GSList *itemscopy = g_slist_copy(feed_get_item_list(feed));
    GSList *items = itemscopy;

    while (items != NULL) {
        itemPtr item = (itemPtr) items->data;
        if (!item->on_server && !item_get_mark(item)) {
            feed_remove_item(feed, item);
        }
        items = g_slist_next(items);
    }
    g_slist_free(itemscopy);
}

/** Compare function used for sorting feed items by time descending.
 *
 * @param a first item
 * @param b second item
 */
static gint compare_by_time_desc(gconstpointer a, gconstpointer b)
{
    return ((itemPtr) b)->time - ((itemPtr) a)->time;
}

/** Delete all but the latest DEFAULT_MAX_ITEMS items of the feed
 *
 * @param feed feed to process
 */
static void ui_feed_delete_all_but_latest(feedPtr feed)
{
    GSList *itemscopy = g_slist_copy(feed_get_item_list(feed));
    GSList *items = itemscopy;
    guint feedcount = 0;

    items = g_slist_sort(items, compare_by_time_desc);

    /* get first 100 elements */
    while (items != NULL) {
        feedcount += 1;
        if (feedcount > DEFAULT_MAX_ITEMS)
            break;
        items = g_slist_next(items);
    }

    /* if more than 100 elements are found then remove the unneeded elements */
    if (feedcount > DEFAULT_MAX_ITEMS && items != NULL) {
        /* remove the items */
        while (items != NULL) {
            itemPtr item = (itemPtr) items->data;
            feed_remove_item(feed, item);
            items = g_slist_next(items);
        }
    }
    /* free the list */
    g_slist_free(itemscopy);
}

static void updatefeedurl(struct request *request, feedPtr fp)
{
    /* note this is to update the feed URL on permanent redirects */
    if (strcmp(request->source, feed_get_source(fp)) != 0) {
        feed_set_source(fp, request->source);
    }
}


static void setonserverstatus(feedPtr fp, gboolean status)
{

    GSList *items = feed_get_item_list(fp);
    while (items != NULL) {
        itemPtr item = (itemPtr) items->data;
        item->on_server = status;
        items = g_slist_next(items);
    }
}

struct pur_data {
    feedPtr fp_pur;
    feedHandlerPtr fhp_pur;
    gchar *old_title_pur;
    gchar *old_source_pur;
    gint old_update_interval_pur;
    gboolean must_load_selected_pur;
};

void ui_feed_process_update_result(struct request *request)
{
    feedPtr fp = NULL;
    feedHandlerPtr fhp = NULL;
    gchar *old_title = NULL, *old_source = NULL;
    gint old_update_interval = 0;
    gboolean must_load_selected = FALSE;
    gboolean opmlsaved = FALSE;

    if (request->jump) {
        struct pur_data *pdata = NULL;
        pdata = (struct pur_data *) request->pur_data;
        if (pdata) {
            fp = pdata->fp_pur;
            fhp = request->fhp;
            old_title = pdata->old_title_pur;
            old_source = pdata->old_source_pur;
            old_update_interval = pdata->old_update_interval_pur;
            must_load_selected = pdata->must_load_selected_pur;
            g_free(pdata);
            //well, it is not good to use goto, but now I have to :/
            g_get_current_time(&fp->lastPoll);
            goto continue_update_result;
        }
    } else {
        fp = (feedPtr) request->user_data;
        g_get_current_time(&fp->lastPoll);
        must_load_selected = ((feedPtr) displayed_node == fp);
    }

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);


    if ((must_load_selected) && (NULL != request->data)) {
        gtkhtml_stop();
    }

    g_assert(NULL != request);

    feed_load(fp);

    fp->needsCacheSave = TRUE;

    feed_set_available(fp, TRUE);
    switch (request->httpstatus) {
        case 401:
            g_debug("Returned 401: Unauthorized request");
            feed_set_available(fp, FALSE);
            app_data->app_ui_data->errors_in_refreshing = TRUE;
            break;
        case 410:
            ULOG_DEBUG("Returned 410");
            feed_set_available(fp, FALSE);
            feed_set_discontinued(fp, TRUE);
            g_debug("\"%s\" is discontinued. Liferea won't updated it anymore!",
                 feed_get_title(fp));
            app_data->app_ui_data->errors_in_refreshing = TRUE;
            break;
        case 304:
            g_debug("Returned 304");
            /* if the on_server flag of the items is up to date then we can
             * delete the items not online and not kept for later */
            ui_feed_delete_offline_items(fp);
            if (must_load_selected) {
                feed_set_feed_read(fp, -1);
                ui_itemlist_load((nodePtr) fp);
                feed_mark_all_items_read(fp);
            }
            if (feed_get_download_images(fp)) {
                feed_download_images(fp);
            }
            break;
        default:
            if (request->data != NULL) {
                feed_reset_update_counter(fp);
                feed_set_lastmodified(fp, request->lastmodified);
                feed_set_etag(fp, request->etag);

                updatefeedurl(request, fp);
                if (must_load_selected) {
                    ui_itemlist_clear();
                }

                /* we save all properties that should not be overwritten in all cases */
                old_update_interval = feed_get_update_interval(fp);
                old_title = g_strdup(feed_get_title(fp));
                old_source = g_strdup(feed_get_source(fp));

                /* set all items as NOT on server feed_add_item() will mark them
                 * as being on server */
                setonserverstatus(fp, FALSE);

                feed_set_parse_errors(fp, FALSE);

                struct pur_data *pdata = g_new0(struct pur_data, 1);

                pdata->fp_pur = fp;
                pdata->fhp_pur = fhp;
                pdata->old_title_pur = old_title;
                pdata->old_source_pur = old_source;
                pdata->old_update_interval_pur = old_update_interval;
                pdata->must_load_selected_pur = must_load_selected;
                request->pur_data = pdata;

                /* parse the new downloaded feed into new_fp */
                fhp = feed_parse(fp, request->data, request->size,
                                 request->flags & FEED_REQ_AUTO_DISCOVER,
                                 request);
                if (!fhp && request->jump) {
                    return;
                } else {
                    g_free(pdata);
                }

              continue_update_result:

                if ((fhp == NULL) && (fp->available == FALSE)
                    && (app_data->app_ui_data->add_feed_dialog != NULL)) {
                    g_free(old_title);
                    g_free(old_source);
                    old_source = NULL;
                    old_title = NULL;
                    fp = NULL;
                    /* perhaps we could just call newdialog_destroy here instead of: */
                    gtk_widget_destroy(app_data->app_ui_data->
                                       add_feed_dialog);
                    return;
                }
                if (fhp && strcmp(fhp->typeStr, "opml") == 0) {
                    /* shouldnt we destroy the dialog here too ? */
                    g_free(old_title);
                    g_free(old_source);
                    old_source = NULL;
                    old_title = NULL;
                    fp = NULL;
                    return;
                }

                /* If opml gotten from add feed dialog. Just returns, because feed
                 * directory dialog will be popped up by now */

                if (fhp == NULL || feed_get_parse_errors(fp)) {
                    g_debug("Could not detect the type of this feed");
                    feed_set_available(fp, FALSE);
                    app_data->app_ui_data->errors_in_refreshing = TRUE;
                } else if (feed_get_new_subscription(fp) == TRUE
                           && feed_get_feed_directory(fp) == TRUE
                           && app_data->app_ui_data->add_feed_dialog !=
                           NULL) {
                    g_debug("Unnecessary feed directory found. Not adding to the feed tree");
                    request->folder =
                        app_data->app_ui_data->ui_data->parent_folder;
                    newdialog_destroy();
                    feed_set_new_subscription(fp, FALSE);
                    feed_set_available(fp, FALSE);
                    ULOG_DEBUG("Processing as a directory...");
                    app_data->app_ui_data->feed_directory_loading = TRUE;
                    ui_feed_directory_process(request);
                } else {
                    fp->fhp = fhp;

                    g_debug("Feed type found %s", fhp->typeStr);

                    /* Feed has been added from the add feed dialog and has been
                     * found so it is added to the feed tree */
                    if (feed_get_new_subscription(fp)
                        && app_data->app_ui_data->add_feed_dialog != NULL) {
                        folderPtr selected_parent =
                            app_data->app_ui_data->ui_data->parent_folder;
                        folderPtr parent = NULL;
                        parent =
                            (selected_parent !=
                             NULL) ? selected_parent :
                            ui_feedlist_get_root_folder();
                        ui_feedlist_add(parent, (nodePtr) fp, -1);
                        /* ui_feedlist_add calls ui_feedlist_update() as its last entry 
                         * so its slightly silly to call it again here. Think about it when
                         * refactoring! */
                        ui_feedlist_update();
                        if (conf_feedlist_save() == FALSE) {
                            ui_feedlist_remove((nodePtr) fp);
                            ui_feedlist_update();
                            feed_set_new_subscription(fp, FALSE);
                            feed_set_available(fp, FALSE);
                            app_data->app_ui_data->search_mode =
                                SFM_INTERRUPTED;
                            download_cancel_all(TRUE);
                            /* And we need to return sentitivity to add new dialog, so here: */
                            newdialog_return_to_normal();

                        } else {
                            ui_feedlist_select((nodePtr) fp);
                            opmlsaved = TRUE;
                            newdialog_destroy();
                            feed_set_new_subscription(fp, FALSE);

                            /* if we cant save opml file, it propably means that disk is full and its really
                             * unnecessery to download images too so this call is added into this branch */

                        }
                    }
                    if (feed_get_download_images(fp)) {
                        feed_download_images(fp);
                    }
                    /* restore user defined properties if necessary */
                    if (!(request->flags & FEED_REQ_RESET_TITLE)) {
                        feed_set_title(fp, old_title);
                    }

                    if (!(request->flags & FEED_REQ_AUTO_DISCOVER)) {
                        feed_set_source(fp, old_source);
                    }

                    /* uispec0.3: automatic removal changed */
                    /* we don't delete items when the feed is opened but we adhere to 
                     * the item limit */
                    if (!app_data->app_ui_data->auto_refresh_is_ongoing
                        || !must_load_selected) {
                        ui_feed_delete_offline_items(fp);
                    }

                    ui_feed_delete_all_but_latest(fp);

                    if (!(request->flags & FEED_REQ_RESET_UPDATE_INT)) {
                        feed_set_update_interval(fp, old_update_interval);
                    }

                    if (must_load_selected) {
                        feed_set_feed_read(fp, -1);
                        ui_itemlist_load((nodePtr) fp);
                        feed_mark_all_items_read(fp);
                    } else {

                    }

                    if (request->flags & FEED_REQ_SHOW_PROPDIALOG) {
                        g_assert(app_data != NULL);
                        g_assert(app_data->app_ui_data != NULL);
                        ui_feed_propdialog_new(GTK_WINDOW
                                               (app_data->app_ui_data->
                                                main_view), fp);
                    }
                }
                g_free(old_title);
                g_free(old_source);
                old_source = NULL;
                old_title = NULL;

            } else {            /* request->data == NULL */
                g_debug("\"%s\" is not available", feed_get_title(fp));
                feed_set_available(fp, FALSE);
                app_data->app_ui_data->errors_in_refreshing = TRUE;
            }
    }

    ui_feed_set_error_description(fp, request->httpstatus,
                                  request->returncode);
    fp->request = NULL;
    ui_feedlist_update();

    if (feed_get_available(fp)) {
        if (request->flags & FEED_REQ_DOWNLOAD_FAVICON) {
            favicon_download(fp, app_data->app_ui_data);
        }
    }

    if (feed_unload(fp) == FALSE) {
        app_data->app_ui_data->search_mode = SFM_INTERRUPTED;
        download_cancel_all(TRUE);
        return;
    }

    /* If the add feed dialog is displayed and the feed is invalid or it is a 
     * feed directory the feed is freed */
    if ((app_data->app_ui_data->add_feed_dialog != NULL
         && feed_get_available(fp) == FALSE)
        || (feed_get_feed_directory(fp) == TRUE
            && feed_get_available(fp) == FALSE)
        || (adding_feeds_from_opml
            && feed_get_available(fp) == FALSE)) {
        g_debug("Deleting an invalid feed");
        ui_feedlist_delete_((nodePtr) fp);
    }
}

/** updating of a single feed list entry */
void ui_feed_update(feedPtr fp)
{
    GtkTreeModel *model = NULL;
    GtkTreeIter *iter = NULL;
    gchar *label = NULL, *tmp = NULL;
    int unread = 0, marked = 0;

    if (fp->ui_data == NULL)
        return;

    iter = &((ui_data *) fp->ui_data)->row;
    model = GTK_TREE_MODEL(feedstore);

    g_assert(FST_FOLDER != fp->type);

    if (FST_FEED == fp->type)
        rss_max_feed = rss_max_feed + 1;
    unread = feed_get_unread_counter(fp);
    marked = feed_get_mark_counter(fp);

    // let's be simple. why need to unhtmlize? 
    // need to escape though, as textrenderer is using markup


    label = g_strdup(feed_get_title(fp));
    label = utf8_fix(label);
    tmp = g_markup_escape_text(label, -1);
    tmp = g_strstrip(tmp);
    g_free(label);

    if (unread > 0 && marked > 0)
        label =
            g_strdup_printf
            ("<span weight=\"bold\" foreground=\"%s\">%s (%d)</span>",
             "red", tmp, unread);
    else if (marked > 0)
        label = g_strdup_printf("<span foreground=\"%s\">%s</span>",
                                "red",
                                tmp);
    else if (unread > 0)
        label =
            g_strdup_printf
            ("<span foreground=\"%s\" weight=\"bold\">%s (%d)</span>",
             "black", tmp, unread);
    else
        label = g_strdup_printf("<span foreground=\"%s\">%s</span>", "black"
                                , tmp);
    g_free(tmp);

    gtk_tree_store_set(feedstore, iter, FS_LABEL, label,
                       FS_UNREAD, unread, FS_ICON, ui_feed_get_icon(fp), -1);
    g_free(label);
}

/* Temporarly mark the New items as hidden. So we will know for example after 
 * refreshing, what items are realy new, and what was new just before
 * refreshing */
void ui_feed_set_new_items_hidden(feedPtr fp)
{
    GSList *item = NULL;
    itemPtr curitem;
    g_assert(fp != NULL);
    item = fp->items;

    while (NULL != item) {
        curitem = item->data;
        if (fp->newCount) {
            if (TRUE == curitem->newStatus) {
                curitem->hidden = TRUE;
            }
        }
        item = g_slist_next(item);
    }
}

GtkWidget *ui_feed_newdialog_new(GtkWindow * parent, const gchar * source)
{
    GtkWidget *newdialog;
    struct fp_prop_ui_data *ui_data = NULL;
    AppUIData *app_ui_data;
    folderPtr ptr = NULL;

    g_assert(app_data != NULL);

    app_ui_data = app_data->app_ui_data;

    app_ui_data->ui_data = ui_data = g_new0(struct fp_prop_ui_data, 1);

    g_assert(app_ui_data != NULL);

    if (ui_data == NULL) {
        ULOG_DEBUG("Not enough memory for add feed dialog");
        return NULL;
    }

    /* Create the dialog */
    app_ui_data->add_feed_dialog = app_ui_data->dialog = ui_data->dialog =
        newdialog = create_newdialog();

    ui_data->label_widget = lookup_widget(newdialog, "folderName");

    ptr = ui_feedlist_get_target_folder();

    if (ptr != NULL && ptr != ui_feedlist_get_root_folder()) {
        gtk_label_set_text(GTK_LABEL(ui_data->label_widget), ptr->title);

        ui_data->parent_folder = ptr;
    } else {
        ui_data->parent_folder = ui_feedlist_get_root_folder();
    }

    GtkWidget *changefolder = lookup_widget(newdialog, "changefolder");
    g_signal_connect(changefolder, "clicked",
                     G_CALLBACK(display_change_folder_dialog), ui_data);

    GtkWidget *browsebtn = lookup_widget(newdialog, "browsebtn");
    g_signal_connect(browsebtn, "clicked",
                     G_CALLBACK(display_file_chooser_dialog), ui_data);


    gtk_window_set_transient_for(GTK_WINDOW(newdialog), GTK_WINDOW(parent));
    gtk_window_set_modal(GTK_WINDOW(newdialog), TRUE);

    /* Setup source entry */
    ui_data->sourceEntry = lookup_widget(newdialog, "sourceEntry");
    g_signal_connect(G_OBJECT(ui_data->sourceEntry), "changed",
                     G_CALLBACK(enable_sensitive_newfeedbtn), ui_data);
    if (source != NULL) {
        // set it here
        gtk_entry_set_text(GTK_ENTRY(ui_data->sourceEntry),
                           (const gchar *) source);
    }

    gtk_widget_grab_focus(GTK_WIDGET(ui_data->sourceEntry));
    gtk_entry_set_activates_default(GTK_ENTRY(ui_data->sourceEntry), TRUE);

    ui_data->newfeedbtn = lookup_widget(newdialog, "newfeedbtn");
    gtk_widget_set_sensitive(ui_data->newfeedbtn, source != NULL);
    gtk_widget_grab_default(ui_data->newfeedbtn);
    hildon_helper_set_insensitive_message(GTK_WIDGET(ui_data->newfeedbtn),
                                          _
                                          ("rss_ni_no_web_address_for_feed"));
    hildon_helper_set_insensitive_message(GTK_WIDGET(changefolder),
                                          _
                                          ("rss_ib_unable_change_tg_subscribe"));
    g_signal_connect(G_OBJECT(newdialog), "response",
                     G_CALLBACK(on_newdialog_response), ui_data);
    g_signal_connect(G_OBJECT(newdialog), "key-press-event",
                     G_CALLBACK(key_press_for_cancelling_dialog), NULL);
    gtk_widget_show_all(newdialog);

    return newdialog;
}

static void on_feed_prop_changed(GtkWidget * widget, struct fp_prop_ui_data *ui_data)
{
    g_assert(ui_data);

    gtk_widget_set_sensitive(ui_data->propokbtn,
                             feed_prop_can_close(ui_data, FALSE, NULL, NULL));
}

static void on_insensitive_prop(GtkWidget * widget, struct fp_prop_ui_data *ui_data)
{
    g_assert(ui_data);

    feed_prop_can_close(ui_data, TRUE, NULL, NULL);
}

GtkWidget *ui_feed_propdialog_new(GtkWindow * parent, feedPtr fp)
{
    GtkWidget *propdialog;
    struct fp_prop_ui_data *ui_data = NULL;
    AppUIData *app_ui_data;

    g_assert(app_data != NULL);

    app_ui_data = app_data->app_ui_data;

    ui_data = g_new0(struct fp_prop_ui_data, 1);
    ui_data->fp = fp;

    g_assert(app_ui_data != NULL);

    /* Create the dialog */
    app_ui_data->dialog = ui_data->dialog = propdialog =
        create_propdialog(feed_get_nonremovable(fp));
    app_ui_data->propdialog = propdialog;

    GtkWidget *changefolder = lookup_widget(propdialog, "changefolder");
    g_signal_connect(changefolder, "clicked",
                     G_CALLBACK(display_change_folder_dialog), ui_data);
    hildon_helper_set_insensitive_message(GTK_WIDGET(changefolder),
                                          _("rss_ib_unable_to_move_feed"));
    gtk_widget_set_sensitive(changefolder, !feed_get_nonremovable(fp));

    GtkWidget *unsubscribe = lookup_widget(propdialog, "unsubscribeBtn");
    gtk_widget_set_sensitive(unsubscribe, !feed_get_nonremovable(fp)
                             && app_ui_data->search_mode <= SFM_INTERRUPTED);
    app_ui_data->dlg_btn_unsubscribe = unsubscribe;

    gtk_window_set_transient_for(GTK_WINDOW(propdialog), GTK_WINDOW(parent));

    gtk_window_set_modal(GTK_WINDOW(propdialog), TRUE);
    /* If we're not modal, ui_data may be overwritten */
    ui_data->feedNameEntry = lookup_widget(propdialog, "feedNameEntry");
    ui_data->feedLocationEntry =
        lookup_widget(propdialog, "feedLocationEntry");
    if (feed_get_nonremovable(fp)) {
        gtk_label_set_label(GTK_LABEL(ui_data->feedNameEntry),
                            feed_get_title(fp));
        gtk_label_set_label(GTK_LABEL(ui_data->feedLocationEntry),
                            feed_get_source(fp));
        hildon_gtk_entry_set_input_mode(GTK_ENTRY(ui_data->feedNameEntry),
                                        HILDON_GTK_INPUT_MODE_AUTOCAP);

    } else {
        gtk_entry_set_text(GTK_ENTRY(ui_data->feedNameEntry),
                           feed_get_title(fp));
        g_signal_connect(G_OBJECT(ui_data->feedNameEntry),
                         "changed", G_CALLBACK(on_feed_prop_changed),
                         ui_data);
        gtk_entry_set_text(GTK_ENTRY(ui_data->feedLocationEntry),
                           feed_get_source(fp));
        g_signal_connect(G_OBJECT(ui_data->feedLocationEntry), "changed",
                         G_CALLBACK(on_feed_prop_changed), ui_data);
    }

    ui_data->downloadImagesEntry =
        lookup_widget(propdialog, "downloadImagesEntry");
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON
                                 (ui_data->downloadImagesEntry),
                                 feed_get_download_images(fp));

    GtkWidget *subscribedData = lookup_widget(propdialog, "subscribedData");
    gchar *time_str = ui_itemlist_format_date(feed_get_added(fp));
    gtk_label_set_text(GTK_LABEL(subscribedData), time_str);
    g_free(time_str);

    GtkWidget *folderName = lookup_widget(propdialog, "folderName");
    folderPtr parentPtr = (folderPtr) ui_feedlist_get_parent((nodePtr) fp);
    if (parentPtr != NULL) {
        gtk_label_set_text(GTK_LABEL(folderName), parentPtr->title);
    }

    ui_data->propokbtn = lookup_widget(propdialog, "okBtn");
    g_signal_connect(ui_data->propokbtn, "insensitive_press",
                     G_CALLBACK(on_insensitive_prop), ui_data);

    g_signal_connect(G_OBJECT(propdialog), "response",
                     G_CALLBACK(on_propdialog_response), ui_data);
    g_signal_connect(G_OBJECT(propdialog), "key-press-event",
                     G_CALLBACK(key_press_for_cancelling_dialog), NULL);
    ui_data->label_widget = lookup_widget(propdialog, "folderName");

    unsubscribe_insens_messages(app_ui_data);
    gtk_widget_show_all(propdialog);

    return propdialog;
}

void newdialog_return_to_normal()
{
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);
    g_assert(app_data->app_ui_data->ui_data != NULL);


    if (app_data->app_ui_data->add_feed_dialog != NULL) 
        gtk_window_set_modal(GTK_WINDOW(app_data->app_ui_data->add_feed_dialog), TRUE);
    if (app_data->app_ui_data->newfeedbtn != NULL) 
        gtk_widget_set_sensitive(app_data->app_ui_data->newfeedbtn, TRUE);
    if (app_data->app_ui_data->changefolderbtn != NULL) 
        gtk_widget_set_sensitive(app_data->app_ui_data->changefolderbtn, TRUE);
    if (app_data->app_ui_data->browsebtn != NULL) 
        gtk_widget_set_sensitive(app_data->app_ui_data->browsebtn, TRUE);
    if (app_data->app_ui_data->ui_data->sourceEntry != NULL) 
        gtk_widget_set_sensitive(app_data->app_ui_data->ui_data->sourceEntry, TRUE);
}

static gboolean idle_refresh(gpointer data)
{
    gtk_widget_queue_draw(app_data->app_ui_data->html_scrollpane);
    gtk_widget_queue_draw(app_data->app_ui_data->rss_news);

    return FALSE;
}

void newdialog_destroy()
{
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    if (app_data->app_ui_data->fsdialog != NULL) {

        gtk_dialog_response(GTK_DIALOG(app_data->app_ui_data->fsdialog),
                            GTK_RESPONSE_CANCEL);
    }

    gtk_widget_destroy(GTK_WIDGET(app_data->app_ui_data->add_feed_dialog));

    app_data->app_ui_data->dialog = NULL;
    app_data->app_ui_data->add_feed_dialog = NULL;

    g_free(app_data->app_ui_data->ui_data);

    app_data->app_ui_data->ui_data = NULL;

    g_idle_add(idle_refresh, NULL);
}

void enable_sensitive_newfeedbtn(GtkWidget * widget, gpointer user_data)
{
    struct fp_prop_ui_data *data = (struct fp_prop_ui_data *) user_data;

    gtk_widget_set_sensitive(data->newfeedbtn,
                             ui_feed_newdialog_can_close(data, FALSE, NULL));
}
