/**
 *   @file support.c
 *  
 *   Support functions are collected here.
 *
 * Copyright (c) 2005-06 Nokia Corporation. All rights reserved.
 * Contact: Ouyang Qi <qi.ouyang@nokia.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <osso-log.h>

#include <gtk/gtk.h>

#include "support.h"

/* Icon sizes */
#include <hildon/hildon-defines.h>

/************************************************************************/
/*                        PUBLIC FUNCTIONS                              */
/************************************************************************/

GtkWidget *lookup_widget(GtkWidget * widget, const gchar * widget_name)
{
    GtkWidget *parent, *found_widget;

    for (;;) {
        if (GTK_IS_MENU(widget))
            parent = gtk_menu_get_attach_widget(GTK_MENU(widget));
        else
            parent = widget->parent;
        if (!parent)
            parent =
                (GtkWidget *) g_object_get_data(G_OBJECT(widget),
                                                "GladeParentKey");
        if (parent == NULL)
            break;
        widget = parent;
    }

    found_widget = (GtkWidget *) g_object_get_data(G_OBJECT(widget),
                                                   widget_name);
    if (!found_widget)
        g_warning("Widget not found: %s", widget_name);
    return found_widget;
}


/* This is an internally used function to create pixmaps. */
GdkPixbuf *create_pixbuf(const gchar * filename)
{
    GdkPixbuf *pixbuf;
    GError *error = NULL;
    GtkIconTheme *icon_theme = gtk_icon_theme_get_default();

    if (!filename || !filename[0])
        return NULL;

    pixbuf = gtk_icon_theme_load_icon(icon_theme, filename,
                                      HILDON_ICON_PIXEL_SIZE_SMALL, 0,
                                      &error);
    if (!pixbuf) {
        g_warning("Failed to load pixbuf file: %s: %s\n",
                   filename, error->message);

        g_error_free(error);
    }
    return pixbuf;
}

/* This is an internally used function to create pixmaps. */
GdkPixbuf *create_composite_pixbuf(const gchar * filename1,
                                   const gchar * filename2)
{
    GdkPixbuf *result, *plain, *emblem;
    plain = create_pixbuf(filename1);
    emblem = create_pixbuf(filename2);
    result = gdk_pixbuf_copy(plain);
    if (!result)
        return plain;

    gdk_pixbuf_composite(emblem, result, 0, 0,
                         MIN(gdk_pixbuf_get_width(emblem),
                             gdk_pixbuf_get_width(result)),
                         MIN(gdk_pixbuf_get_height(emblem),
                             gdk_pixbuf_get_height(result)), 0, 0, 1, 1,
                         GDK_INTERP_NEAREST, 255);
    /* free stuff */
    g_object_unref(plain);
    g_object_unref(emblem);
    return result;

}


/* This is used to set ATK action descriptions. */
void
glade_set_atk_action_description(AtkAction * action,
                                 const gchar * action_name,
                                 const gchar * description)
{
    gint n_actions = 0, i = 0;

    n_actions = atk_action_get_n_actions(action);
    for (i = 0; i < n_actions; i++) {
        if (!strcmp(atk_action_get_name(action, i), action_name))
            atk_action_set_description(action, i, description);
    }
}

inline GtkWidget *widget_alter_font_size(GtkWidget * w, int dsize, int setIt,
                                         gboolean bold)
{
    char abs = 0;
    PangoFontDescription *PFD_font = NULL;
    signed int font_size_px = -1;

    if (w == NULL)
        return w;

    PFD_font = w->style->font_desc;

    abs = pango_font_description_get_size_is_absolute(PFD_font);
    font_size_px = pango_font_description_get_size(PFD_font);

    if (abs)
        dsize *= PANGO_SCALE;
    else
        font_size_px /= PANGO_SCALE;

    if (setIt)
        font_size_px = dsize;
    else
        font_size_px += dsize;

    if (abs)
        pango_font_description_set_absolute_size(PFD_font, font_size_px);
    else
        pango_font_description_set_size(PFD_font, font_size_px * PANGO_SCALE);

    if (bold)
        pango_font_description_set_weight(PFD_font, PANGO_WEIGHT_BOLD);
    else
        pango_font_description_set_weight(PFD_font, PANGO_WEIGHT_NORMAL);

    pango_font_description_set_family(PFD_font, "Sans");

    gtk_widget_modify_font(w, PFD_font);

    return w;
}
