/**
    @file state_save.c

    Copyright (c) 2004-2006 Nokia Corporation. All rights reserved.
	
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
    02111-1307 USA
*/


#include "state_save.h"
#include "conf.h"

#include "gtkhtml/gtkhtml_plugin.h"
#include <stdlib.h>

/* For functions read and write. */
#include <unistd.h>

#include <osso-log.h>

typedef gchar *gchar_ptr;

extern AppData *app_data;
extern nodePtr displayed_node;

/** Searches the tree store for a node
  *
  * @param store the store to search in
  * @param start_iter start iter
  * @param result_iter result iter
  * @param path path to the current feed
  * @param cur_depth current depth
  * @param depth the search is limited by this parameter
  * @return TRUE if found, FALSE otherwise
  */
static
gboolean search_store(GtkTreeModel * store,
                      GtkTreeIter const *start_iter,
                      GtkTreeIter * result_iter,
                      gchar ** path, gint cur_depth, gint depth)
{
    GtkTreeIter cur_iter;
    GtkTreeIter child_iter;
    gboolean iter_result = FALSE;
    nodePtr node = NULL;
    GValue value;
    const gchar *id = NULL;

    *result_iter = *start_iter;
    if (depth <= 0) {
        return FALSE;
    }

    cur_iter = *start_iter;
    if (cur_depth == 1) {
        if (depth > 1) {
            iter_result = gtk_tree_model_iter_children(store,
                                                       &child_iter,
                                                       &cur_iter);
            if (iter_result) {
                g_debug("searching store with store: %p child_iter: %p",
                           store, &child_iter);
                g_debug("result_iter: %p path: %p cur_depth: %d depth: %d",
                           result_iter, path, cur_depth, depth);
                return search_store(store, &child_iter, result_iter, path,
                                    cur_depth + 1, depth);
            } else {
                return FALSE;
            }
        } else {
            return TRUE;
        }
    } else {
        memset(&value, 0, sizeof(value));
        while (TRUE) {
            gtk_tree_model_get_value(store, &cur_iter, FS_PTR, &value);
            node = (nodePtr) g_value_peek_pointer(&value);

            if (node != NULL) {
                if (node->type == FST_FEED || node->type == FST_VFOLDER) {
                    id = ((feedPtr) node)->id;
                } else if (node->type == FST_FOLDER) {
                    id = ((folderPtr) node)->id;
                } else {
                    g_warning("unknown node type (%d)", node->type);
                    id = NULL;
                }

                if (id != NULL && strcmp(id, path[cur_depth - 1]) == 0) {
                    if (cur_depth < depth) {
                        iter_result = gtk_tree_model_iter_children(store,
                                                                   &child_iter,
                                                                   &cur_iter);
                        if (iter_result) {
                            g_value_unset(&value);
                            return
                                search_store(store,
                                             &child_iter,
                                             result_iter,
                                             path, cur_depth + 1, depth);
                        } else {
                            return FALSE;
                        }
                    }
                    *result_iter = cur_iter;
                    g_value_unset(&value);
                    return TRUE;
                }
            }
            g_value_unset(&value);
            iter_result = gtk_tree_model_iter_next(store, &cur_iter);
            if (!iter_result) {
                break;
            }
        }
    }
    return FALSE;
}


/**
   Check if key file read error has occurred.
   If an error has occurred write an error message to the log,
   free the error and set *result to FALSE. Otherwise do nothing.

   @param error pointer to a GError pointer
   @param result pointer to a boolean variable,
   which is set to FALSE if an error has occurred
 */
static void check_read_error(GError ** error, gboolean * result)
{
    if (*error != NULL) {
        g_warning("Error reading field (%s)\n", (*error)->message);
        g_error_free(*error);
        *error = NULL;
        *result = FALSE;
    }
}

void init_app_state(AppState * app_state)
{
    g_assert(app_state != NULL);
    /* Initialize all fields */
    app_state->current_feed = g_new0(gchar_ptr, STATE_MAX_PATH_DEPTH);
    g_assert(app_state->current_feed != NULL);
    app_state->current_feed[0] = NULL;
    app_state->current_feed_depth = 0;
    app_state->fullscreen = FALSE;
    app_state->vertical_scroll_window_position = 0;
}


AppState *create_app_state(void)
{
    AppState *app_state = NULL;

    /* Allocate AppState object. */
    app_state = g_new0(AppState, 1);
    g_assert(app_state != NULL);
    /* Initialize the object. */
    init_app_state(app_state);
    /* Return the object. */
    return app_state;
}


void destroy_app_state(AppState * app_state)
{
    g_assert(app_state != NULL);

    /* Free the fields of app_state. */
    g_strfreev(app_state->current_feed);

    /* Free app_state. */
    g_free(app_state);
}


void construct_app_state(AppState * app_state)
{
    GtkTreePath *path = NULL;
    GtkTreeIter iter;
    GtkTreeIter cur_iter;
    GtkTreeIter new_iter;
    gchar *path_arr[STATE_MAX_PATH_DEPTH + 1];
    gint index = 0;
    gint count = 0;
    gchar *id = NULL;
    GValue value;
    nodePtr node = NULL;
    gboolean iter_result = FALSE;

    g_assert(app_state != NULL);

    if (displayed_node != NULL) {
        /* Fill the fields of app_state using the values from the
         * UI variables.
         */
        gtk_tree_view_get_cursor(GTK_TREE_VIEW(feedlist), &path, NULL);
        if (path != NULL) {
            gtk_tree_model_get_iter(GTK_TREE_MODEL(feedmodel), &iter, path);

            ui_feedlist_convert_visible_iter_to_store_iter(&cur_iter, &iter);

            index = 0;
            while (index < STATE_MAX_PATH_DEPTH) {
                memset(&value, 0, sizeof(value));

                gtk_tree_model_get_value(GTK_TREE_MODEL(feedstore),
                                         &cur_iter, FS_PTR, &value);

                node = (nodePtr) g_value_peek_pointer(&value);

                if (node == NULL) {
                    id = NULL;
                } else if (node->type == FST_FEED
                           || node->type == FST_VFOLDER) {
                    id = ((feedPtr) node)->id;
                } else if (node->type == FST_FOLDER) {
                    id = ((folderPtr) node)->id;
                } else {
                    g_warning("unknown node type (%d)", node->type);
                    id = NULL;
                }
                if (id != NULL) {
                    path_arr[index] = g_strdup(id);
                } else {
                    path_arr[index] = g_strdup("UNKNOWN");
                }
                g_value_unset(&value);
                iter_result =
                    gtk_tree_model_iter_parent(GTK_TREE_MODEL(feedstore),
                                               &new_iter, &cur_iter);
                cur_iter = new_iter;
                index++;
                if (!iter_result) {
                    /* At top level. */
                    break;
                }
            }
            gtk_tree_path_free(path);
            count = index;
        } else {
            count = 0;
        }
        for (index = 0; index < count; index++) {
            app_state->current_feed[index] = path_arr[count - index - 1];
        }
        app_state->current_feed[count] = NULL;
        app_state->current_feed_depth = count;
    } else {
        app_state->current_feed[0] = NULL;
        app_state->current_feed_depth = 0;
    }

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    app_state->fullscreen = app_data->app_ui_data->fullscreen;
}


void update_app_state(const AppState * app_state)
{
    GtkTreeIter root_iter;
    GtkTreeIter result_iter;
    GtkTreeIter visible_iter;
    GtkTreePath *path = NULL;
    gboolean search_result = FALSE;
    nodePtr node = NULL;
    folderPtr root_folder = NULL;

    g_assert(app_state != NULL);

    g_assert(feedstore != NULL && feedlist != NULL);


    if (app_state->current_feed_depth > 0) {
        /*
         * Update the UI variables.
         */

        root_folder = ui_feedlist_get_root_folder();
        if (root_folder == NULL) {
            g_warning("root folder not set");
        }
        if (root_folder->ui_data == NULL) {
            g_warning("error in root folder (ui_data NULL)");
        }
        root_iter = ((ui_data *) (root_folder->ui_data))->row;
        search_result = search_store(GTK_TREE_MODEL(feedstore),
                                     &root_iter,
                                     &result_iter,
                                     app_state->current_feed,
                                     1, app_state->current_feed_depth);

        if (search_result) {
            ui_feedlist_convert_store_iter_to_visible_iter(&visible_iter,
                                                           &result_iter);
            path = gtk_tree_model_get_path(feedmodel, &visible_iter);
            gtk_tree_view_set_cursor(GTK_TREE_VIEW(feedlist),
                                     path, NULL, FALSE);
            gtk_widget_grab_focus(GTK_WIDGET(feedlist));
            node = ui_feedlist_get_selected();
            if (node != NULL) {
                if (node->type == FST_FEED || node->type == FST_FOLDER) {
                    ui_feedlist_load_selected(node);
                    app_data->app_ui_data->vertical_scroll_window_position =
                        app_state->vertical_scroll_window_position;
                }
            } else {
                g_warning("no valid selection found");
            }
            gtk_tree_path_free(path);
        } else {
           g_warning("selected item not found");
        }
    }

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    ui_set_fullscreen_state(app_data->app_ui_data, FALSE, TRUE);
}


StateSaveResultCode reader_save_state(AppData * app_data,
                                      AppState * app_state)
{
    GKeyFile *key_file = NULL;
    gchar *contents = NULL;
    GError *err = NULL;
    StateSaveResultCode ret = ST_SUCCESS;

    g_assert(app_data != NULL && app_state != NULL);

    /* Save the subscriptions and the folder tree. */
    conf_feedlist_save();

    /* Save state to a key file. */
    key_file = save_app_state_to_key_file(app_state);

    /* Get key file contents. */
    contents = g_key_file_to_data(key_file, NULL, &err);

    g_key_file_free(key_file);
    key_file = NULL;

    /* Handle errors. */
    if (err != NULL) {
        g_warning("error saving application state (%s)", err->message);
        g_free(contents);
        g_error_free(err);
        err = NULL;
        return ST_ERR_KEY_FILE;
    }

    /* Save contents to the state file. */
    g_assert(app_data != NULL);
    ret = do_save_state(app_data->osso, contents);

    g_free(contents);
    return ret;
}


GKeyFile *save_app_state_to_key_file(AppState * app_state)
{
    GKeyFile *key_file = NULL;

    g_assert(app_state != NULL);

    /* Create a key file. */
    key_file = g_key_file_new();
    g_assert(key_file != NULL);
    /* Save the fields of app_data to the key file. */
    /* The third argument is typecasted in order to avoid
     * a compiler warning.
     */
    g_key_file_set_string_list(key_file,
                               STATE_KEY_FILE_GROUP,
                               STATE_KEY_CURRENT_FEED,
                               (const char *const *) app_state->
                               current_feed, app_state->current_feed_depth);

    g_key_file_set_boolean(key_file,
                           STATE_KEY_FILE_GROUP,
                           STATE_KEY_FULLSCREEN, app_state->fullscreen);

    g_key_file_set_integer(key_file, STATE_KEY_FILE_GROUP, STATE_KEY_SCROLL_POSITION, 1);

    return key_file;
}


StateSaveResultCode do_save_state(osso_context_t * osso,
                                  const gchar * contents)
{
    osso_state_t state;
    gint contents_length = 0;
    StateSaveResultCode result = ST_ERR_WRITE;
    osso_return_t ret = OSSO_ERROR;
    gchar buffer[STATE_FILE_LENGTH];

    g_assert(osso != NULL && contents != NULL);

    /* Check if the contents are too long. */
    contents_length = strlen(contents);
    if (contents_length >= STATE_FILE_LENGTH) {
        g_warning("Error writing state file. Too long data (%d).",
                 contents_length);
        return ST_ERR_TOO_LONG;
    }

    /* Initialize the osso_state_t struct. */
    memset(&state, 0, sizeof(state));
    state.state_size = STATE_FILE_LENGTH;
    state.state_data = buffer;

    strncpy(buffer, contents, STATE_FILE_LENGTH);

    /* Ensure that buffer is null terminated. */
    buffer[STATE_FILE_LENGTH - 1] = 0;

    /* Write the state file. */
    ret = osso_state_write(osso, &state);

    if (ret != OSSO_OK) {
        g_warning("Writing state file failed. Error %d.", (gint) ret);
        result = ST_ERR_WRITE;
    } else {
        result = ST_SUCCESS;
    }

    return result;
}


StateSaveResultCode
reader_load_state(AppData * app_data, AppState * app_state)
{
    gchar buffer[STATE_FILE_LENGTH];
    GKeyFile *key_file = NULL;
    GError *err = NULL;
    StateSaveResultCode ret_load = ST_ERR_READ;
    gboolean ret = FALSE;

    g_assert(app_data != NULL && app_state != NULL);

    /* Load the state file contents as string. */
    g_assert(app_data != NULL);
    ret_load = do_load_state(app_data->osso, buffer);

    /* Check errors. */
    if (ret_load != ST_SUCCESS) {
        return ret_load;
    }

    /* Create a key file. */
    key_file = g_key_file_new();

    /* Load key file from the state file contents. */
    ret = g_key_file_load_from_data(key_file,
                                    buffer,
                                    strlen(buffer), G_KEY_FILE_NONE, &err);

    /* Check errors. */
    if ((!ret) || err != NULL) {
        g_warning("Error loading key file from state file contents.\n"
                 "error: %s", err->message);
        g_error_free(err);
        err = NULL;
        return ST_ERR_KEY_FILE;
    }

    /* Load app_state from the key file. */
    ret = load_app_state_from_key_file(app_state, key_file);

    /* Free the key file. */
    g_key_file_free(key_file);

    return ret ? ST_SUCCESS : ST_ERR_KEY_FILE_PARSING;
}


StateSaveResultCode do_load_state(osso_context_t * osso, gchar * buffer)
{
    osso_state_t state;
    osso_return_t ret = OSSO_ERROR;

    g_assert(osso != NULL && buffer != NULL);

    memset(&state, 0, sizeof(state));
    state.state_size = STATE_FILE_LENGTH;
    state.state_data = buffer;

    ret = osso_state_read(osso, &state);

    /* Check errors. */
    if (ret == OSSO_ERROR_NO_STATE) {
        g_debug("Reading state file failed. No state.");
        return ST_ERR_NO_SAVED_STATE;
    } else if (ret == OSSO_ERROR_STATE_SIZE) {
        g_debug("Reading state file failed. Empty state.");
        return ST_ERR_NO_SAVED_STATE;
    } else if (ret != OSSO_OK) {
        g_warning("Reading state file failed. Error %d.", (gint) ret);
        return ST_ERR_READ;
    }

    if (((gchar *) state.state_data)[STATE_FILE_LENGTH - 1]
        != 0) {
        g_warning("Reading state file failed. Corrupted data.");
        return ST_ERR_READ;
    }

    return ST_SUCCESS;
}


gboolean load_app_state_from_key_file(AppState * app_state,
                                      GKeyFile * key_file)
{
    GError *err = NULL;
    gboolean result = TRUE;

    g_assert(app_state != NULL && key_file != NULL);

    /* Load the fields of app_state from the key file. */
    g_strfreev(app_state->current_feed);
    app_state->current_feed =
        g_key_file_get_string_list(key_file,
                                   STATE_KEY_FILE_GROUP,
                                   STATE_KEY_CURRENT_FEED,
                                   &(app_state->current_feed_depth), &err);
    check_read_error(&err, &result);

    app_state->fullscreen =
        g_key_file_get_boolean(key_file,
                               STATE_KEY_FILE_GROUP,
                               STATE_KEY_FULLSCREEN, &err);
    check_read_error(&err, &result);

    app_state->vertical_scroll_window_position =
        g_key_file_get_integer(key_file,
                               STATE_KEY_FILE_GROUP,
                               STATE_KEY_SCROLL_POSITION, &err);

    check_read_error(&err, &result);

    return result;
}


StateSaveResultCode reset_state_file(osso_context_t * osso)
{
    gchar contents[] = "0";
    osso_state_t state;
    osso_return_t ret = OSSO_ERROR;

    /* Initialize the osso_state_t struct. */
    memset(&state, 0, sizeof(state));
    state.state_size = 1;
    state.state_data = contents;

    /* Open the state file. */
    ret = osso_state_write(osso, &state);
    if (ret != OSSO_OK) {
        g_warning("Resetting state file failed. Error %d.", (gint) ret);
        return ST_ERR_WRITE;
    }

    return ST_SUCCESS;
}
