/**
    @file interface.c

    Dialogs and windows for the main application are created here.
*/
/*
 * Copyright (c) 2005-06 Nokia Corporation. All rights reserved.
 * Contact: Ouyang Qi <qi.ouyang@nokia.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <hildon/hildon-defines.h>
#include <hildon/hildon-caption.h>
#include <hildon/hildon-time-editor.h>
#include <hildon/hildon-find-toolbar.h>
#include <hildon/hildon-helper.h>
#include <hildon/hildon-help.h>
#include <osso-log.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <langinfo.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include <osso-rss-feed-reader/settings.h>

#include "main.h"
#include "interface.h"

#include "debug_new.h"

#define SCROLLED_WINDOW_FEEDLIST_SIZE 213
#define NEW_DIALOG_X_SIZE 400
#define REFRESH_DIALOG_X_SIZE 300
#define MANAGE_FOLDERS_DIALOG_XY_SIZE 300
#define CHANGE_FOLDER_DIALOG_XY_SIZE 300
#define PREF_DIALOG_X_SIZE 500
#define LABEL_MAX_WIDTH 23
#define PANED_GUTTER_SIZE 1
#define GUTTER_POSITION 200
#define MINIMAL_GUTTER_POSITION 50

static gboolean key_pressed=FALSE;

extern GtkWidget *newspane;
extern GtkWidget *feedlist_container;
extern AppData *app_data;

GtkTreePath *get_tree_view_last_node_path(GtkTreeView * treeview);
gboolean tree_view_select_first(GtkWidget * tr);
gboolean rename_dialog_key_press(GtkWidget * widget,
                                 GdkEventKey * event, gpointer user_data);

/************************************************************************/
/*                        PRIVATE FUNCTIONS                             */
/************************************************************************/



gboolean rename_dialog_key_press(GtkWidget * widget,
                                 GdkEventKey * event, gpointer user_data)
{
    switch (event->keyval) {
        case GDK_Escape:
            gtk_widget_hide((GtkWidget *) user_data);
            return TRUE;
        default:
            break;
    }
    return FALSE;
}


/** Creates the find toolbar
  *
  * @param app_ui_data application specific data
  */
static void ui_create_find_toolbar(AppUIData * app_ui_data)
{
    HildonFindToolbar *find_toolbar = NULL;

    g_assert(app_ui_data != NULL);
    g_assert(app_ui_data->main_view != NULL);

    find_toolbar = HILDON_FIND_TOOLBAR
        (hildon_find_toolbar_new
         (dgettext(HILDON_LIBS_L10N_PACKAGE, "ecdg_ti_find_toolbar_label")));
    g_assert(find_toolbar != NULL);
    app_ui_data->find_toolbar = find_toolbar;
    g_object_set(G_OBJECT(find_toolbar), "prefix", "", NULL);
    g_signal_connect(G_OBJECT(find_toolbar), "search",
                     G_CALLBACK(find_toolbar_search), NULL);
    g_signal_connect(G_OBJECT(find_toolbar), "close",
                     G_CALLBACK(find_toolbar_close), app_ui_data);
    hildon_window_add_toolbar(app_ui_data->main_view,
                              GTK_TOOLBAR(find_toolbar));
    gtk_widget_hide_all(GTK_WIDGET(find_toolbar));
}


/************************************************************************/
/*                        PUBLIC FUNCTIONS                              */
/************************************************************************/


gboolean
window_focus_out(GtkWidget *widget, GdkEventFocus *event, gpointer user_data)
{
    key_pressed=FALSE;
    return FALSE;
}

/** Selects the first item in the tree view
  *
  * @param tr the tree view
  */
gboolean tree_view_select_first(GtkWidget * tr)
{
    GtkTreeView *treeview = NULL;

    if (tr == NULL)
        return FALSE;

    treeview = GTK_TREE_VIEW(tr);

    gtk_tree_view_set_cursor(treeview, gtk_tree_path_new_first(),
                             NULL, FALSE);
    return TRUE;
}

static void
add_shortcut(GtkWidget * menu, GtkWidget * menu_item, const char *shortcut)
{
    g_return_if_fail(GTK_IS_MENU(menu));
    g_return_if_fail(GTK_IS_MENU_ITEM(menu_item));

    if (shortcut) {
        guint key;
        GdkModifierType mods;

        g_return_if_fail(gtk_menu_get_accel_group(GTK_MENU(menu)) != NULL);

        gtk_accelerator_parse(shortcut, &key, &mods);
        if (key != 0 && mods != 0)
            gtk_widget_add_accelerator(menu_item, "activate",
                                       gtk_menu_get_accel_group(GTK_MENU
                                                                (menu)), key,
                                       mods, GTK_ACCEL_VISIBLE);
        else
            g_warning("Shortcut `%s' not recognized", shortcut);
    }
}

GtkWidget *create_mainwindow(AppUIData * app_ui_data)
{
    GtkWidget *mainwindow;


    GtkWidget *menubar;

    GtkWidget *menuitem_feed;
    GtkWidget *menuitem_feed_menu;
    GtkWidget *menuitem_open;
    GtkWidget *menuitem_add;
    GtkWidget *menuitem_unsubscribe;
    GtkWidget *menuitem_properties;


    GtkWidget *menuitem_edit;
    GtkWidget *menuitem_edit_menu;
    GtkWidget *menuitem_cut;
    GtkWidget *menuitem_copy;
    GtkWidget *menuitem_paste;
    GtkWidget *menuitem_selectall;

    GtkWidget *menuitem_view;
    GtkWidget *menuitem_view_menu;
    GtkWidget *menuitem_folders;
    GtkWidget *menuitem_zoom;
    GtkWidget *menu_zoom;
    GtkWidget *menuitem_50;
    GtkWidget *menuitem_80;
    GtkWidget *menuitem_100;
    GtkWidget *menuitem_120;
    GtkWidget *menuitem_150;
    GtkWidget *menuitem_200;
    GtkWidget *menuitem_sort;
    GtkWidget *menuitem_toolbar;
    GtkWidget *menuitem_toolbar_menu;
    GtkWidget *menuitem_full;
    GtkWidget *menuitem_normal;
    GtkWidget *menuitem_fullscreen;

    GtkWidget *menuitem_tools;
    GtkWidget *menuitem_tools_menu;
    GtkWidget *menuitem_settings;
    GtkWidget *menuitem_separator;
    GtkWidget *menuitem_toolsmenu_separator1;
    GtkWidget *menuitem_find_on_page;
    GtkWidget *menuitem_manage_folders;
    GtkWidget *menuitem_refresh_feeds;
    GtkWidget *menuitem_clear_image_cache;
    GtkWidget *menuitem_toolsmenu_separator2;
    GtkWidget *menuitem_search_results;
    GtkWidget *menuitem_toolsmenu_separator3;
    GtkWidget *menuitem_help;
    GtkWidget *menuitem_feed_directory = NULL;

    GtkWidget *menuitem_close;
    GtkWidget *menuitem;
    GtkWidget *menuitem_sub;
    GSList *zoom_group = NULL;

    g_assert(app_ui_data != NULL);
    ULOG_DEBUG("*** Begin create_mainwindow()");

    mainwindow = GTK_WIDGET(app_ui_data->main_view);

    menubar = GTK_WIDGET(gtk_menu_new());
    gtk_menu_set_accel_group(GTK_MENU(menubar), gtk_accel_group_new());
    ULOG_DEBUG("*** creating menubar - feed");

    menuitem_feed = gtk_menu_item_new_with_label(_("rss_me_feed"));
    gtk_widget_show(menuitem_feed);
    gtk_container_add(GTK_CONTAINER(menubar), menuitem_feed);

    menuitem_feed_menu = gtk_menu_new();
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(menuitem_feed),
                              menuitem_feed_menu);

    menuitem_open = gtk_menu_item_new_with_label(_("rss_me_open"));
    gtk_widget_show(menuitem_open);
    gtk_container_add(GTK_CONTAINER(menuitem_feed_menu), menuitem_open);
    app_ui_data->menuitem_open = GTK_MENU_ITEM(menuitem_open);

    menuitem_add = gtk_menu_item_new_with_label(_("rss_me_add_a_feed"));
    add_shortcut(menubar, menuitem_add, "<Ctrl>n");
    gtk_widget_show(menuitem_add);
    gtk_container_add(GTK_CONTAINER(menuitem_feed_menu), menuitem_add);
    app_ui_data->menuitem_add = GTK_MENU_ITEM(menuitem_add);

    menuitem_unsubscribe =
        gtk_menu_item_new_with_label(_("rss_me_unsubscribe"));
    // TODO: add backspace shortcut
    gtk_widget_show(menuitem_unsubscribe);
    gtk_container_add(GTK_CONTAINER(menuitem_feed_menu),
                      menuitem_unsubscribe);
    app_ui_data->menuitem_unsubscribe = GTK_MENU_ITEM(menuitem_unsubscribe);

    menuitem_properties =
        gtk_menu_item_new_with_label(_("rss_me_edit_details"));
    gtk_widget_show(menuitem_properties);
    gtk_container_add(GTK_CONTAINER(menuitem_feed_menu), menuitem_properties);
    app_ui_data->menuitem_properties = GTK_MENU_ITEM(menuitem_properties);

    ULOG_DEBUG("*** creating menubar - edit");

    menuitem_edit = gtk_menu_item_new_with_label(_("rss_me_edit"));
    gtk_widget_show(menuitem_edit);
    gtk_container_add(GTK_CONTAINER(menubar), menuitem_edit);

    menuitem_edit_menu = gtk_menu_new();
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(menuitem_edit),
                              menuitem_edit_menu);

    menuitem_cut = gtk_menu_item_new_with_label(_("rss_me_cut"));
    add_shortcut(menubar, menuitem_cut, "<Ctrl>x");
    gtk_widget_show(menuitem_cut);
    gtk_container_add(GTK_CONTAINER(menuitem_edit_menu), menuitem_cut);
    app_ui_data->menuitem_cut = GTK_MENU_ITEM(menuitem_cut);
    gtk_widget_set_sensitive(GTK_WIDGET(menuitem_cut), FALSE);

    menuitem_copy = gtk_menu_item_new_with_label(_("rss_me_copy"));
    add_shortcut(menubar, menuitem_copy, "<Ctrl>c");
    gtk_widget_show(menuitem_copy);
    gtk_container_add(GTK_CONTAINER(menuitem_edit_menu), menuitem_copy);
    app_ui_data->menuitem_copy = GTK_MENU_ITEM(menuitem_copy);
    gtk_widget_set_sensitive(GTK_WIDGET(menuitem_copy), FALSE);

    menuitem_paste = gtk_menu_item_new_with_label(_("rss_me_paste"));
    add_shortcut(menubar, menuitem_paste, "<Ctrl>v");
    gtk_widget_show(menuitem_paste);
    gtk_container_add(GTK_CONTAINER(menuitem_edit_menu), menuitem_paste);
    app_ui_data->menuitem_paste = GTK_MENU_ITEM(menuitem_paste);
    gtk_widget_set_sensitive(GTK_WIDGET(menuitem_paste), FALSE);

    menuitem_selectall = gtk_menu_item_new_with_label(_("rss_me_select_all"));
    add_shortcut(menubar, menuitem_selectall, "<Ctrl>a");
    gtk_widget_show(menuitem_selectall);
    gtk_container_add(GTK_CONTAINER(menuitem_edit_menu), menuitem_selectall);
    app_ui_data->menuitem_selectall = GTK_MENU_ITEM(menuitem_selectall);
    gtk_widget_set_sensitive(GTK_WIDGET(menuitem_selectall), FALSE);

    ULOG_DEBUG("*** creating menubar - view");

    menuitem_view = gtk_menu_item_new_with_label(_("rss_me_view"));
    gtk_widget_show(menuitem_view);
    gtk_container_add(GTK_CONTAINER(menubar), menuitem_view);

    menuitem_view_menu = gtk_menu_new();
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(menuitem_view),
                              menuitem_view_menu);

    menuitem_folders =
        gtk_check_menu_item_new_with_label(_("rss_me_folders"));
    gtk_widget_show(menuitem_folders);
    gtk_container_add(GTK_CONTAINER(menuitem_view_menu), menuitem_folders);
    app_ui_data->menuitem_folders = GTK_CHECK_MENU_ITEM(menuitem_folders);
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menuitem_folders),
                                   TRUE);

    menuitem_zoom = gtk_menu_item_new_with_label(_("rss_me_zoom"));
    gtk_widget_show(menuitem_zoom);
    gtk_container_add(GTK_CONTAINER(menuitem_view_menu), menuitem_zoom);

    menu_zoom = gtk_menu_new();
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(menuitem_zoom), menu_zoom);

    /* 60 characters for a string that shows at best 100% ?????? 
     * gchar tmp[60];
     */
    gchar tmp[8];

    snprintf(tmp, sizeof(tmp), _("rss_me_zoom_50"));
    menuitem_50 = gtk_radio_menu_item_new_with_label(zoom_group, tmp);
    zoom_group =
        gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(menuitem_50));
    gtk_widget_show(menuitem_50);
    gtk_menu_append(GTK_CONTAINER(menu_zoom), menuitem_50);
    app_ui_data->menuitem_zoom_50 = GTK_MENU_ITEM(menuitem_50);

    snprintf(tmp, sizeof(tmp), _("rss_me_zoom_80"));
    menuitem_80 = gtk_radio_menu_item_new_with_label(zoom_group, tmp);
    zoom_group =
        gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(menuitem_80));
    gtk_widget_show(menuitem_80);
    gtk_menu_append(GTK_CONTAINER(menu_zoom), menuitem_80);
    app_ui_data->menuitem_zoom_80 = GTK_MENU_ITEM(menuitem_80);

    snprintf(tmp, sizeof(tmp), _("rss_me_zoom_100"));
    menuitem_100 = gtk_radio_menu_item_new_with_label(zoom_group, tmp);
    zoom_group =
        gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(menuitem_100));
    gtk_widget_show(menuitem_100);
    gtk_menu_append(GTK_CONTAINER(menu_zoom), menuitem_100);
    app_ui_data->menuitem_zoom_100 = GTK_MENU_ITEM(menuitem_100);

    snprintf(tmp, sizeof(tmp), _("rss_me_zoom_120"));
    menuitem_120 = gtk_radio_menu_item_new_with_label(zoom_group, tmp);
    zoom_group =
        gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(menuitem_120));
    gtk_widget_show(menuitem_120);
    gtk_menu_append(GTK_CONTAINER(menu_zoom), menuitem_120);
    app_ui_data->menuitem_zoom_120 = GTK_MENU_ITEM(menuitem_120);

    snprintf(tmp, sizeof(tmp), _("rss_me_zoom_150"));
    menuitem_150 = gtk_radio_menu_item_new_with_label(zoom_group, tmp);
    zoom_group =
        gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(menuitem_150));
    gtk_widget_show(menuitem_150);
    gtk_menu_append(GTK_CONTAINER(menu_zoom), menuitem_150);
    app_ui_data->menuitem_zoom_150 = GTK_MENU_ITEM(menuitem_150);

    snprintf(tmp, sizeof(tmp), _("rss_me_zoom_200"));
    menuitem_200 = gtk_radio_menu_item_new_with_label(zoom_group, tmp);
    gtk_widget_show(menuitem_200);
    gtk_menu_append(GTK_CONTAINER(menu_zoom), menuitem_200);
    app_ui_data->menuitem_zoom_200 = GTK_MENU_ITEM(menuitem_200);

    menuitem_separator = gtk_separator_menu_item_new();
    gtk_widget_show(menuitem_separator);
    gtk_container_add(GTK_CONTAINER(menuitem_view_menu), menuitem_separator);
    gtk_widget_set_sensitive(menuitem_separator, FALSE);

    menuitem_sort = gtk_menu_item_new_with_label(_("rss_me_sort"));
    gtk_widget_show(menuitem_sort);
    gtk_container_add(GTK_CONTAINER(menuitem_view_menu), menuitem_sort);

    menuitem_separator = gtk_separator_menu_item_new();
    gtk_widget_show(menuitem_separator);
    gtk_container_add(GTK_CONTAINER(menuitem_view_menu), menuitem_separator);
    gtk_widget_set_sensitive(menuitem_separator, FALSE);

    menuitem_fullscreen =
        gtk_check_menu_item_new_with_label(_("rss_me_full_screen"));
    gtk_widget_show(menuitem_fullscreen);
    gtk_container_add(GTK_CONTAINER(menuitem_view_menu), menuitem_fullscreen);
    app_ui_data->menuitem_fullscreen =
        GTK_CHECK_MENU_ITEM(menuitem_fullscreen);

    menuitem_toolbar = gtk_menu_item_new_with_label(_("rss_me_toolbar"));
    gtk_widget_show(menuitem_toolbar);
    gtk_container_add(GTK_CONTAINER(menuitem_view_menu), menuitem_toolbar);

    menuitem_toolbar_menu = gtk_menu_new();
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(menuitem_toolbar),
                              menuitem_toolbar_menu);

    menuitem_normal =
        gtk_check_menu_item_new_with_label(_("rss_me_normalscreen"));
    gtk_widget_show(menuitem_normal);
    gtk_container_add(GTK_CONTAINER(menuitem_toolbar_menu), menuitem_normal);
    app_ui_data->menuitem_normal = GTK_CHECK_MENU_ITEM(menuitem_normal);

    menuitem_full =
        gtk_check_menu_item_new_with_label(_("rss_me_fullscreen"));
    gtk_widget_show(menuitem_full);
    gtk_container_add(GTK_CONTAINER(menuitem_toolbar_menu), menuitem_full);
    app_ui_data->menuitem_full = GTK_CHECK_MENU_ITEM(menuitem_full);

    ULOG_DEBUG("*** creating menubar - tools");

    menuitem_tools = gtk_menu_item_new_with_label(_("rss_me_tools"));
    gtk_widget_show(menuitem_tools);
    gtk_container_add(GTK_CONTAINER(menubar), menuitem_tools);

    menuitem_tools_menu = gtk_menu_new();
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(menuitem_tools),
                              menuitem_tools_menu);

    menuitem_settings = gtk_menu_item_new_with_label(_("rss_me_settings"));
    gtk_widget_show(menuitem_settings);
    gtk_container_add(GTK_CONTAINER(menuitem_tools_menu), menuitem_settings);

    menuitem_toolsmenu_separator1 = gtk_separator_menu_item_new();
    gtk_widget_show(menuitem_toolsmenu_separator1);
    gtk_container_add(GTK_CONTAINER(menuitem_tools_menu),
                      menuitem_toolsmenu_separator1);
    gtk_widget_set_sensitive(menuitem_toolsmenu_separator1, FALSE);

    menuitem_find_on_page =
        gtk_check_menu_item_new_with_label(_("rss_me_find"));
    add_shortcut(menubar, menuitem_find_on_page, "<Ctrl>f");
    gtk_widget_show(menuitem_find_on_page);
    gtk_container_add(GTK_CONTAINER(menuitem_tools_menu),
                      menuitem_find_on_page);
    app_ui_data->menuitem_find_on_page = GTK_MENU_ITEM(menuitem_find_on_page);

    menuitem_manage_folders =
        gtk_menu_item_new_with_label(_("rss_me_manage_folders"));
    gtk_widget_show(menuitem_manage_folders);
    gtk_container_add(GTK_CONTAINER(menuitem_tools_menu),
                      menuitem_manage_folders);
    app_ui_data->menuitem_manage_folders =
        GTK_MENU_ITEM(menuitem_manage_folders);

    menuitem_refresh_feeds =
        gtk_menu_item_new_with_label(_("rss_me_refresh_feeds"));
    add_shortcut(menubar, menuitem_refresh_feeds, "<Ctrl>r");
    gtk_widget_show(menuitem_refresh_feeds);
    gtk_container_add(GTK_CONTAINER(menuitem_tools_menu),
                      menuitem_refresh_feeds);
    app_ui_data->menuitem_refresh_feeds =
        GTK_MENU_ITEM(menuitem_refresh_feeds);

    menuitem_clear_image_cache =
        gtk_menu_item_new_with_label(_("rss_me_clear_image_cache"));
    gtk_widget_show(menuitem_clear_image_cache);
    gtk_container_add(GTK_CONTAINER(menuitem_tools_menu),
                      menuitem_clear_image_cache);

    menuitem_feed_directory =
        gtk_menu_item_new_with_label(_("rss_me_feed_directory"));
    gtk_widget_show(menuitem_feed_directory);
    gtk_container_add(GTK_CONTAINER(menuitem_tools_menu),
                      menuitem_feed_directory);
    app_ui_data->menuitem_feed_directory =
        GTK_MENU_ITEM(menuitem_feed_directory);

    menuitem_toolsmenu_separator2 = gtk_separator_menu_item_new();
    gtk_widget_show(menuitem_toolsmenu_separator2);
    gtk_container_add(GTK_CONTAINER(menuitem_tools_menu),
                      menuitem_toolsmenu_separator2);
    gtk_widget_set_sensitive(menuitem_toolsmenu_separator2, FALSE);

    menuitem_search_results =
        gtk_menu_item_new_with_label(_("rss_me_search_results"));
    gtk_widget_show(menuitem_search_results);
    gtk_container_add(GTK_CONTAINER(menuitem_tools_menu),
                      menuitem_search_results);
    gtk_widget_set_sensitive(GTK_WIDGET(menuitem_search_results), FALSE);
    app_ui_data->menuitem_search_results =
        GTK_MENU_ITEM(menuitem_search_results);

    menuitem_toolsmenu_separator3 = gtk_separator_menu_item_new();
    gtk_widget_show(menuitem_toolsmenu_separator3);
    gtk_container_add(GTK_CONTAINER(menuitem_tools_menu),
                      menuitem_toolsmenu_separator3);
    gtk_widget_set_sensitive(menuitem_toolsmenu_separator3, FALSE);

    menuitem_help = gtk_menu_item_new_with_label(_("rss_me_help"));
    gtk_widget_show(menuitem_help);
    gtk_container_add(GTK_CONTAINER(menuitem_tools_menu), menuitem_help);

    ULOG_DEBUG("*** creating menubar - close");
    menuitem_close = gtk_menu_item_new_with_label(_("rss_me_close"));
    add_shortcut(menubar, menuitem_close, "<Ctrl>q");
    gtk_widget_show(menuitem_close);
    gtk_container_add(GTK_CONTAINER(menubar), menuitem_close);


    ULOG_DEBUG("*** creating context menu");

    app_ui_data->context_menu = GTK_MENU(gtk_menu_new());

    menuitem = gtk_separator_menu_item_new();
    gtk_widget_hide(menuitem);
    gtk_container_add(GTK_CONTAINER(app_ui_data->context_menu), menuitem);
    app_ui_data->context_sp = GTK_MENU_ITEM(menuitem);

    menuitem = gtk_menu_item_new_with_label(_("rss_me_open"));
    gtk_widget_show(menuitem);
    gtk_container_add(GTK_CONTAINER(app_ui_data->context_menu), menuitem);
    app_ui_data->context_menuitem_open = GTK_MENU_ITEM(menuitem);

    menuitem = gtk_menu_item_new_with_label(_("rss_me_refresh"));
    gtk_widget_show(menuitem);
    gtk_container_add(GTK_CONTAINER(app_ui_data->context_menu), menuitem);
    app_ui_data->context_menuitem_refresh = GTK_MENU_ITEM(menuitem);

    menuitem = gtk_menu_item_new_with_label(_("rss_me_new_folder"));
    gtk_widget_show(menuitem);
    gtk_container_add(GTK_CONTAINER(app_ui_data->context_menu), menuitem);
    app_ui_data->context_menuitem_new_folder = GTK_MENU_ITEM(menuitem);

    menuitem = gtk_menu_item_new_with_label(_("rss_me_sort"));
    gtk_widget_show(menuitem);
    gtk_container_add(GTK_CONTAINER(app_ui_data->context_menu), menuitem);
    app_ui_data->context_menuitem_sort = GTK_MENU_ITEM(menuitem);

    menuitem = gtk_menu_item_new_with_label(_("rss_me_edit_details"));
    gtk_widget_show(menuitem);
    gtk_container_add(GTK_CONTAINER(app_ui_data->context_menu), menuitem);
    app_ui_data->context_menuitem_edit_details = GTK_MENU_ITEM(menuitem);

    menuitem = gtk_menu_item_new_with_label(_("rss_me_rename"));
    gtk_widget_show(menuitem);
    gtk_container_add(GTK_CONTAINER(app_ui_data->context_menu), menuitem);
    app_ui_data->context_menuitem_rename = GTK_MENU_ITEM(menuitem);

    menuitem = gtk_menu_item_new_with_label(_("rss_me_unsubscribe"));
    gtk_widget_show(menuitem);
    gtk_container_add(GTK_CONTAINER(app_ui_data->context_menu), menuitem);
    app_ui_data->context_menuitem_unsubscribe = GTK_MENU_ITEM(menuitem);

    menuitem = gtk_menu_item_new_with_label(_("rss_me_delete"));
    gtk_widget_show(menuitem);
    gtk_container_add(GTK_CONTAINER(app_ui_data->context_menu), menuitem);
    app_ui_data->context_menuitem_delete = GTK_MENU_ITEM(menuitem);

    menuitem = gtk_menu_item_new_with_label(_("rss_me_copy_link"));
    gtk_widget_show(menuitem);
    gtk_container_add(GTK_CONTAINER(app_ui_data->context_menu), menuitem);
    app_ui_data->context_menuitem_copy_link = GTK_MENU_ITEM(menuitem);

    menuitem = gtk_menu_item_new_with_label(_("rss_me_send"));
    gtk_widget_show(menuitem);
    gtk_container_add(GTK_CONTAINER(app_ui_data->context_menu), menuitem);
    app_ui_data->context_menuitem_send = GTK_MENU_ITEM(menuitem);

    menuitem_sub = gtk_menu_new();
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(menuitem), menuitem_sub);

    menuitem = gtk_menu_item_new_with_label(_("rss_me_send_post"));
    gtk_widget_show(menuitem);
    gtk_container_add(GTK_CONTAINER(menuitem_sub), menuitem);
    app_ui_data->context_menuitem_send_post = GTK_MENU_ITEM(menuitem);

    menuitem = gtk_menu_item_new_with_label(_("rss_me_send_via_bluetooth"));
    gtk_widget_show(menuitem);
    gtk_container_add(GTK_CONTAINER(menuitem_sub), menuitem);
    app_ui_data->context_menuitem_send_bt = GTK_MENU_ITEM(menuitem);

    menuitem = gtk_menu_item_new_with_label(_("rss_me_load_image"));
    gtk_widget_show(menuitem);
    gtk_container_add(GTK_CONTAINER(app_ui_data->context_menu), menuitem);
    app_ui_data->context_menuitem_load_image = GTK_MENU_ITEM(menuitem);

    menuitem = gtk_menu_item_new_with_label(_("rss_me_save_image"));
    gtk_widget_show(menuitem);
    gtk_container_add(GTK_CONTAINER(app_ui_data->context_menu), menuitem);
    app_ui_data->context_menuitem_save_image = GTK_MENU_ITEM(menuitem);


    // add accelerator group to the main window
    gtk_window_add_accel_group(GTK_WINDOW(mainwindow),
                               gtk_menu_get_accel_group(GTK_MENU(menubar)));


    ULOG_DEBUG("*** creating newsbox");
    newsbox = gtk_hpaned_new();
    gtk_widget_show(newsbox);
    app_ui_data->rss_news = newsbox;
    gtk_container_set_border_width(GTK_CONTAINER(newsbox), PANED_GUTTER_SIZE);

    gtk_container_add(GTK_CONTAINER(mainwindow), newsbox);


    ULOG_DEBUG("*** creating scrolledwindow_feedlist");

    GtkWidget *scrolledwindow_feedlist =
        app_data->app_ui_data->scrolledwindow =
        gtk_scrolled_window_new(NULL, NULL);
    gtk_widget_show(scrolledwindow_feedlist);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW
                                   (scrolledwindow_feedlist),
                                   GTK_POLICY_AUTOMATIC,
                                   GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW
                                        (scrolledwindow_feedlist),
                                        GTK_SHADOW_NONE);
    gtk_widget_set_size_request(scrolledwindow_feedlist,
                                MINIMAL_GUTTER_POSITION, -1);


    ULOG_DEBUG("*** creating feedlist_container");
    feedlist_container = gtk_hbox_new(FALSE, 0);
    gtk_widget_show(feedlist_container);
    gtk_box_pack_start(GTK_BOX(feedlist_container),
                       scrolledwindow_feedlist, TRUE, TRUE, 0);
    gtk_paned_pack1(GTK_PANED(newsbox), feedlist_container, TRUE,   /*resize-able */
                    FALSE);     /*shrink-able */

    gint pane = getNumericConfValue(LAST_VPANE_POS);

    ULOG_DEBUG("#### Setting pane position: %d ####", pane);
    if (pane)
        gtk_paned_set_position(GTK_PANED(newsbox), pane);
    else
        gtk_paned_set_position(GTK_PANED(newsbox), GUTTER_POSITION);
    ULOG_DEBUG("Pane position: %d\n",
               gtk_paned_get_position(GTK_PANED(newsbox)));
    ULOG_DEBUG("*** creating feedlist");

    feedlist = gtk_tree_view_new();
    gtk_tree_view_set_enable_search(GTK_TREE_VIEW(feedlist), FALSE);
    g_signal_connect(G_OBJECT(feedlist), "drag_data_received",
                     G_CALLBACK(ui_dnd_receive_drag_data), feedlist);
    g_signal_connect(G_OBJECT(feedlist), "drag_drop",
                     G_CALLBACK(ui_dnd_drag_drop), feedlist);
    gtk_widget_show(feedlist);
    gtk_container_add(GTK_CONTAINER(scrolledwindow_feedlist), feedlist);
    gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(feedlist), FALSE);
    gtk_tree_view_set_reorderable(GTK_TREE_VIEW(feedlist), TRUE);

    g_assert(app_data->app_ui_data != NULL);

    ui_create_find_toolbar(app_data->app_ui_data);
    int zoomlevel = (int) (app_ui_data->zoom_level * 100.0);

    switch (zoomlevel) {
        case 50:
        {
            gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menuitem_50),
                                           TRUE);
            break;
        }
        case 80:
        {
            gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menuitem_80),
                                           TRUE);
            break;
        }
        case 100:
        {
            gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menuitem_100),
                                           TRUE);
            break;
        }
        case 120:
        {
            gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menuitem_120),
                                           TRUE);
            break;
        }
        case 150:
        {
            gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menuitem_150),
                                           TRUE);
            break;
        }
        case 200:
        {
            gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menuitem_200),
                                           TRUE);
            break;
        }
    }


    ULOG_DEBUG("*** g_signal_connect_ block begins");

    /* feed menu */

    g_signal_connect((gpointer) menuitem_open, "activate",
                     G_CALLBACK(on_menu_open), NULL);

    hildon_helper_set_insensitive_message(GTK_WIDGET(menuitem_open),
                                          _("rss_ib_no_feed_selected"));

    g_signal_connect((gpointer) menuitem_add, "activate",
                     G_CALLBACK(on_newbtn_clicked_cb), NULL);
    hildon_helper_set_insensitive_message(GTK_WIDGET(menuitem_add),
                                          _("rss_ib_unable_add_feed"));

    g_signal_connect((gpointer) menuitem_unsubscribe, "activate",
                     G_CALLBACK(on_unsubscribe_clicked), NULL);




    g_signal_connect((gpointer) menuitem_properties, "activate",
                     G_CALLBACK(on_menu_properties), NULL);


    /* edit menu */

    g_signal_connect((gpointer) menuitem_edit, "activate",
                     G_CALLBACK(on_menuitem_edit), app_ui_data);

    g_signal_connect((gpointer) menuitem_cut, "activate",
                     G_CALLBACK(on_cut), app_ui_data);

    hildon_helper_set_insensitive_message(GTK_WIDGET(menuitem_cut),
                                          dgettext
                                          (HILDON_COMMON_STRINGS_L10N_PACKAGE,
                                           "ckct_ib_unable_to_cut"));
    //Interface changed for these
    g_signal_connect((gpointer) menuitem_copy, "activate",
                     G_CALLBACK(on_copy), app_ui_data);

    hildon_helper_set_insensitive_message(GTK_WIDGET(menuitem_copy),
                                          dgettext
                                          (HILDON_COMMON_STRINGS_L10N_PACKAGE,
                                           "ckct_ib_unable_to_copy"));

    g_signal_connect((gpointer) menuitem_paste, "activate",
                     G_CALLBACK(on_paste), app_ui_data);


    g_signal_connect((gpointer) menuitem_selectall, "activate",
                     G_CALLBACK(on_selectall), app_ui_data);

    hildon_helper_set_insensitive_message(GTK_WIDGET(menuitem_selectall),
                                          dgettext
                                          (HILDON_COMMON_STRINGS_L10N_PACKAGE,
                                           "ckct_ib_nothing_to_select"));

    g_signal_connect((gpointer) menuitem_find_on_page, "activate",
                     G_CALLBACK(toggle_find_on_page), NULL);

    /* view menu */

    g_signal_connect(G_OBJECT(menuitem_200), "toggled",
                     G_CALLBACK(change_zoom_cb)
                     , (gpointer) ZOOM_200);

    g_signal_connect(G_OBJECT(menuitem_150), "toggled",
                     G_CALLBACK(change_zoom_cb)
                     , (gpointer) ZOOM_150);

    g_signal_connect(G_OBJECT(menuitem_120), "toggled",
                     G_CALLBACK(change_zoom_cb)
                     , (gpointer) ZOOM_120);

    g_signal_connect(G_OBJECT(menuitem_100), "toggled",
                     G_CALLBACK(change_zoom_cb)
                     , (gpointer) ZOOM_100);

    g_signal_connect(G_OBJECT(menuitem_80), "toggled",
                     G_CALLBACK(change_zoom_cb)
                     , (gpointer) ZOOM_80);

    g_signal_connect(G_OBJECT(menuitem_50), "toggled",
                     G_CALLBACK(change_zoom_cb)
                     , (gpointer) ZOOM_50);

    g_signal_connect((gpointer) menuitem_folders, "activate",
                     G_CALLBACK(toggle_folder_list), NULL);

    g_signal_connect((gpointer) menuitem_sort, "activate",
                     G_CALLBACK(on_menuitem_sort), app_ui_data);

    g_signal_connect((gpointer) menuitem_normal, "activate",
                     G_CALLBACK(toggle_toolbar), NULL);

    g_signal_connect((gpointer) menuitem_full, "activate",
                     G_CALLBACK(toggle_toolbar), NULL);

    g_signal_connect((gpointer) menuitem_fullscreen, "activate",
                     G_CALLBACK(on_toggle_fullscreen_mode), NULL);

    /* tools menu */
    /* Added this as feed directory might need to be insensitivized if there 
     * hasn't been any opml action yet
     */
    g_signal_connect((gpointer) menuitem_tools, "activate",
                     G_CALLBACK(on_menuitem_tools), app_ui_data);

    g_signal_connect((gpointer) menuitem_refresh_feeds, "activate",
                     G_CALLBACK(on_refresh_menuitem_selected), app_ui_data);

    g_signal_connect((gpointer) menuitem_clear_image_cache, "activate",
                     G_CALLBACK(on_clear_image_cache), app_ui_data);


    g_signal_connect((gpointer) menuitem_manage_folders, "activate",
                     G_CALLBACK(on_managefoldersbtn_clicked), NULL);
    hildon_helper_set_insensitive_message(GTK_WIDGET(menuitem_manage_folders),
                                          _
                                          ("rss_ib_unable_managefolders_searching"));


    g_signal_connect((gpointer) menuitem_settings, "activate",
                     G_CALLBACK(on_prefbtn_clicked), NULL);

    g_signal_connect((gpointer) menuitem_search_results, "activate",
                     G_CALLBACK(on_searchresults_menuitem_selected), NULL);

    hildon_helper_set_insensitive_message(GTK_WIDGET(menuitem_search_results),
                                          _
                                          ("rss_ib_no_previous_search_results"));

    g_signal_connect((gpointer) menuitem_help, "activate",
                     G_CALLBACK(ui_help_cb), NULL);

    if (menuitem_feed_directory != NULL) {
        g_signal_connect((gpointer) menuitem_feed_directory, "activate",
                         G_CALLBACK(on_menuitem_feed_directory_clicked),
                         NULL);

    }
    /* close menu */

    g_signal_connect((gpointer) menuitem_close, "activate",
                     G_CALLBACK(on_close), NULL);

    /* context menu */

    g_signal_connect((gpointer) app_ui_data->context_menuitem_open,
                     "activate", G_CALLBACK(on_open_cmenuitem_selected),
                     app_ui_data);

    g_signal_connect((gpointer) app_ui_data->context_menuitem_refresh,
                     "activate", G_CALLBACK(on_refresh_menuitem_selected),
                     app_ui_data);

    g_signal_connect((gpointer) app_ui_data->context_menuitem_new_folder,
                     "activate", G_CALLBACK(on_new_folder_cmenuitem_selected),
                     app_ui_data);

    g_signal_connect((gpointer) app_ui_data->context_menuitem_sort,
                     "activate", G_CALLBACK(on_menuitem_sort), app_ui_data);

    g_signal_connect((gpointer) app_ui_data->context_menuitem_edit_details,
                     "activate", G_CALLBACK(on_menu_properties), NULL);

    g_signal_connect((gpointer) app_ui_data->context_menuitem_rename,
                     "activate", G_CALLBACK(on_rename_cmenuitem_selected),
                     app_ui_data);

    g_signal_connect((gpointer) app_ui_data->context_menuitem_unsubscribe,
                     "activate", G_CALLBACK(on_unsubscribe_clicked), NULL);

    g_signal_connect((gpointer) app_ui_data->context_menuitem_delete,
                     "activate", G_CALLBACK(on_delete_cmenuitem_selected),
                     app_ui_data);

    g_signal_connect((gpointer) app_ui_data->context_menuitem_copy_link,
                     "activate", G_CALLBACK(on_copy_link_menuitem_selected),
                     app_ui_data);

    g_signal_connect((gpointer) app_ui_data->context_menuitem_send_post,
                     "activate", G_CALLBACK(on_send_post_menuitem_selected),
                     NULL);

    g_signal_connect((gpointer) app_ui_data->context_menuitem_send_bt,
                     "activate", G_CALLBACK(on_send_post_menuitem_selected),
                     (gpointer) 1);

    g_signal_connect((gpointer) app_ui_data->context_menuitem_load_image,
                     "activate", G_CALLBACK(on_load_image_menuitem_selected),
                     app_ui_data);

    g_signal_connect((gpointer) app_ui_data->context_menuitem_save_image,
                     "activate", G_CALLBACK(on_save_image_menuitem_selected),
                     app_ui_data);

    /* others */

    g_signal_connect((gpointer) app_ui_data->main_view, "delete_event",
                     G_CALLBACK(on_quit), NULL);
    g_signal_connect((gpointer) mainwindow, "window_state_event",
                     G_CALLBACK(on_mainwindow_window_state_event), NULL);
    g_signal_connect((gpointer) feedlist, "drag_end",
                     G_CALLBACK(on_feedlist_drag_end), NULL);
    g_signal_connect((gpointer) feedlist, "drag_begin",
                     G_CALLBACK(on_feedlist_drag_begin), NULL);
    g_signal_connect_after((gpointer) feedlist, "tap-and-hold",
                           G_CALLBACK(on_mainfeedlist_tap_and_hold_event),
                           NULL);
    g_signal_connect_after((gpointer) feedlist, "focus-in-event",
                           G_CALLBACK(on_feedlist_focus_in),
                           NULL);
    gtk_widget_tap_and_hold_setup(feedlist, NULL, NULL,
                                  GTK_TAP_AND_HOLD_NONE);

    /* Activate the Folders check menu item. */
    ui_set_folders_menu_item_state(app_ui_data, TRUE);

    /* Activate the Toolbar check menu item. */
    ui_set_toolbar_normal_menu_item_state(app_ui_data, TRUE);

    /* Activate the Toolbar check menu item. */
    ui_set_toolbar_full_menu_item_state(app_ui_data, TRUE);

    GLADE_HOOKUP_OBJECT(mainwindow, newsbox, "leftpane");

    ULOG_DEBUG("*** gtk_widget_grab_focus2222");
    ULOG_DEBUG("@@@ Pane position: %d\n @@@",
               gtk_paned_get_position(GTK_PANED(newsbox)));

    gtk_widget_show_all(GTK_WIDGET(menubar));
    hildon_program_set_common_menu(app_ui_data->app, GTK_MENU(menubar));

    return newsbox;
}

GtkTreePath *get_tree_view_last_node_path(GtkTreeView * treeview)
{
    GtkTreeIter iter, last;
    GtkTreeModel *model = gtk_tree_view_get_model(treeview);
    GtkTreePath *path = gtk_tree_path_new_root();

    gtk_tree_model_get_iter(model, &iter, path);
    gtk_tree_path_free(path);

    while (1) {
        do {
            last = iter;
        } while (gtk_tree_model_iter_next(model, &iter));

        path = gtk_tree_model_get_path(model, &last);

        if (gtk_tree_view_row_expanded(treeview, path)) {
            gtk_tree_path_down(path);
            gtk_tree_model_get_iter(model, &iter, path);
        } else
            break;
    }

    return path;
}

gboolean tree_view_cycle_up(GtkWidget * tr)
{
    GtkTreePath *path = NULL, *root_path = NULL;

    if (tr == NULL)
        return FALSE;

    gtk_tree_view_get_cursor(GTK_TREE_VIEW(tr), &path, NULL);

    if (path == NULL)
        return FALSE;

    root_path = gtk_tree_path_new_root();
    if (gtk_tree_path_compare(path, root_path) == 0) {
        gtk_tree_path_free(root_path);
        gtk_tree_view_set_cursor(GTK_TREE_VIEW(tr),
                                 get_tree_view_last_node_path(GTK_TREE_VIEW
                                                              (tr)), NULL,
                                 FALSE);
        return TRUE;
    }
    return FALSE;
}

gboolean tree_view_cycle_down(GtkWidget * tr)
{
    GtkTreePath *path = NULL;

    if (tr == NULL)
        return FALSE;

    gtk_tree_view_get_cursor(GTK_TREE_VIEW(tr), &path, NULL);

    if (path == NULL)
        return FALSE;

    if (gtk_tree_path_compare(path,
                              get_tree_view_last_node_path(GTK_TREE_VIEW(tr)))
        == 0) {
        path = gtk_tree_path_new_root();
        gtk_tree_view_set_cursor(GTK_TREE_VIEW(tr), path, NULL, FALSE);
        gtk_tree_path_free(path);
        return TRUE;
    }
    return FALSE;
}

static gint key_press_nr = 0;

gboolean key_press(GtkWidget * widget, GdkEventKey * event,
                   AppUIData * app_ui_data)
{
    nodePtr ptr = NULL;

    ULOG_DEBUG("Got a GDK_KEY_PRESS: %d", event->keyval);

    key_pressed=TRUE;

    /* 
     * if( gtk_window_get_focus(GTK_WINDOW(app_ui_data->main_view)) !=
     * feedlist &&
     * ( event->keyval == GDK_Return || event->keyval == GDK_KP_Enter ) ) {
     * return FALSE;
     * }
     */

    switch (event->keyval) {
        case GDK_Left:
            if (gtk_window_get_focus(GTK_WINDOW(app_ui_data->main_view)) ==
                gtk_bin_get_child(GTK_BIN
                                  (ui_mainwindow_get_active_htmlview()))) {

                gtk_widget_grab_focus(feedlist);

                GtkAdjustment *scrollbar =NULL;

                scrollbar =
                    gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW
                                                        (app_data->
                                                         app_ui_data->
                                                         scrolledwindow));

                gtk_adjustment_set_value(scrollbar,
                                         app_data->app_ui_data->
                                         scrollbarposition);

                gtk_scrolled_window_set_vadjustment(GTK_SCROLLED_WINDOW
                                                    (app_data->app_ui_data->
                                                     scrolledwindow),
                                                    scrollbar);

                return TRUE;

            } else
                if (gtk_window_get_focus(GTK_WINDOW(app_ui_data->main_view))
                    == feedlist) {
                return TRUE;
            } else {
                return FALSE;
            }

            /*
             * if (GTK_WIDGET_VISIBLE(feedlist)) {
             * gtk_widget_grab_focus(feedlist);
             * 
             * GtkAdjustment *scrollbar = GTK_ADJUSTMENT(gtk_adjustment_new(0,0,0,0,0,0));
             * 
             * scrollbar = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(app_data->app_ui_data->scrolledwindow));
             * 
             * gtk_adjustment_set_value(scrollbar, app_data->app_ui_data->scrollbarposition);
             * 
             * gtk_scrolled_window_set_vadjustment(GTK_SCROLLED_WINDOW(app_data->app_ui_data->scrolledwindow), scrollbar);
             * 
             * return TRUE;
             * }
             * break;
             */

        case GDK_Right:
            if (gtk_window_get_focus(GTK_WINDOW(app_ui_data->main_view)) ==
                feedlist) {

                gtk_widget_grab_focus(gtk_bin_get_child
                                      (GTK_BIN
                                       (ui_mainwindow_get_active_htmlview
                                        ())));
                return TRUE;
            } else
                if (gtk_window_get_focus(GTK_WINDOW(app_ui_data->main_view))
                    ==
                    gtk_bin_get_child(GTK_BIN
                                      (ui_mainwindow_get_active_htmlview())))
            {
                return TRUE;
            } else {
                return FALSE;
            }

            /*
             * if (app_ui_data->feed_displayed &&
             * gtk_window_get_focus(GTK_WINDOW(app_ui_data->main_view)) !=
             * ui_mainwindow_get_active_htmlview())
             * {
             * gtk_widget_grab_focus(gtk_bin_get_child(
             * GTK_BIN (ui_mainwindow_get_active_htmlview())));
             * }
             * return TRUE;
             * break;
             */

        case GDK_F6:           //Full screen
            key_press_nr++;
            if (key_press_nr == 1)
                toggle_fullscreen_and_update_menu(widget, app_ui_data);
            return TRUE;

        case GDK_Up:
            if (gtk_window_get_focus(GTK_WINDOW(app_ui_data->main_view)) ==
                feedlist)
                return tree_view_cycle_up(feedlist);
            else {
                ULOG_DEBUG("<<<key-press: On the page now ");
                key_press_nr++;
                //if up_key_press_nr == 1) we will handle in release-key
                if (key_press_nr >= 2) {
                    ULOG_DEBUG
                        ("key-press: YOU  are pressing a LONG DOWN KEY\n\
            There fore i'm scrolling UP");
                    // by returning FALSE. Let the system scroll
                    return FALSE;
                }
            }

            break;
        /*
        case GDK_BackSpace:
            if (gtk_window_get_focus(GTK_WINDOW(app_ui_data->main_view)) != feedlist) {
                ULOG_DEBUG("On the page, we should disable Backspace scrolling!");
                return TRUE;
            }

            break;
        */
        case GDK_Down:
            if (gtk_window_get_focus(GTK_WINDOW(app_ui_data->main_view)) ==
                feedlist)
                return tree_view_cycle_down(feedlist);

            else {
                ULOG_DEBUG("<<<key-press: On the page now ");
                key_press_nr++;
                //if down_key_press_nr == 1) we will handle in release-key
                if (key_press_nr >= 2) {
                    ULOG_DEBUG
                        ("key-press: YOU  are pressing a LONG DOWN KEY\n\
            	There fore i'm scrolling down");
                    // by returning FALSE.Let the system scroll"
                    return FALSE;
                }
            }
            break;


        case GDK_Return:
            if (key_press_nr++ == 0) {
                ptr = (nodePtr) ui_feedlist_get_selected();

                if (gtk_window_get_focus(GTK_WINDOW(app_ui_data->main_view))
                    == feedlist) {
                    if (ptr && ((FST_FEED == ptr->type)
                                || (FST_VFOLDER == ptr->type))) {
                        ui_feedlist_load_selected(ptr);
                        return TRUE;
                    }
                    if (ptr && FST_FOLDER == ptr->type) {
                        ui_folder_set_expansion((folderPtr) ptr,
                                                !ui_is_folder_expanded((folderPtr) ptr));
                    }
                }
            }
            break;

        case GDK_F7:
            if (key_press_nr++ == 0)
                gtkhtml_zoom_in();
            return TRUE;
            break;

        case GDK_F8:
            if (key_press_nr++ == 0)
                gtkhtml_zoom_out();
            return TRUE;
            break;

        default:
            break;
    }
    return FALSE;
}

gboolean key_release(GtkWidget * widget, GdkEventKey * event,
                     AppUIData * app_ui_data)
{
    ULOG_DEBUG("Key release event received");

    if (app_ui_data->dialog_closed) {
        ULOG_DEBUG("Dialog was closed. Ignoring the key-release signal");
        app_ui_data->dialog_closed = FALSE;
        return TRUE;
    }

    if (!key_pressed)
        {
            return FALSE;
        }
    
    key_pressed=FALSE;
    
    ULOG_DEBUG("Got an event != GDK_KEY_PRESS: %d", event->keyval);

    switch (event->keyval) {

        case GDK_Escape:
            ULOG_DEBUG("Escape hit with keyval: %d", event->keyval);
            if (SFM_NORMAL != app_ui_data->search_mode) {
                ULOG_DEBUG("SFM_NORMAL != search_mode: %d",
                           app_ui_data->search_mode);
                if (SFM_SEARCH == app_ui_data->search_mode) {
                    ULOG_DEBUG("Interrupting search");
                    stop_search();
                } else if (SFM_REFRESH == app_ui_data->search_mode) {
                    ULOG_DEBUG("Interrupting refresh");
                    app_ui_data->search_mode = SFM_INTERRUPTED;
                    download_cancel_all(TRUE);
                }
                return FALSE;
                /* Actually not TRUE */
            } else if (app_ui_data->fullscreen) {
                toggle_fullscreen_and_update_menu(widget, app_ui_data);
                return FALSE;
            }
            break;

        case GDK_Return:
            key_press_nr = 0;
            break;

        case GDK_Up:
            if (gtk_window_get_focus(GTK_WINDOW(app_ui_data->main_view)) ==
                feedlist)
                return TRUE;
            else if (!GTK_IS_EDITABLE
                     (gtk_window_get_focus
                      (GTK_WINDOW(app_ui_data->main_view)))) {
                ULOG_DEBUG("....key-release: On the page now ");
                //down_key_press_nr --;
                //we had only 1 short press in release-key
                if (key_press_nr == 1) {
                    ULOG_DEBUG
                        ("key-release: You made a SHORT Up key press. So i'm jumping UP");
                    gtkhtml_jump_to_next_focusable_object(FALSE);
                } else
                    gtkhtml_grab_cond_focus(FALSE);
                //reset counter
                key_press_nr = 0;
                //yes we handled it
                return TRUE;
            }

        case GDK_Down:
            if (gtk_window_get_focus(GTK_WINDOW(app_ui_data->main_view)) ==
                feedlist)
                return TRUE;
            //else if not GTKeditable, it should be gtkhtml focused
            else if (!GTK_IS_EDITABLE
                     (gtk_window_get_focus
                      (GTK_WINDOW(app_ui_data->main_view))))
            {
                ULOG_DEBUG("<<key-press: On the page now ");
                //down_key_press_nr --;
                //we had only 1 short press in release-key
                if (key_press_nr == 1) {
                    ULOG_DEBUG
                        ("key-release: You made a SHORT Down key press. So i'm jumping Down");
                    gtkhtml_jump_to_next_focusable_object(TRUE);
                } else
                    gtkhtml_grab_cond_focus(TRUE);
                //reset counter
                key_press_nr = 0;
                //yes we handled it
                return TRUE;
            }
            break;

        case GDK_F6:           //Full screen
        case GDK_F7:           //Zoom in
        case GDK_F8:           //Zoom out
            key_press_nr = 0;
            return TRUE;
            break;

        default:
            break;
    }
    return FALSE;
}

GtkWidget *create_propdialog(gboolean unmovable)
{
    GtkWidget *propdialog;
    GtkWidget *dialog_vbox13;
    GtkWidget *feedPropTable;
    GtkWidget *folderLabel;
    GtkWidget *folderData;
    GtkWidget *subscribedLabel;
    GtkWidget *subscribedData;
    GtkWidget *titleLabel;
    GtkWidget *locationLabel;
    GtkWidget *feedNameEntry;
    GtkWidget *feedLocationEntry;
    GtkWidget *downloadImagesLabel;
    GtkWidget *downloadImagesEntry;
    GtkWidget *dialog_action_area13;
    GtkWidget *prop_cancel;
    GtkWidget *prop_ok;
    GtkWidget *prop_unsubscribe;
    GtkWidget *prop_changefolder;

    GtkSizeGroup *group =
        GTK_SIZE_GROUP(gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL));

    propdialog = gtk_dialog_new();
    gtk_window_set_title(GTK_WINDOW(propdialog), _("rss_ti_feed_details"));
    gtk_window_set_type_hint(GTK_WINDOW(propdialog),
                             GDK_WINDOW_TYPE_HINT_DIALOG);
    gtk_dialog_set_has_separator(GTK_DIALOG(propdialog), FALSE);

    hildon_help_dialog_help_enable(GTK_DIALOG(propdialog),
                                   RSS_NEWS_READER_HELP_DETAILS,
                                   app_data->osso);

    dialog_vbox13 = GTK_DIALOG(propdialog)->vbox;
    gtk_widget_show(dialog_vbox13);

    feedPropTable = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(feedPropTable);
    gtk_container_add(GTK_CONTAINER(dialog_vbox13), feedPropTable);

    if (unmovable) {
        feedNameEntry = gtk_label_new("");
        gtk_misc_set_alignment(GTK_MISC(feedNameEntry), 0, 0.5);
    } else {
        feedNameEntry = gtk_entry_new();
        hildon_gtk_entry_set_input_mode(GTK_ENTRY(feedNameEntry),
                                        HILDON_GTK_INPUT_MODE_ALPHA |
                                        HILDON_GTK_INPUT_MODE_NUMERIC |
                                        HILDON_GTK_INPUT_MODE_SPECIAL |
                                        HILDON_GTK_INPUT_MODE_AUTOCAP);
        gtk_entry_set_activates_default(GTK_ENTRY(feedNameEntry), TRUE);
    }
    gtk_widget_show(feedNameEntry);

    titleLabel = hildon_caption_new(group, _("rss_ia_feed_title"),
                                    feedNameEntry, NULL,
                                    HILDON_CAPTION_OPTIONAL);
    gtk_widget_show(titleLabel);
    gtk_box_pack_start(GTK_BOX(feedPropTable), titleLabel, FALSE, FALSE, 2);

    if (unmovable) {
        feedLocationEntry = gtk_label_new("");
        gtk_misc_set_alignment(GTK_MISC(feedLocationEntry), 0, 0.5);
    } else {
        feedLocationEntry = gtk_entry_new();
        hildon_gtk_entry_set_input_mode(GTK_ENTRY(feedLocationEntry),
                                        HILDON_GTK_INPUT_MODE_ALPHA |
                                        HILDON_GTK_INPUT_MODE_NUMERIC |
                                        HILDON_GTK_INPUT_MODE_SPECIAL);
        gtk_entry_set_activates_default(GTK_ENTRY(feedLocationEntry), TRUE);
    }
    gtk_widget_show(feedLocationEntry);

    locationLabel = hildon_caption_new(group, _("rss_ia_feed_address"),
                                       feedLocationEntry, NULL,
                                       HILDON_CAPTION_OPTIONAL);
    gtk_widget_show(locationLabel);
    gtk_box_pack_start(GTK_BOX(feedPropTable), locationLabel, FALSE, FALSE,
                       2);

    downloadImagesEntry = gtk_check_button_new();
    //gtk_misc_set_alignment(GTK_MISC(downloadImagesEntry), 0, 0.5);
    gtk_widget_show(downloadImagesEntry);

    downloadImagesLabel = hildon_caption_new(group, _("rss_fi_download_img"),
                                             downloadImagesEntry, NULL,
                                             HILDON_CAPTION_OPTIONAL);
    gtk_widget_show(downloadImagesLabel);
    gtk_box_pack_start(GTK_BOX(feedPropTable), downloadImagesLabel, FALSE,
                       FALSE, 2);

    subscribedData = gtk_label_new("-");
    gtk_misc_set_alignment(GTK_MISC(subscribedData), 0, 0.5);
    gtk_widget_show(subscribedData);

    subscribedLabel = hildon_caption_new(group, _("rss_ia_subscribed"),
                                         subscribedData, NULL,
                                         HILDON_CAPTION_OPTIONAL);
    gtk_widget_show(subscribedLabel);
    gtk_box_pack_start_defaults(GTK_BOX(feedPropTable), subscribedLabel);

    folderData = gtk_label_new("");
    gtk_misc_set_alignment(GTK_MISC(folderData), 0, 0.5);
    gtk_widget_show(folderData);

    folderLabel = hildon_caption_new(group, _("rss_ia_feed_folder"),
                                     folderData, NULL,
                                     HILDON_CAPTION_OPTIONAL);
    gtk_widget_show(folderLabel);
    gtk_box_pack_start_defaults(GTK_BOX(feedPropTable), folderLabel);

    dialog_action_area13 = GTK_DIALOG(propdialog)->action_area;
    gtk_widget_show(dialog_action_area13);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(dialog_action_area13),
                              GTK_BUTTONBOX_END);

    GtkRequisition r;
    gtk_widget_size_request(gtk_label_new(_("rss_bd_ok")), &r);

    prop_ok = gtk_button_new_with_label(_("rss_bd_ok"));
    gtk_widget_show(prop_ok);

    gtk_widget_set_size_request(GTK_WIDGET(prop_ok), r.width + 50, -1);
    gtk_dialog_add_action_widget(GTK_DIALOG(propdialog), prop_ok,
                                 GTK_RESPONSE_OK);

    prop_unsubscribe = gtk_button_new_with_label(_("rss_bd_unsubscribe"));
    gtk_widget_show(prop_unsubscribe);

    gtk_dialog_add_action_widget(GTK_DIALOG(propdialog), prop_unsubscribe,
                                 RSS_RESPONSE_FEED_PROP_UNSUBSCRIBE);
    GTK_WIDGET_SET_FLAGS(prop_unsubscribe, GTK_CAN_DEFAULT);

    prop_changefolder = gtk_button_new_with_label(_("rss_bd_change_folder"));

    gtk_widget_show(prop_changefolder);
    gtk_box_pack_start(GTK_BOX(dialog_action_area13), prop_changefolder,
                       FALSE, FALSE, 0);

    prop_cancel = gtk_button_new_with_label(_("rss_bd_cancel"));
    gtk_widget_show(prop_cancel);

    gtk_widget_size_request(gtk_label_new(_("rss_bd_cancel")), &r);
    gtk_widget_set_size_request(GTK_WIDGET(prop_cancel), r.width + 60, -1);

    gtk_dialog_add_action_widget(GTK_DIALOG(propdialog), prop_cancel,
                                 GTK_RESPONSE_CANCEL);
    GTK_WIDGET_SET_FLAGS(prop_cancel, GTK_CAN_DEFAULT);

    /* Store pointers to all widgets, for use by lookup_widget(). */
    GLADE_HOOKUP_OBJECT_NO_REF(propdialog, propdialog, "propdialog");
    GLADE_HOOKUP_OBJECT_NO_REF(propdialog, dialog_vbox13, "dialog_vbox13");
    GLADE_HOOKUP_OBJECT(propdialog, feedNameEntry, "feedNameEntry");
    GLADE_HOOKUP_OBJECT(propdialog, feedLocationEntry, "feedLocationEntry");
    GLADE_HOOKUP_OBJECT(propdialog, downloadImagesEntry,
                        "downloadImagesEntry");
    GLADE_HOOKUP_OBJECT(propdialog, prop_ok, "okBtn");
    GLADE_HOOKUP_OBJECT(propdialog, prop_unsubscribe, "unsubscribeBtn");
    GLADE_HOOKUP_OBJECT(propdialog, subscribedData, "subscribedData");
    GLADE_HOOKUP_OBJECT(propdialog, folderData, "folderName");
    GLADE_HOOKUP_OBJECT_NO_REF(propdialog, dialog_action_area13,
                               "dialog_action_area13");
    GLADE_HOOKUP_OBJECT(propdialog, prop_cancel, "prop_cancel");
    GLADE_HOOKUP_OBJECT(propdialog, prop_ok, "prop_ok");
    GLADE_HOOKUP_OBJECT(propdialog, prop_changefolder, "changefolder");

    gtk_widget_grab_focus(prop_ok);
    return propdialog;
}

GtkWidget *create_newdialog(void)
{
    GtkWidget *newdialog;
    GtkWidget *dialog_vbox2;
    GtkWidget *sourceEntry;
    GtkWidget *label104;
    GtkWidget *dialog_action_area2;
    GtkWidget *cancelbtn;
    GtkWidget *changefolderbtn;
    GtkWidget *browsebtn;
    GtkWidget *newfeedbtn;
    GtkWidget *labelFolder;
    GtkWidget *folderName;
    int length = 0;

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    GtkSizeGroup *group =
        GTK_SIZE_GROUP(gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL));

    newdialog = gtk_dialog_new();
    gtk_container_set_border_width(GTK_CONTAINER(newdialog), 6);
    gtk_window_set_title(GTK_WINDOW(newdialog), _("rss_ti_add_a_feed"));
    gtk_window_set_default_size(GTK_WINDOW(newdialog), NEW_DIALOG_X_SIZE, -1);
    gtk_window_set_type_hint(GTK_WINDOW(newdialog),
                             GDK_WINDOW_TYPE_HINT_DIALOG);
    gtk_dialog_set_has_separator(GTK_DIALOG(newdialog), FALSE);

    hildon_help_dialog_help_enable(GTK_DIALOG(newdialog),
                                   RSS_NEWS_READER_HELP_ADD_FEED,
                                   app_data->osso);

    dialog_vbox2 = GTK_DIALOG(newdialog)->vbox;
    gtk_widget_show(dialog_vbox2);

    sourceEntry = gtk_entry_new();
    hildon_gtk_entry_set_input_mode(GTK_ENTRY(sourceEntry),
                                    HILDON_GTK_INPUT_MODE_ALPHA |
                                    HILDON_GTK_INPUT_MODE_NUMERIC |
                                    HILDON_GTK_INPUT_MODE_SPECIAL);
    gtk_widget_show(sourceEntry);

    label104 = hildon_caption_new(group, _("rss_ia_feed_address"),
                                  sourceEntry, NULL, HILDON_CAPTION_OPTIONAL);
    gtk_widget_show(label104);
    gtk_box_pack_start_defaults(GTK_BOX(dialog_vbox2), label104);

    folderName = gtk_label_new(_("rss_ia_news_feed_folders"));
    length = g_utf8_strlen(_("rss_ia_news_feed_folders"), -1);
    if (length > LABEL_MAX_WIDTH)
        gtk_label_set_max_width_chars(GTK_LABEL(folderName), length);
    else
        gtk_label_set_max_width_chars(GTK_LABEL(folderName), LABEL_MAX_WIDTH);
    gtk_label_set_ellipsize(GTK_LABEL(folderName), PANGO_ELLIPSIZE_END);
    gtk_widget_show(folderName);
    gtk_misc_set_alignment(GTK_MISC(folderName), 0, 0.5);

    labelFolder = hildon_caption_new(group, _("rss_ia_add_to_folder"),
                                     folderName, NULL,
                                     HILDON_CAPTION_OPTIONAL);
    gtk_widget_show(labelFolder);
    gtk_box_pack_start_defaults(GTK_BOX(dialog_vbox2), labelFolder);

    dialog_action_area2 = GTK_DIALOG(newdialog)->action_area;
    gtk_widget_show(dialog_action_area2);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(dialog_action_area2),
                              GTK_BUTTONBOX_END);

    app_data->app_ui_data->newfeedbtn = newfeedbtn =
        gtk_button_new_with_label(_("rss_bd_ok"));
    gtk_widget_show(newfeedbtn);
    gtk_dialog_add_action_widget(GTK_DIALOG(newdialog), newfeedbtn,
                                 GTK_RESPONSE_OK);
    GTK_WIDGET_SET_FLAGS(newfeedbtn, GTK_CAN_DEFAULT);

    app_data->app_ui_data->changefolderbtn = changefolderbtn =
        gtk_button_new_with_label(_("rss_bd_change_folder"));
    gtk_widget_show(changefolderbtn);
    gtk_box_pack_start(GTK_BOX(dialog_action_area2), changefolderbtn,
                       FALSE, FALSE, 0);
    GTK_WIDGET_SET_FLAGS(changefolderbtn, GTK_CAN_DEFAULT);

    app_data->app_ui_data->browsebtn = browsebtn =
        gtk_button_new_with_label(_("rss_bd_browse"));
    gtk_widget_show(browsebtn);
    gtk_box_pack_start(GTK_BOX(dialog_action_area2), browsebtn,
                       FALSE, FALSE, 0);

    app_data->app_ui_data->cancelbtn = cancelbtn =
        gtk_button_new_with_label(_("rss_bd_cancel"));
    gtk_widget_show(cancelbtn);
    gtk_dialog_add_action_widget(GTK_DIALOG(newdialog), cancelbtn,
                                 GTK_RESPONSE_CANCEL);
    GTK_WIDGET_SET_FLAGS(cancelbtn, GTK_CAN_DEFAULT);

    /* Store pointers to all widgets, for use by lookup_widget(). */

    GLADE_HOOKUP_OBJECT_NO_REF(newdialog, newdialog, "newdialog");
    GLADE_HOOKUP_OBJECT_NO_REF(newdialog, dialog_vbox2, "dialog_vbox2");
    GLADE_HOOKUP_OBJECT(newdialog, sourceEntry, "sourceEntry");
    GLADE_HOOKUP_OBJECT(newdialog, label104, "label104");
    GLADE_HOOKUP_OBJECT_NO_REF(newdialog, dialog_action_area2,
                               "dialog_action_area2");
    GLADE_HOOKUP_OBJECT(newdialog, cancelbtn, "cancelbtn");
    GLADE_HOOKUP_OBJECT(newdialog, newfeedbtn, "newfeedbtn");
    GLADE_HOOKUP_OBJECT(newdialog, browsebtn, "browsebtn");
    GLADE_HOOKUP_OBJECT(newdialog, folderName, "folderName");
    GLADE_HOOKUP_OBJECT(newdialog, changefolderbtn, "changefolder");

    return newdialog;
}

GtkWidget *create_refresh_dialog()
{
    GtkWidget *refresh_dialog;
    GtkWidget *refresh_label;
    GtkWidget *radio_button_all_feeds;
    GtkWidget *radio_button_selected_feed;
    GtkWidget *refresh_ok_button;
    GtkWidget *refresh_cancel_button;
    GtkWidget *vbox;
    GtkWidget *table;

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    refresh_dialog = app_data->app_ui_data->refresh_dialog = gtk_dialog_new();
    gtk_window_set_title(GTK_WINDOW(refresh_dialog),
                         _("rss_ti_refresh_feeds"));
    gtk_window_set_default_size(GTK_WINDOW(refresh_dialog),
                                REFRESH_DIALOG_X_SIZE, -1);
    gtk_window_set_type_hint(GTK_WINDOW(refresh_dialog),
                             GDK_WINDOW_TYPE_HINT_DIALOG);
    gtk_window_set_transient_for(GTK_WINDOW(refresh_dialog),
                                 GTK_WINDOW(app_data->app_ui_data->
                                            main_view));
    gtk_window_set_modal(GTK_WINDOW(refresh_dialog), TRUE);
    gtk_dialog_set_has_separator(GTK_DIALOG(refresh_dialog), FALSE);

    hildon_help_dialog_help_enable(GTK_DIALOG(refresh_dialog),
                                   RSS_NEWS_READER_HELP_REFRESH,
                                   app_data->osso);

    vbox = GTK_DIALOG(refresh_dialog)->vbox;
    gtk_widget_show(vbox);

    table = gtk_table_new(2, 2, FALSE);

    gtk_widget_show(table);

    gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 10);
    refresh_label = gtk_label_new(_("rss_ia_refresh"));
    gtk_widget_show(refresh_label);

    gtk_table_attach(GTK_TABLE(table), refresh_label, 0, 1, 0, 1, GTK_FILL,
                     GTK_FILL, 10, 0);

    radio_button_all_feeds =
        gtk_radio_button_new_with_label(NULL, _("rss_ia_all_feeds"));
    gtk_widget_show(radio_button_all_feeds);
    GSList *rbgroup =
        gtk_radio_button_get_group(GTK_RADIO_BUTTON(radio_button_all_feeds));
    radio_button_selected_feed =
        gtk_radio_button_new_with_label(rbgroup,
                                        _("rss_ia_selected_feed_only"));
    gtk_widget_show(radio_button_selected_feed);

    gtk_table_attach(GTK_TABLE(table), radio_button_all_feeds, 1, 2, 0, 1,
                     GTK_FILL, GTK_FILL, 10, 0);
    gtk_table_attach(GTK_TABLE(table), radio_button_selected_feed, 1, 2, 1,
                     2, GTK_FILL, GTK_FILL, 10, 0);

    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radio_button_all_feeds),
                                 TRUE);

    refresh_ok_button = gtk_button_new_with_label(_("rss_bd_ok"));
    gtk_widget_show(refresh_ok_button);
    gtk_dialog_add_action_widget(GTK_DIALOG(refresh_dialog),
                                 refresh_ok_button, GTK_RESPONSE_OK);

    refresh_cancel_button = gtk_button_new_with_label(_("rss_bd_cancel"));
    gtk_widget_show(refresh_cancel_button);
    gtk_dialog_add_action_widget(GTK_DIALOG(refresh_dialog),
                                 refresh_cancel_button, GTK_RESPONSE_CANCEL);

    g_signal_connect(G_OBJECT(refresh_dialog), "key-press-event",
                     G_CALLBACK(key_press_for_cancelling_dialog), NULL);

    GTK_WIDGET_SET_FLAGS(refresh_ok_button, GTK_CAN_DEFAULT);

    GLADE_HOOKUP_OBJECT_NO_REF(refresh_dialog, refresh_dialog,
                               "refresh_dialog");
    GLADE_HOOKUP_OBJECT_NO_REF(refresh_dialog, vbox, "vbox");
    GLADE_HOOKUP_OBJECT(refresh_dialog, radio_button_selected_feed,
                        "radio_button_selected_feed");

    return refresh_dialog;
}

GtkWidget *create_manage_folders_dialog(void)
{
    GtkWidget *mf_dialog;
    GtkWidget *dialog_vbox;
    GtkWidget *dialog_action_area;
    GtkWidget *mf_newbtn;
    GtkWidget *mf_renamebtn;
    GtkWidget *mf_deletebtn;
    GtkWidget *mf_closebtn;
    GtkWidget *folder_view;
    GtkWidget *folder_scrollwindow;

    ULOG_DEBUG(__FUNCTION__);
    app_data->app_ui_data->manage_folders_displayed = TRUE;

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    mf_dialog = gtk_dialog_new();
    gtk_window_set_title(GTK_WINDOW(mf_dialog), _("rss_ti_manage_folders"));
    gtk_window_set_default_size(GTK_WINDOW(mf_dialog),
                                MANAGE_FOLDERS_DIALOG_XY_SIZE,
                                MANAGE_FOLDERS_DIALOG_XY_SIZE);
    gtk_window_set_type_hint(GTK_WINDOW(mf_dialog),
                             GDK_WINDOW_TYPE_HINT_DIALOG);
    gtk_window_set_transient_for(GTK_WINDOW(mf_dialog),
                                 GTK_WINDOW(app_data->app_ui_data->
                                            main_view));
    gtk_window_set_modal(GTK_WINDOW(mf_dialog), TRUE);
    gtk_dialog_set_has_separator(GTK_DIALOG(mf_dialog), FALSE);

    hildon_help_dialog_help_enable(GTK_DIALOG(mf_dialog),
                                   RSS_NEWS_READER_HELP_MANAGE,
                                   app_data->osso);

    dialog_vbox = GTK_DIALOG(mf_dialog)->vbox;
    gtk_widget_show(dialog_vbox);

    folder_scrollwindow = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW
                                   (folder_scrollwindow), GTK_POLICY_NEVER,
                                   GTK_POLICY_AUTOMATIC);
    gtk_widget_show(folder_scrollwindow);

    folder_view = gtk_tree_view_new();
    gtk_tree_view_set_enable_search(GTK_TREE_VIEW(folder_view), FALSE);
    gtk_widget_show(folder_view);
    gtk_container_add(GTK_CONTAINER(folder_scrollwindow), folder_view);
    gtk_container_add(GTK_CONTAINER(dialog_vbox), folder_scrollwindow);

    gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(folder_view), FALSE);
    gtk_tree_view_set_reorderable(GTK_TREE_VIEW(folder_view), FALSE);

    dialog_action_area = GTK_DIALOG(mf_dialog)->action_area;
    gtk_widget_show(dialog_action_area);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(dialog_action_area),
                              GTK_BUTTONBOX_END);

    mf_newbtn = gtk_button_new_with_label(_("rss_bd_new"));
    gtk_widget_show(mf_newbtn);
    gtk_dialog_add_action_widget(GTK_DIALOG(mf_dialog), mf_newbtn,
                                 RSS_RESPONSE_NEW_FOLDER);

    mf_renamebtn = gtk_button_new_with_label(_("rss_bd_rename"));
    gtk_widget_show(mf_renamebtn);
    gtk_dialog_add_action_widget(GTK_DIALOG(mf_dialog), mf_renamebtn,
                                 RSS_RESPONSE_RENAME_FOLDER);

    mf_deletebtn = gtk_button_new_with_label(_("rss_bd_delete"));
    gtk_widget_show(mf_deletebtn);
    gtk_dialog_add_action_widget(GTK_DIALOG(mf_dialog), mf_deletebtn,
                                 RSS_RESPONSE_DELETE_FOLDER);

    mf_closebtn = gtk_button_new_with_label(_("rss_bd_close"));
    gtk_widget_show(mf_closebtn);
    gtk_dialog_add_action_widget(GTK_DIALOG(mf_dialog), mf_closebtn,
                                 GTK_RESPONSE_CANCEL);

    g_signal_connect(G_OBJECT(mf_dialog), "key-press-event",
                     G_CALLBACK(key_press_for_cancelling_dialog), NULL);

    GTK_WIDGET_SET_FLAGS(mf_renamebtn, GTK_CAN_DEFAULT);
    GTK_WIDGET_SET_FLAGS(mf_deletebtn, GTK_CAN_DEFAULT);
    GTK_WIDGET_SET_FLAGS(mf_closebtn, GTK_CAN_DEFAULT);

    GLADE_HOOKUP_OBJECT_NO_REF(mf_dialog, mf_dialog, "mf_dialog");
    GLADE_HOOKUP_OBJECT_NO_REF(mf_dialog, dialog_vbox, "vbox");
    GLADE_HOOKUP_OBJECT(mf_dialog, folder_view, "folder_view");
    GLADE_HOOKUP_OBJECT(mf_dialog, mf_newbtn, "mf_newbtn");
    GLADE_HOOKUP_OBJECT(mf_dialog, mf_renamebtn, "mf_renamebtn");
    GLADE_HOOKUP_OBJECT(mf_dialog, mf_deletebtn, "mf_deletebtn");

    return mf_dialog;
}

GtkWidget *create_change_folder_dialog(GtkWindow * parent)
{
    GtkWidget *cf_dialog;
    GtkWidget *dialog_vbox;
    GtkWidget *dialog_action_area;
    GtkWidget *cf_okbtn;
    GtkWidget *cf_newbtn;
    GtkWidget *cf_cancelbtn;
    GtkWidget *folder_view;
    GtkWidget *folder_scrollwindow;

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    cf_dialog = gtk_dialog_new();

    hildon_help_dialog_help_enable(GTK_DIALOG(cf_dialog),
                                   RSS_NEWS_READER_HELP_CHFOLDER,
                                   app_data->osso);

    gtk_window_set_title(GTK_WINDOW(cf_dialog), _("rss_ti_change_folder"));
    gtk_window_set_default_size(GTK_WINDOW(cf_dialog),
                                CHANGE_FOLDER_DIALOG_XY_SIZE,
                                CHANGE_FOLDER_DIALOG_XY_SIZE);
    gtk_window_set_type_hint(GTK_WINDOW(cf_dialog),
                             GDK_WINDOW_TYPE_HINT_DIALOG);
    gtk_window_set_transient_for(GTK_WINDOW(cf_dialog), parent);
    gtk_window_set_modal(GTK_WINDOW(cf_dialog), TRUE);
    gtk_dialog_set_has_separator(GTK_DIALOG(cf_dialog), FALSE);

    dialog_vbox = GTK_DIALOG(cf_dialog)->vbox;
    gtk_widget_show(dialog_vbox);

    folder_scrollwindow = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW
                                   (folder_scrollwindow), GTK_POLICY_NEVER,
                                   GTK_POLICY_AUTOMATIC);
    gtk_widget_show(folder_scrollwindow);

    folder_view = gtk_tree_view_new();
    gtk_tree_view_set_enable_search(GTK_TREE_VIEW(folder_view), FALSE);
    gtk_widget_show(folder_view);
    gtk_container_add(GTK_CONTAINER(folder_scrollwindow), folder_view);
    gtk_container_add(GTK_CONTAINER(dialog_vbox), folder_scrollwindow);

    gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(folder_view), FALSE);
    gtk_tree_view_set_reorderable(GTK_TREE_VIEW(folder_view), TRUE);


    dialog_action_area = GTK_DIALOG(cf_dialog)->action_area;
    gtk_widget_show(dialog_action_area);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(dialog_action_area),
                              GTK_BUTTONBOX_END);

    cf_okbtn = gtk_button_new_with_label(_("rss_bd_ok"));
    gtk_widget_show(cf_okbtn);
    gtk_dialog_add_action_widget(GTK_DIALOG(cf_dialog), cf_okbtn,
                                 GTK_RESPONSE_OK);

    cf_newbtn = gtk_button_new_with_label(_("rss_bd_new"));
    gtk_widget_show(cf_newbtn);
    gtk_dialog_add_action_widget(GTK_DIALOG(cf_dialog), cf_newbtn,
                                 RSS_RESPONSE_NEW_FOLDER);

    cf_cancelbtn = gtk_button_new_with_label(_("rss_bd_cancel"));
    gtk_widget_show(cf_cancelbtn);
    gtk_dialog_add_action_widget(GTK_DIALOG(cf_dialog), cf_cancelbtn,
                                 GTK_RESPONSE_CANCEL);

    g_signal_connect((gpointer) folder_view, "drag_end",                           
                     G_CALLBACK(on_feedlist_drag_end), NULL);

    GLADE_HOOKUP_OBJECT_NO_REF(cf_dialog, cf_dialog, "cf_dialog");
    GLADE_HOOKUP_OBJECT_NO_REF(cf_dialog, dialog_vbox, "vbox");
    GLADE_HOOKUP_OBJECT(cf_dialog, folder_view, "folder_view");

    return cf_dialog;
}


GtkWidget *create_newfolderdialog(folderPtr folder,
                                  GtkTreeView * parent_treeview,
                                  GtkWindow * parent)
{
    GtkWidget *newfolderdialog;
    GtkWidget *dialog_vbox6;
    GtkWidget *vbox10;
    GtkWidget *label20;
    GtkWidget *foldertitleentry;
    GtkWidget *dialog_action_area6;
    GtkWidget *button3;
    GtkWidget *button2;
    GtkWidget *labelFolder;
    GtkWidget *location;
    GtkWidget *hboxLocation;
    GtkWidget *folderIcon;
    GtkWidget *alignment;
    folderPtr parent_ptr = NULL;
    int length = 0;

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    GtkSizeGroup *group =
        GTK_SIZE_GROUP(gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL));

    newfolderdialog = gtk_dialog_new();
    if (!newfolderdialog)
        return NULL;

    /* Set parent_folder so that we can create the new folder to
     * correct place.
     */

    g_object_set_data(G_OBJECT(newfolderdialog), "parent_folder", folder);
    g_object_set_data(G_OBJECT(newfolderdialog), "folder_treeview",
                      parent_treeview);
    g_object_set_data(G_OBJECT(newfolderdialog), "folder_treemodel",
                      gtk_tree_view_get_model(parent_treeview));


    gtk_window_set_title(GTK_WINDOW(newfolderdialog), _("rss_ti_new_folder"));
    gtk_window_set_type_hint(GTK_WINDOW(newfolderdialog),
                             GDK_WINDOW_TYPE_HINT_DIALOG);
    gtk_window_set_transient_for(GTK_WINDOW(newfolderdialog), parent);
    gtk_window_set_modal(GTK_WINDOW(newfolderdialog), TRUE);
    gtk_dialog_set_has_separator(GTK_DIALOG(newfolderdialog), FALSE);

    hildon_help_dialog_help_enable(GTK_DIALOG(newfolderdialog),
                                   RSS_NEWS_READER_HELP_NEWFOLDER,
                                   app_data->osso);

    dialog_vbox6 = GTK_DIALOG(newfolderdialog)->vbox;
    gtk_widget_show(dialog_vbox6);

    vbox10 = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(vbox10);
    gtk_box_pack_start(GTK_BOX(dialog_vbox6), vbox10, TRUE, TRUE, 0);

    foldertitleentry = gtk_entry_new();
    //hildon_gtk_entry_set_input_mode(GTK_ENTRY(foldertitleentry),HILDON_GTK_INPUT_MODE_AUTOCAP);

    /* Check, if the default foldername exists */
    gchar *new_foldername = NULL;
    parent_ptr = (folder != NULL)
        ? folder : ui_feedlist_get_root_folder();
    GtkTreeIter *parent_iter = &((ui_data *) (parent_ptr->ui_data))->row;
    new_foldername =
        g_strdup(dgettext
                 (HILDON_FM_L10N_PACKAGE, "ckdg_va_new_folder_name_stub"));
    if (ui_folder_check_if_exists(parent_iter, new_foldername)) {
        gint i = 0;
        do {
            i++;
            if (new_foldername) {
                g_free(new_foldername);
            }
            new_foldername =
                g_strdup_printf("%s (%d)",
                                dgettext(HILDON_FM_L10N_PACKAGE,
                                         "ckdg_va_new_folder_name_stub"), i);
        }
        while (ui_folder_check_if_exists(parent_iter, new_foldername));
    }
    gtk_entry_set_text(GTK_ENTRY(foldertitleentry), new_foldername);
    g_free(new_foldername);
    gtk_editable_select_region(GTK_EDITABLE(foldertitleentry), 0, 1000);
    gtk_widget_show(foldertitleentry);
    gtk_entry_set_activates_default(GTK_ENTRY(foldertitleentry), TRUE);

    label20 = hildon_caption_new(group, _("rss_ia_name"),
                                 foldertitleentry, NULL,
                                 HILDON_CAPTION_OPTIONAL);
    gtk_widget_show(label20);
    gtk_box_pack_start(GTK_BOX(vbox10), label20, FALSE, FALSE, 0);

    if (folder != NULL) {
        if (folder->title != NULL) {
            location = gtk_label_new(folder->title);
        } else {
            /* This is probably an error. */
            location = gtk_label_new("");
        }
    } else {
        location = gtk_label_new(_("rss_ia_news_feed_folders"));
    }
    length = g_utf8_strlen(_("rss_ia_news_feed_folders"), -1);
    if (length > LABEL_MAX_WIDTH)
        gtk_label_set_max_width_chars(GTK_LABEL(location), length);
    else
        gtk_label_set_max_width_chars(GTK_LABEL(location), LABEL_MAX_WIDTH);
    gtk_label_set_ellipsize(GTK_LABEL(location), PANGO_ELLIPSIZE_END);
    gtk_widget_show(location);

    folderIcon = gtk_image_new_from_pixbuf(icons[ICON_FOLDER_CLOSED]);
    gtk_widget_show(folderIcon);

    hboxLocation = gtk_hbox_new(FALSE, HILDON_MARGIN_DEFAULT);
    gtk_widget_show(hboxLocation);
    gtk_box_pack_start_defaults(GTK_BOX(hboxLocation), folderIcon);
    gtk_box_pack_start_defaults(GTK_BOX(hboxLocation), location);

    alignment = gtk_alignment_new(0, 0.5, 0, 0);
    gtk_widget_show(alignment);
    gtk_container_add(GTK_CONTAINER(alignment), hboxLocation);

    labelFolder = hildon_caption_new(group, _("rss_ia_location"),
                                     alignment, NULL,
                                     HILDON_CAPTION_OPTIONAL);
    gtk_widget_show(labelFolder);
    gtk_box_pack_start_defaults(GTK_BOX(vbox10), labelFolder);

    dialog_action_area6 = GTK_DIALOG(newfolderdialog)->action_area;
    gtk_widget_show(dialog_action_area6);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(dialog_action_area6),
                              GTK_BUTTONBOX_END);

    button2 = gtk_button_new_with_label(_("rss_bd_ok"));
    gtk_widget_show(button2);
    gtk_dialog_add_action_widget(GTK_DIALOG(newfolderdialog), button2,
                                 GTK_RESPONSE_OK);
    GTK_WIDGET_SET_FLAGS(button2, GTK_CAN_DEFAULT);

    button3 = gtk_button_new_with_label(_("rss_bd_cancel"));
    gtk_widget_show(button3);
    gtk_dialog_add_action_widget(GTK_DIALOG(newfolderdialog), button3,
                                 GTK_RESPONSE_CANCEL);
    GTK_WIDGET_SET_FLAGS(button3, GTK_CAN_DEFAULT);

    g_signal_connect_swapped((gpointer) newfolderdialog, "delete_event",
                             G_CALLBACK(gtk_widget_hide),
                             GTK_OBJECT(newfolderdialog));
    g_signal_connect_swapped((gpointer) button3, "clicked",
                             G_CALLBACK(gtk_widget_hide),
                             GTK_OBJECT(newfolderdialog));
    g_signal_connect((gpointer) button2, "clicked",
                     G_CALLBACK(on_newfolderbtn_clicked), newfolderdialog);

    /* Store pointers to all widgets, for use by lookup_widget(). */
    GLADE_HOOKUP_OBJECT_NO_REF(newfolderdialog, newfolderdialog,
                               "newfolderdialog");
    GLADE_HOOKUP_OBJECT_NO_REF(newfolderdialog, dialog_vbox6, "dialog_vbox6");
    GLADE_HOOKUP_OBJECT(newfolderdialog, vbox10, "vbox10");
    GLADE_HOOKUP_OBJECT(newfolderdialog, label20, "label20");
    GLADE_HOOKUP_OBJECT(newfolderdialog, foldertitleentry,
                        "foldertitleentry");
    GLADE_HOOKUP_OBJECT_NO_REF(newfolderdialog, dialog_action_area6,
                               "dialog_action_area6");
    GLADE_HOOKUP_OBJECT(newfolderdialog, button3, "button3");
    GLADE_HOOKUP_OBJECT(newfolderdialog, button2, "button2");

    gtk_widget_grab_default(button2);
    return newfolderdialog;
}

GtkWidget *create_foldernamedialog(void)
{
    GtkWidget *foldernamedialog;
    GtkWidget *dialog_vbox7;
    GtkWidget *foldernameentry;
    GtkWidget *foldernamecaption;
    GtkWidget *dialog_action_area7;
    GtkWidget *cancelbutton1;
    GtkWidget *foldernamechangebtn;

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    foldernamedialog = gtk_dialog_new();
    gtk_window_set_title(GTK_WINDOW(foldernamedialog),
                         _("rss_ti_rename_folder"));
    gtk_window_set_type_hint(GTK_WINDOW(foldernamedialog),
                             GDK_WINDOW_TYPE_HINT_DIALOG);
    gtk_window_set_transient_for(GTK_WINDOW(foldernamedialog),
                                 GTK_WINDOW(app_data->app_ui_data->
                                            main_view));
    gtk_window_set_modal(GTK_WINDOW(foldernamedialog), TRUE);
    gtk_dialog_set_has_separator(GTK_DIALOG(foldernamedialog), FALSE);

    hildon_help_dialog_help_enable(GTK_DIALOG(foldernamedialog),
                                   RSS_NEWS_READER_HELP_RENAMEFOLDER,
                                   app_data->osso);

    dialog_vbox7 = GTK_DIALOG(foldernamedialog)->vbox;
    gtk_widget_show(dialog_vbox7);

    foldernameentry = gtk_entry_new();
    //hildon_gtk_entry_set_input_mode(GTK_ENTRY(foldernameentry),HILDON_GTK_INPUT_MODE_AUTOCAP);
    gtk_widget_show(foldernameentry);
    foldernamecaption = hildon_caption_new(NULL, _("rss_ia_name"),
                                           foldernameentry, NULL,
                                           HILDON_CAPTION_OPTIONAL);
    gtk_widget_show(foldernamecaption);
    gtk_box_pack_start(GTK_BOX(dialog_vbox7), foldernamecaption, FALSE,
                       FALSE, 12);
    gtk_entry_set_activates_default(GTK_ENTRY(foldernameentry), TRUE);

    dialog_action_area7 = GTK_DIALOG(foldernamedialog)->action_area;
    gtk_widget_show(dialog_action_area7);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(dialog_action_area7),
                              GTK_BUTTONBOX_END);

    foldernamechangebtn = gtk_button_new_with_label(_("rss_bd_ok"));
    gtk_widget_show(foldernamechangebtn);
    gtk_dialog_add_action_widget(GTK_DIALOG(foldernamedialog),
                                 foldernamechangebtn, 0);
    GTK_WIDGET_SET_FLAGS(foldernamechangebtn, GTK_CAN_DEFAULT);

    cancelbutton1 = gtk_button_new_with_label(_("rss_bd_cancel"));
    gtk_widget_show(cancelbutton1);
    gtk_dialog_add_action_widget(GTK_DIALOG(foldernamedialog),
                                 cancelbutton1, 0);
    GTK_WIDGET_SET_FLAGS(cancelbutton1, GTK_CAN_DEFAULT);

    g_signal_connect_swapped((gpointer) foldernamedialog, "delete_event",
                             G_CALLBACK(gtk_widget_hide),
                             GTK_OBJECT(foldernamedialog));
    g_signal_connect_swapped((gpointer) cancelbutton1, "clicked",
                             G_CALLBACK(gtk_widget_hide),
                             GTK_OBJECT(foldernamedialog));
    g_signal_connect((gpointer) foldernamechangebtn, "clicked",
                     G_CALLBACK(on_foldernamechangebtn_clicked), NULL);
    g_signal_connect((gpointer) foldernamedialog, "key-press-event",
                     G_CALLBACK(rename_dialog_key_press), foldernamedialog);

    /* Store pointers to all widgets, for use by lookup_widget(). */
    GLADE_HOOKUP_OBJECT_NO_REF(foldernamedialog, foldernamedialog,
                               "foldernamedialog");
    GLADE_HOOKUP_OBJECT_NO_REF(foldernamedialog, dialog_vbox7,
                               "dialog_vbox7");
    GLADE_HOOKUP_OBJECT(foldernamedialog, foldernameentry, "foldernameentry");
    GLADE_HOOKUP_OBJECT_NO_REF(foldernamedialog, dialog_action_area7,
                               "dialog_action_area7");
    GLADE_HOOKUP_OBJECT(foldernamedialog, cancelbutton1, "cancelbutton1");
    GLADE_HOOKUP_OBJECT(foldernamedialog, foldernamechangebtn,
                        "foldernamechangebtn");

    gtk_widget_grab_default(foldernamechangebtn);

    GtkRequisition r;
    gtk_widget_size_request(foldernamedialog, &r);
    gtk_widget_set_size_request(foldernamedialog, r.width + 100, r.height);

    return foldernamedialog;
}

void set_feed_properties_insens_messages(AppUIData * app_ui_data)
{
    if (SFM_SEARCH == app_ui_data->search_mode) {
        hildon_helper_set_insensitive_message(GTK_WIDGET(app_ui_data->
                                                         feed_properties_button),
                                              _
                                              ("rss_ib_unable_edit_searching"));
        hildon_helper_set_insensitive_message(GTK_WIDGET
                                              (app_ui_data->
                                               menuitem_properties)
                                              ,
                                              _
                                              ("rss_ib_unable_edit_searching"));
    } else {
        // Should be on_feed_insensitive, but here for safety.
        hildon_helper_set_insensitive_message(GTK_WIDGET(app_ui_data->
                                                         feed_properties_button),
                                              _("rss_ib_no_feed_selected"));
        hildon_helper_set_insensitive_message(GTK_WIDGET
                                              (app_ui_data->
                                               menuitem_properties)
                                              , _("rss_ib_no_feed_selected"));
    }
}

void unsubscribe_insens_messages(AppUIData * app_ui_data)
{
    if (SFM_SEARCH == app_data->app_ui_data->search_mode) {
        hildon_helper_set_insensitive_message(GTK_WIDGET
                                              (app_ui_data->
                                               menuitem_unsubscribe),
                                              _
                                              ("rss_ib_unable_unsubscribe_searching"));
        hildon_helper_set_insensitive_message(GTK_WIDGET
                                              (app_ui_data->
                                               unsubscribe_button),
                                              _
                                              ("rss_ib_unable_unsubscribe_searching"));
        if (app_ui_data->dlg_btn_unsubscribe) {
            hildon_helper_set_insensitive_message(GTK_WIDGET
                                                  (app_ui_data->
                                                   dlg_btn_unsubscribe),
                                                  _
                                                  ("rss_ib_unable_unsubscribe_searching"));
        }
    } else if (SFM_REFRESH == app_data->app_ui_data->search_mode) {
        hildon_helper_set_insensitive_message(GTK_WIDGET
                                              (app_ui_data->
                                               menuitem_unsubscribe),
                                              _("rss_ib_unable_unsubscribe"));
        hildon_helper_set_insensitive_message(GTK_WIDGET
                                              (app_ui_data->
                                               unsubscribe_button),
                                              _("rss_ib_unable_unsubscribe"));
        if (app_ui_data->dlg_btn_unsubscribe) {
            hildon_helper_set_insensitive_message(GTK_WIDGET
                                                  (app_ui_data->
                                                   dlg_btn_unsubscribe),
                                                  _
                                                  ("rss_ib_unable_unsubscribe"));
        }
    } else {
        hildon_helper_set_insensitive_message(GTK_WIDGET
                                              (app_ui_data->
                                               menuitem_unsubscribe),
                                              _("rss_ib_no_feed_selected"));
        hildon_helper_set_insensitive_message(GTK_WIDGET
                                              (app_ui_data->
                                               unsubscribe_button),
                                              _("rss_ib_no_feed_selected"));
        if (app_ui_data->dlg_btn_unsubscribe) {
            hildon_helper_set_insensitive_message(GTK_WIDGET
                                                  (app_ui_data->
                                                   dlg_btn_unsubscribe),
                                                  _
                                                  ("rss_ib_no_feed_selected"));
        }
    }


}

void new_insens_messages(AppUIData * app_ui_data)
{
    DMSG("Entering %s with %d\n", __FUNCTION__,
         app_data->app_ui_data->search_mode);
    if (SFM_SEARCH == app_data->app_ui_data->search_mode) {
        hildon_helper_set_insensitive_message(GTK_WIDGET
                                              (app_ui_data->new_feed_button),
                                              _
                                              ("rss_ib_unable_add_feed_searching"));
        hildon_helper_set_insensitive_message(GTK_WIDGET
                                              (app_ui_data->
                                               menuitem_feed_directory),
                                              _
                                              ("rss_ib_unable_add_feed_searching"));
        hildon_helper_set_insensitive_message(GTK_WIDGET
                                              (app_ui_data->
                                               menuitem_add),
                                              _
                                              ("rss_ib_unable_add_feed_searching"));
    } else {
        hildon_helper_set_insensitive_message(GTK_WIDGET
                                              (app_ui_data->new_feed_button),
                                              _("rss_ib_unable_add_feed"));
        hildon_helper_set_insensitive_message(GTK_WIDGET
                                              (app_ui_data->
                                               menuitem_feed_directory),
                                              _("rss_ib_unable_add_feed"));
        hildon_helper_set_insensitive_message(GTK_WIDGET
                                              (app_ui_data->
                                               menuitem_add),
                                              _("rss_ib_unable_add_feed"));
    }


}

void refresh_insens_messages(AppUIData * app_ui_data)
{
    DMSG("Entering %s with %d\n", __FUNCTION__,
         app_data->app_ui_data->search_mode);
    if (SFM_REFRESH == app_ui_data->search_mode) {
        hildon_helper_set_insensitive_message(GTK_WIDGET
                                              (app_ui_data->
                                               menuitem_refresh_feeds),
                                              _
                                              ("rss_ib_refresh_already_started"));
        hildon_helper_set_insensitive_message(GTK_WIDGET
                                              (app_ui_data->refresh_button),
                                              _
                                              ("rss_ib_refresh_already_started"));
    } else if (SFM_NORMAL != app_ui_data->search_mode) {
        hildon_helper_set_insensitive_message(GTK_WIDGET
                                              (app_ui_data->
                                               menuitem_refresh_feeds),
                                              _
                                              ("rss_ib_cannot_start_refresh"));
        hildon_helper_set_insensitive_message(GTK_WIDGET
                                              (app_ui_data->refresh_button),
                                              _
                                              ("rss_ib_cannot_start_refresh"));
    } else if (app_ui_data->low_mem) {
        hildon_helper_set_insensitive_message(GTK_WIDGET
                                              (app_ui_data->
                                               menuitem_refresh_feeds),
                                              dgettext("ke-recv",
                                                       "memr_ib_operation_disabled"));
        hildon_helper_set_insensitive_message(GTK_WIDGET
                                              (app_ui_data->refresh_button),
                                              dgettext("ke-recv",
                                                       "memr_ib_operation_disabled"));
    }
}

void paste_insens_messages(AppUIData * app_ui_data)
{
    GtkClipboard *clipboard = gtk_clipboard_get(GDK_SELECTION_CLIPBOARD);
    if (gtk_clipboard_wait_is_text_available(clipboard)) {
        hildon_helper_set_insensitive_message(GTK_WIDGET
                                              (app_ui_data->menuitem_paste),
                                              _("rss_ib_unable_to_paste"));

    } else {
        hildon_helper_set_insensitive_message(GTK_WIDGET
                                              (app_ui_data->menuitem_paste),
                                              _("rss_ib_nothing_to_paste"));

    }
}

void close_all_opened_dialog(AppData * app_data)
{
    g_assert(app_data);
    AppUIData *app_ui_data = app_data->app_ui_data;
    if (app_ui_data->feed_directory_dialog) {
        gtk_dialog_response(GTK_DIALOG(app_ui_data->feed_directory_dialog),
                            GTK_RESPONSE_CANCEL);
    }

    if (app_ui_data->cfdialog != NULL) {

        if (app_ui_data->nfdialog != NULL) {
            gtk_widget_destroy(GTK_WIDGET(app_ui_data->nfdialog));
            app_ui_data->nfdialog = NULL;
        }

        gtk_widget_destroy(GTK_WIDGET(app_ui_data->cfdialog));
        app_ui_data->cfdialog = NULL;

    }

    if (app_ui_data->mfdialog != NULL) {
        ui_folder_close_rename_dialog();
        if (app_ui_data->nfdialog != NULL) {
            gtk_widget_destroy(GTK_WIDGET(app_ui_data->nfdialog));
            app_ui_data->nfdialog = NULL;
        }

        app_ui_data->dialog = NULL;
        gtk_widget_destroy(GTK_WIDGET(app_ui_data->mfdialog));
        app_ui_data->mfdialog = NULL;
        app_ui_data->manage_folders_displayed = FALSE;

    }

    if (app_ui_data->propdialog != NULL) {

        app_ui_data->dlg_btn_unsubscribe = NULL;
        app_ui_data->dialog = NULL;
        gtk_widget_destroy(GTK_WIDGET(app_ui_data->propdialog));
        app_ui_data->propdialog = NULL;

    }

    if (app_ui_data->add_feed_dialog != NULL)
        newdialog_destroy();


    if (app_ui_data->fsdialog != NULL) {

        gtk_dialog_response(GTK_DIALOG(app_ui_data->fsdialog),
                            GTK_RESPONSE_CANCEL);
        app_ui_data->fsdialog = NULL;

    }
    if (app_ui_data->refresh_dialog != NULL) {
        gtk_widget_destroy(GTK_WIDGET(app_ui_data->refresh_dialog));
        app_ui_data->refresh_dialog = NULL;
        app_ui_data->dialog = NULL;
    }
    settings_dialog_close();
}

/* Updating edit menu items; called from idle */
gboolean ui_update_edit_menu( gpointer user_data )
{
    AppUIData *app_ui_data = (AppUIData *) user_data;
    GtkWidget *focus = NULL;
    GtkClipboard *clipboard = gtk_clipboard_get(GDK_SELECTION_CLIPBOARD);

    g_assert(NULL != user_data);
    g_assert(NULL != clipboard);

    /* first, pretend everything is not click-able */
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->menuitem_cut), FALSE);
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->menuitem_copy), FALSE);
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->menuitem_paste), FALSE);
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->menuitem_selectall),
                             FALSE);

    if (NULL != app_ui_data->dialog)
        focus = gtk_window_get_focus(GTK_WINDOW(app_ui_data->dialog));
    else
        focus = gtk_window_get_focus(GTK_WINDOW(app_ui_data->main_view));

    if (GTK_IS_EDITABLE(focus)) {
        if (gtk_editable_get_selection_bounds
            (GTK_EDITABLE(focus), NULL, NULL)) {
            gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->menuitem_cut),
                                     TRUE);
            gtk_widget_set_sensitive(GTK_WIDGET
                                     (app_ui_data->menuitem_copy), TRUE);
        }
        if (gtk_clipboard_wait_is_text_available(clipboard)) {
            gtk_widget_set_sensitive(GTK_WIDGET
                                     (app_ui_data->menuitem_paste), TRUE);
        }
        gchar *tmp = gtk_editable_get_chars(GTK_EDITABLE(focus), 0, -1);
        if (strlen(tmp) != 0)   //there is something to be select-all'ed
            gtk_widget_set_sensitive(GTK_WIDGET
                                     (app_ui_data->menuitem_selectall), TRUE);
        g_free(tmp);
    }

    /* htmlwidget checking... */
    else {
        ULOG_DEBUG("on_menuitem_edit: gtkeditable is not the focus");
        //if ((focus != feedlist) &&
        if ((GTK_WIDGET(focus) != GTK_WIDGET(feedlist)) &&
            htmlwidget_has_selection(app_data->app_ui_data->html_scrollpane))
        {
            ULOG_DEBUG("on_menuitem_edit: htmlwidget has selection.");
            //can set to COPY sensitive.
            gtk_widget_set_sensitive(GTK_WIDGET
                                     (app_ui_data->menuitem_copy), TRUE);

        }
        //TODO: This will be safe.        
        //gboolean has_focus;
        //g_object_get(G_OBJECT(app_ui_data->htmlwidget), "focus", &has_focus, NULL);

        //not gtkeditable, not feedlist, it should be gtkhtml

        //if ( focus != feedlist)
        if (GTK_WIDGET(focus) != GTK_WIDGET(feedlist))
            gtk_widget_set_sensitive(GTK_WIDGET
                                     (app_ui_data->menuitem_selectall), TRUE);
    }
    paste_insens_messages(app_ui_data);

    return FALSE;
}
