/**
 * @file htmlview.c common interface for browser module implementations
 * and module loading functions
 *
 * Copyright (C) 2003, 2004 Lars Lindner <lars.lindner@gmx.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <sys/stat.h>
#include <string.h>
#include <glib.h>
#include <gmodule.h>

#include <osso-rss-feed-reader/settings.h>

#include "htmlview.h"

#include <ossoemailinterface.h>
#include <osso-log.h>

extern AppData *app_data;

extern char *proxyname;
extern char *proxyusername;
extern char *proxypassword;
extern int proxyport;

GtkWidget *ui_htmlview_new()
{
    return gtkhtml_new();
}

void ui_htmlview_write(GtkWidget * htmlview, gchar * string,
                       gchar * base, nodePtr node)
{
    if (string != NULL) {
        if (!g_utf8_validate((const gchar *) string, -1, NULL)) {

            /* Its really a bug if we get invalid encoded UTF-8 here!!! */
            g_warning("Invalid encoded UTF8 buffer passed to HTML widget!");

            gchar *buffer = g_strdup(string);

            /* to prevent crashes inside the browser */
            buffer = utf8_fix(buffer);
	    gtkhtml_write_html(htmlview, buffer, base, node);
	    g_free(buffer);
	    return;
        }
    }
    gtkhtml_write_html(htmlview, string, base, node);
}

GtkWidget *ui_htmlview_get_buffer()
{
    return NULL;
}

void ui_htmlview_clear(GtkWidget * htmlview)
{
    ui_htmlview_write(htmlview, "", NULL, NULL);
}

gboolean ui_htmlview_scroll()
{
    return gtkhtml_scroll_pagedown(ui_mainwindow_get_active_htmlview());
}

void ui_htmlview_set_proxy(gchar * hostname, int port, gchar * username,
                           gchar * password)
{
    /* unused with gtkhtml */
}

int send_text_via_email(const gchar * post)
{
    g_assert(NULL != post);
    g_assert(app_data != NULL);

    osso_return_t retcode = osso_email_url_email(app_data->osso,
                                                 post);

    /* CHECK: There is no error feedback defined in UI spec. */
    switch (retcode) {
        case OSSO_OK:
            ULOG_DEBUG("osso_email_url_email() returned OK!\n");
            break;
        case OSSO_ERROR:
            ULOG_DEBUG("osso_email_url_email() returned ERROR!\n");
            break;
        case OSSO_INVALID:
            ULOG_DEBUG("osso_email_url_email() returned INVALID!\n");
            break;
        case OSSO_RPC_ERROR:
            ULOG_DEBUG("osso_email_url_email() returned OSSO_RPC_ERROR!\n");
            break;
        default:
            ULOG_DEBUG("osso_email_url_email() returned %d!\n", retcode);
            break;
    }

    return 0;
}

int send_item(itemPtr item, gboolean bluetooth)
{
    gchar *post = NULL;
    gchar *esc = NULL;
    gchar *desc = NULL;

    g_assert(NULL != item);

    g_assert(app_data != NULL);
    g_assert(app_data != NULL);

    if (item->source)
        esc = g_markup_escape_text(item->source, -1);
    else
        esc = g_markup_escape_text(item->title, -1);

    desc = unhtmlize(g_strdup(item->description));

    post = g_strconcat(esc, "\n\n", desc, NULL);

    post = g_strstrip(post);

    if (bluetooth)
        bt_send(item->title, post, strlen(post));
    else
        send_text_via_email(post);

    g_free(post);
    g_free(esc);
    g_free(desc);

    return 0;
}

void ui_htmlview_render_item(itemPtr ip)
{
    gtkhtml_searchview_append_item(ip);
}
