/**
 * @file feed_cache.c cash related feed handling
 * 
 * Copyright (C) 2003, 2004 Lars Lindner <lars.lindner@gmx.net>
 * Copyright (C) 2004 Nathan J. Conrad <t98502@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. 
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "feed_cache.h"

#include <osso-log.h>
#include <unistd.h>
#include <string.h>

#include "common.h"
#include "metadata.h"
#include "favicon_cache.h"
#include <osso-rss-feed-reader/cache_handling.h>

extern gboolean just_empty_items;



rsslib_cache_item_data *cache_write_feed_item(itemPtr ip);

itemPtr cache_read_feed_item(rsslib_cache_item_data * item_data);
/* Function which is called to load a feed's into memory. This function
 * might be called multiple times even if the feed was already loaded.
 * Each time the method is called a reference counter is incremented. */
gboolean feed_load(feedPtr fp)
{
    GList *items = NULL;
    rsslib_cache_item_data *item_data = NULL;


    rsslib_cache_init_data *cache_init_data = NULL;

    g_assert(NULL != fp);
    g_assert(NULL != fp->id);

    if (0 != (fp->loaded)++) {
        return TRUE;
    }

    if (FST_VFOLDER == feed_get_type(fp)) {
        debug0(DEBUG_CACHE, "it's a vfolder, nothing to do...");
        fp->loaded++;
        return TRUE;
    }


    debug1(DEBUG_CACHE, "loading cache file \"%s\"", fp->id);
    ULOG_DEBUG("\n--------------feed_load: loading cache file \"%s\"",
               fp->id);

    fp->unreadCount = 0;
    fp->markCount = 0;
    fp->newCount = 0;
    metadata_list_free(fp->metadata);
    fp->metadata = NULL;

    if ((cache_init_data = rsslib_cache_open_file(fp->id)) == NULL) {
        debug0(DEBUG_CACHE, "A cache file could not be loaded\n");
        return FALSE;           //TRUE?
    }


    /* CHECK: Feed title can be set manually in OPML, so we should override that.
     * However, can this cause problems somewhere? Alternate solution is to
     * save feed cache after title has been changed. */
    if (NULL == feed_get_title(fp)) {
        ULOG_WARN("Feed %s title was somehow NULL, setting %s", fp->id,
                  cache_init_data->feed_title);
        feed_set_title(fp, cache_init_data->feed_title);
    }
    feed_set_source(fp, cache_init_data->feed_source);
    feed_set_description(fp, cache_init_data->feed_description);

    feed_set_image_url(fp, cache_init_data->feed_image_url);
    feed_set_available(fp, cache_init_data->feed_available);
    feed_set_discontinued(fp, cache_init_data->feed_discontinued);

    feed_set_added(fp, cache_init_data->feed_added);

    feed_set_lastmodified(fp, cache_init_data->feed_last_modifed);
    feed_set_newest_post(fp, cache_init_data->feed_newest_post);
    feed_set_feed_read(fp, cache_init_data->feed_read);


    while ((item_data = rsslib_cache_get_next_item(cache_init_data)) != NULL) {
        itemPtr ip = cache_read_feed_item(item_data);
        if (ip != NULL) {
            items = g_list_append(items, ip);
        }
        item_data = NULL;
    }


    feed_add_items(fp, items);
    favicon_load(fp);
    rsslib_cache_init_data_free(cache_init_data);

    debug_exit("feed_load");
    return TRUE;
}

/* Only some feed informations are kept in memory to lower memory
 * usage. This method unloads everything besides necessary infos. 
 * 
 * If the feed parameter is NULL the function is called for all feeds.
 * 
 * Each time this function is called the reference counter of all
 * feeds is decremented and if it zero the unnecessary feed infos are 
 * free'd */
gboolean feed_unload(feedPtr fp)
{
    gint unreadCount = 0;

    g_assert(NULL != fp);
    g_assert(0 <= fp->loaded);  /* could indicate bad loaded reference counting */
    if (0 != fp->loaded) {
        if (feed_save(fp) == FALSE) {   /* save feed before unloading */
            return FALSE;
        }
        if (conf_keep_feeds_in_memory) {
            debug_enter("feed_unload");
            if (1 == fp->loaded) {
                if (FST_FEED == feed_get_type(fp)) {
                    /* free items */
                    unreadCount = fp->unreadCount;
                    feed_clear_item_list(fp);
                    fp->unreadCount = unreadCount;
                } else {
                    debug1(DEBUG_CACHE, "not unloading vfolder (%s)",
                           feed_get_title(fp));
                }
            } else {
                debug2(DEBUG_CACHE,
                       "not unloading (%s) because it's used (%d references)...",
                       feed_get_source(fp), fp->loaded);
            }
            fp->loaded--;
            debug_exit("feed_unload");
        }
    }
    return TRUE;
}


/*
 * Feeds caches are marked to be saved at a few different places:
 * (1) Inside whe feed_set_* functions where an item is marked or made read or unread
 * (2) Inside of feed_process_result
 * (3) The callback where items are removed from the itemlist
 */
gboolean feed_save(feedPtr fp)
{
    rsslib_cache_init_data *save_data = NULL;
    GSList *itemlist = NULL;

    itemPtr ip = NULL;
    gint saveCount = 0;
    gint saveMaxCount = 0;

    if (fp->needsCacheSave == FALSE) {
        return TRUE;
    }

    save_data = g_new0(rsslib_cache_init_data, 1);

    debug1(DEBUG_CACHE, "saving feed: %s", fp->title);
    g_assert(0 != fp->loaded);

    saveMaxCount = fp->cacheLimit;
    if (saveMaxCount == CACHE_DEFAULT)
        saveMaxCount = DEFAULT_MAX_ITEMS;

    save_data->saveMaxCount = saveMaxCount;

    save_data->id = fp->id;
    save_data->feed_title = g_strdup(feed_get_title(fp));
    save_data->feed_source = g_strdup(feed_get_source(fp));
    save_data->feed_description = g_strdup(fp->description);
    save_data->feed_image_url = g_strdup(feed_get_image_url(fp));
    save_data->feed_available = feed_get_available(fp);
    save_data->feed_discontinued = feed_get_discontinued(fp);
    save_data->feed_added = feed_get_added(fp);
    save_data->feed_last_modifed = feed_get_lastmodified(fp);
    save_data->feed_newest_post = feed_get_newest_post(fp);
    save_data->feed_read = feed_get_feed_read(fp);


    if (!just_empty_items) {
        itemlist = feed_get_item_list(fp);
        for (itemlist = feed_get_item_list(fp); itemlist != NULL;
             itemlist = g_slist_next(itemlist)) {
            ip = itemlist->data;
            g_assert(NULL != ip);

            if (saveMaxCount != CACHE_UNLIMITED &&
                saveCount >= saveMaxCount &&
                (fp->fhp == NULL || fp->fhp->directory == FALSE) &&
                !item_get_mark(ip)) {

                continue;
            }

            save_data->feed_items =
                g_slist_append(save_data->feed_items,
                               cache_write_feed_item(ip));
            saveCount++;
        }
    }

    fp->needsCacheSave = FALSE;

    gboolean result = rsslib_cache_save_feed(save_data, just_empty_items);
    rsslib_cache_init_data_free(save_data);
    return result;
}


rsslib_cache_item_data *cache_write_feed_item(itemPtr ip)
{
    GSList *old = NULL;
    GSList *new = NULL;
    rsslib_cache_item_data *item_data = NULL;
    item_data = g_new0(rsslib_cache_item_data, 1);
    item_data->item_title = g_strdup(item_get_title(ip));
    item_data->item_description = g_strdup(item_get_description(ip));
    item_data->item_source = g_strdup(item_get_source(ip));
    item_data->item_real_source_title =
        g_strdup(item_get_real_source_title(ip));
    item_data->item_real_source_url = g_strdup(item_get_real_source_url(ip));
    item_data->item_id = g_strdup(item_get_id(ip));
    item_data->nr = ip->nr;
    item_data->item_read_status = item_get_read_status(ip);
    item_data->marked = item_get_mark(ip);
    item_data->item_time = item_get_time(ip);
    // item_data->enclosure_metadata=metadata_list_get(ip->metadata,"enclosure");
    old = metadata_list_get(ip->metadata, "enclosure");
    while (old != NULL) {
        struct enclosure_attribute *old_enclosure =
            (struct enclosure_attribute *) old->data;
        struct enclosure_attribute *new_enclosure =
            g_new0(struct enclosure_attribute, 1);
        if (new_enclosure != NULL) {
            if (old_enclosure->type != NULL) {
                new_enclosure->type = g_strdup(old_enclosure->type);
            }
            if (old_enclosure->ul != NULL) {
                new_enclosure->ul = g_strdup(old_enclosure->ul);
            }
            new_enclosure->length = old_enclosure->length;
            new = g_slist_append(new, new_enclosure);
        }
        old = g_slist_next(old);
    }
    item_data->enclosure_metadata = new;

    item_data->on_server = ip->on_server;
    return item_data;
}


itemPtr cache_read_feed_item(rsslib_cache_item_data * item_data)
{
    itemPtr ip = item_new();
    GSList *encl_list = NULL;
    ip->newStatus = FALSE;


    item_set_title(ip, item_data->item_title);  /* can be null */

    item_set_description(ip, item_data->item_description);  /* can be null */
    item_set_source(ip, item_data->item_source);    /* can be null */
    item_set_real_source_title(ip, item_data->item_real_source_title);  /* can be null */
    item_set_real_source_url(ip, item_data->item_real_source_url);  /* can be null */
    item_set_id(ip, item_data->item_id);    /* can be null */
    ip->nr = item_data->nr;

    item_set_read_status(ip, item_data->item_read_status);

    /* we don't call item_set_mark here because it would
     * update the UI 
     item_set_mark(ip,b);*/
    ip->marked = item_data->marked;
    item_set_time(ip, item_data->item_time);

    if (item_data->enclosure_metadata) {
        encl_list = item_data->enclosure_metadata;
        struct enclosure_attribute *attrib;
        while (encl_list) {
            attrib = encl_list->data;
            ip->metadata =
                metadata_list_append_enclosure_attr(ip->metadata, "enclosure",
                                                    attrib);
            encl_list = g_slist_next(encl_list);

        }
        g_slist_free(item_data->enclosure_metadata);
        item_data->enclosure_metadata = NULL;
    }


    ip->on_server = item_data->on_server;

    rsslib_cache_item_data_free(item_data);

    return ip;
}
