/**
	@file settings_database.c

	Database stores the values based on configuration keys.
	
	This version is the multi-instance version (synchronizes data
	through GConf).
	
	Copyright (c) 2004-2007 Nokia Corporation.
	Parts of code are from Liferea.
	Liferea (C) 2003,2004 Lars Lindner <lars.lindner@gmx.net>

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License 
	version 2 or later, as published by the Free Software Foundation. 

	This program is distributed in the hope that it will be useful, but
	WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
	General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
	02110-1301 USA
*/

#include <glib.h>
#include <osso-log.h>
#include <gconf/gconf-client.h>

#include <assert.h>
#include <string.h>

#include "settings_database.h"
#include "interface.h"

/* Local interface */

/**
	Removes value from the database.	
	
	@param key	 Key of the key/value-pair
	@param value	 Value of the key/value-pair
	@param user_data Always NULL
	@return Always TRUE
*/
static guint _remove_value(gpointer key,
                          gpointer value,
                          gpointer user_data);

/**
	A function that sets the value (for a key) in local database.
	In addition for this, value has also be set in GConf and this
	is done by _set_gconf. setting_set_value calls this function.
	
	@param key	A key, for which the value is saved.
	@param value	The value
	@return TRUE, if setting was succesful
*/
static gboolean _set_local(const gchar* key, 
			   Value* value);
			   
/**
	A function that sets the value (for a key) in GConf state
	store. In addition for this, value has also be set in 
	local database, which is done by _set_local.
	setting_set_value calls this function.
	
	@param key	A key, for which the value is saved.
	@param value	The value
	@param TRUE, if setting was succesful
*/
static gboolean _set_gconf(const gchar* key, 
			   Value* value);

/**
	Gets a SettingsDatabase (local) value according to a given key.
	
	@param key	The key, for which the value is returned.
	@return value corresponding the given key.
*/
static Value* _get_local(const gchar* key);

/**
	Gets a GConf (remote) value according to a given key.
	
	@param key	The key, for which the value is returned.
	@return value corresponding the given key.
*/
static Value* _get_gconf(const gchar* key);

/*
	Gets GConf-key for a given key.
	
	@param key	Key in SettingsDatabase.
	@return GConf-key.

	static gchar* _get_gconf_key(const gchar* key);*/

/**
	Gets a local key for a given GConf-key.
	
	@param gconf_key	Key in GConf.
	@return local key for given GConf key, or NULL, if error occurs.
*/
static gchar* _get_local_key(const gchar* gconf_key);

/**
	Removes SettingsDatabase-related GConf values.	
*/
static void _remove_gconf_values(void);

/**
	A function to be used with many foreach-functions.
	Frees the given data.
	
	@param data		Data to be freed.
	@param user_data	Not used.
*/
static void _free_data(gpointer data, gpointer user_data);
      
/**
	Callback to handle a change in GConf.
	
	@param client	
	@param cnxn_id	
	@param entry	
	@param user_data	
*/
static void _change_in_gconf(GConfClient *client,
			     guint cnxn_id,
			     GConfEntry *entry,
			     gpointer user_data);
			     
/**
	Transfers value from GConfValue to SettingsDatabase Value.
	
	@param gconf_value	Source GConfValue.
	@param value	Destination SettingsDatabaseValue.
	
	@return TRUE, if value was transferred.
*/
static gboolean _transfer_value(GConfValue* gconf_value, Value* value);

/**
	Refreshes the UI. Used as a timeout.
	
	@param data	NULL
	
	@return FALSE
*/
static gboolean _refresh_ui(gpointer data);

/* implementation */

static GHashTable* database = NULL;
static GList* keylist = NULL;
static guint gconf_notify = 0;
static GConfClient *gclient = NULL;
static guint event_source_id = 0;

static guint _remove_value(gpointer key,
                          gpointer value,
                          gpointer user_data)
{
	
	/* String has to be freed exliciptly */
	if (value != NULL && ((Value*)value)->type == TYPE_STR) 
	{
		g_free(((Value*)value)->str);
	}
	
	/* Free key and value */
	g_free(key);
	g_free(value);
	
	return TRUE;
}


gboolean settings_init(gboolean* state_exists)
{
	Value value;
	Value* retvalue = NULL;
	/* Init database */
	database = NULL;
	keylist = NULL;
	gconf_notify = 0;
	gclient = NULL;
	event_source_id = 0;
	
	gclient = gconf_client_get_default();
	if (gclient == NULL)
	{
		ULOG_ERR("Unable to connect GConf.");
		return FALSE;
	}
	
	/* Clear GConfClient's cache */
	/*
	gconf_client_clear_cache(gclient);
	*/
	
	/* Create local database */
	database = g_hash_table_new(g_str_hash, g_str_equal);
	keylist = NULL;
	
	/* Check state */
	if (state_exists != NULL)
	{
		
		/* Get if state exists from the GConf */
		*state_exists = FALSE;
		retvalue = _get_gconf(ACTIVATION_KEY);
		
		if (retvalue != NULL && retvalue->type == TYPE_BOOLEAN)
		{
			*state_exists = retvalue->boolean;
		}
	} 
	
	value.type = TYPE_BOOLEAN;
	value.boolean = TRUE;
	_set_gconf(ACTIVATION_KEY, &value);
	
	/* Add notifications to KEY_BASE */
	gconf_client_add_dir(gclient, 
			     KEY_BASE,
			     GCONF_CLIENT_PRELOAD_NONE,
			     NULL);
	gconf_notify = gconf_client_notify_add(gclient, KEY_BASE,
					       _change_in_gconf,
						NULL, NULL, NULL);
	
	if (gconf_notify == 0) {
		ULOG_ERR("Unable to register GConf notifications");
		gconf_client_remove_dir(gclient, 
					KEY_BASE,
					NULL);
		
		g_object_unref(gclient);
		return FALSE;		
	}
	
	return TRUE;
}


gboolean settings_set_value(const gchar* key, Value* value) 
{
	gboolean ret = TRUE;

	if (g_hash_table_lookup(database, key) != NULL)
	{
		value->changed = TRUE;
	} else {
		value->changed = FALSE;
	}
	
	/* Set value & key locally */
	ret = _set_local(key, value) & ret;
	
	if (ret == TRUE)
	{
		/* Set value to GConf */
		ret = _set_gconf(key, value) & ret;
		#ifdef GCONFDOESNTFAIL
		ret = TRUE;
		#endif
	}
	
	return ret;
}


static gboolean _set_gconf(const gchar* key, Value* value)
{
	gboolean ret = TRUE;
	gchar* gconf_key = NULL;
	
	assert(gclient != NULL);
	
	/* Set value in GConf */
	gconf_key = _get_gconf_key(key);
		
	/* Set value in GConf and assume it is the same type as
	   last time */
	gconf_client_remove_dir(gclient, KEY_BASE, NULL);
	switch(value->type)
	{ 
		case TYPE_INTEGER:
			ret = gconf_client_set_int(gclient,
						   gconf_key,
						   value->integer,
						   NULL) & ret;
			break;
		case TYPE_BOOLEAN:
			ret = gconf_client_set_bool(gclient,
						    gconf_key,
						    value->boolean,
						    NULL) & ret;
			break;
		case TYPE_STR:
			ret = gconf_client_set_string(gclient,
						      gconf_key,
						      value->str,
						      NULL) & ret;
			break;
	}
	gconf_client_add_dir(gclient, KEY_BASE,
			     GCONF_CLIENT_PRELOAD_NONE,
			     NULL);
	
	g_free(gconf_key);
	
	return ret;
}


gboolean _set_local(const gchar* key, Value* value) 
{	
	gpointer add_value;
	gpointer add_key;
	
	add_value = NULL;
	add_key = NULL;
	
	/* Test to see, if database already has the key, value pair */
	if (g_hash_table_lookup_extended(database, key, &add_key, 
		                         &add_value)) 
	{
		if (add_value != NULL && ((Value*)add_value)->changed)
			value->changed = TRUE;
		
		/* remove the entry in the hash table */
		g_hash_table_remove(database, add_key);

		/* just free the key and value */
		_remove_value(add_key, add_value, NULL);		
	}
	

	/* Add new value (and key) */
	add_value = g_memdup((gconstpointer)value, sizeof(Value));
	add_key = g_strdup(key);
	if (value->type == TYPE_STR && add_value != NULL) 
	{
		/* If value type is string, string must also be duplicated */
		((Value*)add_value)->str = g_strdup(value->str);
		
		if (((Value*)add_value)->str == NULL) 
		{
			ULOG_ERR("Error on allocating string for a value.");
		}
	}
	
	/* Log errors and exit on error */
	if (add_value == NULL) 
	{
		if (add_key) {
			g_free(add_key);
		}
		ULOG_ERR("Allocating memory for new value failed.");
		return FALSE;
	}
	
	if (add_key == NULL) 
	{
		g_free(add_value);
		ULOG_ERR("Allocating memory for new key failed.");
		return FALSE;
	}
	
	/* Insert value to database */
	g_hash_table_insert(database, add_key, add_value);
	
	return TRUE;
}
 

Value* settings_get_value(const gchar* key) 
{
	Value* value = _get_local(key);
	
	if (value == NULL)
	{
		value = _get_gconf(key);
	}
	
	return value;
}


Value* _get_local(const gchar* key)
{
	Value* value = NULL;

	/* Get value from database */
	value = (Value*)g_hash_table_lookup(database, key);
	
	return value;
}


Value* _get_gconf(const gchar* key)
{
	Value value;
	gchar* gconf_key = NULL;
	GConfValue* gconf_value;
	
	/* Get GConf key according to a local key */
	gconf_key = _get_gconf_key(key);
	
	/* Get value from GConf */
	gconf_value = gconf_client_get(gclient, gconf_key, NULL);
	g_free(gconf_key); gconf_key = NULL;
		
	if (gconf_value == NULL) 
	{
		return NULL;
	}
	
	/* Convert value to SettingsDatabase value */
	if (!_transfer_value(gconf_value, &value))
	{
		gconf_value_free(gconf_value);
		return NULL;
	}
	gconf_value_free(gconf_value);
	
	/* Cache got value */
	if (!_set_local(key, &value)) {
		return NULL;
	}
	
	/* Free string value */
	if (value.type == TYPE_STR) g_free(value.str);
	
	return _get_local(key);
}


static gboolean _transfer_value(GConfValue* gconf_value, Value* value)
{
	switch(gconf_value->type)
	{
		case GCONF_VALUE_INT:
			value->type = TYPE_INTEGER;
			value->integer = 
				gconf_value_get_int(gconf_value);
			break;
		case GCONF_VALUE_BOOL:
			value->type = TYPE_BOOLEAN;
			value->boolean = 
				gconf_value_get_bool(gconf_value);
			break;
		case GCONF_VALUE_STRING:
			value->type = TYPE_STR;
			value->str = g_strdup(
				gconf_value_get_string(gconf_value));
			break;
		default:
			return FALSE;
	}
	
	/* Switch changed to FALSE in order to make sure */
	value->changed = FALSE;
	
	return TRUE;
}


gboolean settings_destroy(void)
{
	Value value;

	/* If GConfClient doesn't exist, so don't we */
	if (gclient == NULL) return FALSE;
	
	if (event_source_id > 0) {
		g_source_remove(event_source_id);
	}

	/* Remove notifications */
	if (gconf_notify != 0)
	{
		gconf_client_notify_remove(gclient, gconf_notify);
		gconf_client_remove_dir(gclient, KEY_BASE, NULL);
	}
	
	/* Clean up GConf, set active and unref GconfClient */
	_remove_gconf_values();
	
	value.type = TYPE_BOOLEAN;
	value.boolean = FALSE;
	_set_gconf(ACTIVATION_KEY, &value);
	
	/* Clear GConfClient's cache, synchronize and unref */
	gconf_client_suggest_sync(gclient, NULL);
	g_object_unref(gclient); gclient = NULL;
	
	/* Delete keys, values and database */
	g_list_foreach(keylist, _free_data, NULL);
	g_list_free(keylist); keylist = NULL;
	g_hash_table_foreach_remove(database, (GHRFunc)_remove_value, NULL);
	g_hash_table_destroy(database); database = NULL;
	return TRUE;
}


static gchar* _get_local_key(const gchar* gconf_key)
{
	if (!g_str_has_prefix(gconf_key, KEY_BASE)) return NULL;
		
	gconf_key += strlen(KEY_BASE) + 1;
	
	return g_strdup(gconf_key);
}


static void _remove_gconf_values(void)
{
	GSList* gconf_data = NULL;
	GSList* item = NULL;
	
	gpointer gconf_key_orig = NULL;
	gpointer gconf_entry_orig = NULL;
	
	/* Iterate over keys in GConf and remove */
	
	/* Remove actual data */
	gconf_data = gconf_client_all_entries(gclient, KEY_BASE, NULL);
	item = gconf_data;
	while (item != NULL)
	{
		/* Unset key in GConf */
		gconf_client_unset(gclient, 
				   ((GConfEntry*)item->data)->key,
				   NULL);
		
		/* Remove value from GConf's cache */
		if (g_hash_table_lookup_extended(gclient->cache_hash,
					 ((GConfEntry*)item->data)->key,
					 &gconf_key_orig,
      					 &gconf_entry_orig))
		{
			g_hash_table_remove(gclient->cache_hash,
					    ((GConfEntry*)item->data)->key);
			/* gconf_key_orig must not be freed or 
			   gconf_entry_free hangs! */
			/*
			gconf_entry_free((GConfEntry*)gconf_entry_orig);
			*/
		}
		
		gconf_entry_free((GConfEntry*)item->data);
		item = item->next;
	}
        
        
	g_slist_free(gconf_data);	

}


static void _change_in_gconf(GConfClient *client,
			     guint cnxn_id,
			     GConfEntry *entry,
			     gpointer user_data)
{
	Value db_value;
	GConfValue* gval = NULL;
	gchar* local_key = NULL;
	
	memset(&db_value, 0, sizeof(Value));
	
	/* Cache value to local values */
	
	g_return_if_fail(client);
	g_return_if_fail(entry);
	
	gval = gconf_entry_get_value(entry);
	if (gval == NULL) return;
		
	local_key = _get_local_key(gconf_entry_get_key(entry));
	
	/* Convert value to SettingsDatabase value */
	if (!_transfer_value(gval, &db_value))
	{
		g_free(local_key); local_key = NULL;
		return;
	}
	
	/* Cache got value */
	if (!_set_local(local_key, &db_value)) {
		g_free(local_key); local_key = NULL;
		return;
	}
	g_free(local_key); local_key = NULL;
	
	/* Free string value */
	if (db_value.type == TYPE_STR) g_free(db_value.str);
	
	/* Check if activation of applet has been put to FALSE and quit the 
	    applet dialog, if this is the case */
	if (g_strstr_len(gconf_entry_get_key(entry),
              strlen(gconf_entry_get_key(entry)), ACTIVATION_KEY) != NULL && 
	      gconf_value_get_bool(gval) == FALSE)
	{
		
//		ui_interface_kill_softly_only = TRUE;
		ui_destroy();
//		ui_interface_kill_softly_only = FALSE; // start_app call will kill all
		
	} else {
		/* Value in GConf was changed, and we have no way of knowing, if
		   we changed it ourselves, so we install a timeout handler to
		   make UI refresh, if handler isn't already installed */
		if (event_source_id == 0)
		{
			event_source_id = g_timeout_add(REFRESH_INTERVAL,
							_refresh_ui,
							NULL);
		}
	}
}


static void _free_data(gpointer data, gpointer user_data)
{
	g_free(data);
}


static gboolean _refresh_ui(gpointer data)
{
	event_source_id = 0;
	ui_refresh();
	return FALSE;
}
