/**
    @file cache_handling.c

    Copyright (c) 2004-2005 Nokia. All rights reserved.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
    02111-1307 USA
*/

#include <sys/stat.h>

#include "cache_handling.h"
#include "osso-log.h"

static gchar *lifereaUserPath = NULL;





void initCachePath(void);

gboolean rsslib_cache_write_item(rsslib_cache_init_data *save_data,rsslib_cache_item_data *item_data);

gchar *cache_read_str(FILE * cache)
{
    int len = 0;
    fread(&len, sizeof(len), 1, cache);    /* read length */

    if (feof(cache)) {
    return NULL;
    }

    if (len == 0) {
    return NULL;
    }
    
    gchar *string = g_malloc(len + 1);
    fread(string, len, 1, cache);
    string[len] = 0;
    return string;
}

int cache_read_int(FILE * cache)
{
    int data = 0;
    fread(&data, sizeof(data), 1, cache);
    return data;
}

long cache_read_long(FILE * cache)
{
    long data = 0;
    fread(&data, sizeof(data), 1, cache);
    return data;
}

gboolean cache_read_boolean(FILE * cache)
{
    char c = 0;
    fread(&c, sizeof(c), 1, cache);

    if (c == 0)
    return FALSE;

    return TRUE;
}

gboolean cache_write_str(FILE * cache, const gchar * string)
{
    if (string == NULL) {
        int i = 0;
        if (fwrite(&i, sizeof(i), 1, cache)!=1)      /* write just a zero */
        {
          return TRUE;
        } else {
          return FALSE;
        }
    }

    int len = strlen(string);
    if (fwrite(&len, sizeof(len), 1, cache)!=1)    /* write the size of string */
    {
      return TRUE;
    }
    if (fwrite(string, len, 1, cache)!=1)    /* trailing zero is not written */
    {
      return TRUE;
    }
    return FALSE;
}

gboolean cache_write_int(FILE * cache, int data)
{
    if (fwrite(&data, sizeof(data), 1, cache)!=1)
    {
      return TRUE;
    }
  return FALSE;
}

gboolean cache_write_long(FILE * cache, long data)
{
    if (fwrite(&data, sizeof(data), 1, cache)!=1)
    {
      return TRUE;
    }
    return FALSE;
}

gboolean cache_write_boolean(FILE * cache, gboolean data)
{
    char c = (data ? 1 : 0);

    if (fwrite(&c, sizeof(c), 1, cache)!=1)
    {
      return TRUE;
    }
    return FALSE;
}


void rsslib_cache_item_data_free(rsslib_cache_item_data *item)
{
    if (item)
    {
        if (item->item_title)
            g_free(item->item_title);
        if (item->item_description)
            g_free(item->item_description);
        if (item->item_source)
            g_free(item->item_source);
        if (item->item_real_source_title)
            g_free(item->item_real_source_title);
        if (item->item_real_source_url)
            g_free(item->item_real_source_url);
        if (item->item_id)
            g_free(item->item_id);
        if (item->enclosure_metadata)
            g_slist_free(item->enclosure_metadata);
        g_free(item);
    }
}

void rsslib_cache_init_data_free(rsslib_cache_init_data *init_data)
{
    if (init_data)
    {
        if (init_data->feed_title)
            g_free(init_data->feed_title);
        if (init_data->feed_source)
            g_free (init_data->feed_source);
        if (init_data->feed_description)
            g_free(init_data->feed_description);
        if (init_data->feed_image_url)
            g_free(init_data->feed_image_url);
        if (init_data->cache) {
            fclose(init_data->cache);
            init_data->cache = NULL;
        }

        if (init_data->feed_items) {
	        GSList *item = init_data->feed_items;
                for (; item; item = item->next)
            		rsslib_cache_item_data_free(item->data);
    	    g_slist_free(init_data->feed_items);
    	}
        g_free(init_data);
    }
}

rsslib_cache_item_data *rsslib_cache_get_next_item(rsslib_cache_init_data *init_data)
{
    FILE *cache=NULL;
    gchar *str = NULL;
    gint count_enclosure=0, length=0;
    gint pos=0;
    struct enclosure_attribute *attrib=NULL;
    rsslib_cache_item_data *item_data=NULL;
   
    if (init_data == NULL) {
       return NULL;
    } 

    cache=init_data->cache;
    if (!cache)
    {
        return NULL;
    }
    if (feof(cache))
    {
        fclose(cache);
        init_data->cache=NULL;
        return NULL;
    }
    
    item_data=g_new0(rsslib_cache_item_data,1);
    

    if (init_data->ver > 0)
        pos = cache_read_long(cache);

    READSTR; item_data->item_title= str;    /* can be null */
    if (feof(cache)) {
        fclose(cache);
        init_data->cache=NULL;
        g_free(str);
        g_free(item_data);
        return NULL;
    }

    READSTR; item_data->item_description=str;    /* can be null */
    READSTR; item_data->item_source=str;    /* can be null */
    READSTR; item_data->item_real_source_title=str;    /* can be null */
    READSTR; item_data->item_real_source_url=str;    /* can be null */
    READSTR; item_data->item_id=str;    /* can be null */
    item_data->nr = cache_read_long(cache);

    gboolean b;

    b = cache_read_boolean(cache);
    item_data->item_read_status=b;

    b = cache_read_boolean(cache);
    item_data->marked = b;
    //ip->delete_count=cache_read_int(cache);
    item_data->item_time=cache_read_long(cache);

    if (init_data->ver >= VERSION_2) {
        /* Load the enclosure datas too, if it has some*/
        count_enclosure=cache_read_int(cache);

        while (count_enclosure)
        {
            attrib=g_new0(struct enclosure_attribute,1);
            READSTR; attrib->ul=str;
            str=NULL; READSTR;
            if (str)
            {
                attrib->type=str;
                str=NULL;
            }
            length=cache_read_int(cache);
            attrib->length=length;
            item_data->enclosure_metadata=g_slist_append(item_data->enclosure_metadata,attrib);
                length=0;
            count_enclosure--;
        }
    }

    /* if supported, restore the on_server flag from the cache */
    if (init_data->ver >= VERSION_3) {
	    gboolean on_server = cache_read_boolean(cache);
	    item_data->on_server = on_server;
    } else {
	    item_data->on_server = TRUE;
    }
    
    if (init_data->ver && init_data->pos < init_data->size)
        fseek(cache, pos, SEEK_SET);

    return item_data;
    
    
}

static time_t rsslib_cache_strdate_to_epoch(const gchar *strdate)
{
	time_t result_epoch=0;
	if (strdate)
	{
		gchar **str_tokens=NULL;
		//the lastmodified feed date was stored in a strin like this: "Wed, 13 Jun 2007 14:14:07 GMT"
		str_tokens=g_strsplit(strdate,", ",2);
		if (str_tokens && str_tokens[1])
		{
			struct tm mod_date;
			strptime(str_tokens[1],"%d %h %Y %X %Z",&mod_date);
			result_epoch=mktime(&mod_date);
			g_message("The mod time: %d %s",(gint)result_epoch,strdate);
			g_strfreev(str_tokens);
		}
	}
	return result_epoch;
}

rsslib_cache_init_data *rsslib_cache_open_file(gchar *fname)
{
    gchar *filename = NULL;
    gchar *str=NULL;
    rsslib_cache_init_data *cache_init_data=NULL;
    FILE *cache;
    gint tmp;
    
    
    cache=NULL;
    filename =
        common_create_cache_filename("cache" G_DIR_SEPARATOR_S "feeds",
                     fname, NULL);
    cache = fopen(filename, "rb");
    if (cache == NULL) {
        g_free(filename);
        return NULL;
    }
    cache_init_data=g_new0(rsslib_cache_init_data,1);
    cache_init_data->cache=cache;
    fseek(cache, 0, SEEK_END);
    cache_init_data->size = ftell(cache);
    fseek(cache, 0, SEEK_SET);

    tmp = cache_read_long(cache);
    if (tmp == C_MAGICHEAD) {
        cache_init_data->ver = cache_read_long(cache);
        cache_init_data->pos = cache_read_long(cache);
    } else {
        fseek(cache, 0, SEEK_SET);
    }
    
    
    
    
    
    
    READSTR;
    /* CHECK: Feed title can be set manually in OPML, so we should override that.
    * However, can this cause problems somewhere? Alternate solution is to
    * save feed cache after title has been changed. */
    cache_init_data->feed_title=str;
    READSTR; cache_init_data->feed_source=str;
    READSTR; cache_init_data->feed_description = str;
    /* no need to free this as it's not duplicated
     * - is there a reason we don't use feed_set_description? */
    

    READSTR; cache_init_data->feed_image_url=str;
    cache_init_data->feed_available=cache_read_boolean(cache);
    cache_init_data->feed_discontinued=cache_read_boolean(cache);
    
    cache_init_data->feed_added=cache_read_long(cache);
	
	if (cache_init_data->ver<=VERSION_3)
	{
    	READSTR;
		cache_init_data->feed_last_modifed=rsslib_cache_strdate_to_epoch(str);
	}
	else
	{
		tmp=cache_read_long(cache);
		cache_init_data->feed_last_modifed=tmp;
	}
    cache_init_data->feed_newest_post=cache_read_long(cache);
    cache_init_data->feed_read=cache_read_long(cache);

    if (cache_init_data->ver && cache_init_data->pos < cache_init_data->size)
      fseek(cache, cache_init_data->pos, SEEK_SET);

    
    
    
    
    
    g_free(filename);
    return cache_init_data;
}

gboolean rsslib_cache_write_item(rsslib_cache_init_data *save_data,rsslib_cache_item_data *item_data)
{
    GSList *enclosures=NULL;
    gint pos, tmp;
    FILE *cache;
    cache=save_data->cache;
    pos = ftell(cache);
    if (cache_write_long(cache, 0))
      return TRUE;
    if (cache_write_str(cache, item_data->item_title))    /* can be null */
      return TRUE;
    if (cache_write_str(cache, item_data->item_description))    /* can be null */
      return TRUE;
    if (cache_write_str(cache, item_data->item_source))    /* can be null */
      return TRUE;
    if (cache_write_str(cache, item_data->item_real_source_title))    /* can be null */
      return TRUE;
    if (cache_write_str(cache, item_data->item_real_source_url))    /* can be null */
      return TRUE;
    if (cache_write_str(cache, item_data->item_id))    /* can be null */
      return TRUE;
    if (cache_write_long(cache, item_data->nr))
      return TRUE;
    if (cache_write_boolean(cache, item_data->item_read_status))
      return TRUE;
    if (cache_write_boolean(cache, item_data->marked))
      return TRUE;
    if (cache_write_long(cache, item_data->item_time))
      return TRUE;

    /* Save the enclosure datas too*/
    enclosures=item_data->enclosure_metadata;
    /*Write out how long is the enclosure list*/
    if (enclosures)
    {
        int foobar = g_slist_length(enclosures);
        //enclosures=enclosures->data;
        if (cache_write_int(cache,foobar)) {
            return TRUE;
        }
    }
    else
    {
        if (cache_write_int(cache, 0))
          return TRUE;
    }
    /* Write the enclosures by one by one */
    while(enclosures)
    {
        struct enclosure_attribute *current_enclosure=(struct enclosure_attribute *)enclosures->data;
        /*Write the URL*/
        if (cache_write_str(cache,current_enclosure->ul))
          return TRUE;

        /*Write the type*/
        if (cache_write_str(cache,current_enclosure->type))
          return TRUE;

        /*Write the length*/
        if (cache_write_int(cache,current_enclosure->length))
          return TRUE;

        enclosures=g_slist_next(enclosures);
    }

    /* save the on_server flag */
    cache_write_boolean(cache, item_data->on_server);
    
    tmp = ftell(cache);
    fseek(cache, pos, SEEK_SET);
    if (cache_write_long(cache, tmp))
      return TRUE;
    fseek(cache, tmp, SEEK_SET);
    return FALSE;
}

gboolean rsslib_cache_save_feed(rsslib_cache_init_data *save_data,gboolean just_empty_items)
{
    GSList *itemlist = NULL;
    gchar *filename = NULL;
    gchar *filename_temp = NULL;
  
    rsslib_cache_item_data *item_data = NULL;
    gint saveCount = 0;
    gint pos, tmp;

    filename =
    common_create_cache_filename("cache" G_DIR_SEPARATOR_S "feeds",
                     save_data->id, NULL);
    if (filename != NULL) {
        filename_temp=g_strdup_printf("%s~",filename);
    } else {
        return FALSE;
    }
   
    FILE *cache = fopen(filename_temp, "wb");
    if (!cache) {
        g_free(filename_temp);
        g_free(filename);
        perror("no space on device");
        return FALSE;
    }
    
    save_data->cache=cache;
    if (cache_write_long(cache, C_MAGICHEAD))
      RETURN_IF_ERROR
      
    if (cache_write_long(cache, C_VERSION))
      RETURN_IF_ERROR
    pos = ftell(cache);
    if (cache_write_long(cache, 0))
      RETURN_IF_ERROR
    if (cache_write_str(cache, save_data->feed_title))
      RETURN_IF_ERROR
    if (cache_write_str(cache, save_data->feed_source))
      RETURN_IF_ERROR
    if (cache_write_str(cache, save_data->feed_description))    //can be null
      RETURN_IF_ERROR
    if (cache_write_str(cache, save_data->feed_image_url))    //can be null
      RETURN_IF_ERROR
    if (cache_write_boolean(cache, save_data->feed_available))
      RETURN_IF_ERROR
    if (just_empty_items) {
        if (cache_write_boolean(cache, FALSE))
          RETURN_IF_ERROR
        if (cache_write_long(cache, save_data->feed_added))
          RETURN_IF_ERROR
        if (cache_write_str(cache, 0))
          RETURN_IF_ERROR
        if (cache_write_long(cache, 0))
          RETURN_IF_ERROR
        if (cache_write_long(cache, 0))
          RETURN_IF_ERROR
    } else {
        if (cache_write_boolean(cache, save_data->feed_discontinued))
          RETURN_IF_ERROR
        if (cache_write_long(cache, save_data->feed_added))
          RETURN_IF_ERROR
        if (cache_write_long(cache, save_data->feed_last_modifed))    //can be null
          RETURN_IF_ERROR
        if (cache_write_long(cache, save_data->feed_newest_post))
          RETURN_IF_ERROR
        if (cache_write_long(cache, save_data->feed_read))
          RETURN_IF_ERROR
    }

    tmp = ftell(cache);
    fseek(cache, pos, SEEK_SET);
    if (cache_write_long(cache, tmp))
      RETURN_IF_ERROR
    fseek(cache, tmp, SEEK_SET);

    if (!just_empty_items) {
        for (itemlist = save_data->feed_items; itemlist != NULL; itemlist = g_slist_next(itemlist)) {
            item_data = itemlist->data;
            g_assert(NULL != item_data);

            if (save_data->saveMaxCount != CACHE_UNLIMITED &&
                saveCount >= save_data->saveMaxCount &&
                (save_data->fhp_null|| save_data->is_directory == FALSE) &&
                !item_data->marked) {
                continue;
            }

            if (rsslib_cache_write_item(save_data,item_data)) 
              RETURN_IF_ERROR
            saveCount++;
        }
    }

    fclose(cache);
    save_data->cache = NULL;
    if (rename(filename_temp,filename)<0)
    {
      g_free(filename);
      g_free(filename_temp);
      return FALSE;
    }
    g_free(filename);
    g_free(filename_temp);
    
    return TRUE;

}

void initCachePath(void)
{
    gchar *cachePath = NULL;
    gchar *feedCachePath = NULL;
    gchar *faviconCachePath = NULL;

    lifereaUserPath =
	g_strdup_printf("%s" G_DIR_SEPARATOR_S ".osso_rss_feed_reader", g_get_home_dir());
    if (!g_file_test(lifereaUserPath, G_FILE_TEST_IS_DIR)) {
    	if (0 != mkdir(lifereaUserPath, S_IRUSR | S_IWUSR | S_IXUSR)) {
    	    
    	}
    }

    cachePath = g_strdup_printf("%s" G_DIR_SEPARATOR_S "cache", lifereaUserPath);
    if (!g_file_test(cachePath, G_FILE_TEST_IS_DIR)) {
    	if (0 != mkdir(cachePath, S_IRUSR | S_IWUSR | S_IXUSR)) {
	    
    	}
    }

    feedCachePath = g_strdup_printf("%s" G_DIR_SEPARATOR_S "feeds", cachePath);
    if (!g_file_test(feedCachePath, G_FILE_TEST_IS_DIR)) {
    	if (0 != mkdir(feedCachePath, S_IRUSR | S_IWUSR | S_IXUSR)) {
    	}
    }

    faviconCachePath = g_strdup_printf("%s" G_DIR_SEPARATOR_S "favicons", cachePath);
    if (!g_file_test(faviconCachePath, G_FILE_TEST_IS_DIR)) {
    	if (0 != mkdir(faviconCachePath, S_IRUSR | S_IWUSR | S_IXUSR)) {
    	}
    }

    g_free(cachePath);
    g_free(feedCachePath);
    g_free(faviconCachePath);

}


gchar *common_get_cache_path(void)
{
    if (NULL == lifereaUserPath)
	initCachePath();

    return lifereaUserPath;
}


gchar *common_create_cache_filename(const gchar * folder,
				    const gchar * key,
				    const gchar * extension)
{
    gchar *filename = NULL;

    filename =
	g_strdup_printf("%s" G_DIR_SEPARATOR_S "%s%s%s%s%s",
			common_get_cache_path(),
			(folder != NULL) ? folder : "",
			(folder != NULL) ? G_DIR_SEPARATOR_S : "", key,
			(extension != NULL) ? "." : "",
			(extension != NULL) ? extension : "");

    return filename;
}
