/**
 * @file ossoemailinterface.c
 *
 * This file contains the OSSO Email Interface API implementation
 *
 * Copyright (C) 2005, 2006 Nokia Corporation.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#define DBUS_API_SUBJECT_TO_CHANGE

#include <dbus/dbus.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <gconf/gconf-client.h>

#include <string.h>
#include "libosso.h"
#include <log-functions.h>
#include "osso-log.h"
#include "ossoemailinterface.h"
#include "ossoemailinterface-modest.h"

#define MODEST_IS_DEFAULT_KEY "/apps/modest/is_default"

/* check if modest is the default mailer */
static gboolean
modest_is_default (void)
{
	gboolean is_default;
	
	GConfClient *conf = gconf_client_get_default ();
	if (!conf) {
		g_warning ("%s: could not get gconf client", __FUNCTION__);
		return FALSE;
	}

		/* this wil return FALSE in case of any error */
	is_default = gconf_client_get_bool (conf, MODEST_IS_DEFAULT_KEY, NULL);
	
	if (conf)
		g_object_unref (conf);

	return is_default;
}



static osso_return_t
osso_email_call_rpc_method(const gchar * service,
			   const gchar * method, const gchar * args, osso_context_t * ctx);

/**
   This function will send a top D-Bus message to the email
   application with a list of gnomeVFS URI's as argutments.
 
   @param osso The library context as returned by #osso_initialize.
   @param list A GSList of strings -- URI's in GnomeVFS format.
   
   @return #OSSO_OK if all goes well, #OSSO_ERROR if an error occurred or
    #OSSO_INVALID if some parameter is invalid, #OSSO_RPC_ERROR if an error
    occured while sending the dbus message.
*/
osso_return_t osso_email_files_email(osso_context_t * osso, GSList * list)
{
	////////////////////////////////////////////////////////////////////
	if (modest_is_default())
		return osso_email_files_email_modest (osso, list);
	////////////////////////////////////////////////////////////////////
	
	GSList *next = NULL;
	gchar *args = NULL;
	osso_return_t ret = OSSO_OK;
	gchar *tmp = NULL;

	d_log(LOG_D, "send as files to email  application \n");

	if ((osso == NULL) || (list == NULL)) {
		return OSSO_INVALID;
	}

	if (osso_get_dbus_connection(osso) == NULL) {
		DLOG_OPEN("libosso");
		DLOG_ERR_F("error: no D-BUS connection!");
		LOG_CLOSE();
		return OSSO_INVALID;
	}

	args = g_strconcat("--files=", (gchar *) (list->data), NULL);
	if (args == NULL) {
		return OSSO_ERROR;
	}
	for (next = g_slist_next(list); next != NULL; next = g_slist_next(next)) {
		tmp = g_strconcat(args, ",", (gchar *) (next->data), NULL);
		g_free(args);
		args = tmp;
		if (args == NULL) {
			return OSSO_ERROR;
		}
	}
	ret = osso_email_call_rpc_method(OSSO_EMAIL, SEND_FILES_METHOD, args, osso);
	g_free(args);
	return ret;
}

/**
   This function will cause the email application to be topped
   with an empty email to the specified recipient opened. This is
   implemented with the top D-BUS message to the email
   application.
 
   @param osso The library context as returned by #osso_initialize.
   @param to A GSList of mailto URI strings.
 
   @return #OSSO_OK if all goes well, #OSSO_ERROR if an error occurred or
    #OSSO_INVALID if some parameter is invalid, #OSSO_RPC_ERROR if an error
    occured while sending the dbus message.
 */

osso_return_t osso_email_url_email(osso_context_t * osso, const gchar * url)
{
	////////////////////////////////////////////////////////////////////
	if (modest_is_default())
		return osso_email_url_email_modest (osso, url);
	////////////////////////////////////////////////////////////////////
	
	gchar *args = NULL;
	osso_return_t ret = OSSO_OK;

	d_log(LOG_D, "send url link to email application \n");

	if ((osso == NULL) || (url == NULL)) {
		return OSSO_INVALID;
	}

	if (osso_get_dbus_connection(osso) == NULL) {
		DLOG_OPEN("libosso");
		DLOG_ERR_F("error: no D-BUS connection!");
		LOG_CLOSE();
		return OSSO_INVALID;
	}

	args = g_strconcat("--url=", url, NULL);
	if (args == NULL) {
		return OSSO_ERROR;
	}
	ret = osso_email_call_rpc_method(OSSO_EMAIL, SEND_URL_METHOD, args, osso);
	g_free(args);
	return ret;
}


/**
   This function will cause the email application to be topped
   with an empty email to the specified recipient opened. This is
   implemented with the top D-BUS message to the email
   application.
 
   @param osso The library context as returned by #osso_initialize.
   @param to A GSList of mailto URI strings.
 
   @return #OSSO_OK if all goes well, #OSSO_ERROR if an error occurred or
    #OSSO_INVALID if some parameter is invalid, #OSSO_RPC_ERROR if an error
    occured while sending the dbus message.
*/

osso_return_t osso_email_emailto_email(osso_context_t * osso, GSList * to)
{
	////////////////////////////////////////////////////////////////////
	if (modest_is_default())
		return osso_email_emailto_email_modest (osso, to);
	////////////////////////////////////////////////////////////////////

	GSList *next = NULL;
	gchar *args = NULL;
	osso_return_t ret = OSSO_OK;
	gchar *tmp = NULL;

	d_log(LOG_D, "send mailto parameters to email application \n");

	if ((osso == NULL) || (to == NULL) || (to->data == NULL)) {
		return OSSO_INVALID;
	}

	if (osso_get_dbus_connection(osso) == NULL) {
		DLOG_OPEN("libosso");
		DLOG_ERR_F("error: no D-BUS connection!");
		LOG_CLOSE();
		return OSSO_INVALID;
	}
	if (strncmp(to->data, "mailto:", strlen("mailto:")) != 0) {
		d_log(LOG_D, "Input doesn't start with 'mailto' string \n");
		return OSSO_INVALID;
	}

	args = g_strconcat("--to=", (gchar *) (to->data), NULL);

	if (args == NULL) {
		return OSSO_ERROR;
	}

	for (next = g_slist_next(to); next != NULL; next = g_slist_next(next)) {
		tmp = g_strconcat(args, ",", (gchar *) (next->data), NULL);
		g_free(args);
		args = tmp;
		if (args == NULL) {
			return OSSO_ERROR;
		}
	}

	ret = osso_email_call_rpc_method(OSSO_EMAIL, SEND_MAILTO_METHOD, args, osso);
	g_free(args);
	return ret;
}


/**
   This function will is called to send DBus request to email application 
   
   @param service Email application service name (osso_email) 
   @param method  method to be invoked using using message 
   @param args    arguments to bo sent in the Dbus message 
   @param cxt as returned by #osso_initialize.
   @param address The email address is URI notation.
 
   @return #OSSO_OK if all goes well, #OSSO_ERROR if an error occurred or
    #OSSO_INVALID if some parameter is invalid, #OSSO_RPC_ERROR if an error
    occured while sending the dbus message.
*/

static osso_return_t
osso_email_call_rpc_method(const gchar * service,
			   const gchar * method, const gchar * args, osso_context_t * ctx)
{
	osso_return_t rv = OSSO_OK;
	osso_rpc_t rpc_retval;

	d_log(LOG_D, " OSSOEMAIL Interface rpc method = %s \n", method);

	if (args != NULL) {
		rv = osso_rpc_run_with_defaults(ctx, service,
						method, &rpc_retval,
						DBUS_TYPE_STRING, args, DBUS_TYPE_INVALID);
	} else {
		rv = osso_rpc_run_with_defaults(ctx, service,
						method, &rpc_retval, DBUS_TYPE_INVALID);
	}
	if (rv != OSSO_OK) {
		if (rpc_retval.type == DBUS_TYPE_STRING) {
			osso_log(LOG_ERR, "Error in service %s rpc method %s:%s",
				 service, method, rpc_retval.value.s);
			g_free((gchar *) rpc_retval.value.s);
		} else {
			osso_log(LOG_ERR, "Error in service %s rpc method %s", service, method);
		}
	}
	return rv;
}

/**
   This function is called to get the email folder list to other applications
   (like Global search etc) 

   @param osso The library context as returned by #osso_initialize.
   @return GSList of floders or NULL incase of error
*/

GSList *osso_email_get_folder_list(osso_context_t * ctx)
{
	////////////////////////////////////////////////////////////////////
	if (modest_is_default())
		return osso_email_get_folder_list_modest (ctx);
	////////////////////////////////////////////////////////////////////

	gint idx = 0;
	GSList *folder_list = NULL;
	gchar **list = NULL;
	DBusMessage *msg = NULL, *ret = NULL;
	DBusError err;
	DBusConnection *conn = NULL;
	DBusMessageIter iter;
	gchar *flmsg = NULL;
	const gchar *get_email_folders_list = "--get_folder_list=";

	if ((ctx == NULL) || (osso_get_dbus_connection(ctx) == NULL)) {
		osso_log(LOG_ERR, "error: no D-BUS connection!");
		return NULL;
	}
	dbus_error_init(&err);
	conn = (DBusConnection *) osso_get_dbus_connection(ctx);
	/*Activate Browser in background first */
	msg = dbus_message_new_method_call(OSSO_EMAIL_SERVICE,
					   OSSO_EMAIL_OBJ_PATH,
					   OSSO_EMAIL_SERVICE, GET_FOLDERS_LIST);
	/*dbus_message_set_auto_activation(msg, TRUE); */
	/* dbus_bus_activate_service(conn, OSSO_EMAIL_SERVICE, 0, &reason, &err); */
	dbus_message_set_no_reply(msg, FALSE);

	if (msg == NULL) {
		dbus_error_free(&err);
		return NULL;
	}
	dbus_message_append_args(msg, DBUS_TYPE_STRING, &get_email_folders_list, DBUS_TYPE_INVALID);
	ret = dbus_connection_send_with_reply_and_block(conn, msg, 60000,	/* Time out 1 minute */
							&err);
	if (!ret) {
		dbus_error_free(&err);
	} else {
		switch (dbus_message_get_type(ret)) {
		case DBUS_MESSAGE_TYPE_METHOD_RETURN:
			dbus_message_iter_init(ret, &iter);
			if (dbus_message_iter_get_arg_type(&iter) == DBUS_TYPE_STRING) {
				dbus_message_iter_get_basic(&iter, &flmsg);
				if(flmsg == NULL)
					break;	
				list = g_strsplit(flmsg, "/", 0);
				for (idx = 0; list[idx] != NULL; idx++) {
					folder_list =
					    g_slist_append(folder_list, g_strdup(list[idx]));
				}
				g_strfreev(list);
			}
			break;
		default:
			break;
		}
		dbus_message_unref(ret);
	}
	dbus_message_unref(msg);
	dbus_error_free(&err);
	return folder_list;
}
