
/**
 * This file is part of osso-email-common
 *
 * Copyright (c) 2006 Nokia Corporation.
 * 
 * Contact: Dirk-Jan Binnema <dirk-jan.binnema@nokia.com>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#ifndef __VALID_H__
#define __VALID_H__

#include <glib.h>
#include <sys/stat.h>
#include <openssl/crypto.h>
#include <openssl/pkcs7.h>
#include <stdio.h>
#include <openssl/pem.h>
#include <openssl/err.h>
#include <openssl/rand.h>
#include <openssl/objects.h>
#include <openssl/x509.h>
#include <openssl/x509v3.h>
#include <smime_security.h>
#include <cst.h>

#ifdef __cplusplus
extern "C" {
#endif

	typedef struct _RecipientInfo RecipientInfo;



#define VALID        1
#define NOTVALIDYET     2
#define EXPIRED      4
#define REVOKED      8
#define TEMP_DIR    "tmp"
#define G_DIR_SEP       "/"
#define TEMP_FILE   "tmpmsg"
#define MAXPATH 4095
#define F_WRITE    "w+"
#define F_READ     "r+"
#define DES3	   "des3"
#define DES	   "des"
#define RC2	   "rc2"
#define APPS_DIR   "apps"
#define EMAIL_DIR  "email"



	struct _RecipientInfo {
		gchar *email_id;
		guint cert_id;
		gint cert_valid;
	};


/** 
  This function is called when verifying a signed message
  This function checks the trust setting of every signer
  certificate and even if one of the certificates is not
  trusted makes the signer certificate untrusted.
  
  @param all_signers - Signer of the message
  @return TRUE if the certificate is trusted else
                returns false
*/
	gboolean get_trust_setting(CST * store, STACK_OF(X509) * all_signers);

/** 
  This function is called from verify_message
  Checks the certificate validity for signer
  certificates. Returns invalid even if one of the
  certificate is invalid

  @param all_signers - List of all signer certs
  @return gint - 0 for valid, 1, 2, 3 for invalid
*/
	gint get_cert_state(CST * store, STACK_OF(X509) * all_signers);


/** 
  Function is called from verification_details. The function
  checks if any new certificates are present in the incoming
  message. If so returns all the new certificates in a stack

  @param p7 - Pointer to PKCS7 structure
  @param signer_certs - Certificates from CM store
  @param certs_to_save - Any certificates to be saved
  @param flags - Flags in case any
  @param signers - List of all signers
  @return STACK_OF(X509)* - Stack of X509 certificates
*/
	 STACK_OF(X509) * get_certs_to_save(CST * store, PKCS7 * p7,
					    STACK_OF(X509) * signer_certs,
					    STACK_OF(X509) * certs_to_save, gint flags,
					    STACK_OF(X509) ** signers);


/** 
  This function is called from verification_details.
  It saves a list of incoming certificates in a file path.
  This file path will be given to Certificate Manager UI.
                                  
  @param save_signers - X509 certs to be saved
  @param cert_path - Path to save certs
  @return gboolean - Returns TRUE on success and FALSE on failure
*/
	gboolean save_certs(CST * store, STACK_OF(X509) * save_signers, const gchar * cert_path);


/** 
  This function is called by UI to get private certificate ID
  and validity of the certificate.
  
  @param email_id - Email ID whose default
        private certificate needs to be fetched
  @cert_state - validity of the certificate fetched
  @return private certificate ID
*/
	guint get_private_certificate(gchar * email_id, gint * cert_state);

/** 
  This function is called from UI, to get the list
  of recipient certificate IDs. List of recipient 
  email IDs is passed and certificate IDs are
  fetched for the recipients. Certificate state
  is also set for every certificate.

  @param recip_list - List of RecipientInfo structure
  @return gboolean, TRUE for success and FALSE for failure
*/
	gboolean get_recipient_certitifcateID(GSList * recip_list);


/**
  This function gets the certificate ID of the signer certificate
  This ID will be passed onto UI.

  @param store - Store pointer
  @param all_signers - List of signers of the mail
  @return gint - returns certificate ID
*/
	gint get_sender_cert_id(CST * store, STACK_OF(X509) * all_signers);


/**
  This function is called from the UI process to check the certificate
  state after the certificate has been saved

  @param cert_ids - List of certificate IDs that have been saved
  @return gint - whether certificates are valid or not. Even if one
  certificate is not valid, the function returns invalid
*/
	gint get_changed_cert_state(GSList * cert_ids);


/**
  This function is called from the UI process to check the certificate
  trust after the certificate has been saved or if trust is edited

  @param cert_ids - List of certificate IDs that have been saved
  @return gboolean - whether certificate is trusted or not
*/
	gboolean get_changed_trust_setting(GSList * certs_ids);

/** 
  This functions creates a temporary file
  path for creating files. The files
  will be deleted by the function which calls
  this function

  @param tmp - File path
  @param tmplen - Length of file path
  @return TRUE on success and FALSE on failure
*/
	gboolean get_file_path(gchar * tmp, gint tmplen);


/**         
 Returns temporary file path

 @return gchar* containing the temporary file path
*/
	gchar *get_temp_dir(void);


/**
  This function takes in a certificate file path and checks
  if multiple certificates are there in the file. If so
  every certificate is parsed and put into a different file

  @param fp - File pointer to the file containing all certificates
  @return gchar* - Certificate path of every certificate
*/
	gchar *get_cert_path(FILE * fp);


/** Function removes files from the specified path if
   they are temporary files     
                                
   @param file - Name of the file to be removed
   @return void
*/
	void remove_file(const gchar * file);

/**
  This function is called for saving signer capabilities in
  the CM store. These capabilities will later be used when
  sending an encrypted mail

  @param store - CM store pointer
  @param p7 - PKCS7 structure obtained while verifying
  @param signers - List of signers signing the message
  @return void - No return type
*/
	void val_get_signer_cap(CST * store, PKCS7 * p7, STACK_OF(X509) * signers);

/**
  This function is called to get the list of common
  algorithms supported by multiple recipients when
  sending an encrypted mail
  
  @param cert_ids - Certificate IDs of recipient
  @return GSList - List of common algorithms
*/
	GSList *get_common_algorithms(GSList * cert_ids);

/**
  This function is called to find out the correct password for a 
  given private key. 

  @param cert_id - Certificate ID of private certificate
  @param password - List of passwords
  @return gchar* - certificate password
*/
	gchar *get_password(guint cert_id, GSList * password);
#endif				/* __VALID_H__ */
