
/**
 * This file is part of osso-email-common
 *
 * Copyright (c) 2006 Nokia Corporation.
 * 
 * Contact: Dirk-Jan Binnema <dirk-jan.binnema@nokia.com>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#include "validations.h"

/**
  This function sets the capabilities of signer in the store.
  These capabilities will then be used for sending encrypted mail

  @param store - CM store pointer
  @param signer_info - Signer Information like capabilities etc
  @param signer - signer certificate 
  @return gint - 0 on Success and -1 on failure
*/
static
gint set_capabilities(CST * store, PKCS7_SIGNER_INFO * signer_info, X509 * signer);


/**
  This function gets list of supported algorithm
  DES3, DES, RC are supported algorithms

  @return list of supported algorithms
*/
static
GSList *get_supported_algorithms(void);


/**
  This function returns list of recipients whose
  capabilities is stored in CM store

  @param store - store pointer
  @param cert_ids - List of recipientcertificates
  @return GSList - list of recipients with capabilities
*/
static
GSList *get_recip_with_capab(CST * store, GSList * cert_ids);


/**
  Function to convert ASN1_OBJECT to gchar*

  @param algo - ASN1_OBJECT
  @return gchar - algorithm in gchar* format
*/
static
gchar *get_enc_algo(ASN1_OBJECT * algo);


gboolean get_trust_setting(CST * store, STACK_OF(X509) * all_signers)
{
	gboolean trust = TRUE;
	gint count = 0;
	gint len = 0;
	guint uid = 0;
	X509 *certificate = NULL;

	CHECK_PARAMETER((all_signers == NULL), "Input param check failed", return FALSE);
	CHECK_PARAMETER((store == NULL), "Input param check failed", return FALSE);

	osso_log(LOG_DEBUG, "Getting trust settings\n");

	len = sk_X509_num(all_signers);

	for (count = 0; count < len; count++) {
		certificate = sk_X509_value(all_signers, count);

		/* Makes a call to certificate manager API */
		uid = CST_search_by_X509(store, certificate);
		trust = CST_is_purpose(store, uid, CST_PURPOSE_SMIME_SGN);
		if (trust == FALSE) {
			break;
		}
	}
	osso_log(LOG_INFO, "Trust of signer cert:%d\n", trust);
	return trust;
}

gint get_cert_state(CST * store, STACK_OF(X509) * all_signers)
{
	gint expired = 0;
	X509 *cert = NULL;
	gint state = 0;
	gint length = 0;

	osso_log(LOG_DEBUG, "Getting cert state\n");

	CHECK_PARAMETER((all_signers == NULL), "Input param check failed", return 0);
	CHECK_PARAMETER((store == NULL), "Input param check failed", return 0);

	length = sk_X509_num(all_signers);

	for (expired = 0; expired < length; expired++) {
		cert = sk_X509_value(all_signers, expired);

		/* Calls CM store function to return cert state */
		if (cert != NULL) {
			state = CST_get_state(store, cert);
			if (state != VALID)
				break;
		}
	}

	osso_log(LOG_INFO, "Certificate state: %d\n", state);
	return state;
}

STACK_OF(X509) * get_certs_to_save(CST * store, PKCS7 * p7,
				   STACK_OF(X509) * signer_certs,
				   STACK_OF(X509) * certs_to_save, gint flags,
				   STACK_OF(X509) ** signers)
{
	gint counter = 0;
	X509 *save_cert = NULL;
	guint uid = 0;
	gint len = 0;

	CHECK_PARAMETER((store == NULL), "Input param check failed", return NULL);
	CHECK_PARAMETER((p7 == NULL), "Input param check failed", return NULL);

	osso_log(LOG_DEBUG, "get_certs_to_save\n");

	/* PKCS7_get0_signers is an openssl function to get list of signer
	 * certificates. If the signer certificate is not available it will
	 * look for the same from signer_certs. If unavailable in both
	 * it will return NULL in case of error */
	if (NULL == (*signers = (PKCS7_get0_signers(p7, signer_certs, flags)))) {
		osso_log(LOG_DEBUG, "Unable to extract certificates\n");
		osso_log(LOG_ERR, "Errno: %d\n", ERR_get_error());
		return certs_to_save;
	}

	len = sk_X509_num(*signers);
	for (counter = 0; counter < len; counter++) {
		save_cert = sk_X509_value(*signers, counter);
		uid = CST_search_by_X509(store, save_cert);
		if (uid == 0) {
			sk_X509_push(certs_to_save, save_cert);
		}
	}

	osso_log(LOG_INFO, "%d cert to be saved\n", sk_X509_num(certs_to_save));
	return certs_to_save;
}

gboolean save_certs(CST * store, STACK_OF(X509) * save_signers, const gchar * cert_path)
{
	gint count = 0;
	FILE *save_fp = NULL;
	gint length = 0;

	osso_log(LOG_DEBUG, "Saving certificates\n");

	CHECK_PARAMETER((store == NULL), "Input param check failed", return FALSE);
	CHECK_PARAMETER((cert_path == NULL), "Input param check failed", return FALSE);

	length = sk_X509_num(save_signers);

	if (length < 1) {
		osso_log(LOG_ERR, "No certificates to save\n");
		return FALSE;
	}

	osso_log(LOG_INFO, "Certs will be saved in: %s\n", cert_path);

	if ((save_fp = fopen(cert_path, F_WRITE)) == NULL) {
		osso_log(LOG_ERR, "Unable to open the file for saving\n");
		return FALSE;
	}

	for (count = 0; count < length; count++) {
		/* This openssl function writes every cert in save_signers to
		 * save_fp 
		 */
		if (PEM_write_X509(save_fp, sk_X509_value(save_signers, count)) == 0) {
			osso_log(LOG_ERR, "Unable to write certs to file\n");
			return FALSE;
		}

		if (fputs("\n", save_fp) == EOF) {
			fclose(save_fp);
			return FALSE;
		}
	}

	fclose(save_fp);
	return TRUE;
}

guint get_private_certificate(gchar * email_id, gint * cert_state)
{
	guint cert_id = 0;
	X509 *cert = NULL;
	CST *cert_st = NULL;
	gint key_id = 0;

	osso_log(LOG_DEBUG, "Getting private certificate ID\n");

	cert_st = CST_open(FALSE, NULL);

	if (cert_st == NULL) {
		osso_log(LOG_ERR, "Cannot open the cert store\n");
		return 0;
	}

	if (email_id == NULL) {
		osso_log(LOG_ERR, "Invalid email_id\n");
		CST_free(cert_st);
		return 0;
	}

	/* CM store call return the default certificate ID for
	 * the given email ID
	 */
	cert_id = CST_default_cert_id(cert_st, email_id);

	if (cert_id == 0) {
		osso_log(LOG_ERR, "Invalid cert_id\n");
		CST_free(cert_st);
		return 0;
	}

	/* check if the certificate has a private key attached */
	key_id = CST_get_assigned_key(cert_st, cert_id);
	if (key_id == 0) {
		CST_free(cert_st);
		osso_log(LOG_ERR, "No private key associated with cert\n");
		return 0;
	}

	/* gets the default certificate */
	cert = CST_get_cert(cert_st, cert_id);
	if (cert == NULL) {
		CST_free(cert_st);
		osso_log(LOG_ERR, "Invalid certificate\n");
		return 0;
	}

	/*gets the certificate state of the default 
	 * certificate
	 */
	*cert_state = CST_get_state(cert_st, cert);
	osso_log(LOG_INFO, "Default certificate state is %d", *cert_state);
	osso_log(LOG_INFO, "Default certificate ID is %d\n", cert_id);

	X509_free(cert);
	CST_free(cert_st);
	return cert_id;
}

gboolean get_recipient_certitifcateID(GSList * recip_list)
{
	RecipientInfo *recip_info = NULL;
	gint counter = 0;
	X509 *cert = NULL;
	gint len = 0;
	CST *cert_st = NULL;

	osso_log(LOG_DEBUG, "get_recipient_certitifcateID\n");

	cert_st = CST_open(FALSE, NULL);
	if (cert_st == NULL) {
		osso_log(LOG_ERR, "Cannot open the certe store\n");
		return 0;
	}

	len = g_slist_length(recip_list);

	if (len == 0) {
		osso_log(LOG_ERR, "Null list recip_list\n");
		CST_free(cert_st);
		return FALSE;
	}

	for (counter = 0; counter < len; counter++) {
		osso_log(LOG_INFO, "Counter is:%d\n", counter);
		recip_info = (RecipientInfo *) g_slist_nth_data(recip_list, counter);
		if (recip_info->email_id != NULL) {
			osso_log(LOG_INFO, "Email ID is:%s\n", recip_info->email_id);
			recip_info->cert_id = CST_default_cert_id(cert_st, recip_info->email_id);
			if (recip_info->cert_id != 0) {
				osso_log(LOG_DEBUG, "cert_id is not null\n");
				cert = CST_get_cert(cert_st, recip_info->cert_id);
				if (cert != NULL) {
					osso_log(LOG_DEBUG, "Cert is NULL\n");
					recip_info->cert_valid = CST_get_state(cert_st, cert);
				}
			}
		}
	}
	if (cert != NULL) {
		X509_free(cert);
	}
	CST_free(cert_st);
	return TRUE;
}

gint get_sender_cert_id(CST * store, STACK_OF(X509) * all_signers)
{
	gint cert_id = 0;
	gint pos = 0;
	X509 *cert = NULL;

	osso_log(LOG_DEBUG, "In function get_sender_cert_id\n");

	CHECK_PARAMETER((all_signers == NULL), "Input param check failed", return 0);

	cert = sk_X509_value(all_signers, pos);
	if (cert == NULL) {
		return cert_id;
	}

	/* calls CM store function to get the certificate ID */
	cert_id = CST_search_by_X509(store, cert);

	return cert_id;
}

gint get_changed_cert_state(GSList * cert_ids)
{
	gint state = 0;
	gint length = 0;
	gint count = 0;
	gint cert_id = 0;
	X509 *cert = NULL;
	CST *store = NULL;

	osso_log(LOG_DEBUG, "In function get_changed_cert_state\n");

	CHECK_PARAMETER((cert_ids == NULL), "Input parameter NULL\n", return 0);
	length = g_slist_length(cert_ids);
	CHECK_PARAMETER((length == 0), "Input param check failed\n", return 0);

	/* Memory freed at end of the function */
	store = CST_open(FALSE, NULL);
	CHECK_PARAMETER((store == NULL), "Store not opened\n", return 0);

	for (count = 0; count < length; count++) {
		cert_id = GPOINTER_TO_INT(g_slist_nth_data(cert_ids, count));

		/* Memory freed at end of the function */
		cert = CST_get_cert(store, cert_id);
		if (cert != NULL) {
			/* Calls CM store function to return cert state */
			state = CST_get_state(store, cert);
			if (state != VALID)
				break;
		}
	}

	X509_free(cert);
	CST_free(store);
	return state;
}

gboolean get_changed_trust_setting(GSList * cert_ids)
{
	gboolean trust = TRUE;
	gint count = 0;
	gint len = 0;
	guint uid = 0;
	CST *store = NULL;

	osso_log(LOG_DEBUG, "In function get_changed_t_setting\n");

	CHECK_PARAMETER((cert_ids == NULL), "Input parameter NULL\n", return 0);
	len = g_slist_length(cert_ids);
	CHECK_PARAMETER((len == 0), "Input param check failed\n", return 0);

	store = CST_open(FALSE, NULL);
	CHECK_PARAMETER((store == NULL), "Store not opened\n", return 0);

	for (count = 0; count < len; count++) {
		uid = GPOINTER_TO_INT(g_slist_nth_data(cert_ids, count));
		if (uid != 0) {
			/* Calls CM store function to return cert state */
			trust = CST_is_purpose(store, uid, CST_PURPOSE_SMIME_SGN);
			if (trust == FALSE)
				break;
		}
	}

	CST_free(store);
	return trust;
}

gboolean get_file_path(gchar * tmp, gint tmplen)
{

	osso_log(LOG_DEBUG, "Getting temporary file path\n");
	guint rand = (guint) random();
	if (tmp == NULL) {
		return FALSE;
	}
	memset(tmp, '\0', tmplen);
	g_snprintf(tmp, tmplen - 1, "%s%ctmpmsg.%d", get_temp_dir(), G_DIR_SEPARATOR, rand);
	osso_log(LOG_INFO, "Temporary smime file to write msgs : %s \n", tmp);
	return TRUE;
}

gchar *get_cert_path(FILE * fp)
{
	gchar path[MAXPATH];
	gchar buffer[MAXPATH];
	FILE *outfp = NULL;
	gchar *str = NULL;
	gchar *out_path = NULL;

	osso_log(LOG_DEBUG, "In function get_cert_path\n");
	CHECK_PARAMETER((fp == NULL), "Input param is NULL", return NULL);

	if (FALSE == (get_file_path(path, MAXPATH + 1))) {
		osso_log(LOG_ERR, "Unable to get path\n");
		return NULL;
	}

	outfp = fopen(path, F_WRITE);
	if (outfp == NULL) {
		osso_log(LOG_ERR, "Unable to open file\n");
		return NULL;
	}

	memset(buffer, '\0', sizeof(buffer));

	str = fgets(buffer, sizeof(buffer), fp);
	if (str == NULL) {
		fclose(outfp);
		remove_file(path);
		return NULL;
	}
	while (NULL != str) {
		if (buffer[0] == '\r' || buffer[0] == '\n') {
			fputs("\n", outfp);
			break;
		}
		if (fputs(buffer, outfp) == EOF) {
			osso_log(LOG_ERR, "Unable to write to file\n");
			fclose(outfp);
			return NULL;
		}
		memset(buffer, '\0', sizeof(buffer));
		str = fgets(buffer, sizeof(buffer), fp);

	}

	fclose(outfp);
	out_path = g_strdup(path);
	return out_path;
}

gchar *get_temp_dir(void)
{
	static gchar *tmp_dir = NULL;
	if (!tmp_dir) {
		tmp_dir = g_strdup_printf("%s%c%s%c%s%c%s",
					  g_get_home_dir(),
					  G_DIR_SEPARATOR, APPS_DIR,
					  G_DIR_SEPARATOR, EMAIL_DIR, G_DIR_SEPARATOR, TEMP_DIR);
	}
	return tmp_dir;
}

void remove_file(const gchar * file)
{
	if ((g_strrstr(file, TEMP_FILE) != NULL) && (g_strrstr(file, get_temp_dir()) != NULL)) {
		if (unlink(file) != SMIME_OK) {
			osso_log(LOG_INFO, "Unable to remove file:%s\n", file);
		}
	}
}

void val_get_signer_cap(CST * store, PKCS7 * p7, STACK_OF(X509) * signers)
{

	PKCS7_SIGNER_INFO *signer_info = NULL;
	STACK_OF(PKCS7_SIGNER_INFO) * sinfos = NULL;
	X509 *signer = NULL;
	gint num = 0, count = 0, num_signers = 0, result = 0;

	osso_log(LOG_DEBUG, "In function get_capabilities\n");

	CHECK_PARAMETER((store == NULL), "Input parameter is NULL", return);
	CHECK_PARAMETER((p7 == NULL), "Input parameter is NULL", return);
	CHECK_PARAMETER((signers == NULL), "Input parameter is NULL", return);

	num_signers = sk_X509_num(signers);
	CHECK_PARAMETER((num_signers == 0), "Input parameter is NULL", return);

	/*Get the information about all signers which
	 *includes a list of the supported capabilities
	 */
	sinfos = PKCS7_get_signer_info(p7);
	num = sk_PKCS7_SIGNER_INFO_num(sinfos);
	if ((sinfos == NULL) || (num == 0) || (num_signers != num)) {
		osso_log(LOG_DEBUG, "No signer information to save\n");
		return;
	}

	/* Associate every signer with capabilities */
	for (count = 0; count < num; count++) {
		signer_info = sk_PKCS7_SIGNER_INFO_value(sinfos, count);
		signer = sk_X509_value(signers, count);
		result = set_capabilities(store, signer_info, signer);
		if (result != 0) {
			osso_log(LOG_ERR, "Unable to set capabilities\n");
		}
	}
}

static
gint set_capabilities(CST * store, PKCS7_SIGNER_INFO * signer_info, X509 * signer)
{

	STACK_OF(X509_ALGOR) * algorithm_list = NULL;
	X509_ALGOR *algo = NULL;
	gint num = 0, count = 0, nident = 0, result = -1, cert_id = 0;

	CHECK_PARAMETER((store == NULL), "Input parameter is NULL", return result);
	CHECK_PARAMETER((signer_info == NULL), "Input parameter is NULL", return result);
	CHECK_PARAMETER((signer == NULL), "Input parameter is NULL", return result);

	algorithm_list = PKCS7_get_smimecap(signer_info);
	num = sk_X509_ALGOR_num(algorithm_list);

	if ((algorithm_list == NULL) || (num == 0)) {
		osso_log(LOG_DEBUG, "No algorithm to save\n");
		return result;
	}

	for (count = 0; count < num; count++) {
		algo = sk_X509_ALGOR_value(algorithm_list, count);
		nident = OBJ_obj2nid(algo->algorithm);
		if (0 == (cert_id = CST_search_by_X509(store, signer))) {
			sk_X509_ALGOR_pop_free(algorithm_list, X509_ALGOR_free);
			return result;
		}

		if (FALSE == (CST_is_capability(store, cert_id, algo->algorithm))) {
			result = CST_set_capability(store, cert_id, algo->algorithm, NULL, 0);
		}
	}

	sk_X509_ALGOR_pop_free(algorithm_list, X509_ALGOR_free);
	return result;
}

GSList *get_common_algorithms(GSList * cert_ids)
{
	GSList *supported_algorithms = NULL;
	GSList *recip_with_capab = NULL;
	GSList *common_algos = NULL;
	CST *store = NULL;
	gboolean supported = TRUE;
	gint count = 0, counter = 0, num_recips = 0, num_algos = 0;
	ASN1_OBJECT *algo = NULL;
	gint cert_id = 0;
	gchar *enc_algo = NULL;

	osso_log(LOG_DEBUG, "In funtion get_common_algorithm\n");

	CHECK_PARAMETER((cert_ids == NULL), "Input param NULL", return NULL);
	CHECK_PARAMETER((g_slist_length(cert_ids) == 0), "Input param NULL", return NULL);

	store = CST_open(FALSE, NULL);
	CHECK_PARAMETER((store == NULL), "Store not opened\n", return NULL);

	/* freed at end of funtion */
	supported_algorithms = get_supported_algorithms();
	num_algos = g_slist_length(supported_algorithms);

	recip_with_capab = get_recip_with_capab(store, cert_ids);

	if ((recip_with_capab == NULL) || (g_slist_length(recip_with_capab) == 0)) {
		common_algos = g_slist_append(common_algos, DES3);
		g_slist_foreach(supported_algorithms, (GFunc) ASN1_OBJECT_free, NULL);
		g_slist_free(supported_algorithms);
		g_slist_free(recip_with_capab);
		CST_free(store);
		return common_algos;
	}

	num_recips = g_slist_length(recip_with_capab);
	for (count = 0; count < num_algos; count++) {
		algo = g_slist_nth_data(supported_algorithms, count);

		for (counter = 0; counter < num_recips; counter++) {
			cert_id = GPOINTER_TO_INT(g_slist_nth_data(recip_with_capab, counter));
			if (FALSE == CST_is_capability(store, cert_id, algo)) {
				supported = FALSE;
				break;
			}
		}

		if (supported == TRUE) {
			enc_algo = get_enc_algo(algo);
			common_algos = g_slist_append(common_algos, enc_algo);
		}
	}

	g_slist_foreach(supported_algorithms, (GFunc) ASN1_OBJECT_free, NULL);
	g_slist_free(supported_algorithms);
	g_slist_free(recip_with_capab);
	CST_free(store);
	return common_algos;
}

static
gchar *get_enc_algo(ASN1_OBJECT * algo)
{
	gint nid = 0;

	CHECK_PARAMETER((algo == NULL), "Parameter is NULL", return NULL);

	nid = OBJ_obj2nid(algo);

	switch (nid) {
	case NID_des_cbc:
		return DES3;
	case NID_des_ede3_cbc:
		return DES;
	case NID_rc2_cbc:
		return RC2;
	default:
		return DES3;
	}
}

static
GSList *get_recip_with_capab(CST * store, GSList * cert_ids)
{
	GSList *recip_with_capab = NULL;
	gint num = 0, count = 0;
	gint cert_id = 0;
	STACK_OF(ASN1_OBJECT) * algos = NULL;

	CHECK_PARAMETER((cert_ids == NULL), "Param NULL", return NULL);

	num = g_slist_length(cert_ids);
	CHECK_PARAMETER((num == 0), "Param NULL", return NULL);

	for (count = 0; count < num; count++) {
		cert_id = GPOINTER_TO_INT(g_slist_nth_data(cert_ids, count));
		if (cert_id == 0) {
			continue;
		}
		algos = CST_get_capabilities(store, cert_id);
		if ((algos == NULL) || (sk_ASN1_OBJECT_num(algos) < 1)) {
			osso_log(LOG_INFO, "No capab for:%d\n", cert_id);
			continue;
		} else {
			recip_with_capab = g_slist_append(recip_with_capab,
							  GINT_TO_POINTER(cert_id));
		}
		sk_ASN1_OBJECT_pop_free(algos, ASN1_OBJECT_free);
	}
	return recip_with_capab;
}

static
GSList *get_supported_algorithms(void)
{

	GSList *algos = NULL;
	ASN1_OBJECT *algo = NULL;

	algo = OBJ_nid2obj(NID_des_ede3_cbc);
	algos = g_slist_append(algos, algo);

	algo = OBJ_nid2obj(NID_des_cbc);
	algos = g_slist_append(algos, algo);

	algo = OBJ_nid2obj(NID_rc2_cbc);
	algos = g_slist_append(algos, algo);

	return algos;
}

gchar *get_password(guint cert_id, GSList * password)
{
	gchar *passwd = NULL;
	gint counter = 0;
	gint num = 0;
	X509 *cert = NULL;
	CST *cert_st = NULL;
	EVP_PKEY *key = NULL;

	osso_log(LOG_DEBUG, "In function get_password\n");

	CHECK_PARAMETER((cert_id == 0), "Input param check failed", return NULL);
	CHECK_PARAMETER((password == NULL), "Input param checked failed", return NULL);


	cert_st = CST_open(FALSE, NULL);
	if (cert_st == NULL) {
		osso_log(LOG_ERR, "Cannot open the cert store\n");
		return NULL;
	}

	num = g_slist_length(password);

	for (counter = 0; counter < num; counter++) {
		passwd = g_slist_nth_data(password, counter);
		cert = CST_get_cert(cert_st, cert_id);
		if (cert == NULL) {
			CST_free(cert_st);
			osso_log(LOG_ERR, "Invalid cert\n");
			return NULL;
		}
		key = CST_get_priv_key(cert_st, cert, passwd);
		if (key != NULL) {
			osso_log(LOG_DEBUG, "key found\n");
			CST_free(cert_st);
			X509_free(cert);
			EVP_PKEY_free(key);
			return passwd;
		}
	}

	CST_free(cert_st);
	X509_free(cert);
	EVP_PKEY_free(key);
	return NULL;
}
