
/**
 * This file is part of osso-email-common
 *
 * Copyright (c) 2006 Nokia Corporation.
 * 
 * Contact: Dirk-Jan Binnema <dirk-jan.binnema@nokia.com>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#ifndef __SMIME_H__
#define __SMIME_H__

#include <glib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <openssl/crypto.h>
#include <openssl/pkcs7.h>
#include <stdio.h>
#include <openssl/pem.h>
#include <openssl/err.h>
#include <openssl/rand.h>
#include <openssl/objects.h>
#include <openssl/x509.h>
#include <openssl/x509v3.h>
#include "validations.h"
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <log-functions.h>
#include <libosso.h>

#ifdef __cplusplus
extern "C" {
#endif

#define SMIME_NONE 0
#define SMIME_SIGN 1
#define SMIME_ENCRYPT 2
#define SMIME_SIGN_ENCRYPT 3


/*S/MIME errors to be shown by UI */

#define SMIME_OK 0
#define SMIME_ERROR_BASE 5500
#define SMIME_NO_CERTIFICATE_FOR_SIGNING (SMIME_ERROR_BASE+1)
#define SMIME_NO_PRIVATE_KEY_FOR_SIGNING (SMIME_ERROR_BASE+2)
#define SMIME_UNABLE_TO_RETRIEVE_CERTIFICATE (SMIME_ERROR_BASE+3)
#define SMIME_SIGNING_PROCESSING_ERROR (SMIME_ERROR_BASE+4)
#define SMIME_UNABLE_TO_SIGN (SMIME_ERROR_BASE+5)
#define SMIME_ENCRYPT_PROCESSING_ERROR (SMIME_ERROR_BASE+6)
#define SMIME_NO_PRIVATE_CERT_FOR_SIGNING (SMIME_ERROR_BASE + 7)
#define SMIME_UNKNOWN_ENCRYPTION_ALGO (SMIME_ERROR_BASE + 8)
#define SMIME_NO_CERT_FOR_ENCRYPTION (SMIME_ERROR_BASE + 9)
#define SMIME_DECRYPT_PROCESSING_ERROR (SMIME_ERROR_BASE + 10)
#define SMIME_UNABLE_TO_DECRYPT (SMIME_ERROR_BASE + 11)
#define SMIME_SIGNATURE_VERIF_FAILED (SMIME_ERROR_BASE + 12)
#define SMIME_UNABLE_TO_ENCRYPT (SMIME_ERROR_BASE + 13)
#define SMIME_VERIF_PROCESSING_ERROR (SMIME_ERROR_BASE + 14)
#define SMIME_CANNOT_SAVE_CERTS_ERROR (SMIME_ERROR_BASE + 15)
#define SMIME_NO_CERT_FOR_DECRYPTION_ERROR (SMIME_ERROR_BASE + 16)
#define SMIME_SIGNER_CERT_NOT_AVAILABLE_ERROR (SMIME_ERROR_BASE + 17)
#define SMIME_INTERNAL_PASS_ERROR (SMIME_ERROR_BASE + 18)

#define CHECK_PARAMETER(condition, message, iffail)  \
{                                                    \
	if(condition)  {                             \
	osso_log(LOG_ERR, "%s\n", message);          \
	iffail;                                      \
	}                                            \
}

/** 
   This function will be called by smtp to create signed
   mail with certificate unique ID. Certificate verifications 
   are already done at the UI level. All the memory allocated
   in this function is freed in clean_sign_memory function.
   
   @param in_filename - File to be signed
   @param out_filename - File after signing
   @param cert_id - unique ID as passed by ui to engine 
   @return integer 0 for success or error value as defined
                        in smime_security.h
*/
	gint sign_message(const gchar * in_filename, const gchar * out_filename,
			  GSList * certs_id, gchar * password);


/**
   encrypt_message is called from smtp when an encrypted
   message is to be composed.This function will interface
   with certificate store to fetch all the recipient 
   certificates for encrypting. 
   
   @param recip_cert_uid - List of recipient certificate ID
   @param recip_algo  - Common algorithms supported by all recips
   @param out_filename - Output which with encrypted content
   @param in_filename - Content to be encrypted
   @return gint - 0 for success and error code for failure
*/
	gint encrypt_message(GSList * recip_cert_uid, GSList * recip_algo,
			     const gchar * out_filename, const gchar * in_filename);


/**
   This function is used for both signing and encrypting a mail.
   This is called from the engine for signing and encrypting
   a composed mail. This function internally calls sign_message
   and encrypt_message functions respectively. All the 
   parameters passed to this function are parameters required
   for sign_message function and encrypt_message function
   Refer to the respective functions for explanation
   on the parameters. 

   @param recip_cert_uid - UIDs of recipient certs to use
   encryption of the message
   @param recip_algo - Recipient algorithm to use
   @param in_filename - File path containing message to be
   encrypted
   @param out_filename - File path to place the encrypted
                         content
   @return gint - 0 on success. Error code on failure 
*/
	gint sign_encrypt(GSList * recip_cert_uid, GSList * recip_algo,
			  const gchar * in_filename, const gchar * out_filename, gchar * password);



/**
   This function is called from encrypt_message to get
   list of all recipient certificates. The UIDs of all
   recipient certificates are sent by UI<->engine and
   then engine<->S/MIME handler. For every UID, the
   corresponding X509 certificate is obatined. CM
   store functions are called for this purpose. All
   the certificates are returned as a stack

   @param recip_certs_uid - List of all recipient's
   UIDs as sent by UI
   @param encerts - A certificate stack which will 
                        hold all the X509 certificates
   @return STACK_OF(X509)* - Recipient certs
*/
	 STACK_OF(X509) * get_recip_certs(GSList * recip_certs_uid);


/**
   This function is called for verifying a signed message
   This function is called by S/MIME handler function when
   an S/MIME message is received. Memory allocated in this
   function is freed using clean_verify_memory function.
   
   @param infile - Input file containing signed message
   @param outfile - Output file after verifying signed message
   @param detached file - File with content in clear signed mess
   @param saved - Whether certificate needs to be saved or not
   @param email_id - Signer email ID
   @param cert_state - validity of cert (valid, notyetvalid, expired, revoked)
   @param signature - Signature authentic or not
   @param trust - certificate is trusted or not
   @param cert_path - path of certificate to be saved
   @return gint - 0 for success, error code for failure
*/
	gint verify_message(const gchar * infile, const gchar * outfile,
			    const gchar * detached_file, gboolean * saved,
			    const gchar * email_id, gint * cert_state, gboolean * signature,
			    gboolean * trust, const gchar * cert_path, gint * sender_cert_id);



/** 
   This function is called by S/MIME handler function
   when decrypting a message. After decrypting if the
   resulting message is signed, then verify_message
   function will be called. 
   @param in_decrypt - The encrypted content
   @param out_decrypt - File where decrypted mail will be
   stored. This is the output file which will be passed
   onto engine. Memory allocated in this function is freed
   by clean_decrypted_memory function.

   @param email_id - The recipient private certificate will be
        fetched based on this email ID. This is the email ID 
        associated with default account. In case of archived
        mails there is no default account. Hence all private
        certificates will be fetched.
   @param cert_state - This will indicate whether the
        encryption certificate is still valid or not
   @param detached - File where contents of clear signed message
        will be stored
   @param saved - If cert saving is required or not
   @param signer_email_id - Signer email ID
   @param signature - Signature valid or not
   @param trust - Whether signer cert is trusted or not
   @param cert_path - Path of cert that needs to be saved
   @return gint 0 on success and error code on failure
*/
	gint decrypt_message(const gchar * in_decrypt, const gchar * out_decrypt,
			     const gchar * email_id, gint * cert_state, const gchar * detached,
			     gboolean * saved, const gchar * signer_email_id,
			     gint * signer_cert_state, gboolean * signature,
			     gboolean * trust, const gchar * cert_path, gboolean decr_verify,
			     gint * smime, gint * sender_cert_id, gint * recip_cert_id);



/**
  This function is called when a signed and encrypted mail is
  received. Refer to decrypt_message and verify_message
  for parameter description. This function calls 
  decrypt_message and verify_message to acheive the required
  functionality

  @param in_decrypt - Input file
  @param recip_email - Email ID of the recipient
  @param private_cert_state - Expiry status of decryption certificate
  @param outfile - Output file after decrypting and verifying the contents
  @param detached_file - File with content in clear signed message
  @param saved - Whether certificate needs to be saved or not
  @param signer_email - Signer email ID 
  @param signer_cert_state - validity of cert
  @param signature - Signature authentic or not 
  @param trust - certificate is trusted or not
  @param cert_path - path of certificate to be saved
  @return gint - 0 for success, error code for failure
*/
	gint decrypt_verify(const gchar * in_decrypt, const gchar * recip_email,
			    gint * private_cert_state, gchar * outfile, gchar * detached_file,
			    gboolean * saved, const gchar * signer_email,
			    gint * signer_cert_state, gboolean * signature,
			    gboolean * trust, gchar * cert_path, gint * sender_cert_id,
			    gint * recip_cert_id);



/** 
  This functions is called from al as a part
  of email application initialization. This functions
  loads all the required algorithms, ciphers to be used
  It also opens a pointer to the certificate store. Using 
  this pointer the certificate store will be accessed.
  @return gboolean - returns TRUE for successful execution 
                     else returns FALSE
*/
	gboolean smime_init(void);



/** 
  The following function return the
  S/MIME type of the contents in the
  given file. 
  @param file - Input file
  @return gint - returns SMIME_NONE when contents
        are not of S/MIME type
        returns SMIME_SIGN for signed messages
        returns SMIME_ENCRYPT for encrypted messages
        returns SMIME_SIGN_ENCRYPT for signed and
        encrypted messages
*/
	gint find_smime_type(const gchar * file);



/**
  This function performs the cleanup activity. It
  frees the certificate store handler
*/
	void smime_cleanup(void);


#endif
