
/**
 * This file is part of osso-email-common
 *
 * Copyright (c) 2006 Nokia Corporation.
 * 
 * Contact: Dirk-Jan Binnema <dirk-jan.binnema@nokia.com>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#include "smime_security.h"

/* Certificate store pointer */
CST *gl_smime_st = NULL;

/*certificate id of private certificate*/
guint gl_cert_id = 0;

/*certificate password*/
gchar *gl_cert_pass = NULL;

/** 
   This function is called from verify_message after signature
   verification. Certain information like certificate validity,
   trust, saving of new certificate, path of such a certificate
   needs to be passed on to UI for saving the certificate. 
   Memory allocated here is freed in clean_verify_memory.
   
   @param certs_to_save - List containing certificates to save
   @param all_signers - List of all signer certificates
   @param signer_certs - Signer certs obtained from CM store
   @param p7 - PKCS7 structure
   @param gint - Flags for message
   @param cert_path - Path of certificate to be saved
   @param trust - Signer cert trust setting
   @param saved - Whether certificate needs to be saved or not
   @param cert_state - State of signer certificate
   @return gint - 0 on success error code on failure
*/
static gint verification_details(PKCS7 * p7, STACK_OF(X509) * signer_certs,
				 gint flags, const gchar * cert_path, gboolean * trust,
				 gboolean * saved, gint * cert_state, gint * sender_cert_id);

/**
   This function is called after decrypting a message.
   This is called from decrypt_message. The function
   checks if the decrypted message is signed. In case
   the decrypted message is signed it calls verify_message
   function to verify the signature and then then outputs 
   the final file
   
   @param inter_file - Input file containing decrypted message
   @param outfile - Output file containing final message
   @param detached - File containing detached content
   @param saved - Certs to save or not
   @param signer_email_id - Sender email ID
   @param signer_cert_state - Certificate state
   @param signature - Validity of signature
   @param trust - Sender certificate trusted or not
   @param cert_path - Certificate path in case it needs to be saved.
   @return gint - 0 on success and error code on failure
*/
static gint complete_decrypt(gchar * inter_file, const gchar * outfile,
			     const gchar * detached, gboolean * saved,
			     const gchar * signer_email_id,
			     gint * signer_cert_state, gboolean * signature,
			     gboolean * trust, const gchar * cert_path,
			     gboolean decr_verify, gint * smime, gint * sender_cert_id);


/** 
   This function will clear all the memory used up while
   verifying a signed mail. 

   @param in_verify - Input file
   @param out_verify - Output file
   @param indata_verify - Clear_signed message file
   @param signer_certs - Cert list from CM for signer
   @param certs_to_save - Certs to save into CM
   @param all_signers - Signers of the message
   @param store - Store of certificates
   @param p7 - PKCS7 structure
   @return void
*/
static void clean_verify_memory(BIO * in_verify, BIO * out_verify,
				BIO * indata_verify, STACK_OF(X509) * signer_certs,
				X509_STORE * store, PKCS7 * p7);

/**
  This function frees up all memory allocated
  during the decryption process
  @param in - input BIO file
  @param out - output BIO file
  @param p7 - PKCS7 structure
  @param certs - Recipient certificates
  @param pkey - Private key for decryption
  @return void
*/
static void clean_decrypted_memory(BIO * in, BIO * out, PKCS7 * p7,
				   STACK_OF(X509) * certs, EVP_PKEY * pkey);


/** 
   The function below clears all the memory allocated 
   during the signing process. 

   @param in - input file with content in BIO format
   @param out - output BIO with signed content
   @param cert_chain - Entire chain with CA
   @param p7 - PKCS7 signed structure
   @param signer - Signer certificate
   @param key - Private key
   @return void
*/
static void clean_sign_memory(BIO * in, BIO * out, STACK_OF(X509) * cert_chain,
			      PKCS7 * p7, X509 * signer, EVP_PKEY * key);


/**
  This function is called by decrypt_message. 
  It gets all the certificates associated with a particular 
  email ID from CM store. Then takes the certificates from 
  GSList and puts them into STACK_OF(X509)* list

  @param email_id - Email address
  @return STACK_OF(X509)* - Certificates in X509 format
*/
static STACK_OF(X509) * get_all_certificates(const gchar * email_id);


/**
  This function returns all private certificates associated in the 
  CM store. This function is called by decrypt_message

  @return Stack of all certificates in X509 format
*/
static STACK_OF(X509) * get_all_private_certificates(void);


/** 
  This function changes the file permission 

  @param fp - File pointer whose file permission 
                needs to be changed
  @param file - File path 
  @return integer value of success or failure
*/
static gint change_file_perm_rw(FILE * fp, const gchar * file);

/**
 Function removes carriage return in a str

 @param - str in which carriage return should be removed
 @return gchar* - str without carriage return
*/
static gchar *smime_strcrchomp(gchar * str);

/**
  This function is called by sign_message. The actually
  signing of the mail is performed in this function

  @param signer - Signer certificate
  @param key - Private key of the signer
  @param cert_chain - Certificate chain
  @param in - Input File pointer in BIO format
  @param out - Out file pointer in BIO format
  @param flags - flags to be set while signing
  @param p7 - the p7 structure after signing
  @return gint - 0 on SUCESS and appropriate error
                value on failure
*/
static gint smime_sign(X509 * signer, EVP_PKEY * key, STACK_OF(X509) * cert_chain,
		       BIO * in, BIO * out, gint flags, PKCS7 ** p7);


/**
  This function is called from encrypt_message. 
  The content is encrypted in this function
  
  @param encerts - Stack of certificates with which
                   message is encrypted
  @param in - Input file in BIO format
  @param out - Output file in BIO format
  @param cipher - Algorithm for encryption
  @param flags - flags for encryption
  @return gint - 0 on success and 1 on failure
*/
static gint smime_encrypt(STACK_OF(X509) * encerts, BIO * in, BIO * out,
			  const EVP_CIPHER * cipher, gint flags);


/**
  This function is called from verify_message. This function
  creates the required BIO files. It gets the signer certificates
  from the certificate store

  @param infile - File with contents to be verified
  @param outfile - Output file
  @param in_verify - File in BIO format
  @param out_verify - File in BIO format for output
  @param email_id - signers email ID
  @param signer_certs - Signers certificates
  @return gint - 0 on success and error code on failure
*/
static gint prepare_verify_message(const gchar * infile, const gchar * outfile,
				   BIO ** in_verify, BIO ** out_verify,
				   const gchar * email_id, STACK_OF(X509) ** signer_certs);

/**
  This function extracts a p7 structure from
  a given file.
  
  @param filename - Input file containing a p7
  structure
  @return returns a pointer to a PKCS7 structure
*/
static PKCS7 *extract_p7(const gchar * filename);

/**
   This function will fetch from CM store
   all private certificates associated with 
   email ID or all private certificates if there
   is no email ID. This function is called from
   decrypt_message function

   @param email_id - Email address whose certificates are
                     to be fetched
   @param all_certs - certificates associated with the email ID
   @return gint - 0 on success and error message on falure
*/
static gint get_decryption_certs(const gchar * email_id, STACK_OF(X509) ** all_certs);

/**
   This is functions returns the best encryption algorithm
   when encrypting a mail. The input to this function
   is a list of algorithms supported by all recipients
   This list is sent by CM store to UI. Then UI to engine
   and then engine to S/MIME handler. Fom this list
   strongest algorithm is chosen. Refer to Feasibility
   report for supported algorithms 

   @param recip_algo - List of recipient algorithms
   @return EVP_CIPHER - Algorithm for encryption
*/
static const EVP_CIPHER *get_encryption_algorithm(GSList * recip_algo);


/**
   This function gets the private certificate required for 
   decryption. 

   @return STACK_OF(X509)* - Returns the private certificate
*/
static STACK_OF(X509) * get_certificates();

gint sign_message(const gchar * in_filename, const gchar * out_filename,
		  GSList * certs_id, gchar * password)
{

	gint smime_errno = 0;
	PKCS7 *p7 = NULL;
	X509 *signer = NULL;
	STACK_OF(X509) * cert_chain = NULL;
	BIO *in = NULL, *out = NULL;
	EVP_PKEY *key = NULL;

	/* PKCS7_DETACHED flag set creates clear-signed message */
	gint flags = PKCS7_DETACHED;
	gint pos = 0;
	guint cert_id = 0;

	osso_log(LOG_DEBUG, "Signing message in file: %s\n", in_filename);

	CHECK_PARAMETER((in_filename == NULL), "Input param check failed",
			return SMIME_SIGNING_PROCESSING_ERROR);
	CHECK_PARAMETER((out_filename == NULL), "Input param check failed",
			return SMIME_SIGNING_PROCESSING_ERROR);
	CHECK_PARAMETER((gl_smime_st == NULL), "Input param check failed",
			return SMIME_SIGNING_PROCESSING_ERROR);
	CHECK_PARAMETER((g_slist_length(certs_id) == 0),
			"No certs for signing", return SMIME_NO_CERTIFICATE_FOR_SIGNING);

	/* Fetching the certificate from certificate manager */
	cert_id = GPOINTER_TO_UINT(g_slist_nth_data(certs_id, pos));
	if (NULL == (signer = CST_get_cert(gl_smime_st, cert_id))) {
		osso_log(LOG_ERR, "Certificate: %d not retrived\n", cert_id);
		return SMIME_UNABLE_TO_RETRIEVE_CERTIFICATE;
	}

	/* Fetch private key of certificate from certificate manager.
	 * The private key will be unencrypted
	 */
	if (NULL == (key = CST_get_priv_key(gl_smime_st, signer, password))) {
		osso_log(LOG_ERR, "Private key: %d not retrieved\n", cert_id);
		clean_sign_memory(in, out, cert_chain, p7, signer, key);
		return SMIME_NO_PRIVATE_KEY_FOR_SIGNING;
	}


	/* The input file whose content is to be signed 
	 * is converted to BIO format.
	 */
	if (NULL == (in = BIO_new_file(in_filename, F_READ))) {
		osso_log(LOG_ERR, "Unable to create BIO for %s\n", in_filename);
		clean_sign_memory(in, out, cert_chain, p7, signer, key);
		return SMIME_SIGNING_PROCESSING_ERROR;
	}


	/* The output BIO file which will have the final signed contents. */
	if (NULL == (out = BIO_new_file(out_filename, F_WRITE))) {
		osso_log(LOG_ERR, "Unable to create BIO for %s\n", out_filename);
		clean_sign_memory(in, out, cert_chain, p7, signer, key);
		return SMIME_SIGNING_PROCESSING_ERROR;
	}


	/* Fetch certificate chain for private certificate
	 * with all CAs in the chain.
	 * If this is available with CM the whole
	 * chain will be included in the signed message
	 * else only signer certificate will be included.
	 */
	cert_chain = CST_get_chain(gl_smime_st, signer);

	if ((smime_errno = smime_sign(signer, key, cert_chain, in, out, flags, &p7)) != SMIME_OK) {
		osso_log(LOG_ERR, "Unable to sign %s  message \n", in_filename);
	}

	clean_sign_memory(in, out, cert_chain, p7, signer, key);
	return smime_errno;
}

static
gint smime_sign(X509 * signer, EVP_PKEY * key, STACK_OF(X509) * cert_chain,
		BIO * in, BIO * out, gint flags, PKCS7 ** p7)
{

	gint smime_errno = 0;

	osso_log(LOG_DEBUG, "In smime_sign\n");

	CHECK_PARAMETER((signer == NULL), "Input param check failed",
			return SMIME_SIGNING_PROCESSING_ERROR);
	CHECK_PARAMETER((key == NULL), "Input param check failed",
			return SMIME_SIGNING_PROCESSING_ERROR);
	CHECK_PARAMETER((in == NULL), "Input param check failed",
			return SMIME_SIGNING_PROCESSING_ERROR);
	CHECK_PARAMETER((out == NULL), "Input param check failed",
			return SMIME_SIGNING_PROCESSING_ERROR);


	/* This is openssl call which will create a signed message.
	 * signer is private certificate. key is private key.
	 * Flag will be PKCS7_DETACHED which will create clear
	 * signed message
	 */
	if (NULL == (*p7 = PKCS7_sign(signer, key, cert_chain, in, flags))) {
		osso_log(LOG_ERR, "Errno: %d\n", ERR_get_error());
		return SMIME_UNABLE_TO_SIGN;
	}


	/* The above openssl function produces pkcs7 structure.
	 * The input BIO file is reset to point to beginning 
	 * of file so that plain message can be included in a clear
	 * text message along with MIME headers.
	 */
	if (BIO_reset(in) != 0 && (flags & PKCS7_DETACHED)) {
		osso_log(LOG_ERR, "Unable to reset flag\n");
		return SMIME_SIGNING_PROCESSING_ERROR;
	}


	/* openssl API to add S/MIME headers to create 
	 * an S/MIME message
	 */
	if ((SMIME_write_PKCS7(out, *p7, in, flags)) == 0) {
		osso_log(LOG_ERR, "Unable to add S/MIME Headers\n");
		return SMIME_SIGNING_PROCESSING_ERROR;
	}

	return smime_errno;
}

static
void clean_sign_memory(BIO * in, BIO * out, STACK_OF(X509) * cert_chain,
		       PKCS7 * p7, X509 * signer, EVP_PKEY * key)
{
	osso_log(LOG_DEBUG, "Freeing memory used for signing\n");

	if (in != NULL) {
		BIO_free(in);
	}
	if (out != NULL) {
		BIO_free(out);
	}
	if (cert_chain != NULL) {
		sk_X509_pop_free(cert_chain, X509_free);
	}
	if (signer != NULL) {
		X509_free(signer);
	}
	if (p7 != NULL) {
		PKCS7_free(p7);
	}
	if (key != NULL) {
		EVP_PKEY_free(key);
	}

}

gint encrypt_message(GSList * recip_cert_uid, GSList * recip_algo,
		     const gchar * out_filename, const gchar * in_filename)
{
	gint flags = 0;
	const EVP_CIPHER *cipher = NULL;
	STACK_OF(X509) * encerts = NULL;
	BIO *in = NULL, *out = NULL;
	gint smime_errno = 0;

	osso_log(LOG_DEBUG, "Encrypting message: %s\n", in_filename);

	CHECK_PARAMETER((in_filename == NULL), "Input param check failed",
			return SMIME_ENCRYPT_PROCESSING_ERROR);
	CHECK_PARAMETER((out_filename == NULL), "Input param check failed",
			return SMIME_ENCRYPT_PROCESSING_ERROR);
	CHECK_PARAMETER((gl_smime_st == NULL), "Input param check failed",
			return SMIME_ENCRYPT_PROCESSING_ERROR);
	CHECK_PARAMETER((g_slist_length(recip_cert_uid) == 0),
			"No certs for signing", return SMIME_ENCRYPT_PROCESSING_ERROR);

	/* From list of algorithms passed on by UI<->engine,
	 * engine<->S/MIME handler, need to choose one
	 * encryption algorithm of highest security
	 */
	if (NULL == (cipher = get_encryption_algorithm(recip_algo))) {
		osso_log(LOG_ERR, "Unknown encryption algorithm\n");
		sk_X509_pop_free(encerts, X509_free);
		return SMIME_UNKNOWN_ENCRYPTION_ALGO;
	}


	/* For every UID in recip_cert_uid corresponding 
	 * encryption certificate is fetched. 
	 */
	encerts = get_recip_certs(recip_cert_uid);

	if ((encerts == NULL) || (sk_X509_num(encerts) < 1)) {
		osso_log(LOG_ERR, "No certificates for encryption\n");
		sk_X509_free(encerts);
		return SMIME_NO_CERT_FOR_ENCRYPTION;
	}


	/* Input file whose contents are to be encrypted
	 * in BIO format
	 */
	if (NULL == (in = BIO_new_file(in_filename, F_READ))) {
		osso_log(LOG_ERR, "Unable to create BIO for %s\n", in_filename);
		sk_X509_pop_free(encerts, X509_free);
		return SMIME_ENCRYPT_PROCESSING_ERROR;
	}


	/* Output file which will contain the encrypted
	 * contents in BIO format
	 */
	if (NULL == (out = BIO_new_file(out_filename, F_WRITE))) {
		osso_log(LOG_ERR, "Unable to create BIO:  %s\n", out_filename);
		sk_X509_pop_free(encerts, X509_free);
		BIO_free(in);
		return SMIME_ENCRYPT_PROCESSING_ERROR;
	}

	if ((smime_errno = smime_encrypt(encerts, in, out, cipher, flags)) != SMIME_OK) {
		osso_log(LOG_ERR, "Unable to encrypt: %s\n", in_filename);
	}

	sk_X509_pop_free(encerts, X509_free);
	BIO_free(in);
	BIO_free(out);
	return smime_errno;
}

gint smime_encrypt(STACK_OF(X509) * encerts, BIO * in, BIO * out,
		   const EVP_CIPHER * cipher, gint flags)
{

	gint smime_errno = 0;
	PKCS7 *p7 = NULL;

	osso_log(LOG_DEBUG, "In function smime_encrypt\n");

	CHECK_PARAMETER((encerts == NULL), "Input param check failed",
			return SMIME_ENCRYPT_PROCESSING_ERROR);
	CHECK_PARAMETER((sk_X509_num(encerts) < 1), "Input param check failed",
			return SMIME_NO_CERT_FOR_ENCRYPTION);
	CHECK_PARAMETER((in == NULL), "Input param check failed",
			return SMIME_NO_CERT_FOR_ENCRYPTION);
	CHECK_PARAMETER((out == NULL), "Input param check failed",
			return SMIME_NO_CERT_FOR_ENCRYPTION);
	CHECK_PARAMETER((cipher == NULL), "Input param check failed",
			return SMIME_NO_CERT_FOR_ENCRYPTION);


	/* This is an openssl call to create an encrypted PKCS7
	 * structure. encerts is list all recipient certificates.
	 * in is the input file in BIO format. cipher is the
	 * encryption algorithm to be used. flags are 
	 * optional settings in the encrypted mail
	 */
	if (NULL == (p7 = PKCS7_encrypt(encerts, in, cipher, flags))) {
		osso_log(LOG_ERR, "Errno: %d\n", ERR_get_error());
		return SMIME_UNABLE_TO_ENCRYPT;
	}

	/* This is an openssl call which will create S/MIME
	 * structure. It will take the previously encrypted
	 * p7 structure and add S/MIME and mime headers.
	 */
	SMIME_write_PKCS7(out, p7, in, flags);


	if (out == NULL) {
		osso_log(LOG_ERR, "Unable to add MIME headers\n");
		smime_errno = SMIME_ENCRYPT_PROCESSING_ERROR;
	}

	PKCS7_free(p7);
	return smime_errno;
}

gint sign_encrypt(GSList * recip_cert_uid, GSList * recip_algo,
		  const gchar * in_filename, const gchar * out_filename, gchar * password)
{


	gchar signed_filename[MAXPATH + 1];
	gint retval = 0;

	osso_log(LOG_DEBUG, "Signing and Encrypting: %s\n", in_filename);

	memset(signed_filename, '\0', MAXPATH);
	g_snprintf(signed_filename, MAXPATH - 1, "%s%ctmpmsg.%d", get_temp_dir(),
		   G_DIR_SEPARATOR, (guint) random());


	/* signed_filename is an intermediate file created
	 * after the message is signed. This file becomes
	 * input for encrypted message.
	 */
	if ((retval = (sign_message(in_filename, signed_filename, recip_cert_uid, password))) != 0) {
		osso_log(LOG_ERR, "Unable to sign: %s\n, in_filename");
		remove_file(signed_filename);
		return retval;
	}

	/* This function encrypts the signed message */
	if ((retval = (encrypt_message(recip_cert_uid, recip_algo,
				       out_filename, signed_filename)) != 0)) {
		osso_log(LOG_ERR, "Unable to encrypt the signed content\n");
	}


	/* The intermediate file signed_filename is deleted. 
	 * This file is input to encrypt_message. This
	 * file will be removed whether encryption is 
	 * complete or not
	 */
	remove_file(signed_filename);
	return retval;
}

STACK_OF(X509) * get_recip_certs(GSList * recip_certs_uid)
{
	X509 *cert = NULL;
	guint uid = 0;
	gint counter = 0;
	gint length = 0;
	STACK_OF(X509) * encerts = NULL;

	osso_log(LOG_DEBUG, "Getting recipients certificates\n");

	CHECK_PARAMETER((recip_certs_uid == NULL), "Input param check failed", return NULL);

	length = g_slist_length(recip_certs_uid);

	/* If there are no recipients return NULL */
	if (length == 0) {
		osso_log(LOG_ERR, "No certificates to fetch\n");
		return NULL;
	}

	/* creating a list which will hold X509
	 * certificates of all recipients. The memory
	 * allocated is freed at the end of this function
	 * on success. If any failure is encountred it is freed
	 * at the failed function.
	 */
	if (NULL == (encerts = sk_X509_new_null())) {
		osso_log(LOG_ERR, "Unable to create X509 list\n");
		return NULL;
	}



	/* Iterate through the list of UIDs and fetch the certificates
	 * The certificates are then pushed into a stack using openssl
	 * call
	 */
	for (counter = 0; counter < length; counter++) {
		uid = GPOINTER_TO_INT(g_slist_nth_data(recip_certs_uid, counter));
		if (uid == 0)
			continue;
		cert = CST_get_cert(gl_smime_st, uid);
		if (cert != NULL)
			sk_X509_push(encerts, cert);
	}

	return encerts;
}

static
const EVP_CIPHER *get_encryption_algorithm(GSList * recip_algo)
{

	if ((g_slist_length(recip_algo) == 0)) {
		return EVP_des_ede3_cbc();
	}

	if ((g_slist_find(recip_algo, "des3")) != NULL) {
		return EVP_des_ede3_cbc();
	}

	if ((g_slist_find(recip_algo, "des")) != NULL) {
		return EVP_des_cbc();
	}

	if ((g_slist_find(recip_algo, "rc2-128")) != NULL) {
		return EVP_rc2_cbc();
	}

	if ((g_slist_find(recip_algo, "rc2-64")) != NULL) {
		return EVP_rc2_64_cbc();
	}

	if ((g_slist_find(recip_algo, "rc2-40")) != NULL) {
		return EVP_rc2_40_cbc();
	}

	return EVP_des_ede3_cbc();
}


gint verify_message(const gchar * infile, const gchar * outfile,
		    const gchar * detached_file, gboolean * saved,
		    const gchar * email_id, gint * cert_state, gboolean * signature,
		    gboolean * trust, const gchar * cert_path, gint * sender_cert_id)
{
	STACK_OF(X509) * signer_certs = NULL;
	X509_STORE *store = NULL;
	BIO *in_verify = NULL, *out_verify = NULL, *indata_verify = NULL;
	PKCS7 *p7 = NULL;
	gint flags = PKCS7_NOVERIFY;
	gint smime_errno = 0;

	osso_log(LOG_DEBUG, "Verifying signed message: %s\n", infile);

	CHECK_PARAMETER((infile == NULL), "Parameter NULL in verify_message",
			return SMIME_VERIF_PROCESSING_ERROR);
	CHECK_PARAMETER((outfile == NULL), "Parameter NULL in verify_message",
			return SMIME_VERIF_PROCESSING_ERROR);
	CHECK_PARAMETER((detached_file == NULL),
			"Parameter NULL in verify_message", return SMIME_VERIF_PROCESSING_ERROR);
	CHECK_PARAMETER((gl_smime_st == NULL),
			"Parameter NULL in verify_message", return SMIME_VERIF_PROCESSING_ERROR);


	if ((smime_errno = (prepare_verify_message(infile, outfile, &in_verify,
						   &out_verify, email_id,
						   &signer_certs))) != SMIME_OK) {
		osso_log(LOG_ERR, "Unable to prepare for verification\n");
		clean_verify_memory(in_verify, out_verify, indata_verify, signer_certs, store, p7);
		return smime_errno;
	}

	/* The incoming message is in S/MIME format. This needs
	 * to be converted into PKCS7 structure. The verification 
	 * functions takes only a PKCS7 structure which needs to
	 * be extracted from the sent message.
	 */
	if (NULL == (p7 = SMIME_read_PKCS7(in_verify, &indata_verify))) {
		osso_log(LOG_ERR, "Unable to read the S/MIME content\n");
		clean_verify_memory(in_verify, out_verify, indata_verify, signer_certs, store, p7);
		return SMIME_VERIF_PROCESSING_ERROR;
	}


	/* This function is called for verification of signed message.
	 * It is an openssl function. p7 is the message to be verified
	 * in PKCS7 format. signer_certs is the signer certificate in
	 * case it is not available in the incoming mail. store is
	 * an optional parameter in case certificate verification
	 * needs to be done. In our case this will always be NULL.
	 */
	if ((PKCS7_verify(p7, signer_certs, store, indata_verify, out_verify, flags)) != 1) {
		osso_log(LOG_ERR, "Signature verification failed\n");
		osso_log(LOG_ERR, "Errno: %d\n", ERR_get_error());
		clean_verify_memory(in_verify, out_verify, indata_verify, signer_certs, store, p7);

		*signature = FALSE;
		return SMIME_SIGNATURE_VERIF_FAILED;
	}

	if ((verification_details(p7, signer_certs,
				  flags, cert_path, trust, saved,
				  cert_state, sender_cert_id) != 0)) {
		osso_log(LOG_ERR, "verification_details failed \n");
		smime_errno = SMIME_SIGNER_CERT_NOT_AVAILABLE_ERROR;
	}

	clean_verify_memory(in_verify, out_verify, indata_verify, signer_certs, store, p7);
	return smime_errno;

}

static
gint prepare_verify_message(const gchar * infile, const gchar * outfile,
			    BIO ** in_verify, BIO ** out_verify,
			    const gchar * email_id, STACK_OF(X509) ** signer_certs)
{
	gint smime_errno = 0;

	osso_log(LOG_DEBUG, "In prepare_verify_message\n");

	CHECK_PARAMETER((infile == NULL), "Input param check failed",
			return SMIME_VERIF_PROCESSING_ERROR);
	CHECK_PARAMETER((outfile == NULL), "Input param check failed",
			return SMIME_VERIF_PROCESSING_ERROR);

	/* Creates a BIO for the message to be verified */
	if (NULL == (*in_verify = BIO_new_file(infile, F_READ))) {
		osso_log(LOG_ERR, "Unable to create BIO for %s\n", infile);
		return SMIME_VERIF_PROCESSING_ERROR;
	}

	/* The contents of verified mail is placed in out_verify
	 * A BIO is created for the output file also.
	 */
	if (NULL == (*out_verify = BIO_new_file(outfile, F_WRITE))) {
		osso_log(LOG_ERR, "Unable to create BIO for %s\n", outfile);
		return SMIME_VERIF_PROCESSING_ERROR;
	}

	/* List of all signer certificates is obtained. 
	 * This is passed onto PKCS7_verify function. If
	 * the signer public certificate is not available 
	 * in the sent message, the PKCS7_verify function
	 * will look up this list to find signer certificate.
	 */

	if (email_id != NULL) {
		*signer_certs = get_all_certificates(email_id);
	}

	return smime_errno;
}

static
gint verification_details(PKCS7 * p7, STACK_OF(X509) * signer_certs, gint flags,
			  const gchar * cert_path, gboolean * trust,
			  gboolean * saved, gint * cert_state, gint * sender_cert_id)
{
	gint ret = 0;
	STACK_OF(X509) * all_signers = NULL;
	STACK_OF(X509) * certs_to_save = NULL;

	osso_log(LOG_DEBUG, "In verification_details\n");


	CHECK_PARAMETER((p7 == NULL), "Input parameter p7 is NULL",
			return SMIME_VERIF_PROCESSING_ERROR);

	/* Memory freed end of this function */
	if (NULL == (certs_to_save = sk_X509_new_null())) {
		osso_log(LOG_ERR, "Memory not allocated\n");
		return SMIME_VERIF_PROCESSING_ERROR;
	}

	/* Memory is freed at end of the function */
	if (NULL == (all_signers = sk_X509_new_null())) {
		osso_log(LOG_ERR, "Memory not allocated\n");
		return SMIME_VERIF_PROCESSING_ERROR;
	}

	certs_to_save = get_certs_to_save(gl_smime_st, p7,
					  signer_certs, certs_to_save, flags, &all_signers);

	if (sk_X509_num(certs_to_save) > 0) {
		*saved = TRUE;
		if (FALSE == (save_certs(gl_smime_st, certs_to_save, cert_path))) {
			osso_log(LOG_ERR, "Unable to save Certificates\n");
			remove_file(cert_path);
			sk_X509_free(certs_to_save);
			sk_X509_free(all_signers);
			cert_path = NULL;
			return SMIME_CANNOT_SAVE_CERTS_ERROR;
		}
	}

	if (*saved == FALSE) {
		/* If the certificate needs to be saved and is not
		 * there in CM store, we cannot fetch the trust
		 */
		*trust = get_trust_setting(gl_smime_st, all_signers);

		/* If the certificate is not saved, we cannot fetch
		 * certificate ID
		 */
		*sender_cert_id = get_sender_cert_id(gl_smime_st, all_signers);

		/* Even if one of the signer certificate is invalid,
		 * cert_state will be invalid
		 */
		*cert_state = get_cert_state(gl_smime_st, all_signers);

		/*Set capabilities for the certificate */
		val_get_signer_cap(gl_smime_st, p7, all_signers);

	}


	sk_X509_free(certs_to_save);
	sk_X509_free(all_signers);
	return ret;
}

static
void clean_verify_memory(BIO * in_verify, BIO * out_verify, BIO * indata_verify,
			 STACK_OF(X509) * signer_certs, X509_STORE * store, PKCS7 * p7)
{
	if (in_verify != NULL) {
		BIO_free(in_verify);
	}
	if (out_verify != NULL) {
		BIO_free(out_verify);
	}
	if (indata_verify != NULL) {
		BIO_free(indata_verify);
	}
	if (store != NULL) {
		X509_STORE_free(store);
	}
	if (p7 != NULL) {
		PKCS7_free(p7);
	}
	if (signer_certs != NULL) {
		sk_X509_pop_free(signer_certs, X509_free);
	}

}

gint decrypt_message(const gchar * in_decrypt, const gchar * out_decrypt,
		     const gchar * email_id, gint * cert_state, const gchar * detached,
		     gboolean * saved, const gchar * signer_email_id,
		     gint * signer_cert_state, gboolean * signature,
		     gboolean * trust, const gchar * cert_path, gboolean decr_verify,
		     gint * smime, gint * sender_cert_id, gint * recip_cert_id)
{
	BIO *in_file_decrypt = NULL, *out_file_decrypt = NULL;
	BIO *indata = NULL;
	PKCS7 *p7 = NULL;
	STACK_OF(X509) * all_certs = NULL;
	EVP_PKEY *pkey = NULL;
	X509 *decrypt_cert = NULL;
	gint flags = 0, ret = 0, count = 0, length = 0;
	gchar inter_file[MAXPATH + 1];

	osso_log(LOG_DEBUG, "Decrypting message in file: %s\n", in_decrypt);

	CHECK_PARAMETER((in_decrypt == NULL), "Parameter NULL",
			return SMIME_DECRYPT_PROCESSING_ERROR);
	CHECK_PARAMETER((out_decrypt == NULL), "Parameter NULL",
			return SMIME_DECRYPT_PROCESSING_ERROR);
	CHECK_PARAMETER((gl_smime_st == NULL), "Parameter NULL",
			return SMIME_DECRYPT_PROCESSING_ERROR);
	CHECK_PARAMETER((gl_cert_id == 0), "Cert ID is null", return SMIME_INTERNAL_PASS_ERROR);

	if (FALSE == (get_file_path(inter_file, MAXPATH + 1))) {
		osso_log(LOG_ERR, "Unable to get path for inter_file\n");
		return SMIME_DECRYPT_PROCESSING_ERROR;
	}

	if ((ret = get_decryption_certs(email_id, &all_certs)) != SMIME_OK) {
		osso_log(LOG_ERR, "unable to get decryption certificates\n");
		clean_decrypted_memory(in_file_decrypt, out_file_decrypt, p7, all_certs, pkey);
		return ret;
	}

	/* This will create a BIO file for the encrypted content */
	if (NULL == (in_file_decrypt = BIO_new_file(in_decrypt, F_READ))) {
		clean_decrypted_memory(in_file_decrypt, out_file_decrypt, p7, all_certs, pkey);
		osso_log(LOG_ERR, "unable to create BIO for:%s\n", in_decrypt);
		return SMIME_DECRYPT_PROCESSING_ERROR;
	}

	/* This will create BIO file for decrypted content */
	if (NULL == (out_file_decrypt = BIO_new_file(inter_file, F_WRITE))) {
		osso_log(LOG_ERR, "unable to create BIO for:%s\n", inter_file);
		clean_decrypted_memory(in_file_decrypt, out_file_decrypt, p7, all_certs, pkey);
		return SMIME_DECRYPT_PROCESSING_ERROR;
	}

	/* This function takes the incoming encrypted content 
	 * in_file_decrypt and extracts the p7 content 
	 */
	if (NULL == (p7 = SMIME_read_PKCS7(in_file_decrypt, &indata))) {
		osso_log(LOG_ERR, "unable to read PKCS7 while decrypting\n");
		clean_decrypted_memory(in_file_decrypt, out_file_decrypt, p7, all_certs, pkey);
		return SMIME_DECRYPT_PROCESSING_ERROR;
	}

	/* This code iterates through a list of private
	 * certificates and tries to decrypt the message
	 * When appropriate certificate is found
	 * encryption is performed. 
	 */
	length = sk_X509_num(all_certs);
	for (count = 0; count < length; count++) {
		decrypt_cert = NULL;
		pkey = NULL;
		decrypt_cert = sk_X509_value(all_certs, count);
		pkey = CST_get_priv_key(gl_smime_st, decrypt_cert, gl_cert_pass);
		if (pkey == NULL) {
			osso_log(LOG_DEBUG, "No key for decryption\n");
			continue;
		}
		if ((PKCS7_decrypt(p7, pkey, decrypt_cert, out_file_decrypt, flags) == 1)) {
			*cert_state = CST_get_state(gl_smime_st, decrypt_cert);
			*recip_cert_id = CST_search_by_X509(gl_smime_st, decrypt_cert);
			clean_decrypted_memory(in_file_decrypt,
					       out_file_decrypt, p7, all_certs, pkey);
			ret = complete_decrypt(inter_file, out_decrypt,
					       detached, saved, signer_email_id,
					       signer_cert_state, signature, trust, cert_path,
					       decr_verify, smime, sender_cert_id);
			return ret;
		}
	}
	clean_decrypted_memory(in_file_decrypt, out_file_decrypt, p7, all_certs, pkey);
	return SMIME_UNABLE_TO_DECRYPT;
}

static STACK_OF(X509) * get_certificates()
{
	STACK_OF(X509) * cert_stack = NULL;
	X509 *cert = NULL;

	osso_log(LOG_DEBUG, "In get_certificates\n");

	if (!(cert_stack = sk_X509_new_null())) {
		osso_log(LOG_ERR, "No memory to create X509 list\n");
		return NULL;
	}

	cert = CST_get_cert(gl_smime_st, gl_cert_id);

	if (cert != NULL) {
		sk_X509_push(cert_stack, cert);
	}

	return cert_stack;
}


static
gint get_decryption_certs(const gchar * email_id, STACK_OF(X509) ** all_certs)
{

	gint ret = 0;

	osso_log(LOG_DEBUG, "Getting certificates for decryption\n");

	*all_certs = get_certificates();

	if ((all_certs == NULL) || (sk_X509_num(*all_certs) < 1)) {
		*all_certs = get_all_private_certificates();
	}

	if ((*all_certs == NULL) || (sk_X509_num(*all_certs) < 1)) {
		osso_log(LOG_ERR, "No certificates for decryption\n");
		ret = SMIME_NO_CERT_FOR_DECRYPTION_ERROR;
	}

	return ret;
}

static
gint complete_decrypt(gchar * inter_file, const gchar * outfile,
		      const gchar * detached, gboolean * saved,
		      const gchar * signer_email_id,
		      gint * signer_cert_state, gboolean * signature,
		      gboolean * trust, const gchar * cert_path, gboolean decr_verify,
		      gint * smime, gint * sender_cert_id)
{
	gint smime_type = 0;
	gchar buffer[MAXPATH + 1];
	FILE *interfp = NULL, *outfp = NULL;
	gint retval = 0;

	CHECK_PARAMETER((inter_file == NULL), "Input parameter NULL",
			return SMIME_DECRYPT_PROCESSING_ERROR);
	CHECK_PARAMETER((outfile == NULL), "Input parameter NULL",
			return SMIME_DECRYPT_PROCESSING_ERROR);


	if (decr_verify == FALSE)
		smime_type = find_smime_type(inter_file);

	switch (smime_type) {

	case SMIME_NONE:
		*smime = SMIME_ENCRYPT;
		if ((interfp = fopen(inter_file, F_READ)) == NULL) {
			osso_log(LOG_ERR, "File opening error\n");
			retval = SMIME_DECRYPT_PROCESSING_ERROR;
			return retval;
		}
		if (change_file_perm_rw(interfp, inter_file) < 0) {
			osso_log(LOG_ERR, "File error\n");
			fclose(interfp);
			retval = SMIME_DECRYPT_PROCESSING_ERROR;
			return retval;
		}
		if ((outfp = fopen(outfile, F_WRITE)) == NULL) {
			osso_log(LOG_ERR, "File opening error\n");
			fclose(interfp);
			retval = SMIME_DECRYPT_PROCESSING_ERROR;
			return retval;
		}
		if (change_file_perm_rw(outfp, outfile) < 0) {
			osso_log(LOG_ERR, "File error\n");
			fclose(interfp);
			fclose(outfp);
			retval = SMIME_DECRYPT_PROCESSING_ERROR;
			return retval;
		}

		memset(buffer, '\0', sizeof(buffer));
		while (fgets(buffer, sizeof(buffer), interfp) != NULL) {
			smime_strcrchomp(buffer);
			if (fputs(buffer, outfp) == EOF) {
				fclose(outfp);
				fclose(interfp);
				remove_file(inter_file);
				return SMIME_DECRYPT_PROCESSING_ERROR;
			}
			memset(buffer, '\0', sizeof(buffer));
		}

		fclose(outfp);
		fclose(interfp);
		remove_file(inter_file);
		return retval;

	case SMIME_SIGN:
		*smime = SMIME_SIGN_ENCRYPT;
		retval = verify_message(inter_file, outfile, detached,
					saved, signer_email_id,
					signer_cert_state, signature, trust,
					cert_path, sender_cert_id);
		remove_file(inter_file);
		return retval;

	default:
		return retval;
	}

}

static
void clean_decrypted_memory(BIO * in, BIO * out, PKCS7 * p7,
			    STACK_OF(X509) * certs, EVP_PKEY * pkey)
{
	if (in != NULL) {
		BIO_free(in);
	}
	if (out != NULL) {
		BIO_free(out);
	}
	if (certs != NULL) {
		sk_X509_pop_free(certs, X509_free);
	}
	if (p7 != NULL) {
		PKCS7_free(p7);
	}
	if (pkey != NULL) {
		EVP_PKEY_free(pkey);
	}
}


gint decrypt_verify(const gchar * in_decrypt, const gchar * recip_email,
		    gint * private_cert_state, gchar * outfile, gchar * detached_file,
		    gboolean * saved, const gchar * signer_email,
		    gint * signer_cert_state, gboolean * signature,
		    gboolean * trust, gchar * cert_path, gint * sender_cert_id,
		    gint * recip_cert_id)
{
	gchar decrypted[MAXPATH + 1];
	gboolean decr_verify = TRUE;
	gint retval = 0;
	gint smime = 0;

	osso_log(LOG_DEBUG, "Decrypting and verifying:%s\n", in_decrypt);

	if (FALSE == (get_file_path(decrypted, MAXPATH + 1))) {
		return SMIME_DECRYPT_PROCESSING_ERROR;
	}

	if ((retval = (decrypt_message(in_decrypt, decrypted, recip_email,
				       private_cert_state, detached_file,
				       saved, signer_email, signer_cert_state,
				       signature, trust, cert_path,
				       decr_verify, &smime, sender_cert_id, recip_cert_id))) != 0) {
		osso_log(LOG_ERR, "Decryption Failed\n");
		remove_file(decrypted);
		return retval;
	}


	if ((retval = (verify_message(decrypted, outfile, detached_file, saved,
				      recip_email, signer_cert_state, signature, trust,
				      cert_path, sender_cert_id))) != 0) {
		osso_log(LOG_ERR, "Verification Failed\n");
		remove_file(decrypted);
		return retval;
	}


	/* The intermediate file created after
	 * decrypting the message. This file
	 * is input to verify_message
	 */
	remove_file(decrypted);

	return retval;
}


static
PKCS7 *extract_p7(const gchar * filename)
{
	BIO *file = NULL, *in_data = NULL;
	PKCS7 *p7;

	osso_log(LOG_DEBUG, "Extracting PKCS7 from: %s\n", filename);

	CHECK_PARAMETER((filename == NULL), "Input parameter is NULL\n", return NULL);

	/* Memory allocated is freed at the end of the function */
	if (NULL == (file = BIO_new_file(filename, F_READ))) {
		osso_log(LOG_ERR, "Cannot create a BIO for extracting PKCS7\n");
		return NULL;
	}

	if (NULL == (p7 = SMIME_read_PKCS7(file, &in_data))) {
		BIO_free(file);
		return NULL;
	}

	BIO_free(file);
	return p7;
}

gint find_smime_type(const gchar * file)
{
	PKCS7 *p7 = NULL;
	gint smime_type = 0;

	osso_log(LOG_DEBUG, "Finding S/MIME type\n");

	CHECK_PARAMETER((file == NULL), "Input parameter is NULL", return smime_type);

	if (NULL == (p7 = extract_p7(file))) {
		osso_log(LOG_INFO, "S/MIME type:%d\n", smime_type);
		return smime_type;
	}

	if (PKCS7_type_is_signedAndEnveloped(p7)) {
		smime_type = SMIME_SIGN_ENCRYPT;
	}

	else if (PKCS7_type_is_signed(p7)) {
		smime_type = SMIME_SIGN;
	}

	else if (PKCS7_type_is_enveloped(p7)) {
		smime_type = SMIME_ENCRYPT;
	}
	PKCS7_free(p7);
	osso_log(LOG_INFO, "S/MIME type:%d\n", smime_type);
	return smime_type;
}

gboolean smime_init(void)
{
	osso_log(LOG_DEBUG, "Initializing ciphers, digests, algorithms\n");
	OpenSSL_add_all_ciphers();
	OpenSSL_add_all_digests();
	OpenSSL_add_all_algorithms();

	/* Memory allocated here will be freed in smime_cleanup() */
	gl_smime_st = CST_open(FALSE, NULL);

	if (gl_smime_st == NULL) {
		osso_log(LOG_ERR, "No certificate store handler\n");
		return FALSE;
	}

	return TRUE;
}

void smime_cleanup(void)
{
	osso_log(LOG_DEBUG, "Closing the CM store handler\n");

	EVP_cleanup();

	if (gl_smime_st != NULL)
		CST_free(gl_smime_st);
}

static STACK_OF(X509) * get_all_certificates(const gchar * email_id)
{

	GSList *certs_list = NULL;
	GSList *count = NULL;
	X509 *cert = NULL;
	STACK_OF(X509) * cert_stack = NULL;

	osso_log(LOG_DEBUG, "Getting all certificates from CM\n");

	if (email_id == NULL) {
		osso_log(LOG_ERR, "Input parameter email_id is NULL\n");
		return NULL;
	}

	/* This certificate store API returns all certificates
	 * associated with this email_id
	 */
	certs_list = CST_search_by_email(gl_smime_st, email_id);

	if (certs_list == NULL) {
		osso_log(LOG_ERR, "No certs associated with email_id\n");
		return NULL;
	}

	if (!(cert_stack = sk_X509_new_null())) {
		osso_log(LOG_ERR, "No memory to create X509 list\n");
		return NULL;
	}

	for (count = certs_list; count != NULL; count = count->next) {
		cert = CST_get_cert(gl_smime_st, GPOINTER_TO_UINT(count->data));
		sk_X509_push(cert_stack, cert);
	}

	g_slist_free(certs_list);
	return cert_stack;
}

static STACK_OF(X509) * get_all_private_certificates(void)
{
	GSList *certs_list = NULL;
	GSList *count = NULL;
	STACK_OF(X509) * cert_stack = NULL;
	X509 *cert = NULL;

	osso_log(LOG_DEBUG, "Getting all private certificate from store\n");
	certs_list = CST_search_by_folder(gl_smime_st, CST_FOLDER_PERSONAL);

	if (certs_list == NULL) {
		osso_log(LOG_ERR, "No certs private certificate available\n");
		return NULL;
	}

	if (!(cert_stack = sk_X509_new_null())) {
		osso_log(LOG_ERR, "No memory to create X509 list\n");
		return NULL;
	}

	for (count = certs_list; count != NULL; count = count->next) {
		cert = CST_get_cert(gl_smime_st, GPOINTER_TO_UINT(count->data));
		sk_X509_push(cert_stack, cert);
	}

	g_slist_free(certs_list);
	return cert_stack;

}

static
gint change_file_perm_rw(FILE * fp, const gchar * file)
{
#if HAVE_FCHMOD
	return fchmod(fileno(fp), S_IRUSR | S_IWUSR);
#else
	return chmod(file, S_IRUSR | S_IWUSR);
#endif
}

static
gchar *smime_strcrchomp(gchar * str)
{
	register gchar *s;
	if (!*str)
		return str;
	s = str + strlen(str) - 1;
	if (*s == '\n' && s > str && *(s - 1) == '\r') {
		*(s - 1) = '\n';
		*s = '\0';
	}
	return str;
}
