
/**
 * This file is part of osso-email-common
 *
 * Copyright (c) 2006 Nokia Corporation.
 * 
 * Contact: Dirk-Jan Binnema <dirk-jan.binnema@nokia.com>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <libgnomevfs/gnome-vfs-xfer.h>
#include <libgnomevfs/gnome-vfs-file-info.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include "xml_cache.h"
#include <libosso.h>
#include <log-functions.h>
#include "utils.h"
#include "defs.h"


gboolean xml_cache_is_up_to_date(gchar * folder_name, gchar * arch_path)
{
	GnomeVFSResult result = GNOME_VFS_OK;
	GnomeVFSFileInfo *stat_buf = NULL;
	gchar *xml_cache = NULL;
	time_t cache_time, folder_time;
	gchar *folder = NULL;

	osso_log(LOG_DEBUG, "Checking if cache is up to date\n");

	XML_CHECK_STATUS((folder_name == NULL), "Input param folder_name NULL", return FALSE);

	if (arch_path == NULL) {
		/* Folder population is not for archival folder */
		folder = folder_name;
	} else {
		/*folder population is for archival folder
		 * Use mounted path
		 */
		folder = arch_path;
	}

	stat_buf = gnome_vfs_file_info_new();

	/* determine mod time for folder */
	if ((result = gnome_vfs_get_file_info(folder, stat_buf,
					      GNOME_VFS_FILE_INFO_FOLLOW_LINKS |
					      GNOME_VFS_FILE_INFO_DEFAULT)) != GNOME_VFS_OK) {
		osso_log(LOG_WARNING, "could not stat folder %s: %s\n",
			 folder_name, strerror(errno));
		gnome_vfs_file_info_unref(stat_buf);
		return FALSE;
	} else
		folder_time = stat_buf->mtime;

	gnome_vfs_file_info_clear(stat_buf);

	/* determine mod time for cache */
	xml_cache = g_strdup_printf("%s/%s", folder_name, XML_CACHE_FILE);
	if ((result = gnome_vfs_get_file_info(xml_cache, stat_buf,
					      GNOME_VFS_FILE_INFO_FOLLOW_LINKS |
					      GNOME_VFS_FILE_INFO_DEFAULT)) != GNOME_VFS_OK) {
		osso_log(LOG_WARNING, "could not stat cache file %s: %s",
			 xml_cache, strerror(errno));
		g_free(xml_cache);
		gnome_vfs_file_info_unref(stat_buf);
		return FALSE;
	}
	cache_time = stat_buf->mtime;
	g_free(xml_cache);
	gnome_vfs_file_info_unref(stat_buf);

	/* are we up to date? */
	return folder_time == cache_time;
}

gchar *xml_cache_get(const gchar * folder_name, gint index)
{
	gchar *xml_cache = NULL, *data = NULL;
	GError *err = NULL;
	gchar *xml_cache_num = NULL;
	gchar *xml_cache_file = NULL;

	osso_log(LOG_DEBUG, "Getting contents of cache file\n");

	XML_CHECK_STATUS((folder_name == NULL), "Input param folder_name NULL", return NULL);

	xml_cache_file = g_strdup_printf("%s/%s", folder_name, XML_CACHE_FILE);

	/* get cache file based on index. Example cache_xml0, cache_xml200 */
	xml_cache_num = g_strdup_printf("%s%d", xml_cache_file, index);

	/* If cache file with a particular index is not available
	   * it means that all contents have been sent and no more 
	   * cache files to be sent. Only NULL message is to be sent
	   * For NULL message use cache file with NULL message (cach.xml) 
	   * else use cache_xml(n) where n is the index
	 */
	if (is_file_entry_exist(xml_cache_num) == TRUE) {
		xml_cache = xml_cache_num;
	} else {
		xml_cache = xml_cache_file;
	}


	if (!g_file_get_contents(xml_cache, &data, NULL, &err)) {
		if (err) {
			osso_log(LOG_WARNING, "could not read cache contents from %s: %s\n",
				 xml_cache, err->message);
			g_error_free(err);
		}

		g_free(data);
		data = NULL;
	}

	g_free(xml_cache_file);
	g_free(xml_cache_num);
	return data;
}

gboolean
xml_cache_update(const gchar * folder_name, const gchar * xmldata,
		 size_t size, gint index, gboolean is_last)
{
	gchar *xml_cache = NULL;
	FILE *fp = NULL;
	gboolean res = FALSE;
	gchar *xml_cache_num = NULL;
	gchar *xml_cache_file = NULL;

	osso_log(LOG_DEBUG, "Updating the cache:%s\n", folder_name);

	XML_CHECK_STATUS((xmldata == NULL), "Input param xmldata NULL", return FALSE);
	XML_CHECK_STATUS((folder_name == NULL), "Input param folder_name NULL", return FALSE);

	xml_cache = g_strdup_printf("%s/%s", folder_name, XML_CACHE_FILE);
	xml_cache_file = g_strdup_printf("%s%d", xml_cache, index);
	osso_log(LOG_DEBUG, "XML cache file is:%s\n", xml_cache);

	/* If index is zero it means cache files re-generation is required
	 * hence remove all existing cache files, so that we dont use them 
	 * for next population
	 */
	if (index == 0) {
		xml_cache_remove(xml_cache);
	}

	/*If its the last cache file to be written, it will have no index
	 * cache.xml will be its name, else it will have an index appended
	 * to its name, example cache.xml0, cache.xml200 etc
	 */
	if (is_last == TRUE) {
		xml_cache_num = xml_cache;
	} else {
		xml_cache_num = xml_cache_file;
	}

	if (!(fp = fopen(xml_cache_num, "w+"))) {
		send_engine_error_to_ui(errno);
		osso_log(LOG_WARNING, "could not open cache file %s for writing: %s\n",
			 xml_cache, strerror(errno));
	} else {
		if (fwrite(xmldata, size, 1, fp) != 1) {
			send_engine_error_to_ui(errno);
			osso_log(LOG_WARNING, "error writing to cache file");
		} else
			res = TRUE;

		fclose(fp);
	}

	g_free(xml_cache);
	g_free(xml_cache_file);
	return res;
}

void xml_cache_remove(gchar * xml_cache)
{
	gint count = 0;
	gboolean is_exist = TRUE;
	gchar *cache_file = NULL;

	while (is_exist == TRUE) {
		cache_file = g_strdup_printf("%s%d", xml_cache, count);
		is_exist = is_file_entry_exist(cache_file);
		if (is_exist == TRUE) {
			unlink(cache_file);
		}
		g_free(cache_file);
		count = count + NUM_MAILS;
	}
	unlink(xml_cache);

}
