
/*
 * LibSylph -- E-Mail client library
 * Copyright (C) 1999-2006 Hiroyuki Yamamoto
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef __UTILS_H__
#define __UTILS_H__

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <glib.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <dirent.h>
#include <time.h>
#include <errno.h>

#if HAVE_ALLOCA_H
#  include <alloca.h>
#endif
#if HAVE_WCHAR_H
#  include <wchar.h>
#endif


#define FILE_PROCESS_SUCCESS 0
#define FILE_PROCESS_FAIL -1

/* The AC_CHECK_SIZEOF() in configure fails for some machines.
 * we provide some fallback values here */

#if !SIZEOF_UNSIGNED_SHORT
#undef SIZEOF_UNSIGNED_SHORT
#define SIZEOF_UNSIGNED_SHORT 2
#endif
#if !SIZEOF_UNSIGNED_INT
#undef SIZEOF_UNSIGNED_INT
#define SIZEOF_UNSIGNED_INT 4
#endif
#if !SIZEOF_UNSIGNED_LONG
#undef SIZEOF_UNSIGNED_LONG
#define SIZEOF_UNSIGNED_LONG 4
#endif

#ifndef HAVE_U32_TYPEDEF
#undef u32			/* maybe there is a macro with this name */
typedef guint32 u32;
#define HAVE_U32_TYPEDEF
#endif

#ifndef BIG_ENDIAN_HOST
#if (G_BYTE_ORDER == G_BIG_ENDIAN)
#define BIG_ENDIAN_HOST 1
#endif
#endif

#define CHDIR_RETURN_IF_FAIL(dir) \
{ \
	if (change_dir(dir) < 0) return; \
}

#define CHDIR_RETURN_VAL_IF_FAIL(dir, val) \
{ \
	if (change_dir(dir) < 0) return val; \
}

#define Xalloca(ptr, size, iffail) \
{ \
	if ((ptr = alloca(size)) == NULL) { \
		g_warning("can't allocate memory\n"); \
		iffail; \
	} \
}

#define Xstrdup_a(ptr, str, iffail) \
{ \
	gchar *__tmp; \
 \
	if ((__tmp = alloca(strlen(str) + 1)) == NULL) { \
		g_warning("can't allocate memory\n"); \
		iffail; \
	} else \
		strcpy(__tmp, str); \
 \
	ptr = __tmp; \
}

#define Xstrndup_a(ptr, str, len, iffail) \
{ \
	gchar *__tmp; \
 \
	if ((__tmp = alloca(len + 1)) == NULL) { \
		g_warning("can't allocate memory\n"); \
		iffail; \
	} else { \
		strncpy(__tmp, str, len); \
		__tmp[len] = '\0'; \
	} \
 \
	ptr = __tmp; \
}

#define Xstrcat_a(ptr, str1, str2, iffail) \
{ \
	gchar *__tmp; \
	gint len1, len2; \
 \
	len1 = strlen(str1); \
	len2 = strlen(str2); \
	if ((__tmp = alloca(len1 + len2 + 1)) == NULL) { \
		g_warning("can't allocate memory\n"); \
		iffail; \
	} else { \
		memcpy(__tmp, str1, len1); \
		memcpy(__tmp + len1, str2, len2 + 1); \
	} \
 \
	ptr = __tmp; \
}

#define FILE_OP_ERROR(file, func) \
{ \
	log_message((gchar *) ("%s:%s error\n"),file,func);	\
}

#define MAX_MSGID_LEN 256

#define GET_MSG_NUM(msgnum,msgid) \
{ \
 gchar *tmp_ptr; \
 tmp_ptr = strrchr(msgid,MSGID_SEPARATOR); \
 tmp_ptr ++; \
 msgnum = atoi(tmp_ptr);\
}


#define GET_MSG_ID(ptr,account_name,folder,msgnum)	\
{	\
	ptr =(gchar *)g_malloc(MAX_MSGID_LEN);	\
	if (ptr!= NULL) {	\
	g_snprintf(ptr, MAX_MSGID_LEN,"%s%c%s%c%d",account_name,MSGID_SEPARATOR ,folder,MSGID_SEPARATOR ,msgnum); \
	\
	}       \
}

#define MAKE_DIR_IF_NOT_EXIST(dir) \
{ \
        if (!is_dir_exist(dir)) { \
                if (is_file_exist(dir)) { \
                        return -1; \
                } \
                if (make_dir(dir) < 0) \
                        return -1; \
        } \
}

#define SET_CB_FLAG \
{ \
  gl_callback_enable_flag = TRUE; \
}

#define UNSET_CB_FLAG \
{ \
  gl_callback_enable_flag = FALSE; \
}

#define CHECK_CB_FLAG \
( \
   gl_callback_enable_flag ==TRUE?1:0\
)
#define SET_MMC_FLAG \
{ \
	  gl_mmc_status_flag = TRUE; \
}

#define UNSET_MMC_FLAG \
{ \
	  gl_mmc_status_flag = FALSE; \
}

#define CHECK_MMC_FLAG \
( \
     gl_mmc_status_flag ==TRUE?1:0\
)
#define CHECK_RETURN_VAL_IF_FAIL(check,ret_val)  \
{                                                \
    if(!(check)){                                \
        return ret_val;                          \
    }	                                         \
}
#define CHECK_RETURN_IF_FAIL(check)              \
{                                                \
    if(!(check)){                                \
        return ;                                 \
    }	                                         \
}

/* for macro expansion */
#define Str(x)	#x
#define Xstr(x)	Str(x)

/*smime changes*/
#define BSIZE 4095
#define LESSER_THAN '<'
#define GREATER_THAN '>'
#define TMP_FILE "tmpmsg"


void list_free_strings(GList * list);
void slist_free_strings(GSList * list);

void hash_free_strings(GHashTable * table);
void hash_free_value_mem(GHashTable * table);

gint str_case_equal(gconstpointer v, gconstpointer v2);
guint str_case_hash(gconstpointer key);

void ptr_array_free_strings(GPtrArray * array);

/* number-string conversion */
gint to_number(const gchar * nstr);
gchar *itos_buf(gchar * nstr, gint n);
gchar *itos(gint n);
gchar *to_human_readable(off_t size);

/* alternative string functions */
gint strcmp2(const gchar * s1, const gchar * s2);
gint path_cmp(const gchar * s1, const gchar * s2);
gchar *strretchomp(gchar * str);
gchar *strtailchomp(gchar * str, gchar tail_char);
gchar *strcrchomp(gchar * str);
gchar *strcasestr(const gchar * haystack, const gchar * needle);
gchar *strncpy2(gchar * dest, const gchar * src, size_t n);

/* wide-character functions */
#if !HAVE_ISWALNUM
int iswalnum(wint_t wc);
#endif
#if !HAVE_ISWSPACE
int iswspace(wint_t wc);
#endif
#if !HAVE_TOWLOWER
wint_t towlower(wint_t wc);
#endif

#if !HAVE_WCSLEN
size_t wcslen(const wchar_t * s);
#endif
#if !HAVE_WCSCPY
wchar_t *wcscpy(wchar_t * dest, const wchar_t * src);
#endif
#if !HAVE_WCSNCPY
wchar_t *wcsncpy(wchar_t * dest, const wchar_t * src, size_t n);
#endif

wchar_t *wcsdup(const wchar_t * s);
wchar_t *wcsndup(const wchar_t * s, size_t n);
wchar_t *strdup_mbstowcs(const gchar * s);
gchar *strdup_wcstombs(const wchar_t * s);
gint wcsncasecmp(const wchar_t * s1, const wchar_t * s2, size_t n);
wchar_t *wcscasestr(const wchar_t * haystack, const wchar_t * needle);
gint get_mbs_len(const gchar * s);

gboolean is_next_nonascii(const guchar * s);
gint get_next_word_len(const gchar * s);

/* functions for string parsing */
gint subject_compare(const gchar * s1, const gchar * s2);
gint subject_compare_for_sort(const gchar * s1, const gchar * s2);
void trim_subject_for_compare(gchar * str);
void trim_subject_for_sort(gchar * str);
void trim_subject(gchar * str);
void eliminate_parenthesis(gchar * str, gchar op, gchar cl);
void extract_parenthesis(gchar * str, gchar op, gchar cl);

void extract_parenthesis_with_skip_quote(gchar * str, gchar quote_chr, gchar op, gchar cl);

void eliminate_quote(gchar * str, gchar quote_chr);
void extract_quote(gchar * str, gchar quote_chr);
void eliminate_address_comment(gchar * str);
gchar *strchr_with_skip_quote(const gchar * str, gint quote_chr, gint c);
gchar *strrchr_with_skip_quote(const gchar * str, gint quote_chr, gint c);
void extract_address(gchar * str);
void extract_list_id_str(gchar * str);

GSList *address_list_append(GSList * addr_list, const gchar * str);
GSList *references_list_append(GSList * msgid_list, const gchar * str);
GSList *newsgroup_list_append(GSList * group_list, const gchar * str);

GList *add_history(GList * list, const gchar * str);

gchar *remove_quote(const gchar * buf);

void remove_return(gchar * str);
void remove_space(gchar * str);
void unfold_line(gchar * str);
void subst_char(gchar * str, gchar orig, gchar subst);
void subst_chars(gchar * str, gchar * orig, gchar subst);
void subst_for_filename(gchar * str);
gboolean is_header_line(const gchar * str);
gboolean is_ascii_str(const guchar * str);
gint get_quote_level(const gchar * str);

gchar *strstr_with_skip_quote(const gchar * haystack, const gchar * needle);
gchar *strchr_parenthesis_close(const gchar * str, gchar op, gchar cl);

gchar **strsplit_parenthesis(const gchar * str, gchar op, gchar cl, gint max_tokens);
gchar **strsplit_with_quote(const gchar * str, const gchar * delim, gint max_tokens);

gchar *get_abbrev_newsgroup_name(const gchar * group, gint len);
gchar *trim_string(const gchar * str, gint len);

GList *uri_list_extract_filenames(const gchar * uri_list);
void decode_uri(gchar * decoded_uri, const gchar * encoded_uri);
gint scan_mailto_url(const gchar * mailto,
		     gchar ** to, gchar ** cc, gchar ** bcc, gchar ** subject, gchar ** body);

/* return static strings */
const gchar *get_home_dir(void);
const gchar *get_rc_dir(void);
const gchar *get_mail_dir(void);
gchar *get_news_cache_dir(void);
const gchar *get_imap_cache_dir(void);
const gchar *get_mime_tmp_dir(void);
const gchar *get_template_dir(void);
const gchar *get_tmp_dir(void);
gchar *get_tmp_file(void);
gchar *get_domain_name(void);
const gchar *get_tem_attach_dir(void);

/* file / directory handling */
off_t get_file_size(const gchar * file);
off_t get_file_size_as_crlf(const gchar * file);
off_t get_left_file_size(FILE * fp);

gboolean file_exist(const gchar * file, gboolean allow_fifo);
gboolean is_dir_exist(const gchar * dir);
gboolean is_file_entry_exist(const gchar * file);
gboolean dirent_is_regular_file(struct dirent *d);
gboolean dirent_is_directory(struct dirent *d);

#define is_file_exist(file)		file_exist(file, FALSE)
#define is_file_or_fifo_exist(file)	file_exist(file, TRUE)
#define MAX_FILENAME_LEN	256

gint change_dir(const gchar * dir);
gint make_dir(const gchar * dir);
gint make_dir_hier(const gchar * dir);
gint remove_all_files(const gchar * dir);
gint remove_numbered_files(const gchar * dir, guint first, guint last);
gint remove_all_numbered_files(const gchar * dir);
gint remove_expired_files(const gchar * dir, guint hours);
gint remove_dir_recursive(const gchar * dir);
gint copy_file(const gchar * src, const gchar * dest, gboolean keep_backup);
gint move_file(const gchar * src, const gchar * dest, gboolean overwrite);
gint copy_file_part(FILE * fp, off_t offset, size_t length, const gchar * dest);

gchar *canonicalize_str(const gchar * str);
gint canonicalize_file(const gchar * src, const gchar * dest);
gint canonicalize_file_replace(const gchar * file);
gint uncanonicalize_file(const gchar * src, const gchar * dest);
gint uncanonicalize_file_replace(const gchar * file);

gchar *normalize_newlines(const gchar * str);

gchar *get_outgoing_rfc2822_str(FILE * fp);
gchar *generate_mime_boundary(const gchar * prefix);

gint change_file_mode_rw(FILE * fp, const gchar * file);
//FILE *my_tmpfile ( gchar * outfile );
FILE *str_open_as_stream(const gchar * str, gchar ** outfile);
gint str_write_to_file(const gchar * str, const gchar * file);
gchar *file_read_to_str(const gchar * file);
gchar *file_read_stream_to_str(FILE * fp);

/* process execution */
gint execute_async(gchar * const argv[]);
gint execute_sync(gchar * const argv[]);
gint execute_command_line(const gchar * cmdline, gboolean async);
gchar *get_command_output(const gchar * cmdline);

/* open URI with external browser */
gint open_uri(const gchar * uri, const gchar * cmdline);

/* time functions */
time_t remote_tzoffset_sec(const gchar * zone);
time_t tzoffset_sec(time_t * now);
gchar *tzoffset(time_t * now);
void get_rfc822_date(gchar * buf, gint len);

gchar *get_new_path(const gchar * folder, const gchar * account_name, const gchar * msgid);
gchar *get_account_name(const gchar * msgid);

gint get_msgid_num(const gchar * msgid);

gchar *get_fold_name(const gchar * msgid);

gboolean file_perm_check(const gchar * file, gboolean read_write);

gboolean is_dir_perm_exist(const gchar * dir, gboolean full_perms_check);

gboolean is_inbox_perm_exist(void);

gint compare_account_name(gconstpointer a, gconstpointer b);

gint get_engine_errno(void);
void set_engine_errno(int error);

/* logging */
void set_log_file(const gchar * filename);
void close_log_file(void);
void log_verbosity_set(gboolean verbose);
void debug_print(const gchar * format, ...) G_GNUC_PRINTF(1, 2);
void log_print(const gchar * format, ...) G_GNUC_PRINTF(1, 2);
void log_message(const gchar * format, ...) G_GNUC_PRINTF(1, 2);
void log_warning(const gchar * format, ...) G_GNUC_PRINTF(1, 2);
void log_error(const gchar * format, ...) G_GNUC_PRINTF(1, 2);

/** 
  This function creates a temporary file path
  
  @param tmp - File path with file name
  @param tmplen - Length of file path
  @return gboolean - TRUE on success and FALSE on failure
*/
gboolean get_temp_smime_file(gchar * tmp, gint tmplen);

/**
  This function gets the sender address from the address field

  @param address - Address string
  @return gchar* sender email ID
*/
gchar *parse_from_address(const gchar * address);

/**
   This function checks if the file is a temporary file. 
   If yes, the file is removed else file is not removed

   @param file - File to be removed
   @return void - No return value
*/
void remove_smime_file(const gchar * file);

/**
    This function enables the device mode flag to TRUE when device is in Normal
    mode
    @param none
    @return none
*/
void enable_device_mode_flag();

/**
  This function disables the device mode flag to FALSE when device is in flight
  mode
  @param none
  @return none
*/
void disable_device_mode_flag();

/**
   This function returns the device mode flag 
   @param none
   @return flag FALSE on FLIGHT MODE, TRUE on NORMAl MODE 
*/
gboolean get_device_mode_flag();


/**
 This is a helper function to compare UTF8 strings. Input strings are
 made case insensitive first by using 'g_utf8_casefold' and then compared
 with 'g_utf8_collate'.

 @param  str1  First string to be compared
 @param  str2  Second string to be compared
 @return gint  < 0 if str1 compares before str2,
              0 if they compare equal,
              > 0 if str1 compares after str2.
*/
gint compare_utf8_strings(const gchar * str1, const gchar * str2);

/*
 * convert data to RFC2822 format; similar to get_outgoing_rfc2822_str
 * but uses tempfile so we don't need to allocate so much memory
 * byte_size will receive the number of bytes written
 */

FILE *get_outgoing_rfc2822_file(FILE * fp, size_t * bytes, gchar ** outfile);

void enable_device_mode_flag(void);
void disable_device_mode_flag(void);
gboolean get_device_mode_flag(void);
void hash_free_value_mem_func(gpointer key, gpointer value, gpointer data);

/**
  This function used to send the engine error code to UI pplication. 
  @param errorno generated in engine module
  return 0 on success 
*/

gint send_engine_error_to_ui(gint errorno);

/** 
  This function is called to generate sound on new mail received 
  by application
  @param  play_file which will be supplied to esd daemon
  @return nothing 
 */

void email_play_sound(const gchar * play_file);


/** 
  This function is called to get the file extension,if it is present 
  @param filename 
  @return pointer to the extension,starting from '.' character. 
 */
const gchar *get_file_extension(const gchar * filename);
void email_display_led (void);
void set_scheduling_op(gboolean scheduled_op);
void email_glow_backlight();

/**
  This function will be called to decode a mhtml file
  @param mhtml_file   mhtml file to decode
  @param msgid   the msgid of the file
  @param edit_flag   the flag idnicates whether message is
  opened in editor
  @param get_html_flag    indicates whether the html part of
  message has to be decoded
 @return decoded file path.
 */
gchar *decode_mhtml_file(const gchar * mhtml_file,
                               const gchar * msgid, gboolean edit_flag,
                               gboolean get_html_flag);

gchar *get_file_name(const gchar * msgid, const gchar * sec_nu);

#endif				/* __UTILS_H__ */
