
/**
 * This file is part of osso-email-common
 *
 * Copyright (c) 2006 Nokia Corporation.
 * 
 * Contact: Dirk-Jan Binnema <dirk-jan.binnema@nokia.com>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif				/* */

#include <glib.h>
#include <errno.h>
    
#include "defs.h"
#include "session.h"
#include "socket.h"
#include "utils.h"
static gboolean session_write_data_cb_fp(SockInfo * source, 
					   GIOCondition condition, 
gpointer data);
static void session_destroy_fp(Session * session);
static void session_destroy_fp(Session * session) 
{
	g_return_if_fail(session);
	if (!session->write_buf_fp)
		return;
	if (session->write_buf_fp) {
		if (fclose(session->write_buf_fp) != 0) {
			debug_print("closing session->write_buf_fp failed \n");
		}
		session->write_buf_fp = NULL;
	}
	session->write_buf_written = 0;
	session->write_buf_len = 0;
}

gint session_send_data_fp(Session * session, FILE * fp, guint size) 
{
	gboolean ret;
	g_return_val_if_fail(session != NULL, -1);
	g_return_val_if_fail(fp != NULL, -1);
	g_return_val_if_fail(size != 0, -1);
	session->state = SESSION_SEND;
	session->write_buf_fp = fp;
	session->write_buf_written = 0;
	session->write_buf_len = size;
	gettimeofday(&session->tv_prev, NULL);
	ret = session_write_data_cb_fp(session->sock, G_IO_OUT, session);
	if (ret == TRUE)
		session->io_tag =
		    sock_add_watch(session->sock, G_IO_OUT, session_write_data_cb_fp, session);
	
	else if (session->state == SESSION_ERROR) {
		session_destroy_fp(session);
		return -1;
	}
	return 0;
}

/* 
 * write the data from the FILE* to the socket
 * the FILE* will be fclosed in when the tranfers finished, has an error
 * or in session_destroy in case of user cancelation
 */ 
static gint session_write_buf_fp(Session * session) 
{
	guint write_len;
	guint to_write_len;
	gint result;
	static gchar write_buf[SESSION_BUFFSIZE];
	
	/* do we need to refill the buffer? (not in case of EAGAIN) */ 
	static gboolean refill_buffer = TRUE;
	g_return_val_if_fail(session->write_buf_fp != NULL, -1);
	if (session->write_buf_len <= 0) {
		g_warning("session->write_buf_len <= 0 (%d)", session->write_buf_len);
		session_destroy_fp(session);
		return -1;
	}
	to_write_len = session->write_buf_len - session->write_buf_written;
	to_write_len = MIN(to_write_len, SESSION_BUFFSIZE);
	
	/* read new bytes if we're not dealing with EAGAIN */ 
	if (refill_buffer) {
		result = fread(write_buf, to_write_len, 1, session->write_buf_fp);
		if (to_write_len && result != 1) {
			g_warning("could not read %d bytes from fp", to_write_len);
			session->state = SESSION_ERROR;
			session_destroy_fp(session);
			return -1;
		}
	}
	refill_buffer = TRUE;
	write_len = sock_write(session->sock, write_buf, to_write_len);
	if (write_len < 0) {
		switch (errno) {
		case EAGAIN:
			g_warning("incomplete write (EAGAIN)");
			refill_buffer = FALSE;	/* try again later with the same buffer */
			write_len = 0;
			break;
		default:
			g_warning("error: sock_write: %s\n", g_strerror(errno));
			session->state = SESSION_ERROR;
			session_destroy_fp(session);
			return -1;
		}
	}
	session->write_buf_written += write_len;
	
	/* are we done yet? */ 
	if (session->write_buf_written < session->write_buf_len) {
		
		    /* no */ 
		    return 1;
	} else {
		
		    /* yes */ 
		    session_destroy_fp(session);
		return 0;
	}
}
static gboolean session_write_data_cb_fp(SockInfo * source, 
					    GIOCondition condition, gpointer data) 
{
	Session * session = SESSION(data);
	gint ret;
	gint write_buf_len;
	g_return_val_if_fail(session->write_buf_fp != NULL, FALSE);
	if (condition != G_IO_OUT) {
		session_destroy_fp(session);
		return FALSE;
	}
	write_buf_len = session->write_buf_len;
	ret = session_write_buf_fp(session);
	if (ret < 0) {
		session->state = SESSION_ERROR;
		session_destroy_fp(session);
		return FALSE;
	} else if (ret > 0) {
		struct timeval tv_cur;
		gettimeofday(&tv_cur, NULL);
		if (tv_cur.tv_sec - session->tv_prev.tv_sec > 0
		     || tv_cur.tv_usec - session->tv_prev.tv_usec > UI_REFRESH_INTERVAL) {
			session_set_timeout(session, session->timeout_interval);
			session->send_data_progressive_notify 
			    (session, session->write_buf_written, session->write_buf_len);
			gettimeofday(&session->tv_prev, NULL);
		}
		return TRUE;
	}
	if (session->io_tag > 0) {
		g_source_remove(session->io_tag);
		session->io_tag = 0;
	}
	
	    /* callback */ 
	    ret = session->send_data_finished(session, write_buf_len);
	session->send_data_notify(session, write_buf_len);
	session_destroy_fp(session);
	return FALSE;
}


