
/*
 * LibSylph -- E-Mail client library
 * Copyright (C) 1999-2006 Hiroyuki Yamamoto
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef __PROCMSG_H__
#define __PROCMSG_H__

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <glib.h>
#include <stdio.h>
#include <time.h>
#include <sys/types.h>
#include <string.h>
#include "common.h"


typedef struct _MsgStatus MsgStatus;
typedef struct _Tag Tag;
typedef struct _MsgHeader MsgHeader;
typedef struct _EmailSpecificSettings EmailSpecificSettings;
typedef struct _MsgInfo MsgInfo;
typedef struct _MsgFlags MsgFlags;
typedef struct _MsgFileInfo MsgFileInfo;
typedef struct _AttachInfo AttachInfo;
typedef struct _RetrMsgData RetrMsgData;
typedef struct _InlineImageInfo InlineImageInfo;


typedef enum {
	DATA_READ,
	DATA_WRITE,
	DATA_APPEND
} DataOpenMode;

#define MSG_CLABEL_SBIT	(7)	/* start bit of color label */
#define MAKE_MSG_CLABEL(h, m, l)	(((h) << (MSG_CLABEL_SBIT + 2)) | \
					 ((m) << (MSG_CLABEL_SBIT + 1)) | \
					 ((l) << (MSG_CLABEL_SBIT + 0)))

#define MSG_CLABEL_NONE		MAKE_MSG_CLABEL(0U, 0U, 0U)
#define MSG_CLABEL_1		MAKE_MSG_CLABEL(0U, 0U, 1U)
#define MSG_CLABEL_2		MAKE_MSG_CLABEL(0U, 1U, 0U)
#define MSG_CLABEL_3		MAKE_MSG_CLABEL(0U, 1U, 1U)
#define MSG_CLABEL_4		MAKE_MSG_CLABEL(1U, 0U, 0U)
#define MSG_CLABEL_5		MAKE_MSG_CLABEL(1U, 0U, 1U)
#define MSG_CLABEL_6		MAKE_MSG_CLABEL(1U, 1U, 0U)
#define MSG_CLABEL_7		MAKE_MSG_CLABEL(1U, 1U, 1U)

#define MSG_CLABEL_ORANGE	MSG_CLABEL_1
#define MSG_CLABEL_RED		MSG_CLABEL_2
#define MSG_CLABEL_PINK		MSG_CLABEL_3
#define MSG_CLABEL_SKYBLUE	MSG_CLABEL_4
#define MSG_CLABEL_BLUE		MSG_CLABEL_5
#define MSG_CLABEL_GREEN	MSG_CLABEL_6
#define MSG_CLABEL_BROWN	MSG_CLABEL_7

/* RESERVED */
#define	MSG_RESERVED		(1U << 31)


#define MSG_CLABEL_FLAG_MASK	(MSG_CLABEL_7)

#define MSG_MOVE		(1U << 0)
#define MSG_COPY		(1U << 1)
#define MSG_QUEUED		(1U << 16)
#define MSG_DRAFT		(1U << 17)
#define MSG_ENCRYPTED		(1U << 18)
#define MSG_IMAP		(1U << 19)
#define MSG_NEWS		(1U << 20)
#define MSG_SIGNED		(1U << 21)
#define MSG_MIME		(1U << 29)
#define MSG_INVALID		(1U << 30)
#define MSG_CACHED		(1U << 31)


#define MSG_CACHED_FLAG_MASK	(MSG_MIME)

#define MSG_SET_FLAGS(msg, flags)	{ (msg) |= (flags); }
#define MSG_UNSET_FLAGS(msg, flags)	{ (msg) &= ~(flags); }
#define MSG_SET_PERM_FLAGS(msg, flags) \
	MSG_SET_FLAGS((msg).perm_flags, flags)
#define MSG_SET_TMP_FLAGS(msg, flags) \
	MSG_SET_FLAGS((msg).tmp_flags, flags)
#define MSG_UNSET_PERM_FLAGS(msg, flags) \
	MSG_UNSET_FLAGS((msg).perm_flags, flags)
#define MSG_UNSET_TMP_FLAGS(msg, flags) \
	MSG_UNSET_FLAGS((msg).tmp_flags, flags)

#define MSG_IS_PURGABLE(msg)		(((msg).tmp_flags & MSG_PURGABLE) != 0)
#define MSG_IS_NEW(msg)			(((msg).perm_flags & MSG_NEW) != 0)
#define MSG_IS_UNREAD(msg)		(((msg).perm_flags & MSG_UNREAD) != 0)
#define MSG_IS_MARKED(msg)		(((msg).perm_flags & MSG_MARKED) != 0)
#define MSG_IS_DELETED(msg)		(((msg).perm_flags & MSG_DELETED) != 0)
#define MSG_IS_REPLIED(msg)		(((msg).perm_flags & MSG_REPLIED) != 0)
#define MSG_IS_FORWARDED(msg)		(((msg).perm_flags & MSG_FORWARDED) != 0)
#define MSG_IS_ATTACH(msg)		(((msg).perm_flags & MSG_ATTACH) != 0)

#define MSG_GET_COLORLABEL(msg)		(((msg).perm_flags & MSG_CLABEL_FLAG_MASK))
#define MSG_GET_COLORLABEL_VALUE(msg)	(MSG_GET_COLORLABEL(msg) >> MSG_CLABEL_SBIT)
#define MSG_SET_COLORLABEL_VALUE(msg, val) \
	MSG_SET_PERM_FLAGS(msg, ((((guint)(val)) & 7) << MSG_CLABEL_SBIT))

#define MSG_IS_MOVE(msg)		(((msg).tmp_flags & MSG_MOVE) != 0)
#define MSG_IS_COPY(msg)		(((msg).tmp_flags & MSG_COPY) != 0)

#define MSG_IS_QUEUED(msg)		(((msg).tmp_flags & MSG_QUEUED) != 0)
#define MSG_IS_DRAFT(msg)		(((msg).tmp_flags & MSG_DRAFT) != 0)
#define MSG_IS_ENCRYPTED(msg)		(((msg).tmp_flags & MSG_ENCRYPTED) != 0)
#define MSG_IS_IMAP(msg)		(((msg).tmp_flags & MSG_IMAP) != 0)
#define MSG_IS_NEWS(msg)		(((msg).tmp_flags & MSG_NEWS) != 0)
#define MSG_IS_MIME(msg)		(((msg).tmp_flags & MSG_MIME) != 0)
#define MSG_IS_INVALID(msg)		(((msg).tmp_flags & MSG_INVALID) != 0)
#define MSG_IS_CACHED(msg)		(((msg).tmp_flags & MSG_CACHED) != 0)

#define WRITE_CACHE_DATA_INT(n, fp) \
	fwrite(&n, sizeof(n), 1, fp)

#define WRITE_CACHE_DATA(data, fp)			\
{							\
	gint len;					\
							\
	if (data == NULL) {				\
		len = 0;				\
		WRITE_CACHE_DATA_INT(len, fp);		\
	} else {					\
		len = strlen(data);			\
		WRITE_CACHE_DATA_INT(len, fp);		\
		if (len > 0)				\
			fwrite(data, len, 1, fp);	\
	}						\
}


typedef enum {
	MSG_NEW = 1 << 0,
	MSG_UNREAD = 1 << 1,
	MSG_MARKED = 1 << 2,
	MSG_DELETED = 1 << 3,
	MSG_REPLIED = 1 << 4,
	MSG_FORWARDED = 1 << 5,
	MSG_ATTACH = 1 << 6,
	MSG_EDITABLE = 1 << 7
} MsgPermFlags;

typedef enum {
	IMAP_CACHED = 1,
	IMAP_UNCACHED = 2,
	MSG_PURGABLE = 3
} MsgTmpFlags;

struct _MsgFlags {
	MsgPermFlags perm_flags;
	MsgTmpFlags tmp_flags;
};

struct _MsgStatus {
	gint msgnum;
	MsgFlags msgflags;
};



struct _Tag {
	RecvProtocol protocol;
	RecvType recvtype;
	FolderType foldertype;
	gboolean fullydownloaded;
	gboolean bodydownloaded;
	gboolean remote_copy;
	time_t timestamp;
};
typedef enum {
	LOW = 5,
	NORMAL = 3,
	HIGH = 1
} Priority;

typedef enum {
	SMIME_NONE = 0,
	SMIME_SIGN,
	SMIME_ENCRYPT,
	SMIME_SIGN_ENCRYPT,
	SMIME_INTERNAL
} SMimeType;

struct _MsgHeader {
	guint msgnum;
	off_t size;
	time_t mtime;
	time_t date_t;
	time_t rx_date;
	MsgFlags flags;
	gchar *fromname;
	gchar *date;
	gchar *from;
	gchar *to;
	gchar *cc;
	gchar *bcc;
	gchar *subject;
	gchar *msgid;
	gchar *replyto;
	gchar *inreplyto;
	gchar *username;
	gchar *password;
	gchar *account_name;
	Tag msgtag;
	Priority recv_priority;
	gboolean read_receipt_request;
	gboolean copy_to_own_address;
	SMimeType smime_type;	/* SMime Support */
	gint attach_tag;
};

struct _InlineImageInfo {
	gchar *uri_path;
	gchar *content_id;
	gchar *image_name;
	gchar *image_path;
};

struct _EmailSpecificSettings {
	Priority priority;
	EmailType type;
	gchar *account;
	gboolean read_receipt_request;
	EmailSending options;
	gboolean copy_to_own_address;
	SMimeType smime_type;	/* SMime Support */
};

struct _MsgInfo {
	MsgHeader *header;
	GSList *image_list;	/* Inline Images list */
	GSList *attach_list;	/* Attchment list */
	EmailSpecificSettings *settings;
	GSList *recip_certs_uid;	/*SMime Support */
	GSList *recip_algos;	/*SMime Support */
	gchar *password;	/*password for key */
};


typedef enum {
	DIRECT = 1,
	EMAIL,
	FILE_IN_MAIL,
	MHTML,
	ATT_VCARD
} AttachmentType;

struct _AttachInfo {
	AttachmentType type;
	gchar *filepath;
	gchar *msgid;
	gchar *attachmentname;
	gchar *email_path;
};
struct _MsgFileInfo {
	gchar *file;
	MsgFlags *flags;
};
struct _RetrMsgData {
	gchar *msgid;
	gboolean flag;
};
#define    LOW_PRIORITY    "Low"
#define    NORMAL_PRIORITY "Normal"
#define    HIGH_PRIORITY   "High"

FILE *procmsg_open_message(MsgHeader * msghdr);

void procmsg_msgheader_free(MsgHeader * msgheader);
void procmsg_attachlist_free(GSList * attachinfo_list);
void procmsg_recip_free(GSList * recip_certs_uid, GSList * recip_algos);
void procmsg_settings_free(EmailSpecificSettings * settings);
void procmsg_msginfo_list_free(GSList * msginfo_list);
void procmsg_msgheader_list_free(GSList * msgheader_list);

void procmsg_ImageInfo_list_free(GSList * image_list);

void procmsg_ImageInfo_free(InlineImageInfo * info);

void procmsg_msginfo_free(MsgInfo * msginfo);
FILE *procmsg_open_mark_file(void *item, DataOpenMode mode);
gchar *procmsg_get_message_file(MsgInfo * msginfo);
void procmsg_get_mark_sum(void *item,
			  gint * new,
			  gint * unread, gint * total, gint * min, gint * max, gint first);
/*
GHashTable *procmsg_msg_hash_table_create	(GSList		*mlist);
void procmsg_msg_hash_table_append		(GHashTable	*msg_table,
						 GSList		*mlist);
GHashTable *procmsg_to_folder_hash_table_create	(GSList		*mlist);

GSList *procmsg_read_cache		(Folder	*item,
					 gboolean	 scan_file);
void	procmsg_set_flags		(GSList		*mlist,
					 Folder	*item);
GSList *procmsg_sort_msg_list		(GSList		*mlist,
					 FolderSortKey	 sort_key,
					 FolderSortType	 sort_type);
gint	procmsg_get_last_num_in_msg_list(GSList		*mlist);
void	procmsg_msg_list_free		(GSList		*mlist);
void	procmsg_write_cache		(MsgInfo	*msginfo,
					 FILE		*fp);
void	procmsg_write_flags		(MsgInfo	*msginfo,
					 FILE		*fp);
void	procmsg_flush_mark_queue	(Folder	*item,
					 FILE		*fp);
void	procmsg_add_flags		(Folder	*item,
					 gint		 num,
					 MsgFlags	 flags);

FILE   *procmsg_open_cache_file		(Folder	*item,
					 DataOpenMode	 mode);


GNode  *procmsg_get_thread_tree		(GSList		*mlist);

gint	procmsg_move_messages		(GSList		*mlist);
gint	procmsg_copy_messages		(GSList		*mlist);


GSList *procmsg_get_message_file_list	(GSList		*mlist);
void	procmsg_message_file_list_free	(GSList		*file_list);
FILE   *procmsg_open_message		(MsgInfo	*msginfo);
#if USE_GPGME
FILE   *procmsg_open_message_decrypted	(MsgInfo	*msginfo,
					 MimeInfo      **mimeinfo);
#endif
gboolean procmsg_msg_exist		(MsgInfo	*msginfo);

void	procmsg_get_filter_keyword	(MsgInfo	  *msginfo,
					 gchar	         **header,
					 gchar	         **key,
					 PrefsFilterType   type);

void	procmsg_empty_trash		(void);
gint	procmsg_send_queue		(Folder	*queue,
					 gboolean	 save_msgs);
gint	procmsg_save_to_outbox		(Folder	*outbox,
					 const gchar	*file,
					 gboolean	 is_queued);
void	procmsg_print_message		(MsgInfo	*msginfo,
					 const gchar	*cmdline);

MsgInfo *procmsg_msginfo_copy		(MsgInfo	*msginfo);
MsgInfo *procmsg_msginfo_get_full_info	(MsgInfo	*msginfo);

gint procmsg_cmp_msgnum_for_sort	(gconstpointer	 a,
					gconstpointer	 b);

*/

void procmsg_write_flags(MsgInfo * msginfo, FILE * fp);

/**
  This function frees list containing RetrMsgData structures

  @param list - GSList of RetrMsgData structure
  @returns nothing
*/
void procmsg_free_retr_data_list(GSList * list);

/**
 Function to free RetrMsgData structure
 
 @param retr_msg - Information to free
 @return nothing
*/
void procmsg_free_retr(RetrMsgData * retr_msg);

#endif				/* __PROCMSG_H__ */
