
/*
 * LibSylph -- E-Mail client library
 * Copyright (C) 1999-2006 Hiroyuki Yamamoto
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "defs.h"

#include <glib.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#include "common.h"
#include "prefs.h"
#include "prefs_account.h"
#include "utils.h"
#include "procmsg.h"
#include "folder.h"

void procmsg_msgheader_free(MsgHeader * msgheader)
{
	if (msgheader == NULL) {
		return;
	}
	if (msgheader->fromname) {
		g_free(msgheader->fromname);
	}
	if (msgheader->to) {
		g_free(msgheader->to);
	}
	if (msgheader->date) {
		g_free(msgheader->date);
	}
	if (msgheader->from) {
		g_free(msgheader->from);
	}
	if (msgheader->cc) {
		g_free(msgheader->cc);
	}
	if (msgheader->bcc) {
		g_free(msgheader->bcc);
	}
	if (msgheader->subject) {
		g_free(msgheader->subject);
	}
	if (msgheader->msgid) {
		g_free(msgheader->msgid);
	}
	if (msgheader->inreplyto) {
		g_free(msgheader->inreplyto);
	}
	if (msgheader->replyto) {
		g_free(msgheader->replyto);
	}
	if (msgheader->username) {
		g_free(msgheader->username);
	}
	if (msgheader->password) {
		g_free(msgheader->password);
	}
	if (msgheader->account_name) {
		g_free(msgheader->account_name);
	}
	g_free(msgheader);
	msgheader = NULL;
}

void procmsg_attachlist_free(GSList * attachinfo_list)
{

	AttachInfo *attachinfo = NULL;

	if (attachinfo_list == NULL) {
		return;
	}
	while (attachinfo_list) {
		attachinfo = (AttachInfo *) attachinfo_list->data;
		if (attachinfo->filepath != NULL) {
			g_free(attachinfo->filepath);
		}
		if (attachinfo->msgid != NULL) {
			g_free(attachinfo->msgid);
		}
		if (attachinfo->attachmentname != NULL) {
			g_free(attachinfo->attachmentname);
		}
		g_free(attachinfo->email_path);

		attachinfo_list = g_slist_remove(attachinfo_list, attachinfo);
		g_free(attachinfo);
		attachinfo = NULL;
	}
	attachinfo_list = NULL;
}

/**
 This function will be called to free the EmailSpecificSettings struct 
 @param settings contains the EmailSpecific settings 
 @return none 
*/

void procmsg_settings_free(EmailSpecificSettings * settings)
{
	if (settings != NULL) {
		g_free(settings->account);
		g_free(settings);
		settings = NULL;
	}
}

/**
 This function will be called to free the InlineImageInfo struct 
 @param info Inline Image Info struct 
 @return none 
*/

void procmsg_ImageInfo_free(InlineImageInfo * info)
{
	if (info != NULL) {
		g_free(info->uri_path);
		g_free(info->content_id);
		g_free(info->image_name);
		g_free(info->image_path);
		g_free(info);
		info = NULL;
	}
}

/**
 This function will be called to free the GSlist of InlineImageIfo list
 @param GSList image_list pointer List of inline images 
 @return none 
*/
void procmsg_ImageInfo_list_free(GSList * image_list)
{
	InlineImageInfo *imageinfo = NULL;
	if (image_list == NULL) {
		return;
	}
	while (image_list != NULL) {
		imageinfo = image_list->data;
		image_list = g_slist_remove(image_list, imageinfo);
		procmsg_ImageInfo_free(imageinfo);
	}
}

/** This function will be called to free recip list
 * @param recip_certs_uid - List of recipient cert IDs
 * @param recip_algos - List of recipient algorithm
 */

void procmsg_recip_free(GSList * recip_certs_uid, GSList * recip_algos)
{
	gchar *algos = NULL;

	if (recip_certs_uid != NULL) {
		g_slist_free(recip_certs_uid);
	}
	recip_certs_uid = NULL;

	if (recip_algos != NULL) {
		while (recip_algos) {
			algos = recip_algos->data;
			g_free(algos);
			recip_algos = g_slist_remove(recip_algos, algos);
			algos = NULL;
		}
		recip_algos = NULL;
	}
}


/**
 This function will be called to free the message info.
 @param msginfo pointer to the msginfo structure 
 @return none 
*/
void procmsg_msginfo_free(MsgInfo * msginfo)
{
	if (msginfo == NULL) {
		return;
	}
	procmsg_msgheader_free(msginfo->header);
	procmsg_settings_free(msginfo->settings);
	procmsg_attachlist_free(msginfo->attach_list);
	procmsg_recip_free(msginfo->recip_certs_uid, msginfo->recip_algos);
	procmsg_ImageInfo_list_free(msginfo->image_list);
	g_free(msginfo);
	msginfo = NULL;
}

/**
 This function will be called to free the list of  message info.
 @param msginfo_list is GSList of msginfo structure 
 @return none 
*/

void procmsg_msginfo_list_free(GSList * msginfo_list)
{
	while (msginfo_list) {
		procmsg_msginfo_free((MsgInfo *) msginfo_list->data);
		msginfo_list = g_slist_remove(msginfo_list, msginfo_list->data);
	}
}

/**
 This function will be called to free the list of  message header.
 @param msgheader_list is GSList of msgheader structure 
 @return none 
*/

void procmsg_msgheader_list_free(GSList * msgheader_list)
{
	while (msgheader_list) {
		procmsg_msgheader_free((MsgHeader *) msgheader_list->data);
		msgheader_list = g_slist_remove(msgheader_list, msgheader_list->data);
	}
}

/**
 This function will be called to get the file pointer to the 
 message file 
 @param msghdr pointer to the MsgHeader structure 
 @return valid file pointer or NULL 
*/
FILE *procmsg_open_message(MsgHeader * msghdr)
{
	FILE *fp = NULL;
	gchar *file = NULL;

	g_return_val_if_fail(msghdr != NULL, NULL);
	g_return_val_if_fail(msghdr->msgid != NULL, NULL);

	file = (gchar *) folder_get_message_file_path(msghdr->msgid);
	g_return_val_if_fail(file != NULL, NULL);

	if ((fp = fopen(file, "rb")) == NULL) {
		log_message(("Mime:Error in opening message file"));
		g_free(file);
		return NULL;
	}
	g_free(file);
	return fp;
}

static FILE *procmsg_open_data_file(const gchar * file, gint version,
				    DataOpenMode mode, gchar * buf, size_t buf_size)
{
	FILE *fp = NULL;
	gint data_ver = 0;

	g_return_val_if_fail(file != NULL, NULL);

	if (mode == DATA_WRITE) {
		if ((fp = fopen(file, "wb")) == NULL) {
			send_engine_error_to_ui(errno);
			FILE_OP_ERROR(file, "fopen");
			return NULL;
		}
		if (change_file_mode_rw(fp, file) < 0)
			FILE_OP_ERROR(file, "chmod");

		WRITE_CACHE_DATA_INT(version, fp);
		return fp;
	}
	/* check version */
	if ((fp = fopen(file, "rb")) == NULL) {
		debug_print("Mark/Cache file not found\n");
	} else {
		if (buf && buf_size > 0)
			setvbuf(fp, buf, _IOFBF, buf_size);
		if (fread(&data_ver, sizeof(data_ver), 1, fp) != 1 || version != data_ver) {
			debug_print("Mark/Cache version is different (%d != %d)."
				    "Discarding it.\n", data_ver, version);
			fclose(fp);
			fp = NULL;
		}
	}

	if (mode == DATA_READ)
		return fp;

	if (fp) {
		/* reopen with append mode */
		fclose(fp);
		if ((fp = fopen(file, "ab")) == NULL) {
			send_engine_error_to_ui(errno);
			FILE_OP_ERROR(file, "fopen");
		}
	} else {
		/* open with overwrite mode if mark file doesn't exist or
		   version is different */
		fp = procmsg_open_data_file(file, version, DATA_WRITE, buf, buf_size);
	}

	return fp;
}



FILE *procmsg_open_mark_file(void *item, DataOpenMode mode)
{
	gchar *markfile = NULL;
	FILE *fp = NULL;

	markfile = folder_get_mark_file((Folder *) item);
	fp = procmsg_open_data_file(markfile, MARK_VERSION, mode, NULL, 0);
	g_free(markfile);

	return fp;
}

struct MarkSum {
	gint *new;
	gint *unread;
	gint *total;
	gint *min;
	gint *max;
	gint first;
};

static void mark_sum_func(gpointer key, gpointer value, gpointer data)
{
	MsgFlags *flags = value;
	gint num = GPOINTER_TO_INT(key);
	struct MarkSum *marksum = data;

	if (marksum->first <= num) {
		if (MSG_IS_NEW(*flags))
			(*marksum->new)++;
		if (MSG_IS_UNREAD(*flags))
			(*marksum->unread)++;
		if (num > *marksum->max)
			*marksum->max = num;
		if (num < *marksum->min || *marksum->min == 0)
			*marksum->min = num;
		(*marksum->total)++;
	}

	g_free(flags);
}



static GHashTable *procmsg_read_mark_file(Folder * item)
{
	FILE *fp = NULL;
	GHashTable *mark_table = NULL;
	gint num;
	MsgFlags *flags;
	MsgPermFlags perm_flags;

	if ((fp = procmsg_open_mark_file(item, DATA_READ)) == NULL)
		return NULL;

	mark_table = g_hash_table_new(NULL, g_direct_equal);

	while (fread(&num, sizeof(num), 1, fp) == 1) {
		if (fread(&perm_flags, sizeof(perm_flags), 1, fp) != 1)
			break;

		flags = g_hash_table_lookup(mark_table, GUINT_TO_POINTER(num));
		if (flags != NULL)
			g_free(flags);

		flags = g_new0(MsgFlags, 1);
		flags->perm_flags = perm_flags;
		g_hash_table_insert(mark_table, GUINT_TO_POINTER(num), flags);
	}

	fclose(fp);
	return mark_table;
}

void
procmsg_get_mark_sum(void *item,
		     gint * new, gint * unread, gint * total, gint * min, gint * max, gint first)
{
	GHashTable *mark_table;
	struct MarkSum marksum;

	*new = *unread = *total = *min = *max = 0;
	marksum.new = new;
	marksum.unread = unread;
	marksum.total = total;
	marksum.min = min;
	marksum.max = max;
	marksum.first = first;

	mark_table = (GHashTable *) procmsg_read_mark_file((Folder *) item);

	if (mark_table) {
		g_hash_table_foreach(mark_table, mark_sum_func, &marksum);
		g_hash_table_destroy(mark_table);
	}
}

gchar *procmsg_get_message_file(MsgInfo * msginfo)
{
	gchar *filename = NULL;
	Folder *folder;

	g_return_val_if_fail(msginfo != NULL, NULL);

	folder = (Folder *) folder_get_msg_folder(msginfo->header->msgid);
	filename = folder_fetch_msg(folder, msginfo->header->msgid);
	if (!filename)
		debug_print(("can't fetch message %d\n"), msginfo->header->msgnum);

	return filename;
}

void procmsg_write_flags(MsgInfo * msginfo, FILE * fp)
{
	MsgPermFlags flags = msginfo->header->flags.perm_flags;

	WRITE_CACHE_DATA_INT(msginfo->header->msgnum, fp);
	WRITE_CACHE_DATA_INT(flags, fp);
}

void procmsg_free_retr_data_list(GSList * list)
{
	if (list == NULL)
		return;
	g_slist_foreach(list, (GFunc) procmsg_free_retr, NULL);
}

void procmsg_free_retr(RetrMsgData * retr_msg)
{
	g_free(retr_msg->msgid);
}
