
/*
 * LibSylph -- E-Mail client library
 * Copyright (C) 1999-2006 Hiroyuki Yamamoto
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef __PROCMIME_H__
#define __PROCMIME_H__

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <glib.h>
#include <stdio.h>
#include "procmsg.h"

typedef struct _MimeType MimeType;
typedef struct _MimeInfo MimeInfo;


#define IS_BOUNDARY(s, bnd, len) \
        (bnd && s[0] == '-' && s[1] == '-' && !strncmp(s + 2, bnd, len))


#define MULTIPART_TYPE "multipart"

#define TEXT_TYPE "text"
#define HTML_TYPE "html"

#define GET_DECODE_FILENAME( msgid, mimeinfo,dec_filename)          \
{                                                                   \
	     dec_filename = g_strdup_printf( "%s%c%s%c%s%c%s",      \
	                              get_mime_tmp_dir(),             \
	                              G_DIR_SEPARATOR, msgid,                    \
	                              SECTION_SEPARATOR,                         \
				      mimeinfo->section_number,                  \
				      SECTION_SEPARATOR, DEC_SUFFIX );           \
}

	
typedef enum {
	ENC_7BIT,
	ENC_8BIT,
	ENC_QUOTED_PRINTABLE,
	ENC_BASE64,
	ENC_X_UUENCODE,
	ENC_UNKNOWN
} EncodingType;

typedef enum {
	MIME_TEXT,
	MIME_TEXT_HTML,
	MIME_MESSAGE_RFC822,
	MIME_APPLICATION,
	MIME_APPLICATION_OCTET_STREAM,
	MIME_MULTIPART,
	MIME_IMAGE,
	MIME_AUDIO,
	MIME_UNKNOWN
} ContentType;

struct _MimeType {
	gchar *type;
	gchar *sub_type;
	gchar *extension;
};

/*
 * An example of MimeInfo structure:
 *
 * multipart/mixed            root  <-+ parent
 *                                    |
 *   multipart/alternative      children <-+ parent
 *                                         |
 *     text/plain                 children  --+
 *                                            |
 *     text/html                  next      <-+
 *
 *   message/rfc822             next  <-+ main
 *                                      |
 *                                sub (capsulated message)
 *
 *   image/jpeg                 next
 */

struct _MimeInfo {
	gchar *encoding;
	EncodingType encoding_type;
	ContentType mime_type;
	gchar *content_type;
	gchar *charset;
	gchar *name;
	gchar *boundary;
	gchar *content_disposition;
	gchar *filename;
	glong fpos;
	guint size;
	MimeInfo *main;
	MimeInfo *sub;
	MimeInfo *next;
	MimeInfo *parent;
	MimeInfo *children;
	gint level;
	gchar *section_number;
	gboolean is_downloaded;
	gboolean is_purged;
	gint cert_state_sender;	/* SMime Support */
	gint cert_state_recip;
	gboolean trust;
	gboolean signature_verify;
	gboolean saving;
	gchar *cert_path;
	SMimeType smime_type;
	GSList *inline_images;
	GSList *file_pos_list;
	gint sender_cert_id;
	gint recip_cert_id;
};

gchar *procmime_get_part(const gchar * infile, MimeInfo * mimeinfo, gboolean body_struct);

MimeInfo *procmime_scan_message(FILE * fp, gint pos, gint level, glong fpos,
				GSList * file_pos_list);

MimeInfo *procmime_mimeinfo_next(MimeInfo * mimeinfo);

void procmime_mimeinfo_free_all(MimeInfo * mimeinfo);

void procmime_scan_content_type(MimeInfo * mimeinfo, const gchar * content_type);

void procmime_scan_content_disposition(MimeInfo * mimeinfo, const gchar * content_disposition);

MimeInfo *procmime_scan_mime_header(FILE * fp, gint pos, glong fpos,
				    GSList * file_pos_list, gchar ** subject_ptr);

FILE *procmime_decode_content(FILE * outfp, FILE * infp, MimeInfo * mimeinfo,
			      const gchar * filename, gchar ** remove_file);

gint procmime_get_content(MimeInfo * mimeinfo, FILE * infp,
			  const gchar * out_file, gboolean body_struct, gboolean edit_flag);

gchar *procmime_get_text_content(MsgHeader * msghdr, gboolean edit_flag, gboolean get_html_flag);

gchar *procmime_get_tmp_file_name(MimeInfo * mimeinfo);
//gchar *procmime_get_tmp_file_name(MimeInfo *mimeinfo,const gchar *msgid); 

ContentType procmime_scan_mime_type(const gchar * mime_type);

gchar *procmime_get_mime_type(const gchar * filename);

EncodingType procmime_get_encoding_for_charset(const gchar * charset);

EncodingType procmime_get_encoding_for_file(const gchar * file);

const gchar *procmime_get_encoding_str(EncodingType encoding);

MimeInfo *procmime_scan_body(const gchar * body_str, gint len);

MimeInfo *procmime_get_text_ptr(MimeInfo * mimeinfo, gboolean get_html);

gboolean procmime_check_attachments(const gchar * filename, gboolean bodytype_flag);

gint procmime_get_body_size(MimeInfo * mime_info, gchar * sec_nu);

gboolean procmime_mime_attachment(MimeInfo * mimeinfo);

/* smime changes */
gchar *procmime_get_smime_text_content(MsgHeader * msghdr, const gchar * file);

/** 
 This function creates a temporary file for the given file 
 with MHTML extension 
 @param  mimeinfo pointer to the main mimeinformation
 @param  msgid    message id of the file 
 @param  src_file    source file for copying 
 @return temporary file in which contents are copied 
*/
gchar *procmime_get_tmp_mhtml_file(const gchar * msgid, const gchar * src_file);
gboolean procmime_is_smime(MimeInfo * mimeinfo);

/**
 This function will be called to copy a part from a full file. 
 @param infp    pointer to the file to read 
 @param outfp    pointer to the file to write 
 @param mimeinfo    mime information pointer
 @param rfc_hdr     whether this part is rfc822 header part 
 @param return PARSE_SUCCESS on success or PARSE_FAIL 
*/
gint procmime_copy_part(FILE * infp, FILE * outfp, MimeInfo * mimeinfo, gboolean rfc_hdr);

/**
 This function will be called to store information about a 
 purged attachment. 
 @param filename    path to the file to store information 
 @param section    section number of the deleted attachment 
 @param return PARSE_SUCCESS on success or PARSE_FAIL 
*/
gint procmime_write_purge_section(const gchar * filename, const gchar * section);

/**
 This function will be called to get a list of purged attachment 
 section numbers. 
 @param filepath    path to the file to read purge information from 
 @param return list of section numbers purged 
*/
GSList *procmime_read_purged(const gchar * filepath);

/**
 This function will be called to sets the is_purged flag 
 for the mimeinformation of attachment,which is purged 
 
 @param partinfo    mimeinformation of the attachment part 
 @param purge_list    list of purged sections for the mesasge 
 @param return none 
*/
void procmime_set_purged(MimeInfo * partinfo, GSList * purge_list);

/**
 This function will be called to check all attachments 
 whether it is purgeable 
 
 @param filename    name of the file to check for 
 @param message_filepath    path of the message 
 @param return wether the attachments can be purged 
*/
gboolean procmime_check_purgable_all(const gchar * filename, const gchar * message_filepath);

/**
  This function will give information on whether a particular file
  contains email attachment or not

  @param mimeinfo - Mime information of the file
  @return gboolean - TRUE if email attach is available else false
*/
gboolean procmime_check_email_attach(MimeInfo * mimeinfo);

/**
  This function willcheck for style tag and if default font is 
  not present,adds it
  @param mhtml_file    the file for the checking 
  @param dest_file    the destination file for modified data 
  @param default_font    the default font of the message 
  @return PARSE_FAIL on error PARSE_SUCCESS on success 
*/
gint procmime_check_for_style_tag(const gchar * mhtml_file,
				  const gchar * dest_file, const gchar * default_font);

/**
  This function will write out file pointer on style check 
  @param str    the string for checking 
  @param style_found_flag   indicates wther the style tag was found 
  @param outfp   file pointer to the destination file
  @param default_font    the default font of the message 
  @return PARSE_FAIL on error PARSE_SUCCESS on success 
*/
void procmime_write_on_stylecheck(gchar * str,
				  gboolean * style_found_flag,
				  FILE * outfp, const gchar * default_font);

MimeInfo *procmime_read_mimeinfo(const gchar * mime_file_name);

/**
  This function  checks the message for any extra part if the mimeinfo 
  is not of mixed type 
  @param mimeinfo    mimeinfo of the message to check for extra parts 
  @return    return TRUE if message has attachments.
*/
gboolean procmime_check_extra_part(MimeInfo * mimeinfo);

/**
  This function  sets the filename variable in mime information to the 
  decoded filename.If the message has any extra part 
  and the decoded  file is NULL, it saves the original filename in name 
  variable in mimeinformation. 
  @param mimeinfo    mimeinfo of the message to set the decoded file 
  @param decfilename    name of file which has decoded contents. 
  @return    none.
*/
void procmime_check_and_set_filename( MimeInfo *mimeinfo,const gchar *decfilename);

MimeInfo *procmime_get_mimeinfo(const gchar * msgid, const gchar * filename,
		                                              gboolean edit_flag, gboolean get_html_flag);

gint procmime_folder_set_header(MsgHeader * header, PrefsAccount * account);

gboolean procmime_set_is_downloaded(MimeInfo * mimeinfo, const gchar * msgid);

#endif				/* __PROCMIME_H__ */
