
/*
 * LibSylph -- E-Mail client library
 * Copyright (C) 1999-2006 Hiroyuki Yamamoto
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "defs.h"

#include <glib.h>
#include <stdio.h>
#include <string.h>
#include <locale.h>
#include <ctype.h>

#include "common.h"
#include "prefs_account.h"
#include "account.h"
#include "procmsg.h"
#include "procmime.h"
#include "procheader.h"
#include "base64.h"
#include "quoted-printable.h"
#include "uuencode.h"
#include "unmime.h"
#include "html.h"
#include "codeconv.h"
#include "utils.h"
#include "parsebody.h"
#include "defs.h"
#include "folder.h"
#include "engine_errno.h"

static GHashTable *procmime_get_mime_type_table(void);
static GList *procmime_get_mime_type_list(const gchar * file);
void procmime_scan_multipart_message(MimeInfo * mimeinfo, FILE * fp, gint pos);
static gchar *procmime_get_name_str(const gchar * str, const gchar * end);
static gint procmime_parse_multipart_fields(gchar * fldptr, const gchar * end, MimeInfo * mimeinfo);
static gint mime_multi_part_body_size(MimeInfo * mime_info, gchar * sec_nu);

static void procmime_get_filename(gchar * ptr, gchar * para_ptr, MimeInfo * mimeinfo);

static void procmime_parse_sub_part(MimeInfo * mimeinfo, gchar ** bodyptr,
				    const gchar * end, const gchar * section_number);

static MimeInfo *procmime_mimeinfo_new(void);

static MimeInfo *procmime_mimeinfo_insert(MimeInfo * parent, MimeInfo * mimeinfo);

static void procmime_scan_encoding(MimeInfo * mimeinfo, const gchar * encoding);


/** This function is called from procmime_scan_multipart_message
    if the contents in the message have S/MIME mail embedded. In
    case of an email attachment which is S/MIME this function
    gets called. It scans the complete S/MIME part and returns
    the mimeinfo of the S/MIME part

    @param mimeinfo - Mimeinfo of the main
    @param fp - Pointer to file containing the information
    @param pos - Position in the file_pos_list where file
                position is to be inserted
    @param boundary - Boundary till which the contents should
                     be scanned
    @return mimeinfo containing S/MIME part
*/
static MimeInfo *procmime_get_smime_mimeinfo(MimeInfo * mimeinfo, FILE * fp, gint pos,
					     gchar * boundary);

/** This function is called to decrypt the S/MIME content and
    get the mimeinfo.

    @param filename - Encrypted file
    @param mimeinfo - Mimeinfo of the parent
    @param pos - position where fpos has to be inserted
    @param fpos - File position
    @param file_pos_list - List of positions
    @return mimeinfo - Final mimeinfo
*/
static MimeInfo *procmime_get_smime_mime(gchar * filename, MimeInfo * mimeinfo, gint pos,
					 glong fpos, GSList * file_pos_list);

static
void procmime_get_subfilename(MimeInfo * maininfo, gchar * subject);

static gboolean procmime_check_purgable(const MimeInfo * partinfo, const gchar * filepath);


static gchar *procmime_check_head_tag(gchar * buf,
				      gboolean * style_found_flag, const gchar * default_font);

/**
 This function will be called to set the section numbers read 
 from the mime header file in the mime information read from 
 completely downloaded mail. 
 @param mimeinfo   mimeinformation from the file  
 @param filepath    path to the mime header file  
 @param return FILE_PROCESS_SUCCESS on SUCCESS or FILE_PROCESS_FAIL 
*/
static gint procmime_sync_mimeinfo(MimeInfo * mimeinfo, const gchar * file_path);

/** This function allocates memory for mime info structure
 @return pointer to mime info structure
*/

static MimeInfo *procmime_mimeinfo_new(void)
{
	MimeInfo *mimeinfo;

	mimeinfo = g_new0(MimeInfo, 1);
	if (mimeinfo == NULL)
		return NULL;
	mimeinfo->mime_type = MIME_UNKNOWN;
	mimeinfo->encoding_type = ENC_UNKNOWN;
	mimeinfo->children = NULL;
	mimeinfo->next = NULL;
	mimeinfo->sub = NULL;
	mimeinfo->main = NULL;
	mimeinfo->parent = NULL;
	mimeinfo->is_downloaded = TRUE;

	return mimeinfo;
}

/** This function frees memory allocated for mime info structure.
	@param mimeinfo mime inforamtion structure pointer
	@return none
*/
void procmime_mimeinfo_free_all(MimeInfo * mimeinfo)
{
	GSList *cur = NULL;
	InlineImageInfo *inline_image_info = NULL;

	while (mimeinfo != NULL) {
		MimeInfo *next;

		g_free(mimeinfo->encoding);
		g_free(mimeinfo->content_type);
		g_free(mimeinfo->charset);
		g_free(mimeinfo->name);
		g_free(mimeinfo->boundary);
		g_free(mimeinfo->content_disposition);
		g_free(mimeinfo->filename);
		g_free(mimeinfo->section_number);
		if (mimeinfo->inline_images != NULL) {
			for (cur = mimeinfo->inline_images; cur != NULL; cur = cur->next) {
				inline_image_info = (InlineImageInfo *) cur->data;
				if (inline_image_info != NULL) {
					procmsg_ImageInfo_free(inline_image_info);
				}
			}
		}
		g_slist_free(mimeinfo->inline_images);
		if (mimeinfo->file_pos_list != NULL) {
			g_slist_free(mimeinfo->file_pos_list);
		}
		procmime_mimeinfo_free_all(mimeinfo->sub);
		procmime_mimeinfo_free_all(mimeinfo->children);
		next = mimeinfo->next;
		g_free(mimeinfo);
		mimeinfo = next;
	}
}

/** This function inserts a node in mime info structure.
	@param mimeinfo pointer to the node to insert
	@param parent mime inforamtion structure pointer of parent
	@return pointer to the mime info structure
*/
static MimeInfo *procmime_mimeinfo_insert(MimeInfo * parent, MimeInfo * mimeinfo)
{
	MimeInfo *child = parent->children;

	if (!child)
		parent->children = mimeinfo;
	else {
		while (child->next != NULL)
			child = child->next;

		child->next = mimeinfo;
	}

	mimeinfo->parent = parent;
	mimeinfo->level = parent->level + 1;

	return mimeinfo;
}

/** This function gets next or child node in mime info structure.
	@param mimeinfo pointer 
	@return pointer to the mime info structure
*/

MimeInfo *procmime_mimeinfo_next(MimeInfo * mimeinfo)
{
	if (!mimeinfo)
		return NULL;

	if (mimeinfo->children)
		return mimeinfo->children;
	if (mimeinfo->sub)
		return mimeinfo->sub;
	if (mimeinfo->next)
		return mimeinfo->next;

	if (mimeinfo->main) {
		mimeinfo = mimeinfo->main;
		if (mimeinfo->next)
			return mimeinfo->next;
	}

	for (mimeinfo = mimeinfo->parent; mimeinfo != NULL; mimeinfo = mimeinfo->parent) {
		if (mimeinfo->next)
			return mimeinfo->next;
		if (mimeinfo->main) {
			mimeinfo = mimeinfo->main;
			if (mimeinfo->next)
				return mimeinfo->next;
		}
	}

	return NULL;
}

/** This function scans a file and forms mime info structure.
	@param fp file pointer
	@return pointer to the mime info structure
*/

MimeInfo *procmime_scan_message(FILE * fp, gint pos, gint level, glong fpos, GSList * file_pos_list)
{
	MimeInfo *mimeinfo;

	g_return_val_if_fail(fp != NULL, NULL);

	mimeinfo = procmime_scan_mime_header(fp, pos, fpos, file_pos_list, NULL);

	/* Added to set correct level for a nested S/MIME mail */
	if (level > 0)
		mimeinfo->level = level;

	if (mimeinfo) {
		mimeinfo->size = get_left_file_size(fp);
		if (mimeinfo->mime_type == MIME_MULTIPART ||
		    mimeinfo->mime_type == MIME_MESSAGE_RFC822)
			procmime_scan_multipart_message(mimeinfo, fp, pos);
	}

	return mimeinfo;
}

/** This function scans a multipart mime information and 
	puts it in mime information.
	@param fp pointer to the file
	@param mimeinfo mime inforamtion structure pointer.
	@return none
*/
void procmime_scan_multipart_message(MimeInfo * mimeinfo, FILE * fp, gint pos)
{
	gchar *p;
	gchar *boundary;
	gint boundary_len = 0;
	gchar buf[BUFFSIZE];
	glong fpos, prev_fpos;
	gboolean smime = FALSE;

	g_return_if_fail(mimeinfo != NULL);
	g_return_if_fail(mimeinfo->mime_type == MIME_MULTIPART ||
			 mimeinfo->mime_type == MIME_MESSAGE_RFC822);

	if (mimeinfo->mime_type == MIME_MULTIPART) {
		g_return_if_fail(mimeinfo->boundary != NULL);
		g_return_if_fail(mimeinfo->sub == NULL);
	}
	g_return_if_fail(fp != NULL);

	boundary = mimeinfo->boundary;

	if (boundary) {
		boundary_len = strlen(boundary);

		/* look for first boundary */
		while ((p = fgets(buf, sizeof(buf), fp)) != NULL)
			if (IS_BOUNDARY(buf, boundary, boundary_len))
				break;
		if (!p)
			return;
	} else if (mimeinfo->parent && mimeinfo->parent->boundary) {
		boundary = mimeinfo->parent->boundary;
		boundary_len = strlen(boundary);
	}

	if ((fpos = ftell(fp)) < 0) {
		perror("ftell");
		return;
	}

	for (;;) {
		MimeInfo *partinfo;
		gboolean eom = FALSE;
		gint len = 0;
		MimeInfo *sub_smime = NULL;
		gchar *subject = NULL;

		prev_fpos = fpos;

		debug_print("prev_fpos: %ld\n", fpos);

		if (mimeinfo->mime_type == MIME_MESSAGE_RFC822) {
			MimeInfo *sub;

			sub = procmime_scan_mime_header(fp, pos, mimeinfo->fpos,
							mimeinfo->file_pos_list, &subject);

			procmime_get_subfilename(mimeinfo, subject);

			if ((!g_ascii_strncasecmp(sub->content_type, SIGNED, 16))
			    || (g_strrstr(sub->content_type, ENCR7) != NULL)
			    || (g_strrstr(sub->content_type, ENCR) != NULL)) {
				sub_smime = procmime_get_smime_mimeinfo(mimeinfo,
									fp, pos, boundary);
				if (sub_smime != NULL) {
					procmime_mimeinfo_free_all(sub);
					sub = sub_smime;
					smime = TRUE;
				}
			}

			mimeinfo->sub = sub;

			if (!sub)
				break;

			sub->level = mimeinfo->level + 1;
			sub->parent = mimeinfo->parent;
			sub->main = mimeinfo;
			partinfo = sub;
			if (smime == TRUE)
				return;

		} else {
			partinfo = procmime_scan_mime_header(fp, pos, mimeinfo->fpos,
							     mimeinfo->file_pos_list, NULL);
			if (!partinfo)
				break;
			procmime_mimeinfo_insert(mimeinfo, partinfo);

			debug_print("content-type: %s\n", partinfo->content_type);
		}

		if (partinfo->mime_type == MIME_MULTIPART ||
		    partinfo->mime_type == MIME_MESSAGE_RFC822) {
			if (partinfo->level < 8)
				procmime_scan_multipart_message(partinfo, fp, pos);
		}

		/* look for next boundary */
		buf[0] = '\0';
		while ((p = fgets(buf, sizeof(buf), fp)) != NULL) {
			if (IS_BOUNDARY(buf, boundary, boundary_len)) {
				if (buf[2 + boundary_len] == '-' &&
				    buf[2 + boundary_len + 1] == '-')
					eom = TRUE;
				break;
			}
		}
		if (p == NULL) {
			/* broken MIME, or single part MIME message */
			buf[0] = '\0';
			eom = TRUE;
		}
		debug_print("boundary: %s\n", buf);

		fpos = ftell(fp);

		debug_print("fpos: %ld\n", fpos);

		len = strlen(buf);
		partinfo->size = fpos - prev_fpos - len;

		debug_print("partinfo->size: %d\n", partinfo->size);

		if (partinfo->sub && !partinfo->sub->sub && !partinfo->sub->children) {
			partinfo->sub->size = fpos - partinfo->sub->fpos - strlen(buf);
			debug_print("partinfo->sub->size: %d\n", partinfo->sub->size);

		}

		if (mimeinfo->mime_type == MIME_MESSAGE_RFC822) {
			if (len > 0 && fseek(fp, fpos - len, SEEK_SET) < 0)
				perror("fseek");
			break;
		}

		if (eom)
			break;
	}
}

/** This function scans encoding and updates mime info structure.
	@param mimeinfo pointer to the mime information
	@param encoding Encoding information pointer
	@return none
*/

static void procmime_scan_encoding(MimeInfo * mimeinfo, const gchar * encoding)
{
	gchar *buf = NULL;

	Xstrdup_a(buf, encoding, return);

	g_free(mimeinfo->encoding);

	mimeinfo->encoding = g_strdup(g_strstrip(buf));
	if (!strcasecmp(buf, "7bit"))
		mimeinfo->encoding_type = ENC_7BIT;
	else if (!strcasecmp(buf, "8bit"))
		mimeinfo->encoding_type = ENC_8BIT;
	else if (!strcasecmp(buf, "quoted-printable"))
		mimeinfo->encoding_type = ENC_QUOTED_PRINTABLE;
	else if (!strcasecmp(buf, "base64"))
		mimeinfo->encoding_type = ENC_BASE64;
	else if (!strcasecmp(buf, "x-uuencode"))
		mimeinfo->encoding_type = ENC_X_UUENCODE;
	else
		mimeinfo->encoding_type = ENC_UNKNOWN;

}

/** This function scans content type and updates mime info structure.
	@param mimeinfo pointer to the mime information
	@param content pointer to Content type structure
	@return none
*/

void procmime_scan_content_type(MimeInfo * mimeinfo, const gchar * content_type)
{
	gchar *delim, *p, *cnttype;
	gchar *buf;

	if (conv_get_current_charset() == C_EUC_JP && strchr(content_type, '\033')) {
		gint len;
		len = strlen(content_type) * 2 + 1;
		Xalloca(buf, len, return);
		conv_jistoeuc(buf, len, content_type);
	} else
		Xstrdup_a(buf, content_type, return);

	g_free(mimeinfo->content_type);
	g_free(mimeinfo->charset);
	g_free(mimeinfo->name);
	mimeinfo->content_type = NULL;
	mimeinfo->charset = NULL;
	mimeinfo->name = NULL;

	if ((delim = strchr(buf, ';')))
		*delim = '\0';
	mimeinfo->content_type = cnttype = g_strdup(g_strstrip(buf));

	mimeinfo->mime_type = procmime_scan_mime_type(cnttype);

	if (!delim)
		return;
	p = delim + 1;

	for (;;) {
		gchar *eq;
		gchar *attr, *value;

		if ((delim = strchr(p, ';')))
			*delim = '\0';

		if (!(eq = strchr(p, '=')))
			break;

		*eq = '\0';
		attr = p;
		g_strstrip(attr);
		value = eq + 1;
		g_strstrip(value);

		if (*value == '"')
			extract_quote(value, '"');
		else {
			eliminate_parenthesis(value, '(', ')');
			g_strstrip(value);
		}

		if (*value) {
			if (!strcasecmp(attr, "charset"))
				mimeinfo->charset = g_strdup(value);
			else if (!strcasecmp(attr, "name")) {
				gchar *tmp;
				size_t len;

				len = strlen(value) + 1;
				Xalloca(tmp, len, return);
				conv_unmime_header(tmp, len, value, NULL);
				mimeinfo->name = g_strdup(tmp);
			} else if (!strcasecmp(attr, "boundary"))
				mimeinfo->boundary = g_strdup(value);
		}

		if (!delim)
			break;
		p = delim + 1;
	}

	if (mimeinfo->mime_type == MIME_MULTIPART && !mimeinfo->boundary)
		mimeinfo->mime_type = MIME_TEXT;
}

/** This function gets content disposition in mime info structure.
	@param mimeinfo pointer to the mime information
	@param content_disposition pointer to the content disposition 
	@return none
*/
void procmime_scan_content_disposition(MimeInfo * mimeinfo, const gchar * content_disposition)
{
	gchar *delim, *p, *dispos;
	gchar *buf;

	if (conv_get_current_charset() == C_EUC_JP && strchr(content_disposition, '\033')) {
		gint len;
		len = strlen(content_disposition) * 2 + 1;
		Xalloca(buf, len, return);
		conv_jistoeuc(buf, len, content_disposition);
	} else
		Xstrdup_a(buf, content_disposition, return);

	if ((delim = strchr(buf, ';')))
		*delim = '\0';
	mimeinfo->content_disposition = dispos = g_strdup(g_strstrip(buf));

	if (!delim)
		return;
	p = delim + 1;

	for (;;) {
		gchar *eq;
		gchar *attr, *value;

		if ((delim = strchr(p, ';')))
			*delim = '\0';

		if (!(eq = strchr(p, '=')))
			break;

		*eq = '\0';
		attr = p;
		g_strstrip(attr);
		value = eq + 1;
		g_strstrip(value);

		if (*value == '"')
			extract_quote(value, '"');
		else {
			eliminate_parenthesis(value, '(', ')');
			g_strstrip(value);
		}

		if (*value) {
			if (!strcasecmp(attr, "filename")) {
				gchar *tmp;
				size_t len;

				len = strlen(value) + 1;
				Xalloca(tmp, len, return);
				conv_unmime_header(tmp, len, value, NULL);
				g_free(mimeinfo->filename);
				mimeinfo->filename = g_strdup(tmp);
				break;
			}
		}

		if (!delim)
			break;
		p = delim + 1;
	}
}

enum {
	H_CONTENT_TRANSFER_ENCODING = 0,
	H_CONTENT_TYPE = 1,
	H_CONTENT_DISPOSITION = 2,
	H_SUBJECT = 3
};

/** This function scans mime header.
	@param fp file pointer
	@return pointer to the mime information structure
*/
MimeInfo *procmime_scan_mime_header(FILE * fp, gint pos, glong fpos,
				    GSList * file_pos_list, gchar ** subject_ptr)
{
	static HeaderEntry hentry[] = { {"Content-Transfer-Encoding:",
					 NULL, FALSE},
	{"Content-Type:", NULL, TRUE},
	{"Content-Disposition:",
	 NULL, TRUE},
	{"Subject:", NULL, TRUE},
	{NULL, NULL, FALSE}
	};
	gchar buf[BUFFSIZE], tmp[BUFFSIZE];
	gint hnum;
	HeaderEntry *hp;
	MimeInfo *mimeinfo;
	gchar *type;
	gchar *ptr;
	gint counter = 0;
	gint len = 0;
	gint value = 0;

	g_return_val_if_fail(fp != NULL, NULL);

	mimeinfo = procmime_mimeinfo_new();
	mimeinfo->mime_type = MIME_TEXT;
	mimeinfo->encoding_type = ENC_7BIT;

	if (pos > -1) {
		mimeinfo->fpos = fpos;
		if (file_pos_list != NULL) {
			len = g_slist_length(file_pos_list);
			for (counter = 0; counter < len; counter++) {
				if (counter == pos)
					break;
				value = GPOINTER_TO_INT(g_slist_nth_data(file_pos_list, counter));
				mimeinfo->file_pos_list =
				    g_slist_insert(mimeinfo->file_pos_list,
						   GINT_TO_POINTER(value), counter);
			}
		}
		mimeinfo->file_pos_list = g_slist_insert(mimeinfo->file_pos_list,
							 GINT_TO_POINTER(ftell(fp)), pos);
	}

	else {
		mimeinfo->fpos = ftell(fp);
	}

	while ((hnum = procheader_get_one_field(buf, sizeof(buf), fp, hentry))
	       != -1) {
		hp = hentry + hnum;

		if (H_CONTENT_TRANSFER_ENCODING == hnum) {
			procmime_scan_encoding(mimeinfo, buf + strlen(hp->name));
		} else if (H_CONTENT_TYPE == hnum) {
			procmime_scan_content_type(mimeinfo, buf + strlen(hp->name));
		} else if (H_CONTENT_DISPOSITION == hnum) {
			procmime_scan_content_disposition(mimeinfo, buf + strlen(hp->name));
		} else if (H_SUBJECT == hnum) {
			if (subject_ptr != NULL) {
				ptr = buf + strlen(hp->name);
				while (*ptr == ' ' || *ptr == '\t') {
					ptr++;
				}
				conv_unmime_header(tmp, sizeof(tmp), ptr, NULL);
				*subject_ptr = g_strdup(tmp);
			}
		}
	}

	if (mimeinfo->mime_type == MIME_APPLICATION_OCTET_STREAM && mimeinfo->name) {
		type = procmime_get_mime_type(mimeinfo->name);
		if (type) {
			mimeinfo->mime_type = procmime_scan_mime_type(type);
			g_free(type);
		}
	}

	if (!mimeinfo->content_type)
		mimeinfo->content_type = g_strdup("text/plain");

	return mimeinfo;
}

/** This function scans decodes the mime content and puts it in a file.
	@param outfp outfile pointer
	@param infp in file pointer
	@param mimeinfo mimeinformation pointer
	@param filename name of the file,which will be updated if outfp is null
	@return outfile pointer
*/
FILE *procmime_decode_content(FILE * outfp, FILE * infp,
			      MimeInfo * mimeinfo, const gchar * filename, gchar ** remove_file)
{
	gchar buf[BUFFSIZE];
	gchar *boundary = NULL;
	gint boundary_len = 0;
	gboolean tmp_file = FALSE;
	gchar *tmp = NULL;

	g_return_val_if_fail(infp != NULL, NULL);
	g_return_val_if_fail(mimeinfo != NULL, NULL);

	if (!outfp) {

		tmp = get_tmp_file();
		outfp = fopen(tmp, "w+b");

		if (!outfp) {
			send_engine_error_to_ui(errno);
			g_free(tmp);
			return NULL;
		}
		tmp_file = TRUE;
	}

	if (mimeinfo->parent && mimeinfo->parent->boundary) {
		boundary = mimeinfo->parent->boundary;
		boundary_len = strlen(boundary);
	}

	if (mimeinfo->encoding_type == ENC_QUOTED_PRINTABLE) {
		while (fgets(buf, sizeof(buf), infp) != NULL &&
		       (!boundary || !IS_BOUNDARY(buf, boundary, boundary_len))) {
			gint len;
			len = qp_decode_line(buf);
			fwrite(buf, len, 1, outfp);
		}
	} else if (mimeinfo->encoding_type == ENC_BASE64) {
		gchar outbuf[BUFFSIZE];
		gint len;
		Base64Decoder *decoder;

		decoder = base64_decoder_new();
		while (fgets(buf, sizeof(buf), infp) != NULL &&
		       (!boundary || !IS_BOUNDARY(buf, boundary, boundary_len))) {
			len = base64_decoder_decode(decoder, buf, outbuf);
			if (len < 0) {
				log_message((const gchar *) ("Bad BASE64 content\n"));
				break;
			}
			fwrite(outbuf, sizeof(gchar), len, outfp);
		}
		base64_decoder_free(decoder);
	} else if (mimeinfo->encoding_type == ENC_X_UUENCODE) {
		gchar outbuf[BUFFSIZE];
		gint len;
		gboolean flag = FALSE;

		while (fgets(buf, sizeof(buf), infp) != NULL &&
		       (!boundary || !IS_BOUNDARY(buf, boundary, boundary_len))) {
			if (!flag && strncmp(buf, "begin ", 6))
				continue;

			if (flag) {
				len = fromuutobits(outbuf, buf);
				if (len <= 0) {
					if (len < 0)
						log_message((const gchar
							     *) ("Bad UUENCODE content(%d)\n"),
							    len);
					break;
				}
				fwrite(outbuf, sizeof(gchar), len, outfp);
			} else
				flag = TRUE;
		}
	} else {
		while (fgets(buf, sizeof(buf), infp) != NULL &&
		       (!boundary || !IS_BOUNDARY(buf, boundary, boundary_len))) {
			fputs(buf, outfp);
		}
	}

	if (tmp_file)
		rewind(outfp);

	*remove_file = tmp;
	return outfp;
}

/** This function gets the attachement or part of the message decoded
	@param mimeinfo pointer to the mime information
	@param infile pointer to the message file
	@return file name in which the decoded contents are put
*/
gchar *procmime_get_part(const gchar * infile, MimeInfo * mimeinfo, gboolean body_struct)
{
	gchar *outfile;
	FILE *infp, *outfp;
	gchar buf[BUFFSIZE];
	gchar *remove_file = NULL;

	g_return_val_if_fail(infile != NULL, NULL);
	g_return_val_if_fail(mimeinfo != NULL, NULL);

	outfile = procmime_get_tmp_file_name(mimeinfo);

	g_return_val_if_fail(outfile != NULL, NULL);

	if ((infp = fopen(infile, "rb")) == NULL) {
		log_message((const gchar *) ("Mime:Error in opening the file"));
		g_free(outfile);
		return NULL;
	}
	if (fseek(infp, mimeinfo->fpos, SEEK_SET) < 0) {
		fclose(infp);
		g_free(outfile);
		log_message((const gchar *) ("Mime:Error in contents of the file"));
		return NULL;
	}
	if ((outfp = fopen(outfile, "wb")) == NULL) {
		send_engine_error_to_ui(errno);
		log_message((const gchar *) ("Mime:Error in opening destination file"));
		fclose(infp);
		g_free(outfile);
		return NULL;
	}
	if (body_struct == FALSE) {
		while (fgets(buf, sizeof(buf), infp) != NULL)
			if (buf[0] == '\r' || buf[0] == '\n')
				break;
	}
	procmime_decode_content(outfp, infp, mimeinfo, NULL, &remove_file);

	fclose(infp);
	if (fclose(outfp) == EOF) {
		log_message((const gchar *) ("Mime:Error in closing destination file"));
		unlink(outfile);
		g_free(outfile);

		if (remove_file != NULL)
			unlink(remove_file);
		g_free(remove_file);
		return NULL;
	}
	if (remove_file != NULL)
		unlink(remove_file);

	g_free(remove_file);
	return (outfile);
}

/** This function decodes the text content.
	@param mimeinfo pointer to the mime information
	@param infp input file pointer
	@param out_file output file name
	@return 0 on success,-1 on failure
*/
gint
procmime_get_content(MimeInfo * mimeinfo, FILE * infp,
		     const gchar * out_file, gboolean body_struct, gboolean edit_flag)
{
	FILE *outfp = NULL, *tmpfp = NULL;
	gchar *src_codeset = NULL;
	gboolean conv_fail = FALSE;
	gchar buf[BUFFSIZE];
	gchar *str = NULL;
	gsize bytes_written;
	gchar *remove_file = NULL;

	g_return_val_if_fail(mimeinfo != NULL, -1);
	g_return_val_if_fail(infp != NULL, -1);
	g_return_val_if_fail(out_file != NULL, -1);
	g_return_val_if_fail(mimeinfo->mime_type == MIME_TEXT ||
			     mimeinfo->mime_type == MIME_TEXT_HTML, -1);
	if (body_struct == FALSE) {
		if (fseek(infp, mimeinfo->fpos, SEEK_SET) < 0) {
			perror("fseek");
			return -1;
		}

		while (fgets(buf, sizeof(buf), infp) != NULL)
			if (buf[0] == '\r' || buf[0] == '\n')
				break;
	}

	tmpfp = procmime_decode_content(NULL, infp, mimeinfo, out_file, &remove_file);

	if (tmpfp == NULL) {
		debug_print("\nFILE ERROR\n");
		unlink(remove_file);
		g_free(remove_file);
		return -1;
	}
	outfp = fopen(out_file, "w+b");
	if (outfp == NULL) {
		debug_print("\nFILE ERROR\n");
		send_engine_error_to_ui(errno);
		unlink(remove_file);
		g_free(remove_file);
		return -1;
	}
	if (edit_flag == TRUE) {

		src_codeset = mimeinfo->charset;

		if (src_codeset == NULL) {
			src_codeset = DEFAULT_CHARSET;
		}
	}
	while (fgets(buf, sizeof(buf), tmpfp) != NULL) {
		if (edit_flag == TRUE) {

			str = convert_to_utf8(buf, strlen(buf), src_codeset, &bytes_written);
			if (str) {
				fputs(str, outfp);
				g_free(str);
			} else {
				conv_fail = TRUE;
				fputs(buf, outfp);
			}
		} else {

			fputs(buf, outfp);
		}
	}

	fclose(tmpfp);
	fclose(outfp);
	unlink(remove_file);
	g_free(remove_file);
	return 0;
}

/** This function decodes the text content, and this is called 
	by the adaptation layer function
	@param msghdr pointer to the message header information
	@return filename in which the decoded contents are put
*/
gchar *procmime_get_text_content(MsgHeader * msghdr, gboolean edit_flag, gboolean get_html_flag)
{
	FILE *infp;
	MimeInfo *mimeinfo, *partinfo;
	gchar *outfile = NULL;
	gchar *tmp_dir = NULL;
	gint ret_val = -1;
	gchar *folder_name = NULL;
	gint pos = -1;
	gint level = 0;
	glong fpos = 0;
	GSList *file_pos_list = NULL;


	if (msghdr == NULL) {
		debug_print("\nThe Header NULL\n");
		return NULL;
	}
	infp = procmsg_open_message(msghdr);

	if (infp == NULL) {
		debug_print("ERROR in opening the file");
		return NULL;
	}

	mimeinfo = procmime_scan_message(infp, pos, level, fpos, file_pos_list);

	if (!mimeinfo) {
		fclose(infp);
		return NULL;
	}
	folder_name = get_fold_name(msghdr->msgid);
	if (get_html_flag == TRUE) {
		partinfo = procmime_get_text_ptr(mimeinfo, TRUE);
		if (partinfo == NULL) {
			partinfo = procmime_get_text_ptr(mimeinfo, FALSE);
		}
	} else {
		partinfo = procmime_get_text_ptr(mimeinfo, FALSE);
	}
	g_free(folder_name);
	tmp_dir = (gchar *)get_tmp_dir();

	if (!is_dir_exist(tmp_dir)) {
		debug_print("TEMPORARY directory does not exist\n");
		tmp_dir = (gchar *) g_get_tmp_dir();
	}
	outfile = g_strdup_printf("%s%c%s", tmp_dir, G_DIR_SEPARATOR, msghdr->msgid);
	if (!outfile) {
		debug_print("\n NOT able to allocate memory for OUT\n");
		fclose(infp);
		procmime_mimeinfo_free_all(mimeinfo);
		return NULL;
	}
	if (partinfo)
		ret_val = procmime_get_content(partinfo, infp, outfile, FALSE, edit_flag);

	/* The file is opened by procmsg_open_msg */
	fclose(infp);
	procmime_mimeinfo_free_all(mimeinfo);

	if (ret_val == -1) {
		g_free(outfile);
		return NULL;
	}
	return outfile;
}

/** This function gets the file name for the attachment or 
	part of mime message
	@param mimeinfo pointer to the mime information
	@return output file name
*/
gchar *procmime_get_tmp_file_name(MimeInfo * mimeinfo)
{
	static guint32 id = 0;
	gchar *base;
	gchar *filename;
	gchar f_prefix[10];

	g_return_val_if_fail(mimeinfo != NULL, NULL);

	g_snprintf(f_prefix, sizeof(f_prefix), "%08x.", id++);

	if (MIME_TEXT_HTML == mimeinfo->mime_type)
		base = "mimetmp.html";
	else {
		base = mimeinfo->filename ? mimeinfo->filename
		    : mimeinfo->name ? mimeinfo->name : "mimetmp";
		base = (gchar *) g_basename(base);
		if (*base == '\0')
			base = "mimetmp";
		Xstrdup_a(base, base, return NULL);
		subst_for_filename(base);
	}

	filename = g_strconcat(get_mime_tmp_dir(), G_DIR_SEPARATOR_S, f_prefix, base, NULL);

	return filename;
}

/** This function scans the mime type and returns the content type.
	@param mime_type the mime type information
	@return enum content type
*/
ContentType procmime_scan_mime_type(const gchar * mime_type)
{
	ContentType type;

	if (!strncasecmp(mime_type, "text/html", 9))
		type = MIME_TEXT_HTML;
	else if (!strncasecmp(mime_type, "text/", 5))
		type = MIME_TEXT;
	else if (!strncasecmp(mime_type, "message/rfc822", 14))
		type = MIME_MESSAGE_RFC822;
	else if (!strncasecmp(mime_type, "message/", 8))
		type = MIME_TEXT;
	else if (!strncasecmp(mime_type, "application/octet-stream", 24))
		type = MIME_APPLICATION_OCTET_STREAM;
	else if (!strncasecmp(mime_type, "application/", 12))
		type = MIME_APPLICATION;
	else if (!strncasecmp(mime_type, "multipart/", 10))
		type = MIME_MULTIPART;
	else if (!strncasecmp(mime_type, "image/", 6))
		type = MIME_IMAGE;
	else if (!strncasecmp(mime_type, "audio/", 6))
		type = MIME_AUDIO;
	else if (!strcasecmp(mime_type, "text"))
		type = MIME_TEXT;
	else
		type = MIME_UNKNOWN;

	return type;
}

static GList *mime_type_list = NULL;

/** This function gets the mime type.
	@param filename	the name of the file
	@return mime type
*/
gchar *procmime_get_mime_type(const gchar * filename)
{
	static GHashTable *mime_type_table = NULL;
	MimeType *mime_type;
	const gchar *p;
	gchar *ext;

	if (!mime_type_table) {
		mime_type_table = procmime_get_mime_type_table();
		if (!mime_type_table)
			return NULL;
	}

	filename = g_basename(filename);
	p = strrchr(filename, '.');
	if (!p)
		return NULL;

	Xstrdup_a(ext, p + 1, return NULL);
	g_strdown(ext);
	mime_type = g_hash_table_lookup(mime_type_table, ext);
	if (mime_type) {
		gchar *str;

		str = g_strconcat(mime_type->type, "/", mime_type->sub_type, NULL);
		return str;
	}

	return NULL;
}

/** This function gets the pointer to the mime type table
	@return pointer to the mime type table
*/
static GHashTable *procmime_get_mime_type_table(void)
{
	GHashTable *table = NULL;
	GList *cur;
	MimeType *mime_type;
	gchar **exts;

	if (!mime_type_list) {
		GList *list;
		gchar *dir;

		if (!mime_type_list) {
			list = procmime_get_mime_type_list("/etc/mime.types");
			mime_type_list = g_list_concat(mime_type_list, list);
		}
		dir = g_strdup_printf("%s%c%s", get_rc_dir(), G_DIR_SEPARATOR, "mime.types");
		list = procmime_get_mime_type_list(dir);
		g_free(dir);
		mime_type_list = g_list_concat(mime_type_list, list);

		if (!mime_type_list) {
			log_message((const gchar *) ("mime.types not found\n"));
			return NULL;
		}
	}

	table = g_hash_table_new(g_str_hash, g_str_equal);

	for (cur = mime_type_list; cur != NULL; cur = cur->next) {
		gint i;
		gchar *key;

		mime_type = (MimeType *) cur->data;

		if (!mime_type->extension)
			continue;

		exts = g_strsplit(mime_type->extension, " ", 16);
		for (i = 0; exts[i] != NULL; i++) {
			/* make the key case insensitive */
			g_strdown(exts[i]);
			/* use previously dup'd key on overwriting */
			if (g_hash_table_lookup(table, exts[i]))
				key = exts[i];
			else
				key = g_strdup(exts[i]);
			g_hash_table_insert(table, key, mime_type);
		}
		g_strfreev(exts);
	}

	return table;
}

/** This function gets the pointer to the mime type list 
	@param pointer to the file 
	@return pointer to the mime type list 
*/
static GList *procmime_get_mime_type_list(const gchar * file)
{
	GList *list = NULL;
	FILE *fp;
	gchar buf[BUFFSIZE];
	gchar *p, *delim;
	MimeType *mime_type;

	if ((fp = fopen(file, "rb")) == NULL)
		return NULL;

	debug_print("Reading %s ...\n", file);

	while (fgets(buf, sizeof(buf), fp) != NULL) {
		p = strchr(buf, '#');
		if (p)
			*p = '\0';
		g_strstrip(buf);

		p = buf;
		while (*p && !isspace(*p))
			p++;
		if (*p) {
			*p = '\0';
			p++;
		}
		delim = strchr(buf, '/');
		if (delim == NULL)
			continue;
		*delim = '\0';

		mime_type = g_new(MimeType, 1);
		mime_type->type = g_strdup(buf);
		mime_type->sub_type = g_strdup(delim + 1);

		while (*p && isspace(*p))
			p++;
		if (*p)
			mime_type->extension = g_strdup(p);
		else
			mime_type->extension = NULL;

		list = g_list_append(list, mime_type);
	}

	fclose(fp);

	if (!list)
		log_message((const gchar *) ("Can't read mime.types\n"));

	return list;
}

EncodingType procmime_get_encoding_for_charset(const gchar * charset)
{
	if (!charset)
		return ENC_8BIT;
	else if (!strncasecmp(charset, "ISO-2022-", 9) || !strcasecmp(charset, "US-ASCII"))
		return ENC_7BIT;
	else if (!strcasecmp(charset, "ISO-8859-5") ||
		 !strncasecmp(charset, "KOI8-", 5) || !strcasecmp(charset, "Windows-1251"))
		return ENC_8BIT;
	else if (!strncasecmp(charset, "ISO-8859-", 9))
		return ENC_QUOTED_PRINTABLE;
	else
		return ENC_8BIT;
}

/** This function gets the encoding type 
	@param pointer to the file 
	@return encoding type 
*/
EncodingType procmime_get_encoding_for_file(const gchar * file)
{
	FILE *fp;
	guchar buf[BUFSIZ];
	size_t len;

	if ((fp = fopen(file, "rb")) == NULL) {
		log_message((const gchar *) ("Mime:error in opening file for encoding"));
		return ENC_UNKNOWN;
	}

	while ((len = fread(buf, sizeof(gchar), sizeof(buf), fp)) > 0) {
		guchar *p;
		gint i;

		for (p = buf, i = 0; i < len; p++, i++) {
			if (*p & 0x80) {
				fclose(fp);
				return ENC_BASE64;
			}
		}
	}

	fclose(fp);
	return ENC_7BIT;
}


/** This function gets the encoding string name from type 
	@param Encoding type 
	@return encoding string 
*/
const gchar *procmime_get_encoding_str(EncodingType encoding)
{
	static const gchar *encoding_str[] = {
		"7bit", "8bit", "quoted-printable", "base64", "x-uuencode",
		NULL
	};

	if (encoding >= ENC_7BIT && encoding <= ENC_UNKNOWN)
		return encoding_str[encoding];
	else
		return NULL;
}

/** This parses the body string and builds mimeinfo structure 
	for the body structure response
	@param resp the pointer to the string containing resonse
	@param len length of the response string
	@return mime information structure,NULL on errors
*/
MimeInfo *procmime_scan_body(const gchar * resp, gint len)
{
	gint section_number = 0;
	gchar *ptr = NULL;
	gchar *end = NULL;
	MimeInfo *mimeinfo = NULL;
	gchar *tmp_ptr = NULL;
	gchar *body_str = NULL;

	if ((resp == NULL) || (len == 0)) {
		return NULL;
	}
	/* remove the escape characters from the response */
	body_str = (gchar *) remove_quote(resp);
	len = strlen(body_str);
	end = body_str + len - 1;
	ptr = strstr(body_str, BODY_STRUCT);
	if (!ptr) {
		g_free(body_str);
		return NULL;
	}
	tmp_ptr = ptr;
	ptr = strchr(tmp_ptr, START_BR);
	if (!ptr) {
		g_free(body_str);
		return NULL;
	}
	ptr++;
	section_number = 0;
	/* check the section  information */
	mimeinfo = procmime_parse_section(&ptr, end, NULL);
	if (mimeinfo == NULL) {
		g_free(body_str);
		return NULL;
	}
	/* If it is a multipart message,get all sections within that */
	if (mimeinfo->mime_type == MIME_MULTIPART) {
		procmime_parse_multipart(mimeinfo, &ptr, end, NULL);
		mimeinfo->section_number = g_strdup_printf("%d", section_number);
	} else {
		mimeinfo->section_number = g_strdup_printf("%d", ++section_number);
	}
	g_free(body_str);
	return mimeinfo;
}

/** This parses the section and checks whether it is multipart 
	@param bodyptr	the pointer to the response string pointer
	@param end pointer to the end of the response string
	@return mime information structure,NULL on errors
*/
static MimeInfo *procmime_parse_section(gchar ** bodyptr, const gchar * end,
					const gchar * section_number)
{
	MimeInfo *mimeinfo = NULL;
	gchar *tmp_ptr = NULL;

	tmp_ptr = *bodyptr;
	if ((*bodyptr == NULL) || (end == NULL)) {
		return NULL;
	}
	if (*tmp_ptr == START_BR) {

		mimeinfo = procmime_mimeinfo_new();

		if (!mimeinfo)
			return NULL;

		mimeinfo->mime_type = MIME_MULTIPART;

	} else {
		mimeinfo = procmime_parse_bodyfields(bodyptr, end, section_number);
	}
	return mimeinfo;
}

/** This parses the multipart information and builds multipart mimeinfo tree
	by linking to the root(main) node 
	@param mimeinfo the mimeinformation of the root
	@param bodyptr	the pointer to the response string pointer
	@param end pointer to the end of the response string
	@param part_section_number the section number of the multipart
	@return none
*/
static void
procmime_parse_multipart(MimeInfo * mimeinfo, gchar ** bodyptr,
			 const gchar * end, const gchar * part_section_number)
{
	MimeInfo *partinfo = NULL;
	gchar *section_end = NULL;
	gchar *section_number = NULL;
	gint section = 0;

	if (*bodyptr == end) {
		return;
	}
	if ((*bodyptr == NULL) || (end == NULL) || (mimeinfo == NULL)) {
		return;
	}

	if (**bodyptr != START_BR) {
		return;
	}
	/* get the end of the section */
	section_end = procmime_get_para_end(1, *bodyptr, end);

	if (section_end == NULL) {
		return;
	}

	(*bodyptr)++;
	/* check for all the sections */
	while (*bodyptr < section_end) {


		/* calculate the section number */
		if (part_section_number) {
			section_number = g_strdup_printf("%s%c%d",
							 part_section_number,
							 SECTION_SEPARATOR, ++section);
		} else {
			section_number = g_strdup_printf("%d", ++section);
		}

		partinfo = procmime_parse_section(bodyptr, end, section_number);

		if (!partinfo) {
			return;
		}

		if (partinfo->mime_type == MIME_MULTIPART) {
			partinfo->section_number = g_strdup_printf("%s", section_number);
			/* call the function recursively for multipart */
			procmime_parse_multipart(partinfo, bodyptr, end, section_number);
			g_free(section_number);
		} else {
			partinfo->section_number = section_number;
		}
		/* insert the mimeinformation in to the tree */
		procmime_mimeinfo_insert(mimeinfo, partinfo);

		if (mimeinfo->sub != NULL) {
			mimeinfo->sub->parent = mimeinfo->parent;
		}
		/* This is for multi section */
		while (*bodyptr < section_end) {
			if ((**bodyptr == SPACE_CHAR) || (**bodyptr == END_BR)) {
				(*bodyptr)++;
			} else {
				break;
			}
		}
		/* if no starting bracket,no more sections are present */
		if (**bodyptr != START_BR) {
			procmime_parse_multipart_fields(*bodyptr, end, mimeinfo);
			*bodyptr = section_end;
			break;
		} else {
			(*bodyptr)++;
		}

	}
}

/** This parses the body string and puts the fields in mimeinfo structure
	@param bodyptr the pointer to the string containing header information
	@param end the pointer to the end of the string
	@return mime information structure,NULL on errors
*/
static MimeInfo *procmime_parse_bodyfields(gchar ** bodyptr, const gchar * end,
					   const gchar * section_number)
{
	MimeInfo *mimeinfo = NULL;
	gint field_count = 0;
	gchar *ptr = NULL;
	gchar *section_end = NULL;
	gboolean parsed_sub = FALSE;

	if ((*bodyptr == NULL) || (end == NULL)) {
		return NULL;
	}
	mimeinfo = procmime_mimeinfo_new();

	if (!mimeinfo) {
		return NULL;
	}

	ptr = *bodyptr;
	section_end = procmime_get_para_end(1, *bodyptr, end);

	for (; *bodyptr < section_end; (*bodyptr)++) {
		if (**bodyptr == START_BR) {

			if ((field_count > ALL_FIELDS) &&
			    (mimeinfo->mime_type == MIME_MESSAGE_RFC822) && (parsed_sub == FALSE)) {

				parsed_sub = TRUE;
				procmime_parse_sub_part(mimeinfo, bodyptr, section_end,
							section_number);
			} else {

				procmime_parse_body_para(mimeinfo, bodyptr, end, section_number);
				field_count++;
			}
		} else if (**bodyptr == SPACE_CHAR) {
			continue;
		} else {
			if (procmime_get_field(mimeinfo, ++field_count, bodyptr,
					       section_end) == PARSE_FAIL) {
				(*bodyptr)++;
			}
		}
	}
	return mimeinfo;

}

/** This function fetches the mime type and returns the content type.
	@param mime_type the mime type information
	@return enum content type
*/
static void
procmime_parse_mime_type(const gchar * mime_type, const gchar * sub_type, MimeInfo * mimeinfo)
{
	ContentType type = MIME_UNKNOWN;

	if ((mime_type == NULL) || (sub_type == NULL)) {
		mimeinfo->mime_type = type;
		return;
	}
	mimeinfo->content_type = g_strdup_printf("%s%c%s", mime_type, TYPE_SEPRTR, sub_type);

	mimeinfo->mime_type = procmime_scan_mime_type(mimeinfo->content_type);
	return;
}

/** This parses the section and extracts different fields like charset
	@param mimeinfo	pointer to the mimeinformation,to be used to set the fields.
	@param field_count the field number
	@param bodyptr the pointer to the response string
	@return 0 on success -1 on errors
*/
static gint
procmime_get_field(MimeInfo * mimeinfo, gint field_count, gchar ** bodyptr, const gchar * end)
{
	gchar tmp_val1[MAX_FIELD_LENGTH];
	gchar tmp_val2[MAX_FIELD_LENGTH];
	gchar *ptr = NULL;

	ptr = *bodyptr;

	if ((ptr == NULL) || (ptr >= end) || (mimeinfo == NULL)) {
		return PARSE_FAIL;
	}

	switch (field_count) {

	case 1:		/* the field is mime type */
		if (sscanf(ptr, "%s %s", tmp_val1, tmp_val2) != 2) {
			return PARSE_FAIL;
		}
		extract_quote(tmp_val1, QUOTE);
		extract_quote(tmp_val2, QUOTE);
		procmime_parse_mime_type(tmp_val1, tmp_val2, mimeinfo);

		ptr = ptr + strlen(tmp_val1) + strlen(tmp_val2) + 1;
		while ((*ptr != SPACE_CHAR) && (ptr < end)) {
			ptr++;
		}
		if (ptr > end)
			return PARSE_FAIL;
		break;

	case 2:		/* any other field,just increment the pointer */
	case 3:
	case 4:
		if (*ptr == QUOTE) {
			ptr++;
			while (ptr != NULL) {
				if ((*ptr == QUOTE) || (ptr >= end)) {
					break;
				}
				ptr++;
			}
			if (ptr == end) {
				return PARSE_FAIL;
			}
			break;
		}
		if (sscanf(ptr, "%s", tmp_val1) != 1) {
			return PARSE_FAIL;
		}
		ptr = ptr + strlen(tmp_val1);
		while ((*ptr != SPACE_CHAR) && (ptr < end)) {
			ptr++;
		}

		if (ptr > end) {
			ptr = (gchar *) end;
		}
		break;

	case 5:		/* The field is encoding */
		if (sscanf(ptr, "%s", tmp_val1) != 1) {
			return PARSE_FAIL;
		}
		ptr = ptr + strlen(tmp_val1);
		extract_quote(tmp_val1, QUOTE);
		procmime_scan_encoding(mimeinfo, tmp_val1);
		if (ptr > end)
			return PARSE_FAIL;
		break;

	case 6:		/* The field is filesize */
		if (sscanf(ptr, "%s", tmp_val1) != 1) {
			return PARSE_FAIL;
		}
		ptr = ptr + strlen(tmp_val1);
		extract_quote(tmp_val1, QUOTE);
		mimeinfo->size = atol(tmp_val1);
		if (ptr > end)
			return PARSE_FAIL;
		break;

	default:
		/* nothing else to be read.Increment the pointer to the end */
		if (*ptr == QUOTE) {

			while ((*(++ptr) != QUOTE) && (ptr < end)) {
				ptr++;
			}
			if (ptr == end) {
				return PARSE_FAIL;
			}
			break;
		}
		sscanf(ptr, "%s", tmp_val1);
		ptr = ptr + strlen(tmp_val1);
		if (ptr > end) {
			ptr = (gchar *) end;
		}
		break;

	}
	*bodyptr = ptr;
	return PARSE_SUCCESS;
}

/** This parses the fields within the parantheses which can vary 
	@param mime information structure pointer
	@param bodyptr	the pointer to the response string pointer
	@param end pointer to the end of the response string
	@param none
*/
static void
procmime_parse_body_para(MimeInfo * mimeinfo, gchar ** bodyptr,
			 const gchar * end, const gchar * section_number)
{
	gchar *ptr = NULL;
	gchar *para_ptr = NULL;
	gchar *tmp = NULL;
	gchar *tmp_name = NULL;
	gchar *value = NULL;
	gchar tmp1[MAX_FIELD_LENGTH];
	gchar *charset = NULL;
	gchar *field = NULL;
	gboolean dispose = FALSE;
	gint len = 0;

	if ((mimeinfo == NULL) || (*bodyptr == NULL) || (*bodyptr >= end)) {
		return;
	}

	para_ptr = *bodyptr;

	if (*para_ptr != START_BR) {
		return;
	}

	ptr = *bodyptr;

	para_ptr = procmime_get_para_end(0, *bodyptr, end);


	/* pointer is now set to the end of the parantheses string */
	if (para_ptr == end) {
		return;
	}

	for (tmp = ptr + 1; tmp < para_ptr; tmp++) {
		if (*tmp == START_BR) {
			dispose = TRUE;
			break;
		}
	}
	if (dispose == TRUE) {

		/* this contains attachment fields */
		procmime_get_filename(ptr, para_ptr, mimeinfo);

	} else {

		/* read the character set field */
		charset = (gchar *) g_strstr_len(ptr, para_ptr - ptr, CAP_CHARSET);

		if (charset == NULL) {
			charset = (gchar *) g_strstr_len(ptr, para_ptr - ptr, CHARSET);
		}

		if (charset != NULL) {
			tmp = charset + strlen(CHARSET);
			if (tmp < para_ptr) {
				tmp++;
				sscanf(tmp, "%s", tmp1);
				extract_quote(tmp1, QUOTE);
				mimeinfo->charset = g_strdup(tmp1);
			}
		}

		field = (gchar *) g_strstr_len(ptr, para_ptr - ptr, CAP_NAME);

		if (field == NULL) {
			field = (gchar *) g_strstr_len(ptr, para_ptr - ptr, NAME);
		}

		if (field != NULL) {

			tmp = field + strlen(NAME);
			if (tmp < para_ptr) {
				tmp++;

				value = procmime_get_name_str(tmp, para_ptr);
				len = strlen(value) + 1;
				Xalloca(tmp_name, len, return);
				conv_unmime_header(tmp_name, len, value, NULL);
				mimeinfo->name = g_strdup(tmp_name);
				g_free(value);
			}
		}
	}
	/* set pointer to the end of the parantheses */

	*bodyptr = para_ptr + 1;

}

/** This parses the end of the section,by ignoring the parantheses in between
	@param left_op starting count for the left operator
	@param para_ptr pointer to the string
	@param end pointer to the end,till the searching has to happen
	@return pointer to the end of the parantheses
*/
static gchar *procmime_get_para_end(gint left_op, gchar * para_ptr, const gchar * end)
{
	gint j;
	gint right_op;

	if ((para_ptr == NULL) || (end == NULL)) {
		return NULL;
	}
	for (j = 0, right_op = 0; para_ptr < end; j++, para_ptr++) {

		if (*para_ptr == START_BR) {
			left_op++;
		} else if (*para_ptr == END_BR) {
			right_op++;
		}

		if (left_op == right_op) {
			break;
		}
	}
	return para_ptr;
}

/** This parses content disposition field 
	@param mime information structure pointer
	@param ptr the pointer to the response string
	@param end pointer to the end of the string for search
	@return none
*/
static void procmime_parse_content_disp(MimeInfo * mimeinfo, const gchar * ptr, const gchar * end)
{
	gchar *attach = NULL;

	if ((mimeinfo == NULL) || (ptr == NULL) || (end == NULL)) {
		return;
	}
	attach = (gchar *) g_strstr_len(ptr, end - ptr, ATTACHMENT);

	if (attach == NULL) {
		attach = (gchar *) g_strstr_len(ptr, end - ptr, CAP_ATTACHMENT);
	}
	if (attach == NULL) {
		attach = (gchar *) g_strstr_len(ptr, end - ptr, INLINE);
	}
	if (attach == NULL) {
		attach = (gchar *) g_strstr_len(ptr, end - ptr, CAP_INLINE);
	}

	if (attach != NULL) {
		mimeinfo->content_disposition = g_strdup(ATTACHMENT);
	}

}

/** This function parses the mimeinformation of the RFC822 Message part
 @param mimeinfo pointer to the main mimeinformation 
 @param ptr pointer to the response string 
 @param end pointer to the end of the section 
 @return mimeinfo pointer to the sub mimeinformation 
*/
MimeInfo *procmime_get_sub(MimeInfo * mimeinfo, gchar * ptr, const gchar * end,
			   const gchar * section_number)
{
	MimeInfo *submimeinfo = NULL;

	ptr++;

	/* check the section  information */
	submimeinfo = procmime_parse_section(&ptr, end, NULL);
	if (submimeinfo == NULL) {
		return NULL;
	}
	/* If it is a multipart message,get all sections within that */
	if (submimeinfo->mime_type == MIME_MULTIPART) {
		procmime_parse_multipart(submimeinfo, &ptr, end, section_number);
	}
	return submimeinfo;
}

/** This function searches the text first text pointer
 @param pointer to the main mime information 
 @return pointer to the mime information,which points to text/html section  
*/
MimeInfo *procmime_get_text_ptr(MimeInfo * mimeinfo, gboolean get_html)
{
	MimeInfo *partinfo = NULL;

	if (mimeinfo == NULL) {
		return NULL;
	}

	if (mimeinfo->children != NULL) {
		partinfo = mimeinfo->children;
	} else {
		partinfo = mimeinfo;

	}
	while (partinfo != NULL) {
		if (partinfo->mime_type == MIME_TEXT) {
			if (get_html == FALSE) {
				break;
			}
		}
		if (partinfo->mime_type == MIME_TEXT_HTML) {
			break;
		}
		if (partinfo->mime_type == MIME_MULTIPART) {
			partinfo = partinfo->children;
		} else {
			partinfo = partinfo->next;

		}

	}
	return partinfo;
}

/** This function if the message has any attachments. 
 @param filename name of the file which contains the message 
 @param bodytype_flag if this flag is true,message is 
		downloaded with receive type 2(file has .mime extension)
 @return TRUE,if the message has attachments,FALSE otherwise 
*/
gboolean procmime_check_attachments(const gchar * filename, gboolean bodytype_flag)
{
	FILE *fp = NULL;
	MimeInfo *mimeinfo = NULL;
	MimeInfo *tmpinfo = NULL;
	guchar buf[BUFSIZ];
	gchar *body_str = NULL;
	size_t len;
	gint pos = -1;
	gint level = 0;
	glong fpos = 0;
	GSList *file_pos_list = NULL;
	gboolean ret_val = FALSE;

	memset(buf, '\0', BUFSIZ);

	if (bodytype_flag == TRUE) {
		fp = fopen(filename, "r");
		if (fp == NULL) {
			return FALSE;
		}
		while ((len = fread(buf, sizeof(gchar), sizeof(buf) - 1, fp)) > 0) {
			body_str = g_strconcat(buf, NULL);
		}
		debug_print("%s", body_str);
		mimeinfo = procmime_scan_body(body_str, strlen(body_str));
		g_free(body_str);
		fclose(fp);

	} else {

		fp = fopen(filename, "r+");
		if (fp == NULL) {
			return FALSE;
		}
		mimeinfo = (MimeInfo *) procmime_scan_message(fp, pos, level, fpos, file_pos_list);
		fclose(fp);
	}
	if (mimeinfo == NULL) {
		return FALSE;
	}
	tmpinfo = mimeinfo;
	/* check for if it is actually a message with attachments */
	if (g_ascii_strcasecmp(mimeinfo->content_type, MULTIPART_MIXED) == 0) {
		ret_val = TRUE;
	} else if (g_ascii_strcasecmp(mimeinfo->content_type, MULTIPART_REPORT) == 0) {
		ret_val = TRUE;

	} else if (procmime_check_extra_part(mimeinfo) == TRUE) {
		ret_val = TRUE;
	}
	procmime_mimeinfo_free_all(tmpinfo);
	return ret_val;
}

/** This function gets the name string name  
	@param response string 
	@param end pointer 
	@return name string 
*/
static gchar *procmime_get_name_str(const gchar * str, const gchar * end)
{
	gchar *temp_ptr = (gchar *) end;
	gchar *val = NULL;
	gint len;

	while ((*temp_ptr == SPACE_CHAR) || (*temp_ptr == END_BR)) {
		temp_ptr--;
		if (temp_ptr >= end) {
			break;
		}
	}
	while ((*str == SPACE_CHAR) && (str < end)) {
		str++;
	}
	if (temp_ptr > str) {
		len = temp_ptr - str + 1;
		val = g_strndup(str, len);
	}
	extract_quote(val, QUOTE);
	return val;
}

/** This function gets the multipart fields  
	@param response string 
	@param end pointer 
	@param mimeinfo 
	@return parsing is successful or not 
*/
static gint procmime_parse_multipart_fields(gchar * fldptr, const gchar * end, MimeInfo * mimeinfo)
{
	gchar tmp[MAX_FIELD_LENGTH];
	gchar val[MAX_FIELD_LENGTH];
	gchar *tmp_ptr = NULL;

	sscanf(fldptr, "%s", tmp);
	extract_quote(tmp, QUOTE);
	mimeinfo->content_type = g_strdup_printf("%s%c%s", MULTIPART_TYPE, TYPE_SEPRTR, tmp);

	tmp_ptr = strchr(fldptr, START_BR);
	if (tmp_ptr == NULL) {
		return PARSE_FAIL;
	}
	tmp_ptr++;
	sscanf(tmp_ptr, "%s%s", tmp, val);
	extract_quote(tmp, QUOTE);
	extract_quote(val, QUOTE);
	if (strncasecmp(tmp, BOUNDARY, strlen(BOUNDARY)) == 0) {
		mimeinfo->boundary = g_strndup(val, strlen(val));
		return PARSE_SUCCESS;
	} else {

		return PARSE_FAIL;
	}
}

/** This function gets the body size  
	@param mimeinfo pointer to the mimeinformation 
	@param sec_nu section number 
	@return size 
*/
gint procmime_get_body_size(MimeInfo * mime_info, gchar * sec_nu)
{
	gint size = 0;

	if (mime_info == NULL) {
		return PARSE_FAIL;
	}
	if (mime_info->mime_type == MIME_MULTIPART) {
		size = mime_multi_part_body_size(mime_info, sec_nu);
	} else {
		size = mime_info->size;
	}
	return size;
}

/** This function gets the multipart body size  
	@param mimeinfo 
	@param sec_nu section number 
	@return size 
*/
static gint mime_multi_part_body_size(MimeInfo * mime_info, gchar * sec_nu)
{
	gint size = 0;
	gint return_size = 0;

	if (mime_info == NULL) {
		return PARSE_FAIL;
	}
	while (mime_info != NULL) {
		if (mime_info->content_disposition != NULL) {
			mime_info = mime_info->next;
			continue;

		}
		if (mime_info->mime_type == MIME_MULTIPART) {
			size = mime_multi_part_body_size(mime_info->children, sec_nu);
			if (size != PARSE_FAIL) {
				return_size += size;
			}
		} else {
			return_size += mime_info->size;
		}
		mime_info = mime_info->next;
	}
	return return_size;
}

/** This function gets the filename from the response
 @param ptr pointer to the response string
 @param ptr pointer to the end of response string parantheses
 @param mimeinfo pointer to the mime information
 @return none
*/
static void procmime_get_filename(gchar * ptr, gchar * para_ptr, MimeInfo * mimeinfo)
{
	gchar *field = NULL;
	gchar *tmp = NULL;
	gchar *tmp_name = NULL;
	gchar *value = NULL;
	size_t len;

	field = (gchar *) g_strstr_len(ptr, para_ptr - ptr, CAP_FILENAME);

	if (!field) {
		field = (gchar *) g_strstr_len(ptr, para_ptr - ptr, FILENAME);
	}

	if (field) {
		tmp = field + strlen(FILENAME);
		if (tmp < para_ptr) {
			tmp++;
			value = procmime_get_name_str(tmp, para_ptr);

			len = strlen(value) + 1;
			Xalloca(tmp_name, len, return);
			conv_unmime_header(tmp_name, len, value, NULL);
			mimeinfo->filename = g_strdup(tmp_name);
			g_free(value);
		}
	}
	/* extract the attachment name if it exists */
	procmime_parse_content_disp(mimeinfo, ptr, para_ptr);
}

/** This function checks if the mimeinfo has any attachments.
 @param mimeinfo mimeinformation pointer 
 @return TRUE,if the message has attachments,FALSE otherwise
*/
gboolean procmime_mime_attachment(MimeInfo * mimeinfo)
{
	if (mimeinfo == NULL) {
		return FALSE;
	}
	/* check for if it is actually a message with attachments */
	if (g_ascii_strcasecmp(mimeinfo->content_type, MULTIPART_MIXED) == 0) {
		return TRUE;
	}
	if (g_ascii_strcasecmp(mimeinfo->content_type, MULTIPART_REPORT) == 0) {

		return TRUE;
	}
	return (procmime_check_extra_part(mimeinfo));
}

/**This function checks if any of the nested emails
   in the main mail has S/MIME mail
   @param mimeinfo - MimeInformation
   @param return - TRUE if any nested mail is S/MIME
  */
gboolean procmime_is_smime(MimeInfo * mimeinfo)
{
	if (mimeinfo == NULL) {
		return FALSE;
	}

	while (mimeinfo != NULL) {
		if ((!g_ascii_strncasecmp(mimeinfo->content_type, SIGNED, 16))
		    || (g_strrstr(mimeinfo->content_type, ENCR) != NULL)
		    || (g_strrstr(mimeinfo->content_type, ENCR7) != NULL)) {
			return TRUE;
		}
		mimeinfo = procmime_mimeinfo_next(mimeinfo);
	}
	return FALSE;
}

/** This function parses the emailMessage attachment mime information
 @param mimeinfo pointer to the main mimeinformation
 @param ptr pointer to the response string
 @param end pointer to the end of the section
 @return mimeinfo pointer to the sub mimeinformation
*/
static void
procmime_parse_sub_part(MimeInfo * mimeinfo, gchar ** bodyptr,
			const gchar * end, const gchar * section_number)
{
	gchar *sub_end = NULL;
	gchar *ptr = *bodyptr;

	mimeinfo->content_disposition = g_strdup(ATTACHMENT);
	while (ptr < end) {
		if (*ptr == START_BR) {
			break;
		}
		ptr++;
	}
	if (ptr < end) {
		sub_end = procmime_get_para_end(0, ptr, end);
		mimeinfo->sub = procmime_get_sub(mimeinfo, ptr, sub_end, section_number);
		if (mimeinfo->sub != NULL) {
			mimeinfo->sub->main = mimeinfo;
		}
		*bodyptr = sub_end;
	} else {
		*bodyptr = (gchar *) end;
	}
}

/** This function decodes the text content, and this is called 
 *  by the adaptation layer function
 *  @param msghdr pointer to the message header information
 *  @param file - File in which text contents should be decoded
 *  @return filename in which the decoded contents are put
 */

gchar *procmime_get_smime_text_content(MsgHeader * msghdr, const gchar * file)
{
	FILE *infp;
	MimeInfo *mimeinfo, *partinfo;
	gchar *outfile = NULL;
	gchar *tmp_dir = NULL;
	gint ret_val = -1;
	gint pos = -1;
	gint level = 0;
	glong fpos = 0;
	GSList *file_pos_list = NULL;


	if (msghdr == NULL) {
		debug_print("\nThe Header NULL\n");
		return NULL;
	}

	if ((infp = fopen(file, "rb")) == NULL) {
		debug_print("Error in opening file\n");
		return NULL;
	}

	mimeinfo = procmime_scan_message(infp, pos, level, fpos, file_pos_list);

	if (!mimeinfo) {
		fclose(infp);
		return NULL;
	}

	partinfo = procmime_get_text_ptr(mimeinfo, FALSE);
	tmp_dir = (gchar *)get_tmp_dir();

	if (!is_dir_exist(tmp_dir)) {
		debug_print("TEMPORARY directory does not exist\n");
		tmp_dir = (gchar *) g_get_tmp_dir();
	}
	outfile = g_strdup_printf("%s%c%s", tmp_dir, G_DIR_SEPARATOR, msghdr->msgid);
	if (!outfile) {
		debug_print("\n NOT able to allocate memory for OUT\n");
		fclose(infp);
		procmime_mimeinfo_free_all(mimeinfo);
		return NULL;
	}

	if (partinfo)
		ret_val = procmime_get_content(partinfo, infp, outfile, FALSE, FALSE);

	fclose(infp);
	procmime_mimeinfo_free_all(mimeinfo);

	if (ret_val == -1) {
		g_free(outfile);
		return NULL;
	}
	return outfile;
}

gchar *procmime_get_tmp_mhtml_file(const gchar * msgid, const gchar * src_file)
{
	gchar *temp_file = NULL;
	static guint random = 0;

	if (msgid == NULL) {
		return NULL;

	}
	temp_file = g_strdup_printf("%s%c%s%04x%c%s",
				    get_tmp_dir(), G_DIR_SEPARATOR, msgid,
				    random++, SECTION_SEPARATOR, MHTML_EXTENSION);

	if (src_file == NULL) {
		return temp_file;
	}
	if (copy_file(src_file, temp_file, TRUE) < 0) {
		g_free(temp_file);
		return NULL;
	}
	return temp_file;

}

static
MimeInfo *procmime_get_smime_mimeinfo(MimeInfo * mimeinfo, FILE * fp, gint pos, gchar * boundary)
{
	gint boundary_len = 0;
	gchar buf[BUFFSIZE];
	FILE *outfp = NULL;
	gchar outfile[BUFFSIZE];
	gchar buffer[BUFFSIZE];
	MimeInfo *mime = NULL;
	gchar *str = NULL;
	glong prev_fpos = 0;
	gint seek = 0;
	gboolean ret = TRUE;
	gint val = 0;

	CHECK_RETURN_VAL_IF_FAIL(mimeinfo != NULL, NULL);
	CHECK_RETURN_VAL_IF_FAIL(fp != NULL, NULL);
	CHECK_RETURN_VAL_IF_FAIL(boundary != NULL, NULL);

	if (pos < 0) {
		seek = mimeinfo->fpos;
	} else {
		seek = GPOINTER_TO_INT(g_slist_nth_data(mimeinfo->file_pos_list, pos));
	}

	val = fseek(fp, seek, SEEK_SET);
	CHECK_RETURN_VAL_IF_FAIL(val != -1, NULL);

	ret = get_temp_smime_file(outfile, BUFFSIZE + 1);
	CHECK_RETURN_VAL_IF_FAIL(ret != FALSE, NULL);
	remove_smime_file(outfile);

	outfp = fopen(outfile, "w+");
	if (outfp == NULL) {
		send_engine_error_to_ui(errno);
		return NULL;
	}

	if (change_file_mode_rw(outfp, outfile) < 0) {
		debug_print("Unable to change file mode for interfile\n");
		fclose(outfp);
		return NULL;
	}

	boundary_len = strlen(boundary);

	memset(buffer, '\0', sizeof(buffer));
	while (fgets(buffer, sizeof(buffer), fp) != NULL) {
		if (buffer[0] == '\r' || buffer[0] == '\n') {
			break;
		}
		memset(buffer, '\0', sizeof(buffer));
	}
	prev_fpos = ftell(fp);

	memset(buf, '\0', sizeof(buffer));
	while (((str = fgets(buf, sizeof(buf), fp)) != NULL) &&
	       (!boundary || !IS_BOUNDARY(buf, boundary, boundary_len))) {
		strcrchomp(buf);
		fputs(buf, outfp);
		memset(buf, '\0', sizeof(buf));
	}

	mimeinfo->size = ftell(fp) - prev_fpos - strlen(buf);

	if (fseek(fp, ftell(fp) - strlen(buf), SEEK_SET) < 0) {
		fclose(outfp);
		perror("fseek");
		return NULL;
	}

	fclose(outfp);

	mime = procmime_get_smime_mime(outfile, mimeinfo, pos, mimeinfo->fpos,
				       mimeinfo->file_pos_list);
	CHECK_RETURN_VAL_IF_FAIL(mime != NULL, NULL);
	return mime;
}

static
MimeInfo *procmime_get_smime_mime(gchar * filename, MimeInfo * mimeinfo, gint pos,
				  glong fpos, GSList * file_pos_list)
{

	gchar outfile[BSIZE];
	gchar detached[BSIZE];
	gboolean saved = FALSE;
	gint private_cert_state = 0;
	gint recip_cert_state = 0;
	gboolean signature = TRUE;
	gboolean trust = FALSE;
	gchar cert_path[BSIZE];
	gint smime = 0;
	MimeInfo *mime = NULL;
	MsgInfo *msg_info = NULL;
	gchar *signer_email_id = NULL;
	gchar *recip_email_id = NULL;
	FILE *fp = NULL;
	FILE *filefp = NULL;
	MsgFlags flags;
	gint level = 0;
	gint sender_cert_id = 0;
	gint recip_cert_id = 0;
	PrefsAccount *prefs_account = NULL;


	if (FALSE == (get_temp_smime_file(outfile, BSIZE + 1))) {
		debug_print("Unable to get temp smime file\n");
		return NULL;
	}

	if (FALSE == (get_temp_smime_file(detached, BSIZE + 1))) {
		debug_print("Unable to get temp smime file\n");
		return NULL;
	}

	if (FALSE == (get_temp_smime_file(cert_path, BSIZE + 1))) {
		debug_print("Unable to get temp smime file\n");
		return NULL;
	}

	remove_smime_file(outfile);
	remove_smime_file(detached);
	remove_smime_file(cert_path);

	flags.perm_flags = MSG_NEW | MSG_UNREAD;
	flags.tmp_flags = 0;

	filefp = fopen(filename, "r+");
	if (filefp == NULL) {
		remove_smime_file(outfile);
		remove_smime_file(detached);
		remove_smime_file(cert_path);
		return NULL;
	}

	msg_info = procheader_parse_stream(filefp, flags, TRUE);

	if (msg_info == NULL) {
		fclose(filefp);
		remove_smime_file(outfile);
		remove_smime_file(detached);
		remove_smime_file(cert_path);
		return NULL;
	}

	signer_email_id = g_strdup(parse_from_address(msg_info->header->from));
	procmsg_msginfo_free(msg_info);

	if ((prefs_account = account_get_default_account()) != NULL) {
		recip_email_id = prefs_account->user_settings.email;
	}

	fclose(filefp);
	if (((folder_process_smime_mail(filename, outfile, detached,
					&saved, signer_email_id, recip_email_id,
					&recip_cert_state, &private_cert_state,
					&signature, &trust, cert_path, &smime,
					&sender_cert_id, &recip_cert_id))) != 0) {
		remove_smime_file(outfile);
		remove_smime_file(detached);
		remove_smime_file(cert_path);
		g_free(signer_email_id);
		return NULL;
	}

	fp = fopen(outfile, "r");
	if (fp == NULL) {
		remove_smime_file(outfile);
		remove_smime_file(detached);
		remove_smime_file(cert_path);
		g_free(signer_email_id);
		return NULL;
	}
	pos = pos + 1;
	level = mimeinfo->level + 1;
	mime = (MimeInfo *) procmime_scan_message(fp, pos, level, fpos, file_pos_list);

	fclose(fp);
	remove_smime_file(filename);
	remove_smime_file(outfile);
	g_free(signer_email_id);

	if (mime == NULL) {
		debug_print("Mimeinfo in NULL\n");
		return NULL;
	}
	mimeinfo->trust = trust;
	mimeinfo->saving = saved;
	mimeinfo->cert_path = g_strdup(cert_path);
	mimeinfo->cert_state_sender = private_cert_state;
	mimeinfo->cert_state_recip = recip_cert_state;
	mimeinfo->signature_verify = signature;
	mimeinfo->smime_type = smime;
	mimeinfo->sender_cert_id = sender_cert_id;
	mimeinfo->recip_cert_id = recip_cert_id;

	remove_smime_file(outfile);
	remove_smime_file(detached);
	return mime;
}

/** 
  This function is called to check the filename of email attachment 
  @param maininfo - Mimeinfo of the parent
  @param subject - subject of the email attachment
  @return none 
*/
static void procmime_get_subfilename(MimeInfo * maininfo, gchar * subject)
{

	if ((maininfo->mime_type == MIME_MESSAGE_RFC822) && (subject != NULL)) {
		g_free(maininfo->filename);
		maininfo->filename = subject;
	}
}
gint procmime_copy_part(FILE * infp, FILE * outfp, MimeInfo * mimeinfo, gboolean rfc_hdr)
{

	gchar buf[BUFFSIZE];
	gchar *boundary = NULL;
	gint boundary_len = 0;

	memset(buf, '\0', BUFFSIZE);
	if (rfc_hdr == TRUE) {
		/* copy Rfc822 headers */

		while (fgets(buf, sizeof(buf), infp) != NULL) {

			fputs(buf, outfp);
			if (buf[0] == '\r' || buf[0] == '\n')
				break;
		}

	} else {
		/* copy part of the file */
		if (fseek(infp, mimeinfo->fpos, SEEK_SET) < 0) {
			return PARSE_FAIL;
		}
		while (fgets(buf, sizeof(buf), infp) != NULL)
			if (buf[0] == '\r' || buf[0] == '\n')
				break;

		if (mimeinfo->parent && mimeinfo->parent->boundary) {
			boundary = mimeinfo->parent->boundary;
			boundary_len = strlen(boundary);
		}
		while ((fgets(buf, sizeof(buf), infp) != NULL) &&
		       (!boundary || !IS_BOUNDARY(buf, boundary, boundary_len))) {
			fputs(buf, outfp);
		}
	}
	return PARSE_SUCCESS;

}

gint procmime_write_purge_section(const gchar * filename, const gchar * section)
{
	FILE *fp = NULL;
	gchar buf[BUFFSIZE];
	gchar *outbuf = NULL;

	memset(buf, '\0', BUFFSIZE);
	fp = fopen(filename, "a");
	if (fp == NULL) {
		send_engine_error_to_ui(errno);
		return PARSE_FAIL;
	}
	outbuf = g_strdup(section);
	fprintf(fp, "%s\n", section);
	g_free(outbuf);
	fclose(fp);
	return PARSE_SUCCESS;

}
static gchar *procmime_check_head_tag(gchar * buf,
				      gboolean * style_found_flag, const gchar * default_font)
{
	gchar *start_ptr = NULL;
	gchar *ptr = NULL;
	gchar *out_buf = NULL;
	gchar *font_str = NULL;
	gchar *html_body = NULL;
	static gboolean head_start = FALSE;
	static gboolean html_start = FALSE;

	if (html_start == FALSE) {

		start_ptr = strstr(buf, HTML_CAPTAG);

		if (start_ptr == NULL) {
			start_ptr = strstr(buf, HTML_TAG);

		}
		if (start_ptr != NULL) {
			html_start = TRUE;
		}
	}
	if (html_start == FALSE) {
		return NULL;
	}
	if (((start_ptr = strstr(buf, HEAD_TAG)) != NULL) ||
	    ((start_ptr = strstr(buf, HEAD_CAPTAG)) != NULL)) {
		head_start = TRUE;

	}

	if (head_start == TRUE) {

		if (((ptr = strstr(buf, STYLE_TAG)) != NULL) ||
		    ((ptr = strstr(buf, STYLE_CAPTAG)) != NULL)) {

			start_ptr = strchr(ptr, '>');
			if (start_ptr == NULL) {
				return NULL;
			}
			start_ptr++;
			if (start_ptr == NULL) {
				return NULL;
			}
			html_body = g_strndup(buf, start_ptr - buf);
			font_str = g_strdup_printf(FONT_STR, default_font);
			if (start_ptr != NULL) {
				out_buf = g_strconcat(html_body, font_str, start_ptr, NULL);
			} else {
				out_buf = g_strconcat(html_body, font_str, NULL);
			}
			g_free(font_str);
			g_free(html_body);
			*style_found_flag = TRUE;
		}

	}
	if (((start_ptr = strstr(buf, HEAD_END_TAG)) != NULL) ||
	    ((start_ptr = strstr(buf, HEAD_END_CAPTAG)) != NULL)) {

		if ((*style_found_flag == FALSE) && (html_start == TRUE)) {
			font_str = g_strdup_printf(STYLE_STR, default_font);

			html_body = g_strndup(buf, start_ptr - buf);
			out_buf = g_strconcat(html_body, font_str, start_ptr, NULL);
			g_free(font_str);
			g_free(html_body);
			*style_found_flag = TRUE;
		}

	}
	if (((start_ptr = strstr(buf, BODY_TAG)) != NULL) ||
	    ((start_ptr = strstr(buf, BODY_CAPTAG)) != NULL)) {

		if ((*style_found_flag == FALSE) && (html_start == TRUE)) {
			font_str = g_strdup_printf(HEAD_STYLE_STR, default_font);

			html_body = g_strndup(buf, start_ptr - buf);
			out_buf = g_strconcat(html_body, font_str, start_ptr, NULL);
			g_free(font_str);
			g_free(html_body);
			*style_found_flag = TRUE;
		}

	}
	if (((start_ptr = strstr(buf, HTML_END_TAG)) != NULL) ||
	    ((start_ptr = strstr(buf, HTML_END_TAG)) != NULL)) {

		head_start = FALSE;
		html_start = FALSE;

	}
	if (*style_found_flag == TRUE) {
		head_start = FALSE;
	}
	return out_buf;

}

GSList *procmime_read_purged(const gchar * filepath)
{
	gchar *purge_filename = NULL;
	GSList *purge_list = NULL;
	gchar buf[BUFFSIZE];
	gchar section[BUFFSIZE];
	FILE *fp = NULL;

	purge_filename = g_strdup_printf("%s%c%s", filepath, SECTION_SEPARATOR, PURGE_SUFFIX);
	fp = fopen(purge_filename, "r");
	if (fp == NULL) {
		g_free(purge_filename);
		return NULL;
	}
	while (fgets(buf, sizeof(buf), fp) != NULL) {
		sscanf(buf, "%s", section);
		purge_list = g_slist_append(purge_list, g_strdup(section));
	}
	fclose(fp);
	return purge_list;
}

void procmime_set_purged(MimeInfo * partinfo, GSList * purge_list)
{
	GSList *cur = NULL;

	/* check whether attachment was purged */
	for (cur = purge_list; cur != NULL; cur = cur->next) {
		if (partinfo->section_number == NULL) {
			break;
		}
		if (strncmp(cur->data, partinfo->section_number,
			    strlen(partinfo->section_number)) == 0) {

			partinfo->is_purged = TRUE;
		}
	}
}
static gboolean procmime_check_purgable(const MimeInfo * partinfo, const gchar * filepath)
{
	gchar *enc_filename = NULL;

	if ((partinfo->section_number == NULL) || (partinfo->content_disposition == NULL)) {
		return FALSE;
	}
	if ((strcmp(partinfo->section_number, HEADER_SECTION) == 0) ||
	    (strcmp(partinfo->section_number, MSG_BODY_SEC) == 0)) {
		return FALSE;
	}
	enc_filename = g_strdup_printf("%s%c%s",
				       filepath, SECTION_SEPARATOR, partinfo->section_number);

	if (is_file_entry_exist(enc_filename) == TRUE) {
		g_free(enc_filename);
		return TRUE;
	}
	g_free(enc_filename);
	return FALSE;
}

gboolean procmime_check_purgable_all(const gchar * filename, const gchar * message_filepath)
{
	FILE *fp = NULL;
	MimeInfo *mimeinfo = NULL;
	MimeInfo *tmpinfo = NULL;
	guchar buf[BUFSIZ];
	gchar *body_str = NULL;
	gboolean purgable_flag = FALSE;
	gint len = 0;

	memset(buf, '\0', BUFSIZ);

	fp = fopen(filename, "r");
	if (fp == NULL) {
		return FALSE;
	}
	while ((len = fread(buf, sizeof(gchar), sizeof(buf) - 1, fp)) > 0) {
		body_str = g_strconcat(buf, NULL);
	}
	mimeinfo = procmime_scan_body(body_str, strlen(body_str));
	g_free(body_str);
	fclose(fp);

	if (mimeinfo == NULL) {
		return FALSE;
	}

	tmpinfo = mimeinfo;

	while (mimeinfo != NULL) {
		if (procmime_check_purgable(mimeinfo, message_filepath) == TRUE) {
			purgable_flag = TRUE;
		}
		mimeinfo = procmime_mimeinfo_next(mimeinfo);
	}
	procmime_mimeinfo_free_all(tmpinfo);
	return purgable_flag;
}

gboolean procmime_check_extra_part(MimeInfo * mimeinfo)
{
	if (mimeinfo == NULL) {
		return FALSE;
	}
	/* this has extra parts */
	if (mimeinfo->next != NULL) {
		return TRUE;
	}
	if ((mimeinfo->mime_type != MIME_TEXT) &&
	    (mimeinfo->mime_type != MIME_TEXT_HTML) && (mimeinfo->mime_type != MIME_MULTIPART)) {
		/* this contains a part that can not be displayed as body */
		return TRUE;
	}
	return FALSE;
}

gboolean procmime_check_email_attach(MimeInfo * mimeinfo)
{
	MimeInfo *partinfo = NULL;

	partinfo = mimeinfo;

	while (partinfo != NULL) {
		if (g_strrstr(partinfo->content_type, MULTIPART_MESSAGE) != NULL) {
			return TRUE;
		}

		partinfo = procmime_mimeinfo_next(partinfo);
	}

	return FALSE;
}
void procmime_write_on_stylecheck(gchar * str,
				  gboolean * style_found_flag,
				  FILE * outfp, const gchar * default_font)
{
	gchar *ch_buf = NULL;

	if ((*style_found_flag == FALSE) && (default_font != NULL)) {

		ch_buf = procmime_check_head_tag(str, style_found_flag, default_font);

	}
	if (ch_buf == NULL) {
		fputs(str, outfp);

	} else {
		fputs(ch_buf, outfp);
		g_free(ch_buf);

	}
}
gint procmime_check_for_style_tag(const gchar * mhtml_file,
				  const gchar * dest_file, const gchar * default_font)
{
	FILE *infp = NULL;
	FILE *outfp = NULL;
	gchar *p = NULL;
	gchar buf[BUFFSIZE];


	memset(buf, sizeof(buf), '\0');
	infp = fopen(mhtml_file, "r");

	if (infp == NULL) {
		return PARSE_FAIL;
	}
	outfp = fopen(dest_file, "w");
	if (outfp == NULL) {
		send_engine_error_to_ui(errno);
		fclose(infp);
		return PARSE_FAIL;
	}
	while ((p = fgets(buf, sizeof(buf), infp)) != NULL) {
		fputs(buf, outfp);
	}
	fclose(infp);
	fclose(outfp);
	return PARSE_SUCCESS;
}

/**
  This function reads mimeinformation from file 
  @param mime_file_name name of file,which has mime contents 
  @return NULL on failure,or mimeinfo read from file.The memory allocated 
  for the mime information has to be freed by the calling function.
  This has to be freed by the caller 
 */
MimeInfo *procmime_read_mimeinfo(const gchar * mime_file_name)
{
	FILE *mime_fp;
	guchar buf[BUFSIZ];
	gchar *body_str = NULL;
	gchar *tmp_str = NULL;
	size_t len;
	MimeInfo *mimeinfo = NULL;

	memset(buf, '\0', BUFSIZ);
	mime_fp = fopen(mime_file_name, "r");
	if (mime_fp == NULL) {
		set_engine_errno(ENGINE_ERROR_NO_FILE_PERM);
		return NULL;
	}
	while ((len = fread(buf, sizeof(gchar), sizeof(buf) - 1, mime_fp)) > 0) {
		if (tmp_str != NULL) {
			body_str = g_strconcat(tmp_str, buf, NULL);
			g_free(tmp_str);
		} else {
			body_str = g_strconcat(buf, NULL);
		}
		if (body_str == NULL) {
			fclose(mime_fp);
			return NULL;
		}
		tmp_str = body_str;
	}

	mimeinfo = procmime_scan_body(body_str, strlen(body_str));
	g_free(body_str);
	fclose(mime_fp);
	return mimeinfo;
}
void procmime_check_and_set_filename(MimeInfo * mimeinfo, const gchar * decfilename)
{
	if (decfilename != NULL) {
		g_free(mimeinfo->filename);
		mimeinfo->filename = (gchar *)decfilename;

	} else if (mimeinfo->filename != NULL) {

		if (procmime_check_extra_part(mimeinfo) == TRUE) {
			g_free(mimeinfo->name);
			mimeinfo->name = mimeinfo->filename;
		} else {
			g_free(mimeinfo->filename);
		}
		mimeinfo->filename = NULL;
	}
}

/**
  This function is used to decode the mail body.
  @param msgid    message id of the message
  @param filename    name of the message file 
  @return mimeinfo the mimeinformation of the body,this has to be 
  freed by caller 
 */
MimeInfo *procmime_get_mimeinfo(const gchar * msgid, const gchar * filename,
			      gboolean edit_flag, gboolean get_html_flag)
{
	FILE *fp = NULL;
	MimeInfo *mimeinfo = NULL;
	MsgInfo *msginfo = NULL;
	MsgFlags flags;
	struct stat s;
	gint msg_num = 0;
	gint pos = -1;
	gint level = 0;
	glong fpos = 0;
	GSList *file_pos_list = NULL;
	gchar *decfilename = NULL;


	if (stat(filename, &s) < 0) {
		return NULL;
	}

	fp = fopen(filename, "r");

	if (fp == NULL) {
		return NULL;
	}
	
	mimeinfo = (MimeInfo *) procmime_scan_message(fp, pos, level, fpos, file_pos_list);
	fclose(fp);

	if (mimeinfo == NULL) {
		return NULL;
	}
	/* if this is IMAP message,add section numbers also */
	if (procmime_sync_mimeinfo(mimeinfo, filename) == FILE_PROCESS_FAIL) {
		return NULL;
	}

	flags.perm_flags = MSG_NEW | MSG_UNREAD;
	flags.tmp_flags = 0;

	msginfo = (MsgInfo *) procheader_parse_file(filename, flags, TRUE);
	if (msginfo == NULL) {
		procmime_mimeinfo_free_all(mimeinfo);
		return NULL;
	}

	if (S_ISREG(s.st_mode) == 0) {
		procmsg_msginfo_free(msginfo);
		procmime_mimeinfo_free_all(mimeinfo);
		return NULL;
	}

	msg_num = get_msgid_num(msgid);
	msginfo->header->msgnum = msg_num;
	msginfo->header->size = s.st_size;
	msginfo->header->mtime = s.st_mtime;

	/* Set message id & account information */
	g_free(msginfo->header->msgid);
	msginfo->header->msgid = g_strdup(msgid);
	decfilename = procmime_get_text_content(msginfo->header, edit_flag, get_html_flag);
	
	procmime_check_and_set_filename(mimeinfo,decfilename);

	procmsg_msginfo_free(msginfo);

	if (folder_message_del_list((gchar *) msgid, UNREAD) != FOLDER_SUCCESS) {
		procmime_mimeinfo_free_all(mimeinfo);
		return NULL;
	}

	return mimeinfo;
}

static gint procmime_sync_mimeinfo(MimeInfo * mimeinfo, const gchar * file_path)
{
	MimeInfo *hdr_mimeinfo = NULL;
	MimeInfo *tmp_mimeinfo = NULL;
	MimeInfo *cur_mimeinfo = NULL;
	gchar *mime_file_name = NULL;
	gint ret_val = FILE_PROCESS_SUCCESS;

	mime_file_name = g_strdup_printf("%s%c%s", file_path, SECTION_SEPARATOR, MIME_SUFFIX);
	/* anyway,the attachment can not be deleted */
	if (is_file_entry_exist(mime_file_name) == FALSE) {
		g_free(mime_file_name);
		return FILE_PROCESS_SUCCESS;
	}
	hdr_mimeinfo = procmime_read_mimeinfo(mime_file_name);
	if (hdr_mimeinfo == NULL) {
		g_free(mime_file_name);
		/* error is already sent through callback */
		return FILE_PROCESS_SUCCESS;
	}
	cur_mimeinfo = mimeinfo;
	tmp_mimeinfo = hdr_mimeinfo;
	while (cur_mimeinfo != NULL) {
		if (tmp_mimeinfo == NULL) {
			ret_val = FILE_PROCESS_FAIL;
			break;
		}
		cur_mimeinfo->section_number = tmp_mimeinfo->section_number;
		tmp_mimeinfo->section_number = NULL;
		cur_mimeinfo = procmime_mimeinfo_next(cur_mimeinfo);
		tmp_mimeinfo = procmime_mimeinfo_next(tmp_mimeinfo);
	}
	g_free(mime_file_name);
	procmime_mimeinfo_free_all(hdr_mimeinfo);
	return ret_val;
}


/**
  This function sets the header fields 
  @param header    pointer to the message header,for which header 
  fields have to be set. 
  @param account    pointer to the account information in global 
  account list, for which the attachment has to be downloaded.
  Global account list created and deleted by account management functions. 
  @return FILE_PROCESS_SUCCESS if successfully updated the message header.
  FILE_PROCESS_FAIL if file could not be found 
 */
gint procmime_folder_set_header(MsgHeader * header, PrefsAccount * account)
{
	gchar *message_file_name = NULL;
	gchar *body_file_name = NULL;
	gchar *mime_file_name = NULL;
	gboolean imap_flag = FALSE;
	MimeInfo *mimeinfo = NULL;

/*
    while moving / copying messages from Inbox to user folders complete message
is downloaded and moved/copied to the user folder. In Inbox only mimeheader file
s exist
*/
	mime_file_name = g_strdup_printf("%s%c%s%c%s%c%s",
					 get_mail_dir(),
					 G_DIR_SEPARATOR, INBOX_DIR,
					 G_DIR_SEPARATOR, header->msgid,
					 SECTION_SEPARATOR, MIME_SUFFIX);

	message_file_name = folder_get_message_file_path(header->msgid);

	if (is_file_exist(mime_file_name) == TRUE) {
		imap_flag = TRUE;
		if (procmime_check_attachments(mime_file_name, TRUE) == TRUE) {
			header->flags.perm_flags |= MSG_ATTACH;
		}
		if (procmime_check_purgable_all(mime_file_name, message_file_name) == TRUE) {
			MSG_SET_TMP_FLAGS(header->flags, MSG_PURGABLE);

		} else if ((is_file_entry_exist(message_file_name) == TRUE) &&
			   MSG_IS_ATTACH(header->flags)) {

			MSG_SET_TMP_FLAGS(header->flags, MSG_PURGABLE);
		}
	} else {
		if (message_file_name && (procmime_check_attachments(message_file_name, FALSE) == TRUE)) {
			header->flags.perm_flags |= MSG_ATTACH;
		}
	}
	if (message_file_name != NULL) {
		if (is_file_entry_exist(message_file_name) == TRUE) {
			header->msgtag.fullydownloaded = TRUE;
			header->msgtag.bodydownloaded = TRUE;
			header->smime_type = folder_get_smime_type(header, message_file_name);
		} else if (imap_flag == TRUE) {
			/* check if the body is downloaded */
			body_file_name = get_file_name(header->msgid, MSG_BODY_SEC);

			if (is_file_entry_exist(body_file_name)) {
				header->msgtag.bodydownloaded = TRUE;
				header->smime_type = folder_get_smime_type(header, body_file_name);
			}
			g_free(body_file_name);
			if ((header->flags.perm_flags & MSG_ATTACH) != 0) {
				mimeinfo = procmime_read_mimeinfo(mime_file_name);
				if (mimeinfo != NULL) {
					header->msgtag.fullydownloaded =
					    procmime_set_is_downloaded(mimeinfo, header->msgid);
					procmime_mimeinfo_free_all(mimeinfo);
				}
			} else if (header->msgtag.bodydownloaded == TRUE) {
				header->msgtag.fullydownloaded = TRUE;
			}
		} else {
			return FILE_PROCESS_FAIL;
		}
		g_free(message_file_name);
	}
	g_free(mime_file_name);
	return FILE_PROCESS_SUCCESS;
}

/**
  This function sets the is_downloaded flag for all parts of mimeinfo 
  @param mimeinfo    pointer to mimeinfo.This memory has to be allocated 
  and freed by the calling function.
  @param msgid    message id 
  @return TRUE if all parts are downloaded  
 */
gboolean procmime_set_is_downloaded(MimeInfo * mimeinfo, const gchar * msgid)
{
	MimeInfo *partinfo = NULL;
	gchar *file_path = NULL;
	gchar *enc_filename = NULL;
	gboolean fully_downloaded = TRUE;
	GSList *purge_list = NULL;

	if ((mimeinfo == NULL) || (msgid == NULL)) {
		return fully_downloaded;
	}

	file_path = folder_get_message_file_path(msgid);
	if (file_path == NULL) {
		return fully_downloaded;
	}
	if (is_file_entry_exist(file_path) == TRUE) {
		return fully_downloaded;
	}
	purge_list = procmime_read_purged(file_path);
	partinfo = mimeinfo;

	while (partinfo != NULL) {
		if (partinfo->section_number != NULL) {

			enc_filename = g_strdup_printf("%s%c%s",
						       file_path, SECTION_SEPARATOR,
						       partinfo->section_number);

			if (is_file_entry_exist(enc_filename) == FALSE) {
				if (partinfo->content_disposition != NULL) {
					fully_downloaded = FALSE;

				}
				partinfo->is_downloaded = FALSE;
				procmime_set_purged(partinfo, purge_list);
			}

			g_free(enc_filename);
		}
		partinfo = procmime_mimeinfo_next(partinfo);
	}
	g_free(file_path);
	slist_free_strings(purge_list);

	return fully_downloaded;
}
