
/*
 * LibSylph -- E-Mail client library
 * Copyright (C) 1999-2006 Hiroyuki Yamamoto
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <glib.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>

#include "intl.h"
#include "common.h"
#include "prefs_account.h"
#include "procheader.h"
#include "procmsg.h"
#include "codeconv.h"
#include "procmime.h"
#include "utils.h"
#include "defs.h"
#include "smime_security.h"
#define BUFFSIZE	8192

static Priority procheader_get_priority(const gchar * priority);

gint procheader_get_one_field(gchar * buf, gint len, FILE * fp, HeaderEntry hentry[])
{
	gint nexthead;
	gint hnum = 0;
	HeaderEntry *hp = NULL;

	if (hentry != NULL) {
		/* skip non-required headers */
		do {
			do {
				if (fgets(buf, len, fp) == NULL)
					return -1;
				if (buf[0] == '\r' || buf[0] == '\n')
					return -1;
			} while (buf[0] == ' ' || buf[0] == '\t');

			for (hp = hentry, hnum = 0; hp->name != NULL; hp++, hnum++) {
				if (!g_ascii_strncasecmp(hp->name, buf, strlen(hp->name)))
					break;
			}
		} while (hp->name == NULL);
	} else {
		if (fgets(buf, len, fp) == NULL)
			return -1;
		if (buf[0] == '\r' || buf[0] == '\n')
			return -1;
	}

	/* unfold the specified folded line */
	if (hp && hp->unfold) {
		gboolean folded = FALSE;
		gchar *bufp = buf + strlen(buf);

		for (; bufp > buf && (*(bufp - 1) == '\n' || *(bufp - 1) == '\r'); bufp--)
			*(bufp - 1) = '\0';

		while (1) {
			nexthead = fgetc(fp);

			/* folded */
			if (nexthead == ' ' || nexthead == '\t')
				folded = TRUE;
			else if (nexthead == EOF)
				break;
			else if (folded == TRUE) {
				if (nexthead == '\r' || nexthead == '\n') {
					folded = FALSE;
					continue;
				}

				if ((len - (bufp - buf)) <= 2)
					break;

				/* replace return code on the tail end
				   with space */
				*bufp++ = ' ';
				*bufp++ = nexthead;
				*bufp = '\0';
				/* concatenate next line */
				if (fgets(bufp, len - (bufp - buf), fp)
				    == NULL)
					break;
				bufp += strlen(bufp);

				for (; bufp > buf &&
				     (*(bufp - 1) == '\n' || *(bufp - 1) == '\r'); bufp--)
					*(bufp - 1) = '\0';

				folded = FALSE;
			} else {
				ungetc(nexthead, fp);
				break;
			}
		}

		return hnum;
	}

	while (1) {
		nexthead = fgetc(fp);
		if (nexthead == ' ' || nexthead == '\t') {
			size_t buflen = strlen(buf);

			/* concatenate next line */
			if ((len - buflen) > 2) {
				gchar *p = buf + buflen;

				*p++ = nexthead;
				*p = '\0';
				buflen++;
				if (fgets(p, len - buflen, fp) == NULL)
					break;
			} else
				break;
		} else {
			if (nexthead != EOF)
				ungetc(nexthead, fp);
			break;
		}
	}

	/* remove trailing return code */
	strretchomp(buf);

	return hnum;
}

gchar *procheader_get_unfolded_line(gchar * buf, gint len, FILE * fp)
{
	gboolean folded = FALSE;
	gint nexthead;
	gchar *bufp;

	if (fgets(buf, len, fp) == NULL)
		return NULL;
	if (buf[0] == '\r' || buf[0] == '\n')
		return NULL;
	bufp = buf + strlen(buf);

	for (; bufp > buf && (*(bufp - 1) == '\n' || *(bufp - 1) == '\r'); bufp--)
		*(bufp - 1) = '\0';

	while (1) {
		nexthead = fgetc(fp);

		/* folded */
		if (nexthead == ' ' || nexthead == '\t')
			folded = TRUE;
		else if (nexthead == EOF)
			break;
		else if (folded == TRUE) {
			if (nexthead == '\r' || nexthead == '\n') {
				folded = FALSE;
				continue;
			}

			if ((len - (bufp - buf)) <= 2)
				break;

			/* replace return code on the tail end
			   with space */
			*bufp++ = ' ';
			*bufp++ = nexthead;
			*bufp = '\0';

			/* concatenate next line */
			if (fgets(bufp, len - (bufp - buf), fp)
			    == NULL)
				break;
			bufp += strlen(bufp);

			for (; bufp > buf && (*(bufp - 1) == '\n' || *(bufp - 1) == '\r'); bufp--)
				*(bufp - 1) = '\0';

			folded = FALSE;
		} else {
			ungetc(nexthead, fp);
			break;
		}
	}

	/* remove trailing return code */
	strretchomp(buf);

	return buf;
}

GSList *procheader_get_header_list_from_file(const gchar * file)
{
	FILE *fp;
	GSList *hlist;

	if ((fp = fopen(file, "rb")) == NULL) {
		FILE_OP_ERROR(file, "fopen");
		return NULL;
	}

	hlist = procheader_get_header_list(fp);

	fclose(fp);
	return hlist;
}

GSList *procheader_get_header_list(FILE * fp)
{
	gchar buf[BUFFSIZE], tmp[BUFFSIZE];
	gchar *p;
	GSList *hlist = NULL;
	Header *header;

	g_return_val_if_fail(fp != NULL, NULL);

	while (procheader_get_unfolded_line(buf, sizeof(buf), fp) != NULL) {
		if (*buf == ':')
			continue;
		for (p = buf; *p && *p != ' '; p++) {
			if (*p == ':') {
				header = g_new(Header, 1);
				header->name = g_strndup(buf, p - buf);
				p++;
				while (*p == ' ' || *p == '\t')
					p++;
				conv_unmime_header(tmp, sizeof(tmp), p, NULL);
				header->body = g_strdup(tmp);

				hlist = g_slist_append(hlist, header);
				break;
			}
		}
	}

	return hlist;
}

GPtrArray *procheader_get_header_array(FILE * fp)
{
	gchar buf[BUFFSIZE], tmp[BUFFSIZE];
	gchar *p;
	GPtrArray *headers;
	Header *header;

	g_return_val_if_fail(fp != NULL, NULL);

	headers = g_ptr_array_new();

	while (procheader_get_unfolded_line(buf, sizeof(buf), fp) != NULL) {
		if (*buf == ':')
			continue;
		for (p = buf; *p && *p != ' '; p++) {
			if (*p == ':') {
				header = g_new(Header, 1);
				header->name = g_strndup(buf, p - buf);
				p++;
				while (*p == ' ' || *p == '\t')
					p++;
				conv_unmime_header(tmp, sizeof(tmp), p, NULL);
				header->body = g_strdup(tmp);

				g_ptr_array_add(headers, header);
				break;
			}
		}
	}

	return headers;
}

GPtrArray *procheader_get_header_array_asis(FILE * fp)
{
	gchar buf[BUFFSIZE], tmp[BUFFSIZE];
	gchar *p;
	GPtrArray *headers;
	Header *header;

	g_return_val_if_fail(fp != NULL, NULL);

	headers = g_ptr_array_new();

	while (procheader_get_one_field(buf, sizeof(buf), fp, NULL) != -1) {
		if (*buf == ':')
			continue;
		for (p = buf; *p && *p != ' '; p++) {
			if (*p == ':') {
				header = g_new(Header, 1);
				header->name = g_strndup(buf, p - buf);
				p++;
				conv_unmime_header(tmp, sizeof(tmp), p, NULL);
				header->body = g_strdup(tmp);

				g_ptr_array_add(headers, header);
				break;
			}
		}
	}

	return headers;
}

void procheader_header_list_destroy(GSList * hlist)
{
	Header *header;

	while (hlist != NULL) {
		header = hlist->data;
		procheader_header_free(header);
		hlist = g_slist_remove(hlist, header);
	}
}

void procheader_header_array_destroy(GPtrArray * harray)
{
	gint i;
	Header *header;

	for (i = 0; i < harray->len; i++) {
		header = g_ptr_array_index(harray, i);
		procheader_header_free(header);
	}

	g_ptr_array_free(harray, TRUE);
}

void procheader_header_free(Header * header)
{
	if (!header)
		return;

	g_free(header->name);
	g_free(header->body);
	g_free(header);
}

void procheader_get_header_fields(FILE * fp, HeaderEntry hentry[])
{
	gchar buf[BUFFSIZE];
	HeaderEntry *hp;
	gint hnum;
	gchar *p;

	if (hentry == NULL)
		return;

	while ((hnum = procheader_get_one_field(buf, sizeof(buf), fp, hentry))
	       != -1) {
		hp = hentry + hnum;

		p = buf + strlen(hp->name);
		while (*p == ' ' || *p == '\t')
			p++;

		if (hp->body == NULL)
			hp->body = g_strdup(p);
		else if (!g_ascii_strcasecmp(hp->name, "To:") ||
			 !g_ascii_strcasecmp(hp->name, "Cc:")) {
			gchar *tp = hp->body;
			hp->body = g_strconcat(tp, ", ", p, NULL);
			g_free(tp);
		}
	}
}

MsgInfo *procheader_parse_file(const gchar * file, MsgFlags flags, gboolean full)
{
	FILE *fp;
	MsgInfo *msginfo;

	if ((fp = fopen(file, "rb")) == NULL) {
		FILE_OP_ERROR(file, "fopen");
		return NULL;
	}

	msginfo = procheader_parse_stream(fp, flags, full);
	fclose(fp);
	return msginfo;
}

MsgInfo *procheader_parse_str(const gchar * str, MsgFlags flags, gboolean full)
{
	FILE *fp;
	MsgInfo *msginfo;
	gchar *outfile = NULL;

	if ((fp = str_open_as_stream(str, &outfile)) == NULL)
		return NULL;

	msginfo = procheader_parse_stream(fp, flags, full);
	fclose(fp);

	unlink(outfile);
	g_free(outfile);
	return msginfo;
}

enum {
	H_DATE = 0,
	H_FROM = 1,
	H_TO = 2,
	H_NEWSGROUPS = 3,
	H_SUBJECT = 4,
	H_MSG_ID = 5,
	H_REFERENCES = 6,
	H_IN_REPLY_TO = 7,
	H_CONTENT_TYPE = 8,
	H_SEEN = 9,
	H_CC = 10,
	H_X_FACE = 11,
	H_X_PRIORITY = 12,
	H_REPLY_TO = 13,
	H_BCC = 14,
	H_RESENT_FROM = 15,
	H_RESENT_TO = 16,
	H_RESENT_CC = 17,
	H_RESENT_BCC = 18,
	H_RESENT_DATE = 19,
	H_RESENT_MSG_ID = 20,
	H_CP_OWN_ADDR = 21,
	H_EDIT_DRAFT = 22,
	H_READ_REQ = 23,
	H_RX_TIME = 24,
	H_IMPORTANCE = 25
};
/* changed as per new MsgInfo*/
MsgInfo *procheader_parse_stream(FILE * fp, MsgFlags flags, gboolean full)
{
	static HeaderEntry hentry_full[] = { {"Date:", NULL, FALSE},
	{"From:", NULL, TRUE},
	{"To:", NULL, TRUE},
	{"Newsgroups:", NULL, TRUE},
	{"Subject:", NULL, TRUE},
	{"Message-Id:", NULL, FALSE},
	{"References:", NULL, FALSE},
	{"In-Reply-To:", NULL, FALSE},
	{"Content-Type:", NULL, FALSE},
	{"Seen:", NULL, FALSE},
	{"Cc:", NULL, TRUE},
	{"X-Face:", NULL, FALSE},
	{"X-Priority:", NULL, FALSE},
	{"Reply-To:", NULL, FALSE},
	{"Bcc:", NULL, TRUE},
	{"Resent-From:", NULL, TRUE},
	{"Resent-To:", NULL, TRUE},
	{"Resent-Cc:", NULL, TRUE},
	{"Resent-Bcc:", NULL, TRUE},
	{"Resent-Date:", NULL, TRUE},
	{"Resent-Message-Id:", NULL, TRUE},
	{"X-Copy-to-own:", NULL, FALSE},
	{"X-draft-edit:", NULL, FALSE},
	{"Return-Receipt-To:", NULL, FALSE},
	{"Received:", NULL, TRUE},
	{"Importance:", NULL, FALSE},
	{NULL, NULL, FALSE}
	};

	static HeaderEntry hentry_short[] = { {"Date:", NULL, FALSE},
	{"From:", NULL, TRUE},
	{"To:", NULL, TRUE},
	{"Newsgroups:", NULL, TRUE},
	{"Subject:", NULL, TRUE},
	{"Message-Id:", NULL, FALSE},
	{"References:", NULL, FALSE},
	{"In-Reply-To:", NULL, FALSE},
	{"Content-Type:", NULL, FALSE},
	{"Seen:", NULL, FALSE},
	{NULL, NULL, FALSE}
	};

	MsgInfo *msginfo;
	gchar buf[BUFFSIZE], tmp[BUFFSIZE];
	gchar *reference = NULL;
	gchar *p = NULL;
	gchar *hp = NULL;
	gchar *date_ptr = NULL;
	HeaderEntry *hentry = NULL;
	gint hnum = 0;
	gint resentFound = 0;
	gboolean rx_date_parse = FALSE;
	gboolean priority_read = FALSE;
	gchar *str = NULL;
	gchar *cnttype = NULL;

	hentry = full ? hentry_full : hentry_short;

	if (MSG_IS_QUEUED(flags)) {
		while (fgets(buf, sizeof(buf), fp) != NULL)
			if (buf[0] == '\r' || buf[0] == '\n')
				break;
	}

	msginfo = g_new0(MsgInfo, 1);
	if (msginfo == NULL) {
		return NULL;
	}
	msginfo->header = g_new0(MsgHeader, 1);
	if (msginfo->header == NULL) {
		g_free(msginfo);
		return NULL;
	}

	msginfo->header->flags = flags;
	msginfo->header->inreplyto = NULL;
	msginfo->header->read_receipt_request = FALSE;
	msginfo->header->copy_to_own_address = FALSE;

	while ((hnum = procheader_get_one_field(buf, sizeof(buf), fp, hentry))
	       != -1) {
		hp = buf + strlen(hentry[hnum].name);
		while (*hp == ' ' || *hp == '\t')
			hp++;
		switch (hnum) {
		case H_RESENT_FROM:
			resentFound = TRUE;
			if (msginfo->header->from != NULL) {
				break;
			}
			if (msginfo->header->from) {
				g_free(msginfo->header->from);
			}
			conv_unmime_header(tmp, sizeof(tmp), hp, NULL);
			msginfo->header->from = g_strdup(tmp);
			msginfo->header->fromname = procheader_get_fromname(tmp);
			break;
		case H_RESENT_TO:
			resentFound = TRUE;
			if (msginfo->header->to != NULL) {
				break;
			}
			conv_unmime_header(tmp, sizeof(tmp), hp, NULL);
			if (msginfo->header->to) {
				p = msginfo->header->to;
				msginfo->header->to = g_strconcat(p, ", ", tmp, NULL);
				g_free(p);
			} else {
				msginfo->header->to = g_strdup(tmp);
			}
			break;
		case H_RESENT_CC:
			resentFound = TRUE;
			if (msginfo->header->cc != NULL) {
				break;
			}
			conv_unmime_header(tmp, sizeof(tmp), hp, NULL);
			if (msginfo->header->cc) {
				p = msginfo->header->cc;
				msginfo->header->cc = g_strconcat(p, ", ", tmp, NULL);
				g_free(p);
			} else {
				msginfo->header->cc = g_strdup(tmp);
			}
			break;
		case H_RESENT_BCC:
			resentFound = TRUE;
			if (msginfo->header->bcc != NULL) {
				break;
			}
			conv_unmime_header(tmp, sizeof(tmp), hp, NULL);
			if (msginfo->header->bcc) {
				p = msginfo->header->bcc;
				msginfo->header->bcc = g_strconcat(p, ", ", tmp, NULL);
				g_free(p);
			} else {
				msginfo->header->bcc = g_strdup(tmp);
			}
			break;
		case H_RESENT_DATE:
			if ((msginfo->header->date != NULL)) {
				break;
			}
			resentFound = TRUE;
			if (msginfo->header->date) {
				g_free(msginfo->header->date);
			}
			msginfo->header->date_t = procheader_date_parse(NULL, hp, 0);
			msginfo->header->date = g_strdup(hp);
			break;
		case H_RESENT_MSG_ID:
			resentFound = TRUE;
			if (msginfo->header->msgid) {
				g_free(msginfo->header->msgid);
			}
			extract_parenthesis(hp, '<', '>');
			remove_space(hp);
			msginfo->header->msgid = g_strdup(hp);
			break;
		case H_DATE:
			if ((resentFound == TRUE)
			    || (msginfo->header->date != NULL)) {
				break;
			}
			msginfo->header->date_t = procheader_date_parse(NULL, hp, 0);
			msginfo->header->date = g_strdup(hp);
			break;
		case H_FROM:
			if ((resentFound == TRUE)
			    || (msginfo->header->from != NULL)) {
				break;
			}
			conv_unmime_header(tmp, sizeof(tmp), hp, NULL);
			msginfo->header->from = g_strdup(tmp);
			msginfo->header->fromname = procheader_get_fromname(tmp);
			break;
		case H_TO:
			if ((resentFound == TRUE) || (msginfo->header->to != NULL)) {
				break;
			}
			conv_unmime_header(tmp, sizeof(tmp), hp, NULL);
			if (msginfo->header->to) {
				p = msginfo->header->to;
				msginfo->header->to = g_strconcat(p, ", ", tmp, NULL);
				g_free(p);
			} else {
				msginfo->header->to = g_strdup(tmp);
			}
			break;
		case H_NEWSGROUPS:
			/* news groups are in the scope
			   if (msginfo->newsgroups) {
			   p = msginfo->newsgroups;
			   msginfo->newsgroups =
			   g_strconcat(p, ",", hp, NULL);
			   g_free(p);
			   } else
			   msginfo->newsgroups = g_strdup(buf + 12);
			 */
			break;
		case H_SUBJECT:
			if (msginfo->header->subject)
				break;
			conv_unmime_header(tmp, sizeof(tmp), hp, NULL);
			msginfo->header->subject = g_strdup(tmp);
			break;
		case H_MSG_ID:
			if ((resentFound == TRUE)) {
				break;
			}
			extract_parenthesis(hp, '<', '>');
			remove_space(hp);
			msginfo->header->msgid = g_strdup(hp);
			break;
		case H_REFERENCES:
		case H_IN_REPLY_TO:
			if (!reference) {
				eliminate_parenthesis(hp, '(', ')');
				if ((p = strrchr(hp, '<')) != NULL && strchr(p + 1, '>') != NULL) {
					extract_parenthesis(p, '<', '>');
					remove_space(p);
					if (*p != '\0')
						reference = g_strdup(p);
				}
			}
			break;
		case H_CONTENT_TYPE:
			if (!g_ascii_strncasecmp(hp, SIGNED, 16) && (strstr(hp, ENCR) != NULL)) {
				msginfo->header->smime_type = SMIME_SIGN;
			} else if (g_strrstr(hp, ENCR) != NULL) {
				msginfo->header->smime_type = SMIME_SIGN_ENCRYPT;
			} else {
				Xstrdup_a(str, hp, break);
				if ((cnttype = strchr(str, ';')))
					*cnttype = '\0';
				cnttype = g_strstrip(str);
				if (!g_ascii_strcasecmp(cnttype, MULTIPART_MIXED)) {
					MSG_SET_PERM_FLAGS(msginfo->header->flags, MSG_ATTACH);
				} else if (!g_ascii_strcasecmp(cnttype, MULTIPART_REPORT)) {
					MSG_SET_PERM_FLAGS(msginfo->header->flags, MSG_ATTACH);
				} else
				    if ((g_ascii_strncasecmp
					 (cnttype, MULTIPART_TYPE, strlen(MULTIPART_TYPE)) != 0)
					&&
					(g_ascii_strncasecmp(cnttype, TEXT_TYPE, strlen(TEXT_TYPE))
					 != 0)) {
					MSG_SET_PERM_FLAGS(msginfo->header->flags, MSG_ATTACH);

				}
			}
			break;
		case H_SEEN:
			/* mnews Seen header */
			MSG_UNSET_PERM_FLAGS(msginfo->header->flags, MSG_NEW | MSG_UNREAD);
			break;
		case H_CC:
			if ((resentFound == TRUE) || (msginfo->header->cc != NULL)) {
				break;
			}
			conv_unmime_header(tmp, sizeof(tmp), hp, NULL);
			if (msginfo->header->cc) {
				p = msginfo->header->cc;
				msginfo->header->cc = g_strconcat(p, ", ", tmp, NULL);
				g_free(p);
			} else
				msginfo->header->cc = g_strdup(tmp);
			break;
		case H_X_FACE:
			/*      if (msginfo->xface) break;
			   msginfo->xface = g_strdup(hp);
			   break;
			 */
		case H_X_PRIORITY:
			priority_read = TRUE;
			msginfo->header->recv_priority = atoi(hp);
			break;
		case H_IMPORTANCE:
			if (priority_read == FALSE) {
				priority_read = TRUE;
				msginfo->header->recv_priority = procheader_get_priority(hp);
			}
			break;
		case H_REPLY_TO:
			conv_unmime_header(tmp, sizeof(tmp), hp, NULL);
			if (tmp != NULL) {
				msginfo->header->replyto = g_strdup(tmp);
			}
			break;
		case H_BCC:
			if ((resentFound == TRUE)) {
				break;
			}
			conv_unmime_header(tmp, sizeof(tmp), hp, NULL);
			if (msginfo->header->bcc) {
				p = msginfo->header->bcc;
				msginfo->header->bcc = g_strconcat(p, ", ", tmp, NULL);
				g_free(p);
			} else
				msginfo->header->bcc = g_strdup(tmp);
			break;
		case H_CP_OWN_ADDR:
			msginfo->header->copy_to_own_address = TRUE;
			break;
		case H_EDIT_DRAFT:
			MSG_SET_PERM_FLAGS(msginfo->header->flags, MSG_EDITABLE);
			break;
		case H_READ_REQ:
			msginfo->header->read_receipt_request = TRUE;
			break;
		case H_RX_TIME:
			date_ptr = strchr(hp, DATE_START);
			if (rx_date_parse == TRUE) {
				break;
			}
			rx_date_parse = TRUE;
			if (date_ptr != NULL) {
				date_ptr++;
				msginfo->header->rx_date = procheader_date_parse(NULL, date_ptr, 0);
			}
			break;
		default:
			break;
		}
	}
	if (priority_read == FALSE) {
		msginfo->header->recv_priority = NORMAL;
	}
	msginfo->header->inreplyto = reference;
	msginfo->header->msgtag.remote_copy = TRUE;
	msginfo->header->msgtag.fullydownloaded = FALSE;
	msginfo->header->msgtag.bodydownloaded = FALSE;

	return msginfo;
}

gchar *procheader_get_fromname(const gchar * str)
{
	gchar *tmp, *name;

	Xstrdup_a(tmp, str, return NULL);

	if (*tmp == '\"') {
		extract_quote(tmp, '\"');
		g_strstrip(tmp);
	} else if (strchr(tmp, '<')) {
		eliminate_parenthesis(tmp, '<', '>');
		g_strstrip(tmp);
		if (*tmp == '\0') {
			strcpy(tmp, str);
			extract_parenthesis(tmp, '<', '>');
			g_strstrip(tmp);
		}
	} else if (strchr(tmp, '(')) {
		extract_parenthesis(tmp, '(', ')');
		g_strstrip(tmp);
	}

	if (*tmp == '\0')
		name = g_strdup(str);
	else
		name = g_strdup(tmp);

	return name;
}

static gint
procheader_scan_date_string(const gchar * str,
			    gchar * weekday, gint * day,
			    gchar * month, gint * year,
			    gint * hh, gint * mm, gint * ss, gchar * zone)
{
	gint result;

	result = sscanf(str, "%10s %d %9s %d %2d:%2d:%2d %5s",
			weekday, day, month, year, hh, mm, ss, zone);
	if (result == 8)
		return 0;

	result = sscanf(str, "%3s,%d %9s %d %2d:%2d:%2d %5s",
			weekday, day, month, year, hh, mm, ss, zone);
	if (result == 8)
		return 0;

	result = sscanf(str, "%d %9s %d %2d:%2d:%2d %5s", day, month, year, hh, mm, ss, zone);
	if (result == 7)
		return 0;

	*zone = '\0';
	result = sscanf(str, "%10s %d %9s %d %2d:%2d:%2d", weekday, day, month, year, hh, mm, ss);
	if (result == 7)
		return 0;

	*ss = 0;
	result = sscanf(str, "%10s %d %9s %d %2d:%2d %5s", weekday, day, month, year, hh, mm, zone);
	if (result == 7)
		return 0;

	result = sscanf(str, "%d %9s %d %2d:%2d %5s", day, month, year, hh, mm, zone);
	if (result == 6)
		return 0;

	return -1;
}

time_t procheader_date_parse(gchar * dest, const gchar * src, gint len)
{
	static gchar monthstr[] = "JanFebMarAprMayJunJulAugSepOctNovDec";
	gchar weekday[11];
	gint day;
	gchar month[10];
	gint year;
	gint hh, mm, ss;
	gchar zone[6];
	GDateMonth dmonth = G_DATE_BAD_MONTH;
	struct tm t;
	gchar *p;
	time_t timer;
	time_t tz_offset;

	if (procheader_scan_date_string(src, weekday, &day, month, &year, &hh, &mm, &ss, zone) < 0) {
		g_warning("Invalid date: %s\n", src);
		if (dest && len > 0)
			strncpy2(dest, src, len);
		return 0;
	}

	/* Y2K compliant :) */
	if (year < 1000) {
		if (year < 50)
			year += 2000;
		else
			year += 1900;
	}

	month[3] = '\0';
	for (p = monthstr; *p != '\0'; p += 3) {
		if (!g_ascii_strncasecmp(p, month, 3)) {
			dmonth = (gint) (p - monthstr) / 3 + 1;
			break;
		}
	}
	if (*p == '\0')
		g_warning("Invalid month: %s\n", month);

	t.tm_sec = ss;
	t.tm_min = mm;
	t.tm_hour = hh;
	t.tm_mday = day;
	t.tm_mon = dmonth - 1;
	t.tm_year = year - 1900;
	t.tm_wday = 0;
	t.tm_yday = 0;
	t.tm_isdst = -1;

	timer = mktime(&t);
	tz_offset = remote_tzoffset_sec(zone);
	if (tz_offset != -1)
		timer += tzoffset_sec(&timer) - tz_offset;

	if (dest)
		procheader_date_get_localtime(dest, len, timer);

	return timer;
}

void procheader_date_get_localtime(gchar * dest, gint len, const time_t timer)
{
	struct tm *lt;
	gchar *default_format = "%y/%m/%d(%a) %H:%M";

	lt = localtime(&timer);

	strftime(dest, len, default_format, lt);
}

/**
 This function copies only MIME headers to the output file
 @param pointer to the message file
 @param pointer to the output file
 @return whether the contents are written successfully or not
**/
gint procheader_skip_headers(FILE * fp, FILE * outfp)
{
	enum {
		HL_CONTENT_TRANSFER_ENCODING = 0,
		HL_CONTENT_TYPE = 1,
		HL_CONTENT_DISPOSITION = 2
	};
	HeaderEntry hentry[] = { {"Content-Transfer-Encoding:",
				  NULL, FALSE},
	{"Content-Type:", NULL, FALSE},
	{"Content-Disposition:", NULL, FALSE},
	{NULL, NULL, FALSE}
	};

	gchar buf[BUFFSIZE];
	gint hnum = 0;
	HeaderEntry *hp = NULL;
	gint nexthead;

	while ((hnum = procheader_get_one_field(buf, sizeof(buf), fp, hentry))
	       != -1) {
		hp = hentry + hnum;

		switch (hnum) {
		case HL_CONTENT_TRANSFER_ENCODING:
			fprintf(outfp, "%s\n", buf);
			break;
		case HL_CONTENT_TYPE:
		case HL_CONTENT_DISPOSITION:
			fprintf(outfp, "%s\n", buf);
			while (1) {

				nexthead = fgetc(fp);
				if (nexthead == '\r' || nexthead == '\n') {
					continue;
				} else if (nexthead == EOF) {
					break;
				}
				if (nexthead == ' ' || nexthead == '\t') {

					if (fgets(buf, sizeof(buf), fp) == NULL) {
						return PROC_SUCCESS;
					} else {
						fprintf(outfp, "%s\n", buf);
					}
				} else {
					ungetc(nexthead, fp);
					break;
				}
			}
			break;
		default:
			return PROC_FAIL;
		}
	}
	if (buf[0] == '\r' || buf[0] == '\n') {
		fprintf(outfp, "%s", buf);

	}
	return PROC_SUCCESS;

}

static Priority procheader_get_priority(const gchar * priority)
{
	if (strncasecmp(priority, NORMAL_PRIORITY, strlen(NORMAL_PRIORITY)) == 0) {
		return NORMAL;
	}
	if (strncasecmp(priority, HIGH_PRIORITY, strlen(HIGH_PRIORITY)) == 0) {
		return HIGH;
	}
	if (strncasecmp(priority, LOW_PRIORITY, strlen(LOW_PRIORITY)) == 0) {
		return LOW;
	}
	return NORMAL;

}

gint procheader_copy_headers(FILE * fp, FILE * outfp)
{
	enum {
		HL_CONTENT_TRANSFER_ENCODING = 0,
		HL_CONTENT_TYPE = 1,
		HL_CONTENT_DISPOSITION = 2
	};
	HeaderEntry hentry[] = { {"Content-Transfer-Encoding:",
				  NULL, FALSE},
	{"Content-Type:", NULL, FALSE},
	{"Content-Disposition:", NULL, FALSE},
	{NULL, NULL, FALSE}
	};

	gchar buf[BUFFSIZE];
	HeaderEntry *hp = NULL;
	gboolean skip_header = FALSE;


	if ((fp == NULL) || (outfp == NULL)) {
		return PROC_FAIL;
	}
	while (fgets(buf, sizeof(buf), fp) != NULL) {
		if ((skip_header == TRUE) && (buf[0] == ' ' || buf[0] == '\t')) {
			continue;
		}
		skip_header = FALSE;
		for (hp = hentry; hp->name != NULL; hp++) {
			if (!g_ascii_strncasecmp(hp->name, buf, strlen(hp->name))) {
				skip_header = TRUE;
			}
		}

		if (buf[0] == '\r' || buf[0] == '\n') {
			break;
		}
		if (skip_header == FALSE) {
			fputs(buf, outfp);

		}
	}

	return PROC_SUCCESS;
}
