
/*
 * LibSylph -- E-Mail client library
 * Copyright (C) 1999-2006 Hiroyuki Yamamoto
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include "defs.h"

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#include "intl.h"
#include "prefs.h"
#include "prefs_account.h"
#include "utils.h"

static PrefsAccount tmp_ac_prefs;

/* Update the following structure with receive,advanced,
 * email scheduling settings 
 * as and when you are adding/using some more fields */

static PrefParam param[] = {
	/* User Settings */
	{"account_name", &tmp_ac_prefs.user_settings.account_name, P_STRING, 64},

	{"is_default", &tmp_ac_prefs.user_settings.is_default, P_BOOL, 0},

	{"userid", &tmp_ac_prefs.user_settings.userid, P_STRING, 64},

	{"passwd", &tmp_ac_prefs.user_settings.passwd, P_STRING, 256},

	{"remember_passwd", &tmp_ac_prefs.user_settings.remember_passwd, P_BOOL, 0},

	{"email", &tmp_ac_prefs.user_settings.email, P_STRING, 64},

	{"alias_name", &tmp_ac_prefs.user_settings.alias_name, P_STRING, 64},

	/* Receive Settings */

	{"protocol", &tmp_ac_prefs.receive_settings.protocol, P_ENUM, 0},

	{"recv_server", &tmp_ac_prefs.receive_settings.recv_server, P_STRING, 64},

	{"port", &tmp_ac_prefs.receive_settings.port, P_USHORT, 0},

	{"passwd_auth_type", &tmp_ac_prefs.receive_settings.passwd_auth_type,
	 P_ENUM, 0},

	{"recv_type", &tmp_ac_prefs.receive_settings.recv_type, P_ENUM, 0},

	{"enable_size_limit", &tmp_ac_prefs.receive_settings.enable_size_limit,
	 P_BOOL, 0},

	{"size_limit", &tmp_ac_prefs.receive_settings.size_limit, P_INT, 0},

	{"recent_mails", &tmp_ac_prefs.receive_settings.recent_mails, P_INT, 0},

	{"last_uid_validity", &tmp_ac_prefs.receive_settings.last_uid_validity,
	 P_INT, 0},

	{"first_msgid", &tmp_ac_prefs.receive_settings.first_msgid, P_INT, 0},

	{"last_msgid", &tmp_ac_prefs.receive_settings.last_msgid, P_INT, 0},

	{"leave_msg", &tmp_ac_prefs.receive_settings.leave_msg, P_BOOL, 0},

	{"is_not_editable", &tmp_ac_prefs.receive_settings.is_not_editable,
	 P_BOOL, 0},
	{"num_of_msgs", &tmp_ac_prefs.receive_settings.num_of_msgs,
	 P_INT, 0},

	/* Send Settings */

	{"smtp_server", &tmp_ac_prefs.send_settings.smtp_server, P_STRING, 64},

	{"smtp_auth_type", &tmp_ac_prefs.send_settings.smtp_auth_type, P_ENUM, 0},

	{"smtp_port", &tmp_ac_prefs.send_settings.smtp_port, P_USHORT, 0},

	{"smtp_username", &tmp_ac_prefs.send_settings.username, P_STRING, 64},

	{"smtp_password", &tmp_ac_prefs.send_settings.password, P_STRING, 256},

/* Advanced Settings */

	{"email_type", &tmp_ac_prefs.advanced_settings.email_type, P_ENUM, 0},

	{"email_sending", &tmp_ac_prefs.advanced_settings.email_sending, P_ENUM, 0},

	{"include_signature", &tmp_ac_prefs.advanced_settings.include_signature,
	 P_BOOL, 0},

	{"signature_file", &tmp_ac_prefs.advanced_settings.signature_file,
	 P_STRING, 256},

	{"signature_flag", &tmp_ac_prefs.advanced_settings.signature_flag,
	 P_BOOL, 0},

	{"request_read_report",
	 &tmp_ac_prefs.advanced_settings.request_read_report,
	 P_BOOL, 0},

	{"copy_to_own_address",
	 &tmp_ac_prefs.advanced_settings.copy_to_own_address,
	 P_BOOL, 0},

	{"original_in_reply",
	 &tmp_ac_prefs.advanced_settings.org_in_reply,
	 P_BOOL, 0},

	{"use_ssl", &tmp_ac_prefs.advanced_settings.use_ssl, P_BOOL, 0},

	{"use_tls", &tmp_ac_prefs.advanced_settings.use_tls, P_BOOL, 0},

	{"smtp_use_ssl", &tmp_ac_prefs.advanced_settings.smtp_use_ssl, P_BOOL, 0},

	{"smtp_use_tls", &tmp_ac_prefs.advanced_settings.smtp_use_tls, P_BOOL, 0},

	{"imap_dir", &tmp_ac_prefs.advanced_settings.imap_dir, P_STRING, 25},
#ifdef CL_USE
	{"iap_name", &tmp_ac_prefs.advanced_settings.iap_name, P_STRING, 256},
	{"use_iap_smtp_serv", &tmp_ac_prefs.advanced_settings.use_iap_smtp_serv, P_BOOL, 0},
#endif
	{"smime_security", &tmp_ac_prefs.advanced_settings.smime_security,
	 P_INT, 0},
	{"display_sign_encrypt_mes", &tmp_ac_prefs.advanced_settings.display_sign_encrypt, P_BOOL,
	 0},
	{"display_trust", &tmp_ac_prefs.advanced_settings.display_trust,
	 P_BOOL, 0},
	{"display_validity", &tmp_ac_prefs.advanced_settings.display_validity,
	 P_BOOL, 0},

/* Schechule Settings */
	{NULL, NULL, P_OTHER, 0}
};

PrefsAccount *prefs_account_new(void)
{
	PrefsAccount *ac_prefs = NULL;

	ac_prefs = g_new0(PrefsAccount, 1);
	if (ac_prefs == NULL) {
		return NULL;
	}
	memset(&tmp_ac_prefs, 0, sizeof(PrefsAccount));
	*ac_prefs = tmp_ac_prefs;
	return ac_prefs;
}

gboolean prefs_account_read_config(PrefsAccount * ac_prefs, const gchar * label)
{
	gboolean retval;
	g_return_val_if_fail(ac_prefs != NULL, FALSE);
	g_return_val_if_fail(label != NULL, FALSE);

	memset(&tmp_ac_prefs, 0, sizeof(PrefsAccount));
	retval = prefs_read_config(param, label, ACCOUNT_RC);
	if (retval == FALSE)
		return FALSE;
	*ac_prefs = tmp_ac_prefs;
	return TRUE;
}


gboolean prefs_account_save_config_all(GSList * account_list)
{
	GSList *cur = NULL;
	gchar *rcpath = NULL;
	PrefFile *pfile = NULL;
	gint index = 1;

	rcpath = g_strconcat(get_rc_dir(), G_DIR_SEPARATOR_S, ACCOUNT_RC, NULL);
	if (rcpath == NULL) {
		return FALSE;
	}
	if ((pfile = (PrefFile *) prefs_file_open(rcpath)) == NULL) {
		g_free(rcpath);
		return FALSE;
	}
	g_free(rcpath);


	for (cur = account_list; cur != NULL; cur = cur->next) {
		tmp_ac_prefs = *((PrefsAccount *) cur->data);
		if (fprintf(pfile->fp, "[Account: %d]\n", index) <= 0 ||
		    prefs_file_write_param(pfile, param) < 0) {
			prefs_file_close_revert(pfile);
			return FALSE;
		}
		if (cur->next) {
			if (fputc('\n', pfile->fp) == EOF) {
				FILE_OP_ERROR(rcpath, "fputc");
				prefs_file_close_revert(pfile);
				return FALSE;
			}
		}

		index++;
	}

	if (prefs_file_close(pfile) < 0) {
		g_warning(("failed to write configuration to file\n"));
		return FALSE;
	}

	return TRUE;
}

void prefs_account_free(PrefsAccount * ac_prefs)
{
	if (!ac_prefs) {
		return;
	}
	tmp_ac_prefs = *ac_prefs;
	prefs_free(param);
	g_free(ac_prefs);
	ac_prefs = NULL;
}
