
/*
 * LibSylph -- E-Mail client library
 * Copyright (C) 1999-2006 Hiroyuki Yamamoto
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>

#include "intl.h"
#include "prefs.h"
#include "utils.h"
#include "defs.h"


typedef enum {
	DUMMY_PARAM
} DummyEnum;


static void prefs_config_parse_one_line(PrefParam * param, const gchar * buf);

/* Needed */

gboolean prefs_read_config(PrefParam * param, const gchar * label, const gchar * rcfile)
{
	FILE *fp;
	gchar buf[PREFSBUFSIZE];
	gchar *rcpath;
	gchar *block_label;

	g_return_val_if_fail(param != NULL, FALSE);
	g_return_val_if_fail(label != NULL, FALSE);
	g_return_val_if_fail(rcfile != NULL, FALSE);

	debug_print(("Reading configuration...\n"));
	rcpath =
	    g_strdup_printf("%s%s%c%s", get_home_dir(), REST_CONFIG_FILE_PATH, G_DIR_SEPARATOR,
			    ACCOUNT_RC);

	if (rcpath == NULL) {
		return FALSE;
	}
	if (is_file_entry_exist(rcpath) == FALSE) {
		g_free(rcpath);
		rcpath =
		    g_strdup_printf("%s%s%c%s", get_home_dir(), CONFIG_FILE_PATH, G_DIR_SEPARATOR,
				    ACCOUNT_RC);
	}
	if ((fp = fopen(rcpath, "rb")) == NULL) {
		if (ENOENT != errno)
			FILE_OP_ERROR(rcpath, "fopen");
		g_free(rcpath);
		return FALSE;
	}
	g_free(rcpath);

	block_label = g_strdup_printf("[%s]", label);

	/* search aiming block */
	gint val = -1;
	while (fgets(buf, sizeof(buf), fp) != NULL) {


		val = strncmp(buf, block_label, strlen(block_label));
		if (val == 0) {
			debug_print(("Found %s\n"), block_label);
			break;
		}
	}
	g_free(block_label);

	if (val == -1)
		return FALSE;

	while (fgets(buf, sizeof(buf), fp) != NULL) {
		strretchomp(buf);
		/* reached next block */
		if (buf[0] == '[')
			break;

		prefs_config_parse_one_line(param, buf);
	}
#if 0
	debug_print(("Finished reading configuration.\n"));
#endif
	fclose(fp);
	return TRUE;
}

/* Needed */
static void prefs_config_parse_one_line(PrefParam * param, const gchar * buf)
{
	gint i;
	gint name_len;
	const gchar *value;

	for (i = 0; param[i].name != NULL; i++) {
		name_len = strlen(param[i].name);
		if (g_ascii_strncasecmp(buf, param[i].name, name_len))
			continue;
		if (buf[name_len] != '=')
			continue;
		value = buf + name_len + 1;
		/* debug_print("%s = %s\n", param[i].name, value); */

		switch (param[i].type) {
		case P_STRING:
			g_free(*((gchar **) param[i].data));
			*((gchar **) param[i].data) =
			    *value ? g_strndup(value, param[i].len) : NULL;
			break;
		case P_INT:
			*((gint *) param[i].data) = (gint) atoi(value);
			break;
		case P_BOOL:
			*((gboolean *) param[i].data) = (*value == '0' || *value == '\0')
			    ? FALSE : TRUE;
			break;
		case P_ENUM:
			*((DummyEnum *) param[i].data) = (DummyEnum) atoi(value);
			break;
		case P_USHORT:
			*((gushort *) param[i].data) = (gushort) atoi(value);
			break;
		default:
			break;
		}
	}
}



gint prefs_file_write_param(PrefFile * pfile, PrefParam * param)
{
	gint i;
	gchar buf[PREFSBUFSIZE];

	for (i = 0; param[i].name != NULL; i++) {
		switch (param[i].type) {
		case P_STRING:
			g_snprintf(buf, sizeof(buf), "%s=%s\n", param[i].name,
				   *((gchar **) param[i].data) ? *((gchar **) param[i].data) : "");
			break;
		case P_INT:
			g_snprintf(buf, sizeof(buf), "%s=%d\n", param[i].name,
				   *((gint *) param[i].data));
			break;
		case P_BOOL:
			g_snprintf(buf, sizeof(buf), "%s=%d\n", param[i].name,
				   *((gboolean *) param[i].data));
			break;
		case P_ENUM:
			g_snprintf(buf, sizeof(buf), "%s=%d\n", param[i].name,
				   *((DummyEnum *) param[i].data));
			break;
		case P_USHORT:
			g_snprintf(buf, sizeof(buf), "%s=%d\n", param[i].name,
				   *((gushort *) param[i].data));
			break;
		default:
			buf[0] = '\0';
		}

		if (buf[0] != '\0') {
			if (fputs(buf, pfile->fp) == EOF) {
				perror("fputs");
				return -1;
			}
		}
	}

	return 0;
}

/* Needed */
PrefFile *prefs_file_open(const gchar * path)
{
	PrefFile *pfile;
	gchar *tmppath;
	FILE *fp;

	g_return_val_if_fail(path != NULL, NULL);

	tmppath = g_strconcat(path, ".tmp", NULL);
	if (tmppath == NULL) {
		return NULL;
	}
	if ((fp = fopen(tmppath, "wb")) == NULL) {
		FILE_OP_ERROR(tmppath, "fopen");
		send_engine_error_to_ui(errno);
		g_free(tmppath);
		return NULL;
	}

	if (change_file_mode_rw(fp, tmppath) < 0)
		FILE_OP_ERROR(tmppath, "chmod");

	g_free(tmppath);

	pfile = g_new(PrefFile, 1);
	if (pfile != NULL) {
		pfile->fp = fp;
		pfile->path = g_strdup(path);
	}
	return pfile;
}

/* Needed */
gint prefs_file_close(PrefFile * pfile)
{
	FILE *fp;
	gchar *path;
	gchar *tmppath;

	g_return_val_if_fail(pfile != NULL, -1);
	fp = pfile->fp;
	path = pfile->path;
	g_free(pfile);

	tmppath = g_strconcat(path, ".tmp", NULL);
	if (tmppath == NULL) {
		g_free(path);
		fclose(fp);
		return -1;
	}
	if (fclose(fp) == EOF) {
		FILE_OP_ERROR(tmppath, "fclose");
		unlink(tmppath);
		g_free(path);
		g_free(tmppath);
		return -1;
	}

	if (rename(tmppath, path) < 0) {
		FILE_OP_ERROR(tmppath, "rename");
		unlink(tmppath);
		g_free(path);
		g_free(tmppath);
		return -1;
	}

	g_free(path);
	g_free(tmppath);
	return 0;
}

/* Needed */
gint prefs_file_close_revert(PrefFile * pfile)
{
	gchar *tmppath;

	g_return_val_if_fail(pfile != NULL, -1);

	tmppath = g_strconcat(pfile->path, ".tmp", NULL);
	if (tmppath == NULL) {
		return -1;
	}
	fclose(pfile->fp);
	if (unlink(tmppath) < 0)
		FILE_OP_ERROR(tmppath, "unlink");
	g_free(tmppath);
	g_free(pfile->path);
	g_free(pfile);

	return 0;
}

/* Needed */

void prefs_free(PrefParam * param)
{
	gint i;

	g_return_if_fail(param != NULL);

	for (i = 0; param[i].name != NULL; i++) {
		if (!param[i].data)
			continue;

		switch (param[i].type) {
		case P_STRING:
			g_free(*((gchar **) param[i].data));
			break;
		default:
			break;
		}
	}
}
