#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <glib.h>
#include <hildon/hildon-notification.h>
#include <libnotify/notify.h>
#include <osso-log.h>

#include "notify.h"
#include "email_gconf.h"
#include "prefs_account.h"
#include "defs.h"
#include "procmsg.h"

/* list of notification ids */
static GSList *notif_list = NULL;

static void
notify_new_mail_arrival(gchar *from, gchar *subject, gchar *msgid);

static void
notify_remove(NotifyNotification *notif, gpointer data);

static void
notify_save_open_ids(void);

static void
notify_load_open_ids(void);

static void
notify_save_open_ids(void)
{
        osso_email_gconf_set_int_list(EMAIL_OPEN_NOTIFICATIONS, notif_list);
}

static void
notify_load_open_ids(void)
{
        GSList *list = NULL;

        g_slist_free(notif_list);
        osso_email_gconf_get_int_list(EMAIL_OPEN_NOTIFICATIONS, &list);
        notif_list = list;
}


/**
 * This function sends a system notification, it's called on
 * new mail arrival
 * @returns TRUE if sending notification was succesful,
 *          FALSE if sending failed or there'are too many open notifications
 */
static gboolean
notify_mail(gchar *from, gchar *subject, gchar *msgid)
{
        HildonNotification *notif = NULL;
        gboolean play_sound = FALSE;
        gint notif_id;
 
        ULOG_DEBUG_F("notify_new_mail_arrival called with msgid=%s", msgid);
        if (g_slist_length(notif_list) >= MAX_OPEN_NOTIFICATIONS) {
                ULOG_DEBUG_F("reached max number of open notifications");
                return FALSE;
        }
        notif = hildon_notification_new(from, subject,
                        NOTIFICATION_ICON,
                        NOTIFICATION_CATEGORY);
        hildon_notification_add_dbus_action(notif,
                        "default",
                        "Cancel",
                        "com.nokia.osso_email",
                        "/com/nokia/osso_email",
                        "com.nokia.osso_email",
                        "open_email",
                        G_TYPE_STRING, msgid,
                        -1);
        notify_notification_set_hint_int32(NOTIFY_NOTIFICATION(notif),
                        "dialog-type", 4);
        notify_notification_set_hint_string(NOTIFY_NOTIFICATION(notif), 
                        "led-pattern", 
                        "PatternCommunicationEmail");
        g_signal_connect(G_OBJECT(notif), "closed", 
                        G_CALLBACK(notify_remove), 
                        NULL);

        if (osso_email_gconf_get_bool(EMAIL_PLAY_SOUND_ON_NEW_MAIL, &play_sound) &&
                play_sound) {
                notify_notification_set_hint_string(NOTIFY_NOTIFICATION(notif),
                                "sound-file", NEW_MAIL_WAV_FILE);
        }
        if(!notify_notification_show(NOTIFY_NOTIFICATION(notif), NULL))
        {
                ULOG_ERR_F("Failed to send notification");
                return FALSE;
        }
        g_object_get(G_OBJECT(notif), "id", &notif_id, NULL);
        notif_list = g_slist_prepend(notif_list,
                        GINT_TO_POINTER(notif_id));
        notify_save_open_ids();
        return TRUE;
}

static void
notify_remove(NotifyNotification *notif, gpointer data) {
        gint notif_id;
        gboolean already_removed = FALSE;

        g_object_get(G_OBJECT(notif), "id", &notif_id, NULL);
        if (!notif_list) {
                already_removed = TRUE;
        } else {
                notif_list = g_slist_remove(notif_list, GINT_TO_POINTER(notif_id));
        }
        g_object_unref(notif);
        ULOG_DEBUG_F("notification %d removed, %d notifications remain",
            notif_id, g_slist_length(notif_list));
        if (!already_removed) {
                notify_save_open_ids();
        }
}

void
notify_remove_and_close_all(void) {
        notify_load_open_ids();
        while (notif_list) {
                if (notif_list->data) {
                        gint notif_id;
                        NotifyNotification *notif;

                        notif_id = GPOINTER_TO_INT(notif_list->data);
                        notif = notify_notification_new("dummy", NULL, NULL, NULL);
                        g_object_set(G_OBJECT(notif), "id", notif_id, NULL);
                        notify_notification_close(notif, NULL);
                        g_object_unref(notif);
                }
                notif_list = g_slist_delete_link(notif_list,
                                notif_list);
        }
        notify_save_open_ids();
}

/**
 * This function calls notify_mail for each message in msg_list
 * @param msg_list is a GSList of MsgInfo pointers
 */
void notify_mail_list(GSList *msg_list)
{
        GSList *link = msg_list;
        MsgInfo *msginf = NULL;

        if (!notif_list) {
                notify_load_open_ids();
        }
        while (link) {
                gboolean succ = FALSE;

                msginf = (MsgInfo *)link->data;
                ULOG_DEBUG_F("notify from %s subjec %s msgid %s",
                    msginf->header->from, msginf->header->subject,
                    msginf->header->msgid);
                succ = notify_mail(msginf->header->from,
                                msginf->header->subject,
                                msginf->header->msgid);
                if (!succ)
                        break;
                link = link->next;
        }
}

